/*
  Copyright 2005 Ronald S. Burkey <info@sandroid.org>
  
  This file is part of yaAGC.

  yaAGC is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  yaAGC is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with yaAGC; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  Filename:	DebuggerHookAGS.c
  Purpose:	This function is called by aea_engine() in order to 
  		implement the interactive debugger.
  Compiler:	GNU gcc.
  Contact:	Ron Burkey <info@sandroid.org>
  Reference:	http://www.ibiblio.org/apollo/yaAGS.html
  Mode:		2005-06-05 RSB	This is an improved version of the 
  				yaAGC debugger, in that it is built into
				aea_engine rather than into the main
				program.
		2005-06-09 RSB	Fixed a bunch of stuff related to i/o
				addresses.
		2005-06-11 RSB	Added the DISASSEMBLE command.
		2005-06-14 RSB	Added "EDIT PC".
		2005-06-18 RSB	Windows-dependent stuff added.
		2005-07-13 RSB	Fixed a possible issue of using too much CPU time
				in Win32.
*/

#include "yaAEA.h"
#include "aea_engine.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef WIN32
#include <windows.h>
#include <sys/time.h>
#define LB "\r\n"
#else
#include <time.h>
#include <sys/times.h>
#define LB ""
#endif

#ifdef WIN32
struct tms {
  clock_t tms_utime;  /* user time */
  clock_t tms_stime;  /* system time */
  clock_t tms_cutime; /* user time of children */
  clock_t tms_cstime; /* system time of children */
};
extern clock_t times (struct tms *p);
#define _SC_CLK_TCK (1000)
#define sysconf(x) (x)
#endif // WIN32

static int DebugStopCounter = -1;

#define MAX_BREAKPOINTS_AGS 64
typedef struct {
  // Type is
  //	0	for breakpoints.
  //	1	for memory watchpoints (change)
  //	2	for input-port watch (change)
  //	3	for output-port watch (change)
  //	4	for "patterns".
  int Type;
  int Address;
  int Value;
} Breakpoint_t;
static Breakpoint_t Breakpoints[MAX_BREAKPOINTS_AGS];
static int NumBreakpointsAGS = 0;
static char BreakCause[128];

static const char *Opcodes[32] = {
  "???", "???", "DVP", "MPY", 
  "STO", "STQ", "LDQ", "???", 
  "CLA", "ADD", "SUB", "MPR",
  "CLZ", "ADZ", "SUZ", "MPZ",
  "TRA", "TIX", "TOV", "TMI",
  "AXT", "LLS", "LRS", "ALS",
  "COM", "ABS", "INP", "OUT",
  "DLY", "TSQ", "???", "???"
};

// Buffer for keyboard input.
static char s[257];

// A function that converts i/o addresses like 2020 to constants like IO_2020.
// Returns -1 on error, or the constant's value otherwise.  We use the special
// (otherwise unused) i/o address for the fictitious discrete-output register
// we use internally in aea_engine.
static const int PowersOfTwo[8] = {
  01, 02, 04, 010, 020, 040, 0100, 0200
};
static int
TranslateIoAddress (int RawAddress)
{
  int Address, i, LowAddress;
  if (RawAddress == 0)
    return (IO_ODISCRETES);
  LowAddress = (RawAddress & 0377);
  Address = (RawAddress & 07400);
  if (Address == 02000)
    Address = IO_2001;
  else if (Address == 06000)
    Address = IO_6001;
  else
    return (-1);
  for (i = 0; i < 8; i++)
    if (PowersOfTwo[i] == LowAddress)
      return (Address + i);
  return (-1);
}

//-----------------------------------------------------------------------------------
// My substitute for fgets, for use when stdin is unblocked.

#define MAX_FROMFILES 11
static FILE *FromFiles[MAX_FROMFILES];
static int NumFromFiles = 1;

static void
rfgetsAGS (ags_t *State, char *Buffer, int MaxSize, FILE * fp)
{
  int c, Count = 0;
  char *s;
  //static int FirstTime = 1;
  MaxSize--;
  while (1)
    {
      // While waiting for character input, continue to look for client connects
      // and disconnects.
      while ((fp != stdin && EOF == (c = fgetc (fp))) ||
            (fp == stdin && Buffer != (s = nbfgets (Buffer, MaxSize))))
	{
	  // If we have redirected console input, and the file of source data is
	  // exhausted, then reattach the console.
	  if (NumFromFiles > 1 && fp == FromFiles[NumFromFiles - 1])
	    {
	      NumFromFiles--;
	      printf ("Keystroke source-file closed.\n");
	      if (NumFromFiles == 1)
		printf ("The keyboard has been reattached.\n> ");
	      fclose (fp);
	      fp = FromFiles[NumFromFiles - 1];
	    }
	  else
	    {
#ifdef WIN32	    
	      Sleep (10);
#else
	      struct timespec req, rem;
	      req.tv_sec = 0;
	      req.tv_nsec = 10000000;
	      nanosleep (&req, &rem);
#endif // WIN32
	    }  
	  //if (FirstTime)
	  //  {
	  //    FirstTime = 0;
	  //    printf ("Non-blocking getchar.\n");
	  //  }
          ChannelRoutineGeneric (State, UpdateAeaPeripheralConnect);
	}
      if (fp == stdin && s != NULL)
        return;
      Buffer[Count] = c;
      if (c == '\n' || Count >= MaxSize)
        {
	  Buffer[Count] = 0;
	  return;
	}
      Count++;
    }
}

//----------------------------------------------------------------------------
// Adds a breakpoint or a watchpoint.

void
AddBreakWatch (int Type, int Address, int Value)
{
  if (NumBreakpointsAGS >= MAX_BREAKPOINTS_AGS)
    {
      printf ("Max break/watch points (%d) defined already.\n", MAX_BREAKPOINTS_AGS);
      return;
    }
  Breakpoints[NumBreakpointsAGS].Type = Type;
  Breakpoints[NumBreakpointsAGS].Address = Address;
  Breakpoints[NumBreakpointsAGS].Value = Value;
  NumBreakpointsAGS++;    
}

// Deletes a breakpoint or watchpoint

void
DeleteBreakWatch (int Type, int Address)
{
  int i;
  for (i = 0; i < NumBreakpointsAGS; i++)
    if (Breakpoints[i].Type == Type && Breakpoints[i].Address == Address)
      {
        printf ("Breakpoint and/or watchpoint(s) deleted\n");
        for (NumBreakpointsAGS--; i < NumBreakpointsAGS; i++)
	  Breakpoints[i] = Breakpoints[i + 1];
	return;
      }
   //printf ("Breakpoint or watchpoint not found.\n");
}

// Dump memory or i/o space.

static int LastAddress = 0, LastN = 1, LastType = 1;

void
DumpAGS (ags_t *State)
{
  int CountOnLine = 0;
  int i;
  int Address;
  Address = LastAddress;
  for (i = 0; i < LastN; i++)
    {
      if (CountOnLine == 0)
        {
	  if (LastType == 2)
	    printf ("I");
	  else if (LastType == 3)
	    printf ("O");
	  printf ("0%04o:", Address);
	}
      if (LastType == 1)
        printf ("\t0%06o", State->Memory[Address++]);
      else if (LastType == 2)
        printf ("\t0%06o", State->InputPorts[Address++]);
      else if (LastType == 3)
        printf ("\t0%06o", State->OutputPorts[Address++]);
      CountOnLine++;
      if (CountOnLine >= 8 || 0 == (Address & 7))
        {
	  printf ("\n");
	  CountOnLine = 0;
	}
    }
  if (CountOnLine)
    printf ("\n");
}

//----------------------------------------------------------------------------
// Disassemble

void
DisassembleAGS (ags_t *State, int Address, int Count)
{
  int i, j, k;
  if (Count > 010000)
    Count = 010000;
  for (k = 0; k < Count; k++)
    {
      j = Address + k;
      if (j < 0 || j > 07777)
        continue;
      printf ("0%04o  0%06o\t", j, State->Memory[j]);
      i = ((State->Memory[j] >> 13) & 037);	// Opcode
      printf ("%s", Opcodes[i]);
      if (i != 031 && i != 030)		// If not ABS or COM, print operand.
	{
	  printf ("\t0%04o", State->Memory[j] & 07777);
	  // The next bit displays the index register as an increment to the address,
	  // as a courtesy.  It turns out that only opcodes from 000 to 036 can be
	  // indexed, and those from 040 to 076 cannot be.  Since i contains the 
	  // opcode divided by 2, the purpose of the condition "i < 020" should be
	  // obvious.
	  if (0 != (State->Memory[j] & 010000) && i < 020)
	    printf (",1\t[0%04o]", State->Index | (State->Memory[j] & 07777));
	}
      printf ("\n");
    }
}

//----------------------------------------------------------------------------

static unsigned long InstructionCounts[0100] = { 0 };

void
DebuggerHookAGS (ags_t *State)
{
  int i, j, k, Stop = 0;
  static int StopForNewInstructionType = 0, DisassembleCount = 24;
  struct tms TimeStruct;
  clock_t StoppedAt;
  
  if (!DebugModeAGS)		// If not in debugging mode, just return.
    return;
  // Collect instruction-type counts.
  i = ((State->Memory[State->ProgramCounter] >> 12) & 077);
  InstructionCounts[i]++;
  if (InstructionCounts[i] == 0)	// Overflow in the count?
    InstructionCounts[i]--;
  if (StopForNewInstructionType && InstructionCounts[i] == 1)
    {
      StopForNewInstructionType = 0;
      Stop = 1;
      strcpy (BreakCause, "previously-unused instruction type.");
    }  
  // Check the various possible conditions for halting execution.  
  if (DebugModeAGS == 2)	// Halt immediately on startup?
    {
      FromFiles[0] = stdin;
      Stop = 1;
      DebugModeAGS = 1;
      strcpy (BreakCause, "program loaded.");
    }
  if (DebugStopCounter > 0)	// Halt if spec'd # of instructions executed.
    DebugStopCounter--;
  if (!Stop)
    {
      if (RealTimeAGS >= NextKeycheckAGS)
	{
	  NextKeycheckAGS = RealTimeAGS + KEYSTROKE_CHECK_AGS;
	  while (s == nbfgets (s, sizeof (s)))
	    {
	      Stop = 1;
	      strcpy (BreakCause, "keypress.");
	    }
	}
    }
  if (!Stop && DebugStopCounter == 0)
    {
      DebugStopCounter = -1;
      Stop = 1;
      strcpy (BreakCause, "step count reached.");
    }
  for (i = 0; !Stop && i < NumBreakpointsAGS; i++)
    switch (Breakpoints[i].Type)
      {
      case 0:		// Breakpoint.
        Stop |= (State->ProgramCounter == Breakpoints[i].Address);
	strcpy (BreakCause, "breakpoint.");
	break;
      case 1:		// Memory watchpoint (change)
        Stop |= (State->Memory[Breakpoints[i].Address] != Breakpoints[i].Value);
	sprintf (BreakCause, "watched memory location 0%o changed from 0%o to 0%o.",
	         Breakpoints[i].Address, Breakpoints[i].Value, 
		 State->Memory[Breakpoints[i].Address]);
	Breakpoints[i].Value = State->Memory[Breakpoints[i].Address];
	break;
      case 2:		// Input-port watchpoint (change)
        Stop |= (State->InputPorts[Breakpoints[i].Address] != Breakpoints[i].Value);
	sprintf (BreakCause, "watched input port 0%o changed from 0%o to 0%o.",
	         Breakpoints[i].Address, Breakpoints[i].Value, 
		 State->InputPorts[Breakpoints[i].Address]);
	Breakpoints[i].Value = State->InputPorts[Breakpoints[i].Address];
	break;
      case 3:		// Output-port watchpoint (change)
        Stop |= (State->OutputPorts[Breakpoints[i].Address] != Breakpoints[i].Value);
	sprintf (BreakCause, "watched output port 0%o changed from 0%o to 0%o.",
	         Breakpoints[i].Address, Breakpoints[i].Value, 
		 State->OutputPorts[Breakpoints[i].Address]);
	Breakpoints[i].Value = State->OutputPorts[Breakpoints[i].Address];
	break;
      case 4:		// Pattern.
        // Note that Breakpoints[i].Value is the mask and Breakpoints[i].Address is the 
	// pattern we're looking for.
        Stop |= (0 == (Breakpoints[i].Value & (Breakpoints[i].Address ^ State->Memory[State->ProgramCounter])));
	break;
      }	
  // If none of the stop-conditions held, then return.
  if (!Stop)
    return;
    
Redraw:
  printf ("\n");
  printf ("Stopped because %s\n", BreakCause);
  // Print registers.
  printf ("A=0%06o\tQ=0%06o\tOverflow=%d\tIndex=%03o\tHalted=%d\nIcount=%lu\tCycles=%Lu\n",
	  State->Accumulator, State->Quotient, State->Overflow, State->Index,
	  State->Halt, 
	  InstructionCounts[077 & (State->Memory[State->ProgramCounter] >> 12)],
	  State->CycleCounter);
  // Print disassembly of the instruction.
  DisassembleAGS (State, State->ProgramCounter, 1);
  
  // Now do our interactive thing.
  while (1)
    {
      char *ss;
      printf ("> ");
      // Get input from the user or a macro-file.
      StoppedAt = times (&TimeStruct);
      rfgetsAGS (State, s, sizeof (s) - 1, stdin);
      RealTimeOffsetAGS += times (&TimeStruct) - StoppedAt;
      while (isspace (s[0]))	// Get rid of leading spaces.
        strcpy (&s[0], &s[1]); 
      for (ss = s; *ss; ss++)
        {
	  // Turn to upper case, get rid of multiple spaces and
	  // end-of-line linefeed.
	  while (isspace (*ss) && isspace (ss[1]))
	    strcpy (ss, ss + 1);
	  *ss = toupper (*ss);
	  if (*ss == '\n')
	    *ss = 0;
	}
      //printf ("\"%s\"\n", s);
      // Parse the input and do stuff with it.
      if (*s == '#')
        continue;
      else if (1 == sscanf (s, "STEP%o", &i) || 1 == sscanf (s, "NEXT%o", &i))
	{
	  if (i >= 1)
	    DebugStopCounter = i;
	  else
	    printf ("The step-count must be 1 or greater.\n");
	  break;
	}
      else if (!strcmp (s, "STEP") || !strcmp (s, "NEXT") ||
	       !strcmp (s, "S") || !strcmp (s, "N"))
	{
	  DebugStopCounter = 1;
	  break;
	}
      else if (!strcmp (s, "QUIT") || !strcmp (s, "EXIT"))
	exit (0);
      else if (1 == sscanf (s, "BREAK%o", &i))
        AddBreakWatch (0, i, 0);
      else if (1 == sscanf (s, "WATCH%o", &i))
        AddBreakWatch (1, i, State->Memory[i]);
      else if (1 == sscanf (s, "WATCH I%o", &i))
        {
	  j = TranslateIoAddress (i);
	  if (j == -1)
	    printf ("0%o is not a supported i/o address.\n", i);
	  else   
            AddBreakWatch (2, j, State->InputPorts[j]);
	}
      else if (1 == sscanf (s, "WATCH O%o", &i))
        {
	  j = TranslateIoAddress (i);
	  if (j == -1)
	    printf ("0%o is not a supported i/o address.\n", i);
	  else   
            AddBreakWatch (3, j, State->OutputPorts[j]);
	}
      else if (2 == sscanf (s, "PATTERN %o %o", &i, &j))
        AddBreakWatch (4, i, j);
      else if (!strcmp (s, "CONT"))
        break;
      else if (!strcmp (s, "CONT-TIL-NEW"))
        {
	  StopForNewInstructionType = 1;
          break;
	}
      else if (1 == sscanf (s, "DELETE%o", &i))
        {
          DeleteBreakWatch (0, i);
	  DeleteBreakWatch (1, i);
	}
      else if (1 == sscanf (s, "DELETE I%o", &i))
        DeleteBreakWatch (2, TranslateIoAddress (i));
      else if (1 == sscanf (s, "DELETE O%o", &i))
        DeleteBreakWatch (3, TranslateIoAddress (i));
      else if (!strcmp (s, "DELETE"))
        NumBreakpointsAGS = 0;
      else if (!strcmp (s, "BREAKPOINTS"))
        {
	  if (NumBreakpointsAGS == 0)
	    printf ("No breakpoints or watchpoints are defined.\n");
	  else
	    {
	      for (i = 0; i < NumBreakpointsAGS; i++)
	        {
		  //printf ("%2d: ", i);
		  switch (Breakpoints[i].Type)
		    {
		    case 0:
		      printf ("Breakpoint");
		      j = Breakpoints[i].Address;
		      printf (" at address 0%04o.\n", j);
		      break;
		    case 1:
		      printf ("Memory watchpoint");
		      j = Breakpoints[i].Address;
		      printf (" at address 0%04o.\n", j);
		      break;
		    case 2:
		      printf ("Input watchpoint");
		      j = Breakpoints[i].Address;
		      if (j >= IO_2001 && j <= IO_2200)
		        j = 02000 + (1 << (j - IO_2001));
		      else if (j >= IO_6001 && j <= IO_6200)
		        j = 06000 + (1 << (j - IO_6001));
		      else
		        j = -1;
		      printf (" at address 0%04o.\n", j);
		      break;
		    case 3:
		      printf ("Output watchpoint");
		      j = Breakpoints[i].Address;
		      if (j == IO_ODISCRETES)
		        j = 0;
		      else if (j >= IO_2001 && j <= IO_2200)
		        j = 02000 + (1 << (j - IO_2001));
		      else if (j >= IO_6001 && j <= IO_6200)
		        j = 06000 + (1 << (j - IO_6001));
		      else
		        j = -1;
		      printf (" at address 0%04o.\n", j);
		      break;
		    case 4:
		      printf ("Pattern 0%011o with mask 0%011o.\n",
		      	      Breakpoints[i].Address,
			      Breakpoints[i].Value);
		      break;
		    }
		}
	    }
	}
      else if (!strcmp (s, "DUMP"))
        DumpAGS (State);
      else if (2 == sscanf (s, "DUMP%o%o", &i, &j))
        {
	  LastType = 1;
	  LastN = i;
	  LastAddress = j;
	  DumpAGS (State);
	}
      else if (1 == sscanf (s, "DUMP%o", &j))
        {
	  LastType = 1;
	  LastN = 1;
	  LastAddress = j;
	  DumpAGS (State);
	}
      else if (1 == sscanf (s, "DUMP I%o", &j))
        {
	  i = TranslateIoAddress (j);
	  if (i == -1)
	    printf ("0%o is not a supported i/o address.\n", j);
	  else
	    {
	      LastType = 2;
	      LastN = 1;
	      LastAddress = i;
	      DumpAGS (State);
	    }
	}
      else if (1 == sscanf (s, "DUMP O%o", &j))
        {
	  i = TranslateIoAddress (j);
	  if (i == -1)
	    printf ("0%o is not a supported i/o address.\n", j);
	  else
	    {
	      LastType = 3;
	      LastN = 1;
	      LastAddress = i;
	      DumpAGS (State);
	    }
	}
      else if (2 == sscanf (s, "EDIT%o%o", &i, &j))
        {
	  if (i < 0 || i > 07777)
	    printf ("Address o%o out of range.\n", i);
	  else
	    State->Memory[i] = (j & 0777777);
	}
      else if (2 == sscanf (s, "EDIT I%o %o", &i, &j))
        {
	  k = TranslateIoAddress (i);
	  if (k == -1)
	    printf ("I/O address o%o is not supported.\n", i);
	  else
	    State->InputPorts[k] = (j & 0777777);
	}
      else if (2 == sscanf (s, "EDIT O%o %o", &i, &j))
        {
	  extern void Output (ags_t *State, int AddressField, int Value);
	  if (i == 0)
	    {
	      State->OutputPorts[IO_ODISCRETES] = (j & 0777777);
              ChannelOutputAGS (040, j & 0777777);
	    }
	  else
	    Output (State, i, j);
	}
      else if (1 == sscanf (s, "EDIT A %o", &j))
        {
	  if (j < 0 || j > 0777777)
	    printf ("Value out of range.\n");
	  else 
	    State->Accumulator = j;
	    goto Redraw;
	}
      else if (1 == sscanf (s, "EDIT Q %o", &j))
        {
	  if (j < 0 || j > 0777777)
	    printf ("Value out of range.\n");
	  else 
	    State->Quotient = j;
	    goto Redraw;
	}
      else if (1 == sscanf (s, "EDIT OVERFLOW %o", &j))
        {
	  if (j < 0 || j > 1)
	    printf ("Value out of range.\n");
	  else 
	    State->Overflow = j;
	    goto Redraw;
	}
      else if (1 == sscanf (s, "EDIT INDEX %o", &j))
        {
	  if (j < 0 || j > 7)
	    printf ("Value out of range.\n");
	  else 
	    State->Index = j;
	    goto Redraw;
	}
      else if (1 == sscanf (s, "EDIT HALT %o", &j))
        {
	  if (j < 0 || j > 1)
	    printf ("Value out of range.\n");
	  else 
	    State->Halt = j;
	    goto Redraw;
	}
      else if (1 == sscanf (s, "EDIT PC %o", &j))
        {
	  if (j < 0 || j > 07777)
	    printf ("Value out of range.\n");
	  else 
	    State->ProgramCounter = j;
	    goto Redraw;
	}
      else if (!strcmp (s, "BACKTRACES"))
	ListBacktracesAGS ();
      else if (1 == sscanf (s, "BACKTRACE%d", &i))
        {
	  RegressToBacktraceAGS (State, i);
	  goto Redraw;
	}
      else if (2 == sscanf (s, "DISASSEMBLE%o%o", &i, &j))
        {
	  DisassembleCount = i;
	  DisassembleAGS (State, j, DisassembleCount);
	}
      else if (1 == sscanf (s, "DISASSEMBLE%o", &j))
        DisassembleAGS (State, j, DisassembleCount);
      else if (!strcmp (s, "DISASSEMBLE"))
        DisassembleAGS (State, State->ProgramCounter, DisassembleCount);	
      else if (!strcmp (s, "COUNTS"))
        {
	  for (i = 0; i < 32; i++)
	    printf ("%02o:\t%s\t%lu\t\t%02o:\t%s,1\t%lu\n",
	              2 * i, Opcodes[i], InstructionCounts[2 * i], 
		      2 * i + 1, Opcodes[i], InstructionCounts[2 * i + 1]);
	}
      else
        printf ("Unrecognized command \"%s\".\n", s);
  }
}

