# Copyright 2005 Stephan Hotto <stephan.hotto@web.de>
#
# This file is part of yaAGC.
#
# yaAGC is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# yaAGC is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with yaAGC; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# ***********************************************************************************************
# **** Program:   (yaAGC by Ron Burkey) LM System Simulator                                  ****
# **** Function:  Traces the yaAGC Output Channels and sets Input Data                       ****
# **** Language:  TCL/TK                                                                     ****
# **** Invoke:    wish lm_system_simulator                                                   ****
# ****            (As a pre-condition the TCL/TK package must be installed)                  ****
# **** Date:      28.05.2005                                                                 ****
# **** Version:   v0.11                                                                      ****
# **** Author:    Stephan Hotto                                                              ****
# ***********************************************************************************************


# --------------------------- Global Variables --------------------------------------------------
set font1         -adobe-courier-medium-r-normal--12-140-75-75-m-90-iso8859-1
set font2         -adobe-courier-medium-r-normal--12-120-75-75-m-70-iso8859-1
set font3         -adobe-courier-bold-b-normal--12-140-75-75-m-90-iso8859-1
set serverIP      "localhost"
set serverSock    19801
set sockChan      ""
set log_out_flag  "0"
set log_in_flag   "1"

set path          "./"
set docpath       "${path}doc/"
set modulpath     "${path}modules/"
set source1       "${modulpath}AGC_Outputs.tcl"
set source2       "${modulpath}AGC_System_Inputs.tcl"
set source3       "${modulpath}AGC_Crew_Inputs.tcl"
set source4       "${modulpath}AGC_IMU.tcl"

# ---------------- Unprogrammed Sequences ----------------------------
# ---- IN ----
set unpSeq(PINC)  000000000000000
set unpSeq(PCDU)  000000000000001
set unpSeq(MINC)  000000000000010
set unpSeq(MCDU)  000000000000011

# ---- OUT ----
set unpSeq(000000000001101) POUT
set unpSeq(000000000001110) MOUT
set unpSeq(000000000001111) ZOUT

# ---------------- IMU Simulation Initialization ---------------------
# ---- 39.55078125'' of arc ----
set ANGLE_INCR        39.55078125/3600.0
set IMUX_ANGLE        0
set CDUX_INC_DEC_FLAG "INC"
set CDUX_COUNT_FLAG   "SINGLE"
set IMUY_ANGLE        0
set CDUY_INC_DEC_FLAG "INC"
set CDUY_COUNT_FLAG   "SINGLE"
set IMUZ_ANGLE        0
set CDUZ_INC_DEC_FLAG "INC"
set CDUZ_COUNT_FLAG   "SINGLE"

# **********************************************************************************************
# **** Function: Build the User Interface                                                   ****
# **********************************************************************************************
proc create_gui {} {
 global font1 font2 font3 log_in_flag log_out_flag cdata wdata bmask
 
 wm title    . "LM System Simulator for yaAGC (LUMINARY 131/1C)"
 wm geometry . +50+50
 wm geometry . 450x500
 wm iconname . "LM Simulator"
 wm minsize  . 450 450
 
 create_menu

 frame .fp1; frame .fp2
 
 frame .fm1; frame .fm2; frame .fm3
 
 frame .f0; frame .f01; frame .f1; frame .f2; frame .f3; frame .f4; frame .f5 ;frame .f6; frame .f7; frame .f8
 frame .f81; frame .f9; frame .f10; frame .f101; frame .f11
 
 frame .f0r; frame .f01r; frame .f1r; frame .f2r; frame .f3r; frame .f4r; frame .f5r; frame .f6r; frame .f7r
 frame .f8r; frame .f81r; frame .f9r; frame .f10r; frame .f101r
 
 pack .fp1 .fp2 -side top -anchor w -pady 1 -padx 2
 
 pack .fm1 .fm2 -in .fp1 -side left -anchor n -pady 1 -padx 10
 pack .fm3 -in .fp2 -side top -anchor w -pady 0 -padx 1
 
 pack .f0 .f01 .f1 .f2 .f3 .f4 .f5 .f6 .f7 .f8 .f81 .f9 .f10 -in .fm1 -side top -anchor w -pady 1 -padx 1
 pack .f0r .f01r .f1r .f2r .f3r .f4r .f5r .f6r .f7r .f8r .f81r .f9r .f10r .f101r -in .fm2 -side top -anchor w -pady 1 -padx 1
 pack .f101 .f11 -in .fm3 -side top -anchor w -pady 2 -padx 2

 # ------------------------------- Output --------------------------------------
 label .l0 -text "Output Channels:" -font ${font3}
 pack .l0 -side left -in .f0
 label .l01 -text "    MSB <-----> LSB" -font ${font1}
 pack .l01 -side left -in .f01
 
 label .l1 -text " 5:" -font ${font1}
 entry .e1 -width 15 -relief groove -bd 2 -textvariable cdata(5) -font ${font1}
 pack .l1 .e1 -side left -in .f1
 
 label .l2 -text " 6:" -font ${font1}
 entry .e2 -width 15 -relief groove -bd 2 -textvariable cdata(6) -font ${font1}
 pack .l2 .e2 -side left -in .f2
 
 label .l3 -text " 7:" -font ${font1}
 entry .e3 -width 15 -relief groove -bd 2 -textvariable cdata(7) -font ${font1}
 pack .l3 .e3 -side left -in .f3
 
 label .l4 -text "10:" -font ${font1}
 entry .e4 -width 15 -relief groove -bd 2 -textvariable cdata(10) -font ${font1}
 pack .l4 .e4 -side left -in .f4
 
 label .l5 -text "11:" -font ${font1}
 entry .e5 -width 15 -relief groove -bd 2 -textvariable cdata(11) -font ${font1}
 pack .l5 .e5 -side left -in .f5 
 
 label .l6 -text "12:" -font ${font1}
 entry .e6 -width 15 -relief groove -bd 2 -textvariable cdata(12) -font ${font1}
 pack .l6 .e6 -side left -in .f6
 
 label .l7 -text "13:" -font ${font1}
 entry .e7 -width 15 -relief groove -bd 2 -textvariable cdata(13) -font ${font1}
 pack .l7 .e7 -side left -in .f7
 
 label .l8 -text "14:" -font ${font1}
 entry .e8 -width 15 -relief groove -bd 2 -textvariable cdata(14) -font ${font1}
 pack .l8 .e8 -side left -in .f8
 
 label .l81 -text "33:" -font ${font1}
 entry .e81 -width 15 -relief groove -bd 2 -textvariable cdata(33) -font ${font1}
 pack .l81 .e81 -side left -in .f81
 
 label .l9 -text "34:" -font ${font1}
 entry .e9 -width 15 -relief groove -bd 2 -textvariable cdata(34) -font ${font1}
 pack .l9 .e9 -side left -in .f9
 
 label .l10 -text "35:" -font ${font1}
 entry .e10 -width 15 -relief groove -bd 2 -textvariable cdata(35) -font ${font1}
 pack .l10 .e10 -side left -in .f10
 
 checkbutton .cb101 -text "Log OUT - ON/OFF " -variable log_out_flag -relief raised -borderwidth 2 -anchor w -font ${font1} -pady 4
 checkbutton .cb102 -text "Log IN - ON/OFF " -variable log_in_flag -relief raised -borderwidth 2 -anchor w -font ${font1} -pady 4
 button .b101 -text "Clear Log" -command {.text delete 0.0 end} -font ${font1} -pady 3
 pack  .cb101 .cb102 .b101 -side left -in .f101 -padx 2
 
 text .text -relief sunken -bd 2 -yscrollcommand ".scrolly set" -xscrollcommand ".scrollx set" -bg white -wrap none -font ${font1}
 scrollbar .scrollx -command ".text xview" -orient horizontal
 scrollbar .scrolly -command ".text yview"
 
 pack .scrolly -in .f11 -side right -fill y
 pack .scrollx -in .f11 -side bottom -fill x
 pack .text -in .f11 -side top -fill x -fill y -ipady 400
 
 # --------------------------------- Input -----------------------------------------
 label .l0r -text "Input Channels:" -font ${font3}
 pack .l0r -side left -in .f0r
 label .l01r -text "    MSB <-----> LSB     Bitmask" -font ${font1}
 pack .l01r -side left -in .f01r
 
 label .l1r -text " 3:" -font ${font1}
 entry .e1r -width 15 -relief sunken -bd 2 -textvariable wdata(3) -bg white -font ${font1}
 bind .e1r <Return> {write_socket w 3}
 entry .e1r1 -width 15 -relief sunken -bd 2 -textvariable bmask(3) -bg white -font ${font1}
 # pack .l1r .e1r .e1r1 -side left -in .f1r
 
 label .l2r -text " 4:" -font ${font1}
 entry .e2r -width 15 -relief sunken -bd 2 -textvariable wdata(4) -bg white -font ${font1}
 bind .e2r <Return> {write_socket w 4}
 entry .e2r1 -width 15 -relief sunken -bd 2 -textvariable bmask(4) -bg white -font ${font1}
 # pack .l2r .e2r .e2r1 -side left -in .f2r 
 
 label .l3r -text "15:" -font ${font1}
 entry .e3r -width 15 -relief sunken -bd 2 -textvariable wdata(15) -bg white -font ${font1}
 bind .e3r <Return> {write_socket w 15}
 entry .e3r1 -width 15 -relief sunken -bd 2 -textvariable bmask(15) -bg white -font ${font1}
 pack .l3r .e3r .e3r1 -side left -in .f3r
 
 label .l4r -text "16:" -font ${font1}
 entry .e4r -width 15 -relief sunken -bd 2 -textvariable wdata(16) -bg white -font ${font1}
 bind .e4r <Return> {write_socket w 16}
 entry .e4r1 -width 15 -relief sunken -bd 2 -textvariable bmask(16) -bg white -font ${font1}
 pack .l4r .e4r .e4r1 -side left -in .f4r
 
 label .l5r -text "30:" -font ${font1}
 entry .e5r -width 15 -relief sunken -bd 2 -textvariable wdata(30) -bg white -font ${font1}
 bind .e5r <Return> {write_socket w 30}
 entry .e5r1 -width 15 -relief sunken -bd 2 -textvariable bmask(30) -bg white -font ${font1}
 pack .l5r .e5r .e5r1 -side left -in .f5r
 
 label .l6r -text "31:" -font ${font1}
 entry .e6r -width 15 -relief sunken -bd 2 -textvariable wdata(31) -bg white -font ${font1}
 bind .e6r <Return> {write_socket w 31}
 entry .e6r1 -width 15 -relief sunken -bd 2 -textvariable bmask(31) -bg white -font ${font1}
 pack .l6r .e6r .e6r1 -side left -in .f6r
 
 label .l7r -text "32:" -font ${font1}
 entry .e7r -width 15 -relief sunken -bd 2 -textvariable wdata(32) -bg white -font ${font1}
 bind .e7r <Return> {write_socket w 32}
 entry .e7r1 -width 15 -relief sunken -bd 2 -textvariable bmask(32) -bg white -font ${font1}
 pack .l7r .e7r .e7r1 -side left -in .f7r
 
 label .l8r -text "33:" -font ${font1}
 entry .e8r -width 15 -relief sunken -bd 2 -textvariable wdata(33) -bg white -font ${font1}
 bind .e8r <Return> {write_socket w 33}
 entry .e8r1 -width 15 -relief sunken -bd 2 -textvariable bmask(33) -bg white -font ${font1}
 pack .l8r .e8r .e8r1 -side left -in .f8r
 
 focus .e1r
 update
 
}

# **********************************************************************************************
# **** Function: Build the Menu                                                             ****
# **********************************************************************************************
proc create_menu {} {
 global font2

 frame .men -relief raised -bd 2

 pack .men -side top -fill x

 # ----------------------------------------- Edit ----------------------------------------------
 menubutton .men.edit -text "Edit" -menu .men.edit.m -font $font2
 pack .men.edit -side left -padx 2m

 menu .men.edit.m
 .men.edit.m add command -label "Clear Message Window"  -command {.text delete 0.0 end} -font $font2
 .men.edit.m add separator
 .men.edit.m add command -label "Exit                " -underline 0 -accelerator Ctrl+x -command {exit_program} -font $font2

 # ------------------------------------- Systems Menu --------------------------------------------
 menubutton .men.ctr -text "Systems" -menu .men.ctr.m -font $font2
 pack .men.ctr -side left -padx 2m

 menu .men.ctr.m
 .men.ctr.m add command -label "AGC Outputs" -command {create_lmsys} -font $font2
 .men.ctr.m add separator
 .men.ctr.m add command -label "AGC LM System Inputs" -command {create_sysinp} -font $font2
 .men.ctr.m add command -label "AGC Crew Inputs" -command {create_crewinp} -font $font2
 .men.ctr.m add separator
 .men.ctr.m add command -label "IMU (Inertial Measurement Unit)" -command {create_imugui} -font $font2
 
 # ------------------------------------- Info Menu -----------------------------------------------
 menubutton .men.inf -text "Info" -menu .men.inf.m -font $font2
 pack .men.inf -side left -padx 2m

 menu .men.inf.m
 .men.inf.m add command -label "About" -command {open_text "about.txt"} -font $font2
 .men.inf.m add command -label "Help" -command {open_text "help.txt"} -font $font2
 .men.inf.m add separator
 .men.inf.m add command -label "License" -command {open_text "license.txt"} -font $font2

}


# *********************************************************************************************
# **** Function: Open Text Files (About, Help, Licence)                                    ****
# *********************************************************************************************
proc open_text {file} {
 global docpath font2

 if {[winfo exists .op] == 1} {destroy .op}

 if {$file == "about.txt"} {
    set wmtxt "ABOUT LM System Simulator"
 } elseif {$file == "help.txt"} {
    set wmtxt "HELP of LM System Simulator"
 } elseif {$file == "license.txt"} {
    set wmtxt "GPL LICENSE of the LM System Simulator"
 } 

 toplevel .op
 wm title .op $wmtxt
 wm geometry .op +300+300
 wm geometry .op 600x400
 wm iconname .op "ABOUT/HELP"

 frame .op.f1

 pack .op.f1 -side top -pady 1m

 text .op.text -relief sunken -bd 2 -yscrollcommand ".op.scrolly set" -xscrollcommand ".op.scrollx set" -bg white -font $font2 -wrap none

 scrollbar .op.scrollx -command ".op.text xview" -orient horizontal
 scrollbar .op.scrolly -command ".op.text yview"

 pack .op.scrolly -in .op.f1 -side right -fill y
 pack .op.scrollx -in .op.f1 -side bottom -fill x
 pack .op.text -in .op.f1 -side top -fill x -fill y -ipadx 200 -ipady 200

 set datei [open ${docpath}${file}]
 while {[gets $datei zeile] >= 0} {
       .op.text insert end "$zeile\n"
 }
 close $datei

return
}


# **********************************************************************************************
# **** Function: Do Initialization                                                          ****
# **********************************************************************************************
proc init {} {
 global wdata bmask b

 # ------------------ Set Initial Data ----------------------------
 set wdata(15)  "000000000000000"; set wdata(16)  "000000000000000"
 set wdata(30)  "011110101001001"; set wdata(31)  "100111111111111"
 set wdata(32)  "001111111111111"; set wdata(33)  "101111011111000"

 # ------------------ Set Initial Input Bit Masks -----------------
 set bmask(15) "000000000000000"; set bmask(16) "000000001111100"
 set bmask(30) "111111111111111"; set bmask(31) "111111111111111"
 set bmask(32) "001111111111111"; set bmask(33) "111111111111111"
 
 foreach i {15 16 30 31 32 33} {
    if {$i != 3 && $i != 4} {write_socket w $i}
    for {set j 0} {$j <= 15} {incr j 1} {
        set b($i,[expr 15-$j]) [string index $wdata($i) $j]   
    }
 }

}


# *********************************************************************************************
# **** Function: Open Client Socket Connection                                             ****
# *********************************************************************************************
proc open_socket {} {
 global serverIP serverSock sockChan
 
 set err [catch {set sockChan [socket $serverIP $serverSock]}]
 if {$err != 0} {ShowMessageBox "error" "Can't connect to yaAGC!"; exit_program}
 fconfigure ${sockChan} -blocking 0 -buffering none -buffersize 0 -encoding binary
 
}


# *********************************************************************************************
# **** Function: Read 4 Bytes (Sync 00) from Socket (Main Loop)                            ****
# *********************************************************************************************
proc read_socket {} {
 global sockChan ChanData
 set sync ""
 set res2 ""
 
 set i 5
 while {1} {
   set res [read $sockChan 1]
   binary scan $res B2 sync
   if {$sync == "00"} {set i 0}
   binary scan $res B8 res2
   set ChanData($i) $res2
   incr i 1
   if {$i == 4} {update_simulation; process_data}
 } 
 
}


# *********************************************************************************************
# **** Function: Convert Binary (7 Bits) to Octal                                          ****
# *********************************************************************************************
proc convert_bin_oct {par1} {

   set dec [string index $par1 6]
   set dec [expr $dec + [string index $par1 5] * 2]
   set dec [expr $dec + [string index $par1 4] * 4]
   set dec [expr $dec + [string index $par1 3] * 8]
   set dec [expr $dec + [string index $par1 2] * 16]
   set dec [expr $dec + [string index $par1 1] * 32]
   set dec [expr $dec + [string index $par1 0] * 64]

   return [format %o $dec]
}


# *********************************************************************************************
# **** Function: Convert Octal to Binary (7 Bits)                                          ****
# *********************************************************************************************
proc convert_oct_bin {par1} {

   set res [format %d 0$par1]
   set res2 [format %c $res]
   binary scan $res2 B8 res
 
   return [string range $res 1 7]
}


# *********************************************************************************************
# **** Function: Process AGC Output (Channel Data and Unprogrammed Sequences)              ****
# *********************************************************************************************
proc process_data {} {
 global ChanData cdata log_out_flag bo unpSeq
 
 set data "$ChanData(0)$ChanData(1)$ChanData(2)$ChanData(3)"
 set t [string index $data 3]
 set channel [string range $data 4 7][string range $data  10 12]
 set octChannel [convert_bin_oct $channel]
 
 if {$t == 0} {
   set cdata($octChannel) [string range $data 13 15][string range $data 18 23][string range $data 26 31]
 
   for {set j 0} {$j <= 15} {incr j 1} {
     set bo($octChannel,[expr 15-$j]) [string index $cdata($octChannel) $j]   
   }
 
   if {$log_out_flag == 1} {
     .text insert end "Out: [format %2d $octChannel]  $channel  $cdata($octChannel)\n"
     .text yview moveto 1
   }
 } else {
     set temp [string range $data 13 15][string range $data 18 23][string range $data 26 31]
     .text insert end "Out: Unprogrammed Sequence:  $unpSeq($temp)  Counter: [format %2d $octChannel]  $channel  $temp\n"
 }
 
 
}


# *********************************************************************************************
# **** Function: Updating Simulation (every read of 4 bytes)                               ****
# *********************************************************************************************
proc update_simulation {} {
  
  update
  imu_count "LOOP"

}


# *********************************************************************************************
# **** Function: Write to Socket (Bitmask; Input Channel and Unprogrammed Sequences)       ****
# *********************************************************************************************
proc write_socket {par0 par1} {
 global sockChan wdata bmask b unpSeq log_in_flag
 
 if {$par0 == "b"} {
   set wdata($par1) ""
   for {set j 15} {$j >= 0} {incr j -1} {
        set wdata($par1) $wdata($par1)$b($par1,$j)   
   }
 }
 
 if {$par0 == "w"} {
   for {set j 0} {$j <= 15} {incr j 1} {
      set b($par1,[expr 15-$j]) [string index $wdata($par1) $j]   
   }
 }
 
 set channel [convert_oct_bin $par1]
 
 if {$par0 == "b" || $par0 == "w"} {
   if {$log_in_flag == 1} {
     .text insert end "In:  [format %2d $par1]  $channel  $wdata($par1)  $bmask($par1)\n"
     .text yview moveto 1
   }
   set cd1 "0010[string range $channel 0 3]"
   set cd2 "01[string range $channel 4 6][string range $bmask($par1) 0 2]"
   set cd3 "10[string range $bmask($par1) 3 8]"
   set cd4 "11[string range $bmask($par1) 9 14]"
 
   puts -nonewline $sockChan [binary format B8 $cd1]
   puts -nonewline $sockChan [binary format B8 $cd2]
   puts -nonewline $sockChan [binary format B8 $cd3]
   puts -nonewline $sockChan [binary format B8 $cd4]
 
   set cd1 "0000[string range $channel 0 3]"
   set cd2 "01[string range $channel 4 6][string range $wdata($par1) 0 2]"
   set cd3 "10[string range $wdata($par1) 3 8]"
   set cd4 "11[string range $wdata($par1) 9 14]"
 } else {
   if {$log_in_flag == 1} {
     .text insert end "IN:  Unprogrammed Sequence:  $par0  Counter: [format %2d $par1]  $channel  $unpSeq($par0)\n"
     .text yview moveto 1
   }
   
   set cd1 "0001[string range $channel 0 3]"
   set cd2 "01[string range $channel 4 6][string range $unpSeq($par0) 0 2]"
   set cd3 "10[string range $unpSeq($par0) 3 8]"
   set cd4 "11[string range $unpSeq($par0) 9 14]"
 } 
 
 puts -nonewline $sockChan [binary format B8 $cd1]
 puts -nonewline $sockChan [binary format B8 $cd2]
 puts -nonewline $sockChan [binary format B8 $cd3]
 puts -nonewline $sockChan [binary format B8 $cd4]
 
 flush $sockChan
 
}


# *********************************************************************************************
# **** Function: Show Message Box                                                          ****
# *********************************************************************************************
proc ShowMessageBox {par1 par2} {
 global font1
  
  set reply [tk_messageBox -default ok -icon ${par1} -message ${par2} -parent . -title "LM System Simulator" -type ok]	
  
  return
}


# *********************************************************************************************
# **** Funktion: Exit Program                                                              ****
# *********************************************************************************************
proc exit_program {} {

 exit 0

return
}


# *********************************************************************************************
# **** Function: Read Sources                                                              ****
# *********************************************************************************************

# ----------------------------- Source: AGC Outputs -------------------------------------------
source $source1

# ----------------------------- Source: AGC System Inputs -------------------------------------
source $source2

# ----------------------------- Source: AGC Crew Inputs ---------------------------------------
source $source3

# ----------------------------- Source: AGC IMU -----------------------------------------------
source $source4

# *********************************************************************************************
# **** Function: Main                                                                      ****
# *********************************************************************************************

# ----------------- Create the GUI and wait for Events ----------------------------------------
create_gui
open_socket
init
read_socket

