/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**
** NAME
**       flx_filebox
**
** PURPOSE
**       File selection dialog
**
** FUNCTION PROTOTYPE     
**       const char *flx_filebox( const char *directory,
**                                const char *pattern, 
**                                const char *default);
**
** INPUTS
**       directory: starting directory
**       pattern:   file filter
**       default:   optional default starting filename (or use "")
**
** OUTPUT
**       Selected filename, or "" if cancelled
**
** RESTRICTIONS
**       Selections are not cached
**       Only one file selector may be visible at any given time
**              
** MODIFICATION HISTORY
**       Written, Robert.Mallozzi@msfc.nasa.gov, September 1997
**
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**
** flx: Extensions for the xforms library
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the Free
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#include "forms.h"
#include "flx.h" 
#include "flx_filebox.h"

#define USE_COLOR 1



 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *flx_filebox(const char *directory, 
                  const char *pattern, 
                  const char *default_file) 
{                  
 
    FORM_S *theForm;    
     
    /* Do I need a static buffer to build the return filename ??? */
    static char return_file[FILENAME_MAX];
        
    const char *help_text = 

	  "The dialog shows a list of directories in the left browser,\n"
	  "and a list of files in the right browser.  Double click on a\n"
	  "file name from the list on the right, or select it and click Ok.\n"
	  "Single clicking on a directory name will display the files\n"
	  "in that directory.  To move up in the directory tree, single\n"
	  "click on the directory \"..\".  You can also type directory and\n"
	  "file filter specifications in the fields labeled \"Location\" and\n"
	  "\"Filter\", or you can type a filename into the \"File\" field.\n\n"

	  "File types are displayed in the following colors:\n\n"
	  "@C0\tfile\n" 
	  "@C0\tdirectory\n" 
	  "@C6\tlink\n" 
	  "@C3\tcharacter device\n"
	  "@C4\tblock device\n" 
	  "@C5\tsocket\n"
	  "@C2\tfifo\n\n"

	  "If you use the filter field, note that the \"Show Hidden\"\n"
	  "option takes precedence over the file filter.  Thus if hidden\n"
	  "files are not currently being shown, adding the filter \".*\"\n"
	  "will not show the hidden files.\n";

    
    /* Build the main form */   
    theForm = build_filebox (directory, pattern, default_file); 

    /* Build the help form */   
    theForm->help = flx_help_dialog (help_text); 

    /* 
    ** Set up modal interaction for the file selector. 
    ** This will break user-code if some forms are inactive when 
    ** flx_filebox() is called, since all forms will be re-activated
    ** on exit.
    */
    fl_deactivate_all_forms();
    fl_activate_form (theForm->main);

    /* Realize the main form */
    fl_show_form (theForm->main, FL_PLACE_MOUSE, FL_FULLBORDER, "flx_filebox");

    while (! have_file)
          fl_check_forms();
    
    /* Reset for next invocation of flx_filebox */
    have_file = 0;

    if (fl_form_is_visible (theForm->help))
       fl_hide_form (theForm->help); 
    fl_hide_form (theForm->main); 

    fl_activate_all_forms();
    
/*              
    return ((char *) fl_get_input (theForm->file_input));    
*/

    if (strcmp(fl_get_input (theForm->file_input), "") == 0)
       sprintf(return_file, "%s", fl_get_input (theForm->file_input));
    else
       sprintf(return_file, "%s/%s", theForm->dir,
           fl_get_input (theForm->file_input));


    /* Free allocated memory */
    free (theForm->dir);
    free (theForm->filter);


    return ((char *) return_file);    

}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
FORM_S *build_filebox(const char *directory,
                      const char *pattern, 
                      const char *default_file) 
{
    FL_OBJECT *obj;
    FORM_S *theForm = (FORM_S *) fl_calloc(1, sizeof(*theForm));  
    char *cwd;      

    /* 
    ** Store current directory and file filter 
    */
    if ((chdir(directory)) == -1) {
       perror("Could not change dir");
       return(0);
    }
    cwd = getcwd(NULL, FILENAME_MAX);       
    theForm->dir = (char *) fl_calloc (1, strlen(cwd) + 1);
    sprintf(theForm->dir, "%s", cwd);
    free (cwd);     
    
    theForm->filter = (char *) fl_calloc (1, strlen(pattern) + 1);
    sprintf(theForm->filter, "%s", pattern);
     

     
    theForm->main = fl_bgn_form(FL_NO_BOX, 440, 390);
  
        obj = fl_add_box(FL_UP_BOX, 0, 0, 440, 390, "");
        

        /* Directories section */
         
        obj = fl_add_text(FL_NORMAL_TEXT, 
            20, 10, 130, 20, "Directories");
        fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);

        theForm->dot_dirs = fl_add_round3dbutton(FL_PUSH_BUTTON, 
            105, 10, 110, 20, "Show Hidden");
        fl_set_object_boxtype(theForm->dot_dirs, FL_NO_BOX);
        fl_set_object_color(theForm->dot_dirs, FL_COL1, FL_BLUE);
        fl_set_object_lstyle(theForm->dot_dirs, FL_BOLD_STYLE);
        fl_set_object_lsize(theForm->dot_dirs, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->dot_dirs, DotfileCB, 0);

        theForm->dot_dirs->u_vdata = (void *) theForm; 


        obj = fl_add_box(FL_NO_BOX, 
            20, 25, 190, 10, "@DnLine");  
 
        theForm->dir_browser = fl_add_browser(FL_HOLD_BROWSER, 
            20, 40, 190, 165, "");
        fl_set_browser_hscrollbar(theForm->dir_browser, FL_ON);
        fl_set_browser_vscrollbar(theForm->dir_browser, FL_ON);
        fl_set_browser_scrollbarsize(theForm->dir_browser, 15, 15);
        fl_set_browser_fontstyle(theForm->dir_browser, FL_BOLD_STYLE);
        fl_set_browser_fontsize(theForm->dir_browser, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->dir_browser, DirCB, 0);
        
        theForm->dir_browser->u_vdata = (void *) theForm; 


        /* Files section */
         
        obj = fl_add_text(FL_NORMAL_TEXT, 
            230, 10, 50, 20, "Files");
        fl_set_object_lalign(obj, FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);

        theForm->dot_files = fl_add_round3dbutton(FL_PUSH_BUTTON, 
            315, 10, 110, 20, "Show Hidden");
        fl_set_object_boxtype(theForm->dot_files, FL_NO_BOX);
        fl_set_object_color(theForm->dot_files, FL_COL1, FL_BLUE);
        fl_set_object_lstyle(theForm->dot_files, FL_BOLD_STYLE);
        fl_set_object_lsize(theForm->dot_files, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->dot_files, DotfileCB, 0);

        theForm->dot_files->u_vdata = (void *) theForm; 

        obj = fl_add_box(FL_NO_BOX, 
            230, 25, 190, 10, "@DnLine");  

        theForm->file_browser = fl_add_browser(FL_HOLD_BROWSER, 
            230, 40, 190, 165, "");
        fl_set_browser_hscrollbar(theForm->file_browser, FL_ON);
        fl_set_browser_vscrollbar(theForm->file_browser, FL_ON);
        fl_set_browser_scrollbarsize(theForm->file_browser, 15, 15);        
        fl_set_browser_fontstyle(theForm->file_browser, FL_BOLD_STYLE);
        fl_set_browser_fontsize(theForm->file_browser, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->file_browser, FileCB, 0);
        fl_set_browser_dblclick_callback(theForm->file_browser, DblFileCB, 0);
        theForm->file_browser->u_vdata = (void *) theForm; 
  
  
        /* Input section */ 
        
        /* 
        ** File name.  Although file input is geometrically below location
        ** and filter input, we add it first so that it has input focus
        ** by default. 
        */
        
        obj = fl_add_text(FL_NORMAL_TEXT, 
            20, 285, 80, 25, "File Name");
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);
        fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
 
        theForm->file_input = fl_add_input(FL_NORMAL_INPUT, 
            110, 285, 310, 25, "");
        fl_set_object_lstyle(theForm->file_input, FL_BOLD_STYLE);
        fl_set_object_lsize(theForm->file_input, FL_NORMAL_SIZE);
        fl_set_object_posthandler(theForm->file_input, change_focus);

        if ((default_file != "") && (default_file != NULL)) {
           fl_set_input(theForm->file_input, default_file);
           flx_justify_input (theForm->file_input, default_file, 
               FL_BOLD_STYLE, FL_NORMAL_SIZE);    
        }              
        theForm->file_input->u_vdata = (void *) theForm; 
        theForm->file_input->u_ldata = 2; 

        /* 
        ** Location 
        */
        obj = fl_add_text(FL_NORMAL_TEXT, 
            20, 215, 80, 25, "Location");
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);
        fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
 
        theForm->loc_input = fl_add_input(FL_NORMAL_INPUT, 
            110, 215, 310, 25, "");
        fl_set_object_lstyle(theForm->loc_input, FL_BOLD_STYLE);
        fl_set_object_lsize(theForm->loc_input, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->loc_input, LocationCB, 0);
        fl_set_input(theForm->loc_input, theForm->dir); 
        flx_justify_input (theForm->loc_input, theForm->dir, 
            FL_BOLD_STYLE, FL_NORMAL_SIZE);    
        fl_set_object_posthandler(theForm->loc_input, change_focus);
        
        theForm->loc_input->u_vdata = (void *) theForm; 
        theForm->loc_input->u_ldata = 0; 

        /* 
        ** Filter
        */
        obj = fl_add_text(FL_NORMAL_TEXT, 
            20, 250, 80, 25, "File Filter");
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);
        fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
 
        theForm->filter_input = fl_add_input(FL_NORMAL_INPUT, 
            110, 250, 310, 25, "");
        fl_set_object_lstyle(theForm->filter_input, FL_BOLD_STYLE);
        fl_set_object_lsize(theForm->filter_input, FL_NORMAL_SIZE);
        fl_set_object_callback(theForm->filter_input, FilterCB, 0);
        fl_set_input(theForm->filter_input, theForm->filter); 
        flx_justify_input (theForm->filter_input, theForm->filter, 
            FL_BOLD_STYLE, FL_NORMAL_SIZE);    
        fl_set_object_posthandler(theForm->filter_input, change_focus);

        theForm->filter_input->u_vdata = (void *) theForm; 
        theForm->filter_input->u_ldata = 1; 
                
         
        /* Button section */
        
        obj = fl_add_box(FL_NO_BOX, 
            0, 325, 440, 10, "@DnLine");  


        /* Designate "Ok" as the default button */
        theForm->ok_def_b = flx_add_return_button(FL_NORMAL_BUTTON,
            20, 345, 70, 30, "Ok");
        fl_set_object_lstyle(theForm->ok_def_b, FL_BOLD_STYLE);
        fl_set_object_callback(theForm->ok_def_b, DoneCB, 0);
        theForm->ok_def_b->u_vdata = (void *) theForm; 
  
        /* Hide for now */
        theForm->ok_b = fl_add_button(FL_NORMAL_BUTTON,
            20, 345, 70, 30, "Ok");
        fl_set_object_lstyle(theForm->ok_b, FL_BOLD_STYLE);
        fl_set_object_callback(theForm->ok_b, DoneCB, 0);
        theForm->ok_b->u_vdata = (void *) theForm; 
        fl_hide_object(theForm->ok_b);
        
        theForm->filter_b = fl_add_button(FL_NORMAL_BUTTON, 
            130, 345, 70, 30, "Filter"); 
        fl_set_object_lstyle(theForm->filter_b, FL_BOLD_STYLE);
        fl_set_object_callback(theForm->filter_b, FilterCB, 0);
        theForm->filter_b->u_vdata = (void *) theForm; 

        /* Hide for now */
        theForm->filter_def_b = flx_add_return_button(FL_NORMAL_BUTTON,
            130, 345, 70, 30, "Filter");
        fl_set_object_lstyle(theForm->filter_def_b, FL_BOLD_STYLE);
        fl_set_object_callback(theForm->filter_def_b, FilterCB, 0);
        theForm->filter_def_b->u_vdata = (void *) theForm; 
        fl_hide_object(theForm->filter_def_b);


        obj = fl_add_button(FL_NORMAL_BUTTON, 
            240, 345, 70, 30, "Cancel"); 
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);
        fl_set_object_callback(obj, DoneCB, 1);
        obj->u_vdata = (void *) theForm; 

        obj = fl_add_button(FL_NORMAL_BUTTON, 
            350, 345, 70, 30, "Help"); 
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);
        fl_set_object_callback(obj, DoHelpCB, 0);
        obj->u_vdata = (void *) theForm; 

 
        /* 
        ** Fill browsers 
        */ 
        parse_dir(theForm);

    fl_end_form();

    flx_scale_form(theForm->main);

    
    return(theForm);
    
} /* end function build_filebox() */



/* * * * * CALLBACK FUNCTIONS * * * * */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Double click on a filename */
void DblFileCB(FL_OBJECT *ob, long data)
{
    have_file = 1;
}    


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DirCB(FL_OBJECT *ob, long data)
{
    const char *thedir;    
    char *cwd;
        
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;
    
    
    /* Remove any selected file */
    fl_set_input(theForm->file_input, "");

    thedir = fl_get_browser_line(theForm->dir_browser, 
        fl_get_browser(theForm->dir_browser)); 

    /* Remove color specification */
    if (thedir[0] == '@')
       thedir += 3;

    if ((chdir(thedir)) == -1) {
       perror("Could not change dir");
       return;
    }


    /* Store current directory */
    cwd = getcwd(NULL, FILENAME_MAX);        
    theForm->dir = (char *) realloc (theForm->dir, strlen(cwd) + 1);
    sprintf(theForm->dir, "%s", cwd);
    free(cwd); 

    /* Set the new location */
    fl_set_input(theForm->loc_input, theForm->dir);
    flx_justify_input (theForm->loc_input, theForm->dir, 
        FL_BOLD_STYLE, FL_NORMAL_SIZE);    
 

    parse_dir(theForm);

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DoneCB(FL_OBJECT *ob, long data)
{
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;

    switch (data) {
    
           case 0: /* Ok */
               if (strcmp(fl_get_input(theForm->file_input), "") == 0) {
                  flx_set_dialog_font(FL_BOLD_STYLE, FL_NORMAL_SIZE);
                  flx_show_dialog("error", "Please select a file to open"); 
                  return; 
               }               
               have_file = 1;
               break;
           
           case 1: /* Cancel - clear file and return */
               fl_set_input(theForm->file_input, "");
               have_file = 1;
               break;
               
           default: /* Error */
               fprintf(stderr, "flx_filebox: Internal Error: DoneCB\n");
               break;
    
    }            


}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DoHelpCB(FL_OBJECT *ob, long data)
{
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;
    
    
    if (!fl_form_is_visible(theForm->help)) 
       fl_show_form(theForm->help, FL_PLACE_HOTSPOT, FL_TRANSIENT, "Help");
    
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DotfileCB(FL_OBJECT *ob, long data)
{
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;

    parse_dir(theForm);


}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void FileCB(FL_OBJECT *ob, long data)
{
    const char *thefile;
    
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;


    thefile = fl_get_browser_line(theForm->file_browser, 
        fl_get_browser(theForm->file_browser));
    
    /* Remove color specification */
    if (thefile[0] == '@')
       thefile += 3;
       
    /* Set the new file */
    fl_set_input(theForm->file_input, thefile); 
    flx_justify_input (theForm->file_input, thefile, 
        FL_BOLD_STYLE, FL_NORMAL_SIZE);    

    /* Set the default button */
    fl_set_object_focus(theForm->main, theForm->file_input);


}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void FilterCB(FL_OBJECT *ob, long data)
{
    char *theFilter;
    char *pattern;

    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;
    

    theFilter = (char *) fl_get_input(theForm->filter_input);
    
    if ((pattern = extract_filter(theFilter, '/')) == NULL)
       pattern = theFilter;
        
    strcpy(theForm->filter, pattern);
    
    parse_dir(theForm);


}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void LocationCB(FL_OBJECT *ob, long data)
{
    const char *thedir;

    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;
    
    
    thedir = (char *) fl_get_input(theForm->loc_input);

    if ((chdir(thedir)) == -1) {
       perror("Could not change dir");
       fl_set_input(theForm->loc_input, theForm->dir);
       flx_justify_input (theForm->loc_input, theForm->dir, 
           FL_BOLD_STYLE, FL_NORMAL_SIZE);    
       return;
    }
    theForm->dir = (char *) realloc (theForm->dir, strlen(thedir) + 1);
    sprintf(theForm->dir, "%s", thedir);

    
    parse_dir(theForm);


}



/* * * * * UTILITY FUNCTIONS * * * * */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void parse_dir (FORM_S *theForm)
{
    extern int errno;
    DIR *dir_ptr;
    struct dirent **dir_s;
    int i, numEntries;
    
    struct stat *stat_buf = (struct stat *) vmalloc (sizeof(struct stat));    
    int filetype;
        
    char *sfile;
    char *file_color     = "@C0"; /* default black */ 
    char *link_color     = "@C0"; 
    char *dir_color      = "@C0"; 
    char *chardev_color  = "@C0"; 
    char *blockdev_color = "@C0"; 
    char *socket_color   = "@C0"; 
    char *fifo_color     = "@C0"; 

    char *currentFile;
    int currentWidth, maxWidth = 0;
    int min_browser_width = 175;     


    if (USE_COLOR) {
       file_color     = "@C0";  /* black   */ 
       link_color     = "@C6";  /* cyan    */ 
       dir_color      = "@C0";  /* black   */ 
       chardev_color  = "@C3";  /* yellow  */
       blockdev_color = "@C4";  /* blue    */ 
       socket_color   = "@C5";  /* magenta */
       fifo_color     = "@C2";  /* green   */
    }
        
    /* 
    ** Open current directory for reading 
    */    
    if ((dir_ptr = opendir(theForm->dir)) == NULL) {
       perror("parse_dir");
       exit (1);
    }
    
    /* 
    ** Freeze the file selector form, and clear the browsers 
    */    
    fl_freeze_form(theForm->main);
    fl_clear_browser(theForm->dir_browser);
    fl_clear_browser(theForm->file_browser);
 
    /*
    ** Scan the directory; sort alphabetically
    */   
    if ((numEntries = scandir(theForm->dir, &dir_s, NULL, alphasort)) < 0) {
       /* Fatal error */
       perror("flx_filebox: parse_dir");
       exit (1);
    }     
    

    /* 
    ** Handle each directory entry 
    */        
    for (i = 0; i< numEntries; i++) {
        
        /* Get the length of the longest filename */
/*
        currentWidth = fl_get_string_width(FL_BOLD_STYLE, FL_NORMAL_SIZE, 
                     dir_s[i]->d_name, strlen(dir_s[i]->d_name));

        if (currentWidth > maxWidth) {
           maxWidth = currentWidth;
*/

        currentFile = (char *) vmalloc (strlen(theForm->dir) + strlen("/") +
            strlen(dir_s[i]->d_name) + 1);
        sprintf(currentFile, "%s/%s", theForm->dir, dir_s[i]->d_name);

        if (lstat(currentFile, stat_buf)) {
           perror("flx_filebox");
           exit (errno);
        }
         
        
        /* Non-POSIX method of testing filetype, for compatibility with BSD */
        filetype = stat_buf->st_mode & S_IFMT;

        /*
        ** Directories are not filtered
        */
        if (filetype != S_IFDIR) {
        
           /* 
           ** Check against file filter
           */ 
           if (fnmatch(theForm->filter, dir_s[i]->d_name, 0) != 0)
              continue; /* for loop */    
        
        }
        
        /* Filename used for display in filebox: add color */
        sfile = (char *) vmalloc (strlen(dir_s[i]->d_name) + 3 + 1);
                
        switch (filetype) {

          case S_IFDIR: /* Directory */
                        /* 
                        ** Always show the ".." directory
                        */
                        sprintf(sfile, "%s%s", dir_color, dir_s[i]->d_name);
                        if (! fl_get_button(theForm->dot_dirs)) {
                           if ((dir_s[i]->d_name[0] != '.') ||
                              ((dir_s[i]->d_name[0] == '.') && 
                               (dir_s[i]->d_name[1] == '.'))) 
                              fl_add_browser_line(theForm->dir_browser, sfile);
                        }
                        else
                           fl_add_browser_line(theForm->dir_browser, sfile);
                        break;
 
          case S_IFCHR: /* Character device file */
                        sprintf(sfile, "%s%s", chardev_color, dir_s[i]->d_name);
                        fl_add_browser_line(theForm->file_browser, sfile);
                        break;
 
          case S_IFBLK: /* Block device file */
                        sprintf(sfile, "%s%s", blockdev_color, dir_s[i]->d_name);
                        fl_add_browser_line(theForm->file_browser, sfile);
                        break;

          case S_IFREG: /* Regular file */
                        sprintf(sfile, "%s%s", file_color, dir_s[i]->d_name);
                        if (! fl_get_button(theForm->dot_files)) {
                           if (dir_s[i]->d_name[0] != '.')
                              fl_add_browser_line(theForm->file_browser, sfile);
                        }
                        else
                           fl_add_browser_line(theForm->file_browser, sfile);
                        break;

          case S_IFLNK: /* Symbolic link */
                        /*
                        ** Sometimes a hidden file is also a symlink.  Make
                        ** hidden property have precedence.
                        */
                        sprintf(sfile, "%s%s", link_color, dir_s[i]->d_name);
                        if (! fl_get_button(theForm->dot_files)) {
                           if (dir_s[i]->d_name[0] != '.')
                              fl_add_browser_line(theForm->file_browser, sfile);
                        } 
                        else
                           fl_add_browser_line(theForm->file_browser, sfile);
                        break;

          case S_IFSOCK: /* Socket */
                         sprintf(sfile, "%s%s", socket_color, dir_s[i]->d_name);
                         fl_add_browser_line(theForm->file_browser, sfile);      
                         break;

          case S_IFIFO: /* Pipe */
                        sprintf(sfile, "%s%s", fifo_color, dir_s[i]->d_name);
                        fl_add_browser_line(theForm->file_browser, sfile);
                        break;

          default: fprintf(stderr, "flx_filebox: Unknown file type: %s\n", 
                       dir_s[i]->d_name);
                   break;
        
        } /* end switch */
        
        free(sfile);
        free(currentFile);
        
    } /* endfor */
    
    closedir(dir_ptr);

    free(stat_buf);


/*
    if (maxWidth > min_browser_width) {
       fl_set_object_geometry(theForm->file_browser, 
           230, 40, wide_string_p + 20, 165);
    }
    else {
       fl_set_object_geometry(theForm->file_browser, 
           230, 40, 190, 165);    
    }
*/

    fl_unfreeze_form(theForm->main);
    

} /* end function parse_dir() */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
char *extract_filter(const char *filename, char delimiter)
{
    char *endchar;
    char *filter = NULL;
    
    if ((filename != NULL) &&
       ((endchar = strrchr(filename, delimiter)) != NULL))
   
       filter = ++endchar;
    
    return(filter); 
    
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* 
**  malloc() a void pointer, and verify success
*/
void *vmalloc(int chunk_size)
{
    void *chunk;

    
    chunk = (void *) malloc (chunk_size);
    if (chunk == NULL) {
       perror("flx_filebox");
       exit (1);
    }
    
    return (chunk);
       
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
int change_focus(FL_OBJECT *ob, int ev, 
    FL_Coord mx, FL_Coord my, int key, void *xev)
{
    /* Retrieve the form structure */
    FORM_S *theForm = (FORM_S *) ob->u_vdata;

    /* Retrieve data key */
    long ob_num = ob->u_ldata;


    if (!fl_form_is_visible(theForm->main))
       return(0);

    /* 
    ** Event handler
    */
    switch (ev) {
            
        case FL_FOCUS: 

            switch (ob_num) { 

	       case 0:
		 fl_hide_object(theForm->ok_def_b);
		 fl_show_object(theForm->ok_b);
		 fl_hide_object(theForm->filter_def_b);
		 fl_show_object(theForm->filter_b);
        	 break;

	       case 1:
		 fl_hide_object(theForm->ok_def_b);
		 fl_show_object(theForm->ok_b);
		 fl_hide_object(theForm->filter_b);
		 fl_show_object(theForm->filter_def_b);
        	 break;
               
	       case 2:
		 fl_hide_object(theForm->ok_b);
		 fl_show_object(theForm->ok_def_b);
		 fl_hide_object(theForm->filter_def_b);
		 fl_show_object(theForm->filter_b);
        	 break;

               default: /* Error */
                 fprintf(stderr, "flx_filebox: Internal Error: change_focus\n");
                 break;

            } /* endswitch */

            break;

        default: break;
 
    } /* endswitch */

    
    return(0);
    
}
