/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**
** NAME
**       flx_return_button
**
** PURPOSE
**       Extension of the xforms library.
**       Draws a button that returns when the <Return> key is pressed. 
**       Look is based on the default return button of the Motif library. 
**
**       Three types of buttons are supported:
**           FLX_NORMAL_BUTTON     Uses an FL_UP_BOX
**           FLX_ROUNDED3D_BUTTON  Uses an FL_ROUNDED3DUPBOX
**           FLX_OVAL3D_BUTTON     Uses an FL_OVAL3D_UPBOX
**
** FUNCTION PROTOTYPE
**       FL_OBJECT *flx_add_return_button(int type, 
**                                        FL_Coord x, FL_Coord y, 
**                                        FL_Coord w, FL_Coord h, 
**                                        const char *label);
**
** CALLING SEQUENCE
**
**       #include "flx_return_button.h"
**
**       obj = flx_add_return_button(FLX_NORMAL_BUTTON, x, y, w, h, "label");
**       obj = flx_add_return_button(FLX_ROUNDED3D_BUTTON, x, y, w, h, "label");
**       obj = flx_add_return_button(FLX_OVAL3D_BUTTON, x, y, w, h, "label");
**
** INPUTS
**       x, y, w, h: Location and size of the button
**       label:      Button label
**
** OUTPUT
**       Pointer to the button object
**
** RESTRICTIONS
**       Border width of the outside box is fixed at border width = -1
**
** CLASS NUMBER
**       3502
**
** MODIFICATION HISTORY
**       Written, Robert.Mallozzi@msfc.nasa.gov, August 1997
**           Based on the examples of new button classes in the XForms manual.
**
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**
** flx: Extensions for the xforms library
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the Free
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "forms.h"
#include "flx.h"

typedef FL_BUTTON_STRUCT SPEC;


 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
static void draw_return_button(FL_OBJECT *ob)
{
    long theColor;
    int outer_box_bw = -1;
    int upBox;
    int downBox;


    /* 
    ** Button color - if pushed, use col2, else use col1 
    */
    theColor = ((SPEC *) (ob->spec))->val ? ob->col2 : ob->col1;
    
    /* 
    ** Hightlight button when mouse is over, if using default colors 
    */
    if (ob->belowmouse && theColor == FL_COL1)
       theColor = FL_MCOL;

    /* 
    ** Get button type
    */      
    switch (ob->u_ldata) {
           
           case FLX_NORMAL_BUTTON: 
               upBox   = FL_UP_BOX; 
               downBox = FL_DOWN_BOX;
               break; 
           
           case FLX_ROUNDED3D_BUTTON:
               upBox   = FL_ROUNDED3D_UPBOX; 
               downBox = FL_ROUNDED3D_DOWNBOX;
               break; 
           
           case FLX_OVAL3D_BUTTON:
               upBox   = FL_OVAL3D_UPBOX; 
               downBox = FL_OVAL3D_DOWNBOX;
               break; 

           default: 
               upBox   = FL_UP_BOX; 
               downBox = FL_DOWN_BOX;
               break; 
    }    
    
    /* 
    ** Draw the outer box 
    */
    fl_drw_box(downBox, ob->x-5, ob->y-5, ob->w+10, ob->h+10,
        ob->col1, outer_box_bw);
    
    /* 
    ** Draw the button 
    */
    if (((SPEC *) (ob->spec))->val)
       fl_drw_box(downBox, ob->x, ob->y, ob->w, ob->h,
           theColor, ob->bw);
    else
       fl_drw_box(upBox, ob->x, ob->y, ob->w, ob->h,
           theColor, ob->bw);
       
    /* 
    ** Draw button label 
    */
    fl_draw_object_label(ob);
                 

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
FL_OBJECT *flx_create_return_button(int type, FL_Coord x, FL_Coord y,
    FL_Coord w, FL_Coord h,  const char *label)
{
    FL_OBJECT *ob;
    
            
    fl_add_button_class(FLX_RETURN_BUTTON, draw_return_button, 0);    
    
    ob = fl_create_generic_button(FLX_RETURN_BUTTON, FL_RETURN_BUTTON, 
        x, y, w, h, label);
    
    ob->col1    = FL_COL1;           /* Default unpushed color     */
    ob->col2    = FL_MCOL;           /* Default pushed color       */
    ob->align   = FL_ALIGN_CENTER;   /* Default label alignment    */
    ob->u_ldata = type;              /* User-specified button type */
     
    return ob;

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
FL_OBJECT *flx_add_return_button(int type, FL_Coord x, FL_Coord y,
    FL_Coord w, FL_Coord h,  const char *label)
{

    FL_OBJECT *ob = flx_create_return_button(type, x, y, w, h, label);   
    fl_add_object(fl_current_form, ob);

    
    return ob;
    
}    
