/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**+
** NAME
**      flx_show_dialog
**
** PURPOSE
**     Popup dialogs to replace some of the xforms goodies
**
** CALLING SEQUENCE
**     status = flx_show_dialog("type", "message_string");
**
**     status (int) is the return value.  All dialogs except question have
**          only an "Ok" button, and return the value 1.  A question dialog
**          has "Ok" and "Cancel" buttons, and returns 1 for "Ok", and 0
**          for "Cancel"
**
**     type can be one of the following strings: 
**          error
**          info
**          question
**          message
**          warning
**
**     message_string is the text to display in the dialog box, and may 
**          contain newline (\n) characters for multi-line messages
**
**     Dialog attributes can be set by calling any of the following functions
**          before the call to fl_show_dialog()
**
**          void flx_set_dialog_lcol (int theLabelColor);
**          void flx_set_dialog_font (int theLabelStyle, int theLabelSize);
**          void flx_set_dialog_color (int theInactiveColor, int theActiveColor);
**          void flx_set_dialog_bgcolor (int theBgColor);
**         
** MODIFICATION HISTORY
**     Written, Robert.Mallozzi@msfc.nasa.gov, February 1997
**                         
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**
** flx: Extensions for the xforms library
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the Free
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include "forms.h"
#include "flx.h"
#include "flx_show_dialog.h" /* Internal to this routine */

#include "error.xpm"
#include "info.xpm"
#include "message.xpm"
#include "warning.xpm"
#include "question.xpm"

static void usage();
void flx_set_dialog_lcol();
void flx_set_dialog_font();
void flx_set_dialog_color();
void flx_set_dialog_bgcolor();

 
 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int flx_show_dialog (const char *type_str, const char *message_str)
{

    FL_FORM *main_form;  
    FL_OBJECT *obj, *obj_ok, *obj_cancel, *ret;

    int button_width  = 70;
    int button_height = 25;
    int min_x         = 180;
    int min_y         = 120;
    int pad           = 10;
    int pm_width      = 50;
    int pm_height     = 50;
                    
    int label_width, label_height;
    int button_label_width, button_label_height;
    int status = 0;
    int form_x, form_y, yoffset, ymax_height;
    char **discriminator;

 
    /* discriminator holds the array of char pointers obtained from the
    ** #include of the pixmaps
    */
    discriminator = NULL;
    
    if (strcmp(type_str, "error") == 0)
       discriminator = error_xpm;
  
    if (strcmp(type_str, "info") == 0)
       discriminator = info_xpm;
  
    if (strcmp(type_str, "message") == 0)
       discriminator = message_xpm;
  
    if (strcmp(type_str, "question") == 0)
       discriminator = question_xpm;
            
    if (strcmp(type_str, "warning") == 0)
       discriminator = warning_xpm;
   
    if ((strcmp(type_str, "error")    != 0) &&
        (strcmp(type_str, "info")     != 0) &&
        (strcmp(type_str, "message")  != 0) &&
        (strcmp(type_str, "question") != 0) &&
        (strcmp(type_str, "warning")  != 0)) {
       usage();
    }


    /* Get the string sizes, then size the form */

    /* The Ok, Cancel buttons */
    fl_get_string_dimension(FLX_DIALOG_LSTYLE, FLX_DIALOG_LSIZE,
                  "Cancel", strlen("Cancel"), 
                  &button_label_width, &button_label_height);

    /* Resize buttons ? */
    if (button_label_width >= button_width)
       button_width = button_label_width + 6;
    if (button_label_height >= button_height)
       button_height = button_label_height + 6;

    /* The main label */
    fl_get_string_dimension(FLX_DIALOG_LSTYLE, FLX_DIALOG_LSIZE,
                  message_str, strlen(message_str), 
                  &label_width, &label_height);

    form_x = pm_width + pad + label_width + pad;
    if (form_x < min_x)
       form_x = min_x;
    
    if (label_height <= pm_height)
       ymax_height = pm_height;
    else
       ymax_height = label_height;
       
    form_y = ymax_height + pad + 10 + pad + button_height + pad; 
    if (form_y < min_y)
       form_y = min_y;

    yoffset = form_y - min_y;
    
   
    main_form = fl_bgn_form(FL_NO_BOX, form_x, form_y);

        obj = fl_add_box(FL_UP_BOX, 
            0, 0, form_x, form_y, "");
        fl_set_object_color(obj, FLX_DIALOG_BGCOLOR, 0);

        obj = fl_add_pixmap(FL_NORMAL_PIXMAP, 
            0, 0, pm_width, pm_height, "");
        fl_set_pixmap_data(obj, discriminator);

        obj = fl_add_box(FL_NO_BOX, 
            pm_width + pad, 0, 
            label_width, ymax_height, 
            message_str);
        fl_set_object_lcol(obj, FLX_DIALOG_LCOL);
        fl_set_object_lstyle(obj, FLX_DIALOG_LSTYLE);
        fl_set_object_lsize(obj, FLX_DIALOG_LSIZE);
  
        obj = fl_add_box(FL_NO_BOX, 
            0, ymax_height + pad, form_x, 10, "@DnLine");

        if (strcmp(type_str, "question") == 0) {
 
           obj_ok = flx_add_return_button(FL_NORMAL_BUTTON, 
               pad, ymax_height + pad + 10 + pad, 
               button_width, button_height, "Ok"); 
           fl_set_object_color(obj_ok, FLX_DIALOG_ICOLOR, FLX_DIALOG_ACOLOR);
           fl_set_object_lcol(obj_ok, FLX_DIALOG_LCOL);
           fl_set_object_lstyle(obj_ok, FLX_DIALOG_LSTYLE);
           fl_set_object_lsize(obj_ok, FLX_DIALOG_LSIZE);
 
           obj_cancel = fl_add_button(FL_NORMAL_BUTTON, 
               form_x - button_width - pad, ymax_height + pad + 10 + pad, 
               button_width, button_height, "Cancel"); 
           fl_set_object_color(obj_cancel, FLX_DIALOG_ICOLOR, FLX_DIALOG_ACOLOR);
           fl_set_object_lcol(obj_cancel, FLX_DIALOG_LCOL);
           fl_set_object_lstyle(obj_cancel, FLX_DIALOG_LSTYLE);
           fl_set_object_lsize(obj_cancel, FLX_DIALOG_LSIZE);
 
        }
        else {

           obj_ok = flx_add_return_button(FL_NORMAL_BUTTON, 
               (form_x - button_width)/2, ymax_height + pad + 10 + pad, 
               button_width, button_height, "Ok"); 
           fl_set_object_color(obj_ok, FLX_DIALOG_ICOLOR, FLX_DIALOG_ACOLOR);
           fl_set_object_lcol(obj_ok, FLX_DIALOG_LCOL);
           fl_set_object_lstyle(obj_ok, FLX_DIALOG_LSTYLE);
           fl_set_object_lsize(obj_ok, FLX_DIALOG_LSIZE);
             
           obj_cancel = obj_ok;
                  
        }

    fl_end_form();
         
    fl_show_form(main_form, FL_PLACE_MOUSE, FL_TRANSIENT, type_str);
    
    ret = fl_do_forms();

    if (ret == obj_cancel)
       status = 0;

    if (ret == obj_ok)
       status = 1;
                    
    fl_hide_form(main_form);
    return (status);


} /* END */

 

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void usage(void)
{
       printf("int flx_show_dialog (const char *type_str, const char *message_str)\n");
       printf("    type_str = error\n");
       printf("               info\n");
       printf("               message\n");
       printf("               question\n");
       printf("               warning\n");
       printf("    message_string = the message drawn in the dialog box\n");
       exit (2);

}


  
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void flx_set_dialog_lcol (int theLabelColor)
{
    FLX_DIALOG_LCOL = theLabelColor;
 
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void flx_set_dialog_font (int theLabelStyle, int theLabelSize)
{
    FLX_DIALOG_LSTYLE = theLabelStyle;
    FLX_DIALOG_LSIZE  = theLabelSize;
    
} 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void flx_set_dialog_color (int theInactiveColor, int theActiveColor)
{
    FLX_DIALOG_ICOLOR = theInactiveColor;
    FLX_DIALOG_ACOLOR = theActiveColor;
 
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void flx_set_dialog_bgcolor (int theBgColor)
{
    FLX_DIALOG_BGCOLOR = theBgColor;
 
}
