/* 
** xfpovray callback functions
*/


#include "forms.h"
#include "xfpovray.h"
#include "pov-defaults.h"
#include "pov-templates.h"

#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
 
#define TIME 0.01

extern int start_x;
extern int start_y;
extern int have_geometry;  

extern struct pov_options pov;
 
extern char *POV_INIFILE;
extern char *POV_INIFILE_DEF;
extern char *STREAM_FILE;
extern char *FATAL_FILE;
extern char *PREVIEW_FILE;
extern char *TEMP_FILE;
 
extern void write_config(char *ini_file);
extern int  povray_pid;
extern int  auto_view;
extern void show_image(void);
extern int  PovCheckCB(XEvent *ev, void *data);
extern void run_program  (int count, ...);
 
struct preview_options {
       float camera[3];
       float camera_lookat[3];
       float light[3];
};
extern struct preview_options preview_setup;
extern struct preview_options preview_setup_def;

extern int show_tips;

extern char *CONFIG_POV_COMMAND;
extern char *CONFIG_POV_LIB_PATH;
extern char *CONFIG_POV_EDITOR;
extern char *CONFIG_DISPLAY_COMMAND;
extern char *CONFIG_POV_DIR;
extern char *CONFIG_SCENE_FILTER;
extern char *CONFIG_IMAGE_FILTER;
extern char *CONFIG_POV_WIDTH; 
extern char *CONFIG_POV_HEIGHT;
 
extern XFDATA_S *xfData;
extern OUTPUT_PAGE_S    *outputPage;
extern TRACING_PAGE_S   *tracingPage;
extern CONFIG_PAGE_S    *configPage;
 


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void AbortCB(FL_OBJECT *ob, long data)
{

    /* MB1: abort; MB2, MB3 are used to hide the render popup form */
    if ((fl_get_button_numb(ob) == 2) || (fl_get_button_numb(ob) == 3)) {

       if (fl_form_is_visible (xfData->renderPupForm)) {
          fl_hide_form (xfData->renderPupForm);
          fl_set_menu_item_mode(xfData->optionsMenu, 4, FL_PUP_BOX);
       }  
       return;        
    }

    if (povray_pid == 0) 
       return;

    kill(povray_pid, SIGINT);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void ClearPartialImageCB(FL_OBJECT *ob, long data)
{
    fl_set_input (outputPage->columnStart, "");
    fl_set_input (outputPage->columnEnd, "");
    fl_set_input (outputPage->rowStart, "");
    fl_set_input (outputPage->rowEnd, "");

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void ChangedFolderCB(FL_OBJECT *ob, long data)
{
    if (fl_get_active_folder(xfData->mainFolder) == configPage->form)
       write_config(NULL);
}    


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Configuration file options */
void ConfigCB(FL_OBJECT *ob, long data)
{

    int i, submenu;
    char *save_config_file;
    char *temp;
    char *path;
    char *basename;
    const int maxTokens = 100;
    int tokenCount;
    char *tokenList[maxTokens];
        
         
    submenu = fl_get_menu(ob);
    switch (submenu) {
    
           case 1: /* Show */
                   write_config(NULL);
                      
                   fl_clear_browser (configPage->configBrowser);
                   fl_load_browser (configPage->configBrowser, POV_INIFILE);    
                   
		   /* Show folder form? */
                   if (! fl_form_is_visible (xfData->folderForm)) {
		      fl_set_object_label (xfData->toggleFolder,"Hide Options");
		      fl_show_form (xfData->folderForm, 
        		  FL_PLACE_SIZE, FL_FULLBORDER, "Render Options");
                   }
		   fl_set_folder (xfData->mainFolder, configPage->form); 
		   fl_show_object (xfData->mainFolder);
                   
                   break;
                   
           case 2: /* Save */
                    
                   flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);   
                   
		   if (strcmp(fl_get_input (xfData->sceneFileInput), "") == 0) {
		   
		      /* No scene file selected */
		      flx_show_dialog ("warning", "No Scene File Specified");
		      path = (char *) fl_calloc(1, strlen(".") + 1);
		      sprintf(path, "."); 
                      basename = (char *) fl_calloc(1, strlen("default.ini")+1);
                      sprintf(basename, "default.ini");
		   
		   }
		   else {
		   
                      /* Break up filename into path and basename */
                      temp = (char *) fl_calloc (1, 
			  strlen(fl_get_input (xfData->sceneFileInput)) + 1);
                      sprintf(temp, fl_get_input (xfData->sceneFileInput));

		      path = (char *) fl_calloc (1, strlen(temp) + 1);
		      sprintf(path, "/"); 

		      tokenCount = tokenize (temp, tokenList, 20);
		      for (i = 0; i < tokenCount - 1; i++) {
			  strcat(path, tokenList[i]); 
        		  strcat(path, "/");
		      }
		      basename = (char *) fl_calloc (1, strlen(tokenList[i])+1);
		      sprintf(basename, tokenList[i]);
                      strtruncate (basename, '.');
                      strcat(basename, ".ini");		      
		      free (temp); 
		   }

                   save_config_file = (char *) flx_filebox (
		       path, "*.ini", basename);
		   
                   /* Cancel selected? */
                   if (strcmp(save_config_file, "") == 0)
                      return;
                   
		   /* Write saved configuration */
		   write_config (save_config_file);

                   free (path);
		   free (basename);
                   break;
                   
           case 3: /* Load */        

        	   temp = (char *) flx_filebox (CONFIG_POV_DIR, "*.ini", "");
                   if (strcmp(temp, "") == 0) 
                      return;

		   /* Read the ini file, and update widget state */
                   if ((read_config (temp)) == 0) {
		   
		      /* 
		      ** Update new configuration 
		      */
		      write_config (NULL);

		      /*
		      ** Update the config folder
		      */
		      fl_freeze_form (configPage->form);
		      fl_clear_browser (configPage->configBrowser);
		      fl_load_browser (configPage->configBrowser, POV_INIFILE);
		      fl_unfreeze_form (configPage->form);
                   
		   }  

                   break;
		    
           default: perror("xfpovray: ConfigCB"); break;
                  
    } /* endswitch */
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* For the fancy "About" widget */
void ContinueCB(FL_OBJECT *ob, long data)
{
    int r, val;
    int num_colors = 10;
    int colors[10] = {FL_RED,
                      FL_GREEN,
                      FL_YELLOW,
                      FL_BLUE,
                      FL_MAGENTA,
                      FL_SLATEBLUE,
                      FL_ORCHID,
                      FL_DARKGOLD,
                      FL_TOMATO,
                      FL_WHITE};
    static FL_Coord dx = 1, dy = 1;
    FL_Coord x, y, w, h;  


    
    fl_set_timer(xfData->aboutTimer, TIME);
    fl_set_object_callback (xfData->aboutTimer, ContinueCB, 0);

    /* Set a random color for tc */
    srand(time(0));
    r = rand();
    val = (int) ( (r / (float) RAND_MAX) * (num_colors) ); 
       
    fl_set_object_lcol (xfData->aboutText, colors[val]);


    /* Set the text position */
    fl_get_object_geometry (xfData->aboutText, &x, &y, &w, &h);
    
    if ( (x + dx) < 0 || (x + w + dx) >= xfData->aboutText->form->w)
       dx = -dx;
    if ( (y + dy) < 0 || (y + (h+140) + dy) >= xfData->aboutText->form->h)
       dy = -dy;
    
    x += dx;
    y += dy;

    fl_set_object_position (xfData->aboutText, x ,y);


}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DismissAboutCB(FL_OBJECT *ob, long data)
{
    fl_hide_form (xfData->aboutForm);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DismissConfiguratorCB(FL_OBJECT *ob, long data)
{

    switch (data) {
            
           case 0: /* Accept */
               
               if (strcmp(fl_get_input (xfData->configOptions[0]), "") != 0)
                  CONFIG_POV_COMMAND = 
		      (char *) fl_get_input (xfData->configOptions[0]);
               else
                  fl_set_input(xfData->configOptions[0], CONFIG_POV_COMMAND);

               if (strcmp(fl_get_input (xfData->configOptions[1]), "") != 0)
                  CONFIG_POV_LIB_PATH = 
		      (char *) fl_get_input (xfData->configOptions[1]);
               else
                  fl_set_input (xfData->configOptions[1], CONFIG_POV_LIB_PATH);
                                       
               if (strcmp(fl_get_input (xfData->configOptions[2]), "") != 0)
                  CONFIG_POV_EDITOR = 
		      (char *) fl_get_input (xfData->configOptions[2]);
               else
                  fl_set_input (xfData->configOptions[2], CONFIG_POV_EDITOR);

               if (strcmp(fl_get_input (xfData->configOptions[3]), "") != 0)
                  CONFIG_DISPLAY_COMMAND = 
		      (char *) fl_get_input (xfData->configOptions[3]);
               else
                  fl_set_input(xfData->configOptions[3], CONFIG_DISPLAY_COMMAND);

               if (strcmp(fl_get_input (xfData->configOptions[4]), "") != 0)
                  CONFIG_POV_DIR = 
		      (char *) fl_get_input (xfData->configOptions[4]);
               else
                  fl_set_input (xfData->configOptions[4], CONFIG_POV_DIR);

               if (strcmp(fl_get_input (xfData->configOptions[5]), "") != 0)
                  CONFIG_SCENE_FILTER = 
		      (char *) fl_get_input (xfData->configOptions[5]);
               else
                  fl_set_input(xfData->configOptions[5], CONFIG_SCENE_FILTER);

               if (strcmp(fl_get_input (xfData->configOptions[6]), "") != 0)
                  CONFIG_IMAGE_FILTER = 
		      (char *) fl_get_input (xfData->configOptions[6]);
               else
                  fl_set_input (xfData->configOptions[6], CONFIG_IMAGE_FILTER);

               if (strcmp(fl_get_input (xfData->configOptions[7]), "") != 0)
                  CONFIG_POV_WIDTH = 
		      (char *) fl_get_input (xfData->configOptions[7]);
               else
                  fl_set_input (xfData->configOptions[7], CONFIG_POV_WIDTH);

               if (strcmp(fl_get_input (xfData->configOptions[8]), "") != 0)
                  CONFIG_POV_HEIGHT = 
		      (char *) fl_get_input (xfData->configOptions[8]);
               else
                  fl_set_input (xfData->configOptions[8], CONFIG_POV_HEIGHT);

               break;
           
           case 1: /* Cancel */
               break;
                   
    }
    
    fl_hide_form (xfData->configuratorForm);

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DismissFolderCB(FL_OBJECT *ob, long data)
{
    fl_call_object_callback (xfData->toggleFolder);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DismissHelpCB(FL_OBJECT *ob, long data)
{
    fl_hide_form (xfData->helpForm);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DismissMessageCB(FL_OBJECT *ob, long data)
{

    switch (data) {
           case 0:  fl_hide_form (xfData->messageForm);    break;
           case 1:  fl_hide_form (xfData->fatalForm);      break;
           default: perror ("xfpovray: DismissMessageCB"); break;
    }
    

}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DisplayHelpCB(FL_OBJECT *ob, long data)
{
    int choice_line; 
    const char *choice;
    char filename[256];

    choice_line = fl_get_browser (ob);
    choice = fl_get_browser_line (ob, choice_line); 

    /* Get rid of the xforms control characters that denote colored text */
    if (*choice == '@') 
       choice = choice + 3;

    sprintf(filename, "%s/%s", HELP_DIR, choice);
    fl_load_browser (xfData->helpBrowser, filename);


}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void DoPreviewCB(FL_OBJECT *ob, long data)
{
    char command[256];
    FILE *fp;
    int choice_line, iwidth, iheight; 
    const char *choice;
    char discriminator[20];


    /* Image height and width */
    if (strcmp(fl_get_input (outputPage->imageHeightInput), "") == 0) 
       iheight = atoi(CONFIG_POV_HEIGHT);
    else
       iheight = atoi(fl_get_input (outputPage->imageHeightInput));

    if (strcmp(fl_get_input (outputPage->imageWidthInput), "") == 0)
       iwidth = atoi(CONFIG_POV_WIDTH); 
    else
       iwidth = atoi(fl_get_input (outputPage->imageWidthInput)); 

    sprintf(command, "%s +W%d +H%d +D +P -V -GA +GF -F +L%s +I%s", 
                     CONFIG_POV_COMMAND, iwidth, iheight, 
                     CONFIG_POV_LIB_PATH, PREVIEW_FILE);


    if ((fp = fopen(PREVIEW_FILE, "w")) == NULL) {
       perror("xfpovray: DoPreviewCB");
       return;
    }

    fprintf(fp, "#include \"colors.inc\"\n"); 

    if (fl_get_button (xfData->previewOption[0]) == 1)
       strcpy(discriminator, "pigment");
            
    if (fl_get_button (xfData->previewOption[1]) == 1) {
       fprintf(fp, "#include \"stones1.inc\"\n"); 
       fprintf(fp, "#include \"stones2.inc\"\n");
       strcpy(discriminator, "texture");
    }

    if (fl_get_button (xfData->previewOption[2]) == 1) {
       fprintf(fp, "#include \"woods.inc\"\n");
       strcpy(discriminator, "texture");
    }

    if (fl_get_button (xfData->previewOption[3]) == 1) {
       fprintf(fp, "#include \"glass.inc\"\n");
       strcpy(discriminator, "texture");
       
       /* For glass, add a checkerboard background */
       fprintf(fp, "plane {\n");
       fprintf(fp, "  <1, 0, 0>, -5\n");
       fprintf(fp, "  pigment {\n");
       fprintf(fp, "    checker color <0, 0, 0>, color <1, 1, 1>\n");
       fprintf(fp, "  }\n");
       fprintf(fp, "}\n");
    }

    if (fl_get_button (xfData->previewOption[4]) == 1) 
       fprintf(fp, "#include \"metals.inc\"\n");
   

    if (fl_get_button (xfData->previewOption[5]) == 1)
       fprintf(fp, "#include \"golds.inc\"\n");
 

    if (fl_get_button (xfData->previewOption[6]) == 1) {
       fprintf(fp, "#include \"finish.inc\"\n");
       strcpy(discriminator, "finish");
    }

    if (fl_get_button (xfData->previewOption[7]) == 1) {
       fprintf(fp, "#include \"stars.inc\"\n");
       fprintf(fp, "#include \"skies.inc\"\n");
       strcpy(discriminator, "texture");
    }

    if (fl_get_button (xfData->previewOption[8]) == 1) {
       fprintf(fp, "#include \"shapes.inc\"\n"); 
       fprintf(fp, "#include \"shapes2.inc\"\n");
       fprintf(fp, "#include \"shapesq.inc\"\n");
       strcpy(discriminator, "geometry");
    }

    if (fl_get_button (xfData->previewOption[9]) == 1) {
       fprintf(fp, "#include \"platon.inc\"\n"); 
       strcpy(discriminator, "geometry");
    }


    /* The user sometimes must select "pigment" or "texture" :-( */
    if (fl_get_button (xfData->previewType[0]) == 1)
       strcpy(discriminator, "pigment");

    if (fl_get_button (xfData->previewType[1]) == 1)
       strcpy(discriminator, "texture");
       
    if (fl_get_button (xfData->previewType[2]) == 1)
       strcpy(discriminator, "finish");

    if (fl_get_button (xfData->previewType[3]) == 1)
       strcpy(discriminator, "geometry");


    choice_line = fl_get_browser (xfData->previewBrowser);
    choice = fl_get_browser_line (xfData->previewBrowser, choice_line); 

    /* Add axes? */
    if ((fl_get_button (xfData->showAxes)) == 1) {
       fprintf(fp, "#declare Axes =\n"); 
       fprintf(fp, "union {\n");
       fprintf(fp, "   cylinder { <0, 0, 0>, <500, 0, 0>, 0.05\n");
       fprintf(fp, "     pigment { rgb <1, 0, 0> }\n");
       fprintf(fp, "   }\n");
       fprintf(fp, "   cylinder { <0, 0, 0>, <0, 500, 0>, 0.05\n");
       fprintf(fp, "     pigment { rgb <0, 1, 0> }\n");
       fprintf(fp, "   }\n");
       fprintf(fp, "   cylinder { <0, 0, 0>, <0, 0, 500>, 0.05\n");
       fprintf(fp, "     pigment { rgb <0, 0, 1> }\n");
       fprintf(fp, "   }\n");
       fprintf(fp, "}\n");
       fprintf(fp, "object { Axes }\n");
    }

    /* Lights, camera... */
    preview_setup.camera[0] = atof(fl_get_input (xfData->previewCamera[0]));
    preview_setup.camera[1] = atof(fl_get_input (xfData->previewCamera[1]));
    preview_setup.camera[2] = atof(fl_get_input (xfData->previewCamera[2]));

    preview_setup.light[0] = atof(fl_get_input (xfData->previewLight[0]));
    preview_setup.light[1] = atof(fl_get_input (xfData->previewLight[1]));
    preview_setup.light[2] = atof(fl_get_input (xfData->previewLight[2]));
    
    
    fprintf(fp, "camera {\n");
    fprintf(fp, "    location <%f, %f, %f>\n", 
                preview_setup.camera[0],
                preview_setup.camera[1],
                preview_setup.camera[2]);
    fprintf(fp, "    look_at  <0, 0,  0>\n");
    fprintf(fp, "}\n");

    fprintf(fp, "light_source {\n");
    fprintf(fp, "    <%f, %f, %f>\n",
                preview_setup.light[0],
                preview_setup.light[1],
                preview_setup.light[2]);
    fprintf(fp, "    color White\n");
    fprintf(fp, "}\n");


    if (strcmp(discriminator, "sky_sphere") == 0) {

       fprintf(fp, "sky_sphere {\n");
       fprintf(fp, "    pigment {\n");
       fprintf(fp, "        %s\n", choice);
       fprintf(fp, "    }\n");
       fprintf(fp, "}\n");
   
    } 
    else
    if (strcmp(discriminator, "geometry") == 0) {

       fprintf(fp, "object {\n");
       fprintf(fp, "    %s\n", choice);
       fprintf(fp, "    pigment {\n");
       fprintf(fp, "        rgb <0.9 0.9 0.9>\n");
       fprintf(fp, "    }\n");
       fprintf(fp, "}\n");    
 
    } 
    else
    if (strcmp(discriminator, "finish") == 0) {
       fprintf(fp, "sphere {\n");
       fprintf(fp, "    <0, 0, 0>, 1.0\n");
       fprintf(fp, "    pigment {rgb <0.9, 0.9, 0.9>}\n");
       fprintf(fp, "    finish {\n");
       fprintf(fp, "        %s\n", choice);
       fprintf(fp, "    }\n");
       fprintf(fp, "}\n");
   
    } 
    else {

       fprintf(fp, "sphere {\n");
       fprintf(fp, "    <0, 0, 0>, 1.0\n");
       fprintf(fp, "    %s {\n", discriminator);
       fprintf(fp, "        %s\n", choice);
       fprintf(fp, "    }\n");
       fprintf(fp, "}\n");
    } 

    fclose(fp);

    system(command);

}


                 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Pops up a file selection box */
void GetFileCB(FL_OBJECT *ob, long data)
{   
    int pid, type;
    char *temp;

  
    switch (data) {
       
       case 0: /* Scene File button */
            temp = pov.input_file_name;

            pov.input_file_name = (char *) flx_filebox ( 
                CONFIG_POV_DIR, CONFIG_SCENE_FILTER, "");  

            if (strcmp(pov.input_file_name, "") == 0) {
               /* Restore old file */
               pov.input_file_name = temp; 
               return;
            }   
            else {
               fl_set_input (xfData->sceneFileInput, pov.input_file_name);
               flx_justify_input (xfData->sceneFileInput, pov.input_file_name, 
                   FL_NORMAL_STYLE, FL_NORMAL_SIZE);
	    }
	    break;
       
       case 1: /* Edit Scene button */
            temp = (char *) flx_filebox ( 
                CONFIG_POV_DIR, CONFIG_SCENE_FILTER, "");  

            if (strcmp(temp, "") == 0) 
               return;
            else
               run_program (2, CONFIG_POV_EDITOR, temp);
            break;
       
       case 2: /* View Image button */
            temp = (char *) flx_filebox ( 
                CONFIG_POV_DIR, CONFIG_IMAGE_FILTER, "");  

            if (strcmp(temp, "") == 0) 
               return;
            else
               run_program (2, CONFIG_DISPLAY_COMMAND, temp);
            break;

       /* Output File button */
/*
       case 3: 
            temp = pov.output_file_name;

            pov.output_file_name = (char *) flx_filebox ( 
                CONFIG_POV_DIR, CONFIG_SCENE_FILTER, "");  

            if (strcmp(pov.output_file_name, "") == 0) {
               pov.output_file_name = temp; 
               return;
            }   
            else
               fl_set_input (xfData->outFileInput, pov.output_file_name);
            break;
*/	    
        default: perror("xfpovray: GetFileCB"); 
	    break;

    } /* endswitch */          

}

 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void HelpCB(FL_OBJECT *ob, long data)
{
    int submenu;
    

    submenu = fl_get_menu(ob);

    switch (submenu) {
           case 1: fl_show_form (xfData->helpForm, 
                       FL_PLACE_CENTERFREE, FL_FULLBORDER, "xfpovray Help");
                   break;
           case 2: fl_show_form (xfData->aboutForm, 
                       FL_PLACE_CENTERFREE, FL_FULLBORDER, "xfpovray");
                   break;
           default: perror("xfpovray: HelpCB");
	           break;
    } 
    
 
} 

 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void LoadClipboardCB(FL_OBJECT *ob, long data)
{
    char *clipboardData;
    long type_unused = 0;
    

    switch (fl_get_menu(ob)) {
    
           /* Finite solids */
           case 11:  clipboardData = T_BLOB           ; break;
           case 12:  clipboardData = T_BOX            ; break;
           case 13:  clipboardData = T_CONE           ; break;
           case 14:  clipboardData = T_CYLINDER       ; break;
           case 15:  clipboardData = T_HEIGHT_FIELD   ; break;
           case 16:  clipboardData = T_JULIA_FRACTAL  ; break;
           case 17:  clipboardData = T_LATHE          ; break;
           case 18:  clipboardData = T_SPHERE         ; break;
           case 19:  clipboardData = T_SUPERELLIPSOID ; break;
           case 110: clipboardData = T_SOR            ; break;           
           case 111: clipboardData = T_TEXT           ; break;
           case 112: clipboardData = T_TORUS          ; break;

           /* Infinite solids */ 
           case 21:  clipboardData = T_XPLANE ; break;
           case 22:  clipboardData = T_YPLANE ; break;
           case 23:  clipboardData = T_ZPLANE ; break;
           case 24:  clipboardData = T_AXES   ; break;

           /* Lights */ 
           case 31:  clipboardData = T_LIGHT_POINT ; break;
           case 32:  clipboardData = T_LIGHT_SPOT  ; break;
           case 33:  clipboardData = T_LIGHT_CYL   ; break;
           case 34:  clipboardData = T_LIGHT_AREA  ; break;

           /* Cameras */ 
           case 41:  clipboardData = T_PERSPECTIVE_CAMERA  ; break;
           case 42:  clipboardData = T_ORTHOGRAPHIC_CAMERA ; break;
           case 43:  clipboardData = T_FISHEYE_CAMERA      ; break;
           case 44:  clipboardData = T_WIDE_ANGLE_CAMERA   ; break;
           case 45:  clipboardData = T_OMNIMAX_CAMERA      ; break;
           case 46:  clipboardData = T_PANORAMIC_CAMERA    ; break;

           /* Include files */ 
           case 51:  clipboardData = T_INC_CHARS   ; break;
           case 52:  clipboardData = T_INC_COLORS  ; break;
           case 53:  clipboardData = T_INC_CONSTS  ; break;
           case 54:  clipboardData = T_INC_FINISH  ; break;
           case 55:  clipboardData = T_INC_GLASS   ; break;
           case 56:  clipboardData = T_INC_GOLDS   ; break;
           case 57:  clipboardData = T_INC_METALS  ; break;
           case 58:  clipboardData = T_INC_SHAPES  ; break;
           case 59:  clipboardData = T_INC_SHAPES2 ; break;
           case 510: clipboardData = T_INC_SHAPESQ ; break;
           case 511: clipboardData = T_INC_SKIES   ; break;
           case 512: clipboardData = T_INC_STARS   ; break;
           case 513: clipboardData = T_INC_STONES  ; break;
           case 514: clipboardData = T_INC_STONES1 ; break;
           case 515: clipboardData = T_INC_STONES2 ; break;
           case 516: clipboardData = T_INC_TEXTURE ; break;
           case 517: clipboardData = T_INC_WOODMAP ; break;
           case 518: clipboardData = T_INC_WOODS   ; break;

           /* Atmospheric effects */ 
           case 61:  clipboardData = T_ATMOSPHERE ; break;
           case 62:  clipboardData = T_BACKGROUND ; break;
           case 63:  clipboardData = T_FOG        ; break;
           case 64:  clipboardData = T_SKY_SPHERE ; break;
           case 65:  clipboardData = T_RAINBOW    ; break;

           /* Global settings */ 
           case 71:  clipboardData = T_G_ADC_BAIL      ; break;
           case 72:  clipboardData = T_G_AMBIENT       ; break;
           case 73:  clipboardData = T_G_GAMMA         ; break;
           case 74:  clipboardData = T_G_HF_GRAY       ; break;
           case 75:  clipboardData = T_G_IRID          ; break;
           case 76:  clipboardData = T_G_MAX_TRACE     ; break;
           case 77:  clipboardData = T_G_MAX_INTERSECT ; break;
           case 78:  clipboardData = T_G_NUM_WAVES     ; break;
           case 79:  clipboardData = T_G_RADIOSITY     ; break;

           /* Scene template */ 
           case 8:  clipboardData = T_SCENE ; break;

           default: break;
    }        

    if (fl_stuff_clipboard(ob, type_unused, (void *) clipboardData, 
       strlen(clipboardData), nothingCB) <= 0) {        
       printf("LoadClipboard: Failed to load clipboard\n");  
    }

}
/* Required callback in the call to fl_stuff_clipboard :-( */
int nothingCB(FL_OBJECT *ob, long data)
{
    return (0);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void OptionsCB(FL_OBJECT *ob, long data)
{
    int item_selected = fl_get_menu(ob);
    FILE *ext;
    char temp[256];
    char buf[256];
    int i, pid, key;
    char *t;
 

    /* Configure */
    if (item_selected == 1) {
       if (!fl_form_is_visible (xfData->configuratorForm))
          fl_show_form (xfData->configuratorForm, 
              FL_PLACE_MOUSE, FL_FULLBORDER, "xfpovray Configurator");
       else
          return;
    }

    /* Save State */
    if (item_selected == 2) {
       save_state();
       return;
    }

    /* Restore State */
    if (item_selected == 3) {
       get_state(1);
       return;
    }

    /* Render popup */
    if (item_selected == 4) {
       if (fl_form_is_visible (xfData->renderPupForm)) {
          fl_set_menu_item_mode (xfData->optionsMenu, 4, FL_PUP_BOX);
          fl_hide_form (xfData->renderPupForm);       
       }
       else {
          fl_set_menu_item_mode (xfData->optionsMenu, 4, FL_PUP_CHECK);
          fl_show_form (xfData->renderPupForm, FL_PLACE_MOUSE, 
	      FL_FULLBORDER, "");
       } 
       return;
    }
        
    /* Tooltips */
/*
    if (item_selected == 5) {    
       if (show_tips) {
          fl_set_menu_item_mode (xfData->optionsMenu, 5, FL_PUP_BOX);
       }
       else {
          fl_set_menu_item_mode (xfData->optionsMenu, 5, FL_PUP_CHECK);
       } 
       show_tips = !show_tips;
       return;
    }
*/
    /* Auto image display */
    /* Edit state.c if menu item number changes */
    if (item_selected == 5) {    
       auto_view = !auto_view;
       return;
    }

    /* Check for external routines */
    sprintf(temp, "%s/.xfpovray/external.routines", getenv("HOME"));
    if ((ext = fopen(temp, "r")) != NULL) {
       
       key = 0;
       while (fgets(buf, sizeof(buf), ext) != NULL) {
             t = strtok (buf, " ");

             if (item_selected - data == key) {  
                run_program (1, t);  
                break;  /* out of while */
             }
             key++; 
       }
       fclose(ext);
    }
    
}


 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Write the config file, then invoke POV-ray */
void PovCB(FL_OBJECT *ob, long data)
{
    char command[256];
    int pid;
    FILE *stream_file;
    FILE *fatal_file;
           

    /* MB1: render; MB2, MB3 are used to hide the render popup form */
    if ((fl_get_button_numb(ob) == 2) || (fl_get_button_numb(ob) == 3)) {

       if (fl_form_is_visible (xfData->renderPupForm)) {
          fl_hide_form (xfData->renderPupForm);
          fl_set_menu_item_mode (xfData->optionsMenu, 4, FL_PUP_BOX);
       }
       return;        
    }
    
    /* Handles direct editting of the text input field */
    pov.input_file_name = (char *) fl_get_input (xfData->sceneFileInput);
    
    if ((pov.input_file_name == NULL) || 
        (strcmp(pov.input_file_name,"") == 0)) {

       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       if (flx_show_dialog ("question", 
          "You have not entered\na scene file. Select scene file now?") == 1) {

          pov.input_file_name = (char *) flx_filebox ( 
              CONFIG_POV_DIR, CONFIG_SCENE_FILTER, "");  
          if (strcmp(pov.input_file_name, "") == 0) 
             return;
          else {
             fl_set_input (xfData->sceneFileInput, pov.input_file_name);
             flx_justify_input (xfData->sceneFileInput, pov.input_file_name, 
                 FL_NORMAL_STYLE, FL_NORMAL_SIZE);
          }
       }
       else 
          return;
    }       

    if ((pov.display_during_render == 0) && (pov.output_to_file == 0)) {
       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       if (flx_show_dialog ("question", 
          "Display is off and file output is off.\nContinue rendering?") == 0)
	  return;
    }
    
    write_config(NULL);

    /* Execute POV-Ray as a child process so we can interrupt a 
    ** render by sending SIGINT. 
    */
    pid = fork();
    
    if (pid == 0) {
       /* In child process; invoke POV-Ray */

       /* Open new files for the current run of POV-Ray.  If these files
       ** have non-zero length when POV-Ray is done, display them
       */ 
       if ((stream_file = fopen(STREAM_FILE, "w")) == NULL) {
          perror("xfpovray: PovCB");
          exit (1);
       }
       fclose(stream_file);
       
       if ((fatal_file = fopen(FATAL_FILE, "w")) == NULL) {
          perror("xfpovray: PovCB");
          exit (1);
       }
       fclose(fatal_file);

       sprintf(command, "%s", CONFIG_POV_COMMAND);
       if (execlp (command, "xfpovray", POV_INIFILE, NULL) == -1) {
        
          printf ("Error executing command\n %s\n", command);
          perror ("xfpovray");
          /* sigchld catcher handles the error */
          _exit (255);
       }
    }
    else if (pid > 0) {
       /* In parent process */
       
       povray_pid = pid; 
                 
    }
    else {
        printf ("Error: PovCB: fork() failed!\n");
        perror ("xfpovray");
        exit (1);
    }

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
int PovCheckCB(XEvent *ev, void *data)
{
    int status, code;
    char temp[256];
    FILE *stream_file;
    FILE *fatal_file;

    
    /* Idle */
    if (povray_pid == 0) {
       return (0);
    }

    status = waitpid (povray_pid, &code, WNOHANG);

    /* If status == 0, POV-Ray is running; if status < 0, POV-Ray is
    ** done or has been aborted.  We need not be concerned with 'code'.
    */ 
    if (status < 0) {

       povray_pid = 0;

       /* If verbose output is on, display a Message box, which will 
       ** include any error messages, else display an Error box if needed.
       */
       if (pov.redirect_on == 1) {

	  if ((stream_file = fopen (STREAM_FILE, "r")) == NULL) {
             perror ("xfpovray: PovCheckCB");
             exit (1);
	  }
	  else {
             if (fgets (temp, sizeof(temp), stream_file) != NULL) {
 		fl_set_form_position (xfData->messageForm, 0, 0);
		fl_show_form (xfData->messageForm, FL_PLACE_POSITION, 
		    FL_TRANSIENT, "Messages");
		fl_load_browser (xfData->messageBrowser, STREAM_FILE);
             }
          fclose(stream_file);    
          }

       } /* endif */
       else {
       
	  if ((fatal_file = fopen (FATAL_FILE, "r")) == NULL) {
             perror ("xfpovray: PovCheckCB");
             exit (1);
	  }
	  else {
             if (fgets(temp, sizeof(temp), fatal_file) != NULL) {
               fl_set_form_position (xfData->fatalForm, 0, 0);
               fl_show_form (xfData->fatalForm, FL_PLACE_POSITION, 
        	   FL_TRANSIENT, "Error");
               fl_load_browser (xfData->fatalBrowser, FATAL_FILE);
             }
             fclose (fatal_file);    
	  }

       } /* endelse */

       /* Display image using viewer? */
/*       if ((auto_view == TRUE) && (pov.display_during_render == FALSE)) */
       if (auto_view == TRUE)
	  show_image();

    } /* endif */   


    return (0);
 
}
 


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void PreviewCB(FL_OBJECT *ob, long data)
{

    fl_show_form (xfData->previewForm, FL_PLACE_SIZE, 
        FL_FULLBORDER, "xfpovray");
 
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void PreviewDismissCB(FL_OBJECT *ob, long data)
{
    fl_hide_form (xfData->previewForm);

}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void PreviewRadioCB(FL_OBJECT *ob, long data)
{
    char temp_file[1024];
    char command[128];
    int i;
    
    char filter[256] = "%s \"#declare\" %s/%s | %s -d\" \" -f2 | %s 's/=//'| %s -v Inc_Temp > %s"; 
        /* grep cut set grep */
    
    char add_filter[256] = "%s \"#declare\" %s/%s | %s -d\" \" -f2 | %s 's/=//' | %s -v Inc_Temp >> %s"; 
        /* grep cut set grep */
     
    
    sprintf(temp_file, "%s", TEMP_FILE);    

    switch (data) {

           case 0: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "colors.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[0], PUSHED);
                   break;
           case 1: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "stones1.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button(xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[1], PUSHED);
                   break;
           case 2: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "woods.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[1], PUSHED);
                   break;
           case 3: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "glass.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[1], PUSHED);
                   break;
           case 4: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "metals.inc", CUT, SED, GREP, temp_file); 
                   break;
           case 5: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "golds.inc", CUT, SED, GREP, temp_file); 
                   break;
           case 6: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "finish.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[2], PUSHED);
                   break;
           case 7: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "skies.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[1], PUSHED);
                   break;
           case 8: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "shapes.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[3], PUSHED);
                   break;
           case 9: sprintf(command, filter, 
                       GREP, CONFIG_POV_LIB_PATH, "platon.inc", CUT, SED, GREP, temp_file); 
                   for (i=0; i<4; i++) 
                       fl_set_button (xfData->previewType[i], RELEASED);
                   fl_set_button (xfData->previewType[3], PUSHED);
                   break;
    }   
 
    system(command);

    /* stones uses a second include file */
    if (data == 1) {
       sprintf(temp_file, "%s", TEMP_FILE);    
       sprintf(command, add_filter, 
           GREP, CONFIG_POV_LIB_PATH, "stones2.inc", CUT, SED, GREP, temp_file); 
       system(command);
    }     
     
    /* shapes uses 3 files */
    if (data == 8) {
       sprintf(temp_file, "%s", TEMP_FILE);    
       sprintf(command, add_filter, 
           GREP, CONFIG_POV_LIB_PATH, "shapes2.inc", CUT, SED, GREP, temp_file); 
       system(command);
       sprintf(temp_file, "%s", TEMP_FILE);    
       sprintf(command, add_filter, 
           GREP, CONFIG_POV_LIB_PATH, "shapesq.inc", CUT, SED, GREP, temp_file); 
       system(command);
    }    

    /* skies/stars uses a second include file */
    if (data == 7) {
       sprintf(temp_file, "%s", TEMP_FILE);    
       sprintf(command, add_filter, 
           GREP, CONFIG_POV_LIB_PATH, "stars.inc", CUT, SED, GREP, temp_file); 
       system(command);
    }     

    fl_load_browser (xfData->previewBrowser, TEMP_FILE); 
 
  
}
     


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void QualityCB(FL_OBJECT *ob, long data)
{
    switch (data) {
        case 0: pov.quality = (int) fl_get_slider_value(ob);  break;
        case 1: pov.aa_depth = (int) fl_get_slider_value(ob); break;
    }
                   
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void QualityResetCB(FL_OBJECT *ob, long data)
{
    switch (data) {
           case 0: fl_set_slider_value (
	               tracingPage->qualitySlider, POV_QUALITY);
                   pov.quality = POV_QUALITY;
                   break;
           case 1: fl_set_slider_value (
	               tracingPage->antialiasSlider, POV_AA_DEPTH);
                   pov.aa_depth = POV_AA_DEPTH;
                   break;
    }

}
 

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void QuitCB(FL_OBJECT *ob, long data)
{ 
    int pid;
    char *temp;
    int menuItem = fl_get_menu(ob);

    
    switch (menuItem) {
    
        /* New */  
        case 1: run_program (1, CONFIG_POV_EDITOR); break;

        /* Open */ 
        case 2:  
            temp = (char *) flx_filebox ( 
                CONFIG_POV_DIR, CONFIG_SCENE_FILTER, "");  

            if (strcmp(temp, "") == 0)
               return;
            else {
               pov.input_file_name = temp;

               fl_set_input (xfData->sceneFileInput, pov.input_file_name);
               flx_justify_input (xfData->sceneFileInput, pov.input_file_name, 
                   FL_NORMAL_STYLE, FL_NORMAL_SIZE);

               run_program (2, CONFIG_POV_EDITOR, pov.input_file_name);	       
            }
	    break;

        /* Quit */ 
        case 3:
	    fl_finish();
            printf ("\n");
	    exit (0);
	    break;
    
    }
    
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Resets Camera, Camera LookAt, or Light on the preview widget */
void ResetPreviewCB(FL_OBJECT *ob, long data)
{
    char temp[256];

                    
    switch (data) {
           
	   case 0: preview_setup.camera[0] = preview_setup_def.camera[0];
                   preview_setup.camera[1] = preview_setup_def.camera[1];
                   preview_setup.camera[2] = preview_setup_def.camera[2];
           
                   sprintf(temp, "%4.1f", preview_setup.camera[0]);
                   fl_set_input (xfData->previewCamera[0], temp);
                   sprintf(temp, "%4.1f", preview_setup.camera[1]);
                   fl_set_input (xfData->previewCamera[1], temp);
                   sprintf(temp, "%4.1f", preview_setup.camera[2]);
                   fl_set_input (xfData->previewCamera[2], temp);
                   break;
           
	   case 1: preview_setup.camera_lookat[0] = 
                       preview_setup_def.camera_lookat[0];
                   preview_setup.camera_lookat[1] = 
                       preview_setup_def.camera_lookat[1];
                   preview_setup.camera_lookat[2] = 
                       preview_setup_def.camera_lookat[2];
           
                   sprintf(temp, "%4.1f", preview_setup.camera_lookat[0]);
                   fl_set_input (xfData->previewCameraLookat[0], temp);
                   sprintf(temp, "%4.1f", preview_setup.camera_lookat[1]);
                   fl_set_input (xfData->previewCameraLookat[1], temp);
                   sprintf(temp, "%4.1f", preview_setup.camera_lookat[2]);
                   fl_set_input (xfData->previewCameraLookat[2], temp);
                   break;
           
	   case 2: preview_setup.light[0] = preview_setup_def.light[0];
                   preview_setup.light[1] = preview_setup_def.light[1];
                   preview_setup.light[2] = preview_setup_def.light[2];
           
                   sprintf(temp, "%4.1f", preview_setup.light[0]);
                   fl_set_input (xfData->previewLight[0], temp);
                   sprintf(temp, "%4.1f", preview_setup.light[1]);
                   fl_set_input (xfData->previewLight[1], temp);
                   sprintf(temp, "%4.1f", preview_setup.light[2]);
                   fl_set_input (xfData->previewLight[2], temp);
                   break;
 	   
	   default: fprintf(stderr, "xfpovray: ResetPreviewCB case error\n");
                   break;   
    }


}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Set font for the Help widget */
void SetHelpFontCB(FL_OBJECT *ob, long data)
{
    switch (data) {

           case 0: fl_set_browser_fontsize(xfData->helpBrowser, FL_NORMAL_SIZE);
                   break;

           case 1: fl_set_browser_fontsize(xfData->helpBrowser, FL_MEDIUM_SIZE);
                   break;

           case 2: fl_set_browser_fontsize(xfData->helpBrowser, FL_LARGE_SIZE);
                   break;

 	   default: fprintf(stderr, "xfpovray: SetHelpFontCB case error\n");
                   break;
    }      

}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* Set image height or width */
void SetImageSizeCB(FL_OBJECT *ob, long data)
{
    double current_ratio;
    int    do_constrain;
    char   temp[10];


    do_constrain  = fl_get_button (outputPage->constrainRatio);
    current_ratio = (float) pov.height / (float) pov.width; 
  
    switch (data) {

       case 0:
	   if (ob->objclass == FL_MENU)
              fl_set_input (outputPage->imageHeightInput, fl_get_menu_text(ob));

	   if (strcmp(fl_get_input (outputPage->imageHeightInput), "") == 0) {
	      fl_set_input (outputPage->imageHeightInput, CONFIG_POV_HEIGHT);
	      pov.height = atoi(CONFIG_POV_HEIGHT);
	   }
	   else
	      pov.height = atoi(fl_get_input (outputPage->imageHeightInput)); 

	   if (do_constrain) {
              pov.width = (int) ( (float) pov.height / (float) current_ratio );
              sprintf(temp, "%d", pov.width);
              fl_set_input (outputPage->imageWidthInput, temp);
	   }         
           break;
          
       case 1:
	   if (ob->objclass == FL_MENU)
              fl_set_input (outputPage->imageWidthInput, fl_get_menu_text(ob));
	      
	   if (strcmp(fl_get_input (outputPage->imageWidthInput), "") == 0) {
	      fl_set_input (outputPage->imageWidthInput, CONFIG_POV_WIDTH);
	      pov.width = atoi(CONFIG_POV_WIDTH);
	   }
	   else
	      pov.width = atoi(fl_get_input (outputPage->imageWidthInput));    
 
	   if (do_constrain) {
	      pov.height = (int) ( (float) pov.width * (float) current_ratio );
 	      sprintf(temp, "%d", pov.height);
	      fl_set_input (outputPage->imageHeightInput, temp);
	   } 
           break; 

       case 2:
           fl_set_input (outputPage->imageWidthInput, CONFIG_POV_WIDTH);
           pov.width = atoi(CONFIG_POV_WIDTH);
           break;

       case 3:      
           fl_set_input (outputPage->imageHeightInput, CONFIG_POV_HEIGHT);
           pov.height = atoi(CONFIG_POV_HEIGHT);
           break;
    
       default: fprintf(stderr, "xfpovray: SetImageSizeCB case error\n");
           break;
	    
    } /* endswitch */
 
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void SetMosaicCB(FL_OBJECT *ob, long data)
{

    if (data == 0)
       fl_set_input (outputPage->mosaicStart, fl_get_menu_text(ob));
    else 
    if (data == 1)
       fl_set_input (outputPage->mosaicEnd, fl_get_menu_text(ob));
   
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void ToggleFolderCB(FL_OBJECT *ob, long data)
{
    if (strcmp(ob->label, "Render Options") == 0) {
    
       fl_set_object_label (ob, "Hide Options");
       fl_show_form (xfData->folderForm, 
           FL_PLACE_SIZE, FL_FULLBORDER, "Render Options");

    } else {

       fl_set_object_label (ob, "Render Options");
       fl_hide_form (xfData->folderForm);
        
    }

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
/* 
** Callback for choice buttons
*/
void YesNoCB(FL_OBJECT *ob, long data)
{
    char *buf = 0;
    char val[16];
        
    
    /* 
    ** case statement for strings
    */
    while (TRUE) {

	if (strcmp(ob->label, "Display During Render") == 0) {
	   pov.display_during_render = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Verbose") == 0) {
           switch (fl_get_choice(ob) - 1) {
 	   	  case 0: pov.verbose = 0;
 	   		  pov.redirect_on = 0;
 	   		  break;
 	   	  case 1: pov.verbose = 1;
 	   		  pov.redirect_on = 0;
 	   		  break;
 	   	  case 2: pov.verbose = 2;
 	   		  pov.redirect_on = 1;
 	   		  break;
 	   	  default: fprintf(stderr, "xfpovray: YesNoCB (%s)", ob->label);
 	                  break;  
	   }
	   break;
	}
	else
	if (strcmp(ob->label, "Buffer Output") == 0) {
	   pov.buffer_output = fl_get_choice(ob) - 1;
           if (pov.buffer_output) {
              sprintf(val, "%d", pov.buffer_size);
              flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
	      buf = (char *) flx_show_input (FL_INT_INPUT,
	          "Enter buffer size (kilobytes)", val);
	      if (strcmp(buf, "") != 0)
                 pov.buffer_size = atoi(buf);
	   }   
	   break;
	}
	else
	if (strcmp(ob->label, "Antialias") == 0) {
	   pov.antialias = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Radiosity") == 0) {
	   pov.radiosity = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Output Image to File") == 0) {
	   pov.output_to_file = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Language Version") == 0) {
	   pov.version = fl_get_choice(ob);
	   break;
	}
	else
	if (strcmp(ob->label, "Remove Bounds") == 0) {
	   pov.remove_bounds = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Split Unions") == 0) {
	   pov.split_unions = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Bounding") == 0) {
	   pov.bounding = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Light Buffer") == 0) {
	   pov.light_buffer = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Vista Buffer") == 0) {
	   pov.vista_buffer = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Method") == 0) {
	   pov.aa_sampling_method = fl_get_choice(ob);
	   break;
	}
	else
	if (strcmp(ob->label, "Cyclic Animation") == 0) {
	   pov.cyclic_animation = fl_get_choice(ob) - 1;
	   break;
	}
	else
	if (strcmp(ob->label, "Field Rendering") == 0) {
	   switch (fl_get_choice(ob) - 1) {
        	  case 0: pov.field_rendering = 0;
                	  break;
        	  case 1: pov.field_rendering = 1;
                	  pov.odd_field = 0;
                	  break;
        	  case 2: pov.field_rendering = 1;
                	  pov.odd_field = 1;
                	  break;
        	  default: fprintf(stderr, "xfpovray: YesNoCB: (%s)",ob->label);
           } 
	   break;
	}
	else
	if (strcmp(ob->label, "Palette") == 0) {
	   pov.palette = fl_get_choice(ob);
	   break;
	}
	else
	if (strcmp(ob->label, "File Type") == 0) {
	   pov.output_filetype = fl_get_choice(ob);
	   break;
	}
	else
	if (strcmp(ob->label, "Jitter") == 0) {
	   pov.jitter = fl_get_choice(ob) - 1;
	   break;
	}
	
	fprintf(stderr, "xfpovray: YesNoCB case error\n");
	break;

    } /* endwhile */

}
