/*
** xfpovray ini file functions
*/

#include "forms.h"
#include "xfpovray.h"
#include "pov-defaults.h"

extern char *xfp_version;
extern struct pov_options pov;

extern char *POV_INIFILE;
extern char *STREAM_FILE;
extern char *FATAL_FILE;

extern char *CONFIG_POV_WIDTH; 
extern char *CONFIG_POV_HEIGHT;

extern XFDATA_S *xfData;
extern PARSING_PAGE_S   *parsingPage;
extern OUTPUT_PAGE_S	*outputPage;
extern TRACING_PAGE_S   *tracingPage;
extern ANIMATION_PAGE_S *animationPage;
extern CONFIG_PAGE_S	*configPage;

 
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* 
** Function to write the POV-Ray configuration file 
*/
void write_config(char *ini_file)
{
    FILE *options_file;
    int i, do_free = FALSE;
   
    
    if (ini_file == NULL) {
       ini_file = (char *) fl_calloc (1, strlen(POV_INIFILE) + 1);
       sprintf(ini_file, "%s", POV_INIFILE);
       do_free = TRUE;
    }
              
    if ((options_file = fopen(ini_file, "w")) == NULL) {
       printf("xfpovray: write_config\n");
       printf("  Could not write config file %s\n", ini_file);
       exit (1);
    }


    fprintf(options_file, "; Generated with xfpovray %s\n", xfp_version);    

    fprintf(options_file, "version=%3.1f\n", (float) pov.version);    
    if (pov.input_file_name == NULL)  
       fprintf(options_file, "input_file_name=NULL\n");
    else
       fprintf(options_file, "input_file_name=%s\n", pov.input_file_name);
    
    /* Library path */
    for (i = 0; i < 5; i++) {
        if (strcmp(fl_get_input (parsingPage->libpathInput[i]), "") != 0)
           fprintf(options_file, "library_path=%s\n", 
               fl_get_input (parsingPage->libpathInput[i]));
    }
               
    /* 
    ** Image height and width. 
    ** Must parse the input boxes in case the user 
    ** made an entry by clicking in the input box instead
    ** of using the pulldown menu.
    */
    if (strcmp(fl_get_input (outputPage->imageHeightInput), "") == 0) {
       fl_set_input (outputPage->imageHeightInput, CONFIG_POV_HEIGHT);
       pov.height = atoi(CONFIG_POV_HEIGHT);
    }
    else
       pov.height = atoi(fl_get_input (outputPage->imageHeightInput));

    if (strcmp(fl_get_input (outputPage->imageWidthInput), "") == 0) {
       fl_set_input (outputPage->imageWidthInput, CONFIG_POV_WIDTH);
       pov.width = atoi(CONFIG_POV_WIDTH);
    }
    else
       pov.width = atoi(fl_get_input (outputPage->imageWidthInput)); 
       
    fprintf(options_file, "height=%d\n", pov.height); 
    fprintf(options_file, "width=%d\n", pov.width); 

    fprintf(options_file, "verbose=%d\n", pov.verbose);

    fprintf(options_file, "buffer_output=%d\n", pov.buffer_output);
    if (pov.buffer_output)
       fprintf(options_file, "buffer_size=%d\n", pov.buffer_size);

    /* 
    ** Redirection options.                   
    ** Turn off redirect to console; redirect to file if selected,
    ** then display file in message browser.
    */
    fprintf(options_file, "all_console=%d\n", 0);
    if (pov.redirect_on == 1)
       fprintf(options_file, "all_file=%s\n", STREAM_FILE); 
    else
       fprintf(options_file, "fatal_file=%s\n", FATAL_FILE);
 
    /* Partial image height and width; parse the input boxes */ 
    if (strcmp(fl_get_input (outputPage->columnStart), "") != 0)
       fprintf(options_file, "start_column=%s\n", 
       fl_get_input (outputPage->columnStart)); 
    if (strcmp(fl_get_input (outputPage->columnEnd), "") != 0)
       fprintf(options_file, "end_column=%s\n", 
       fl_get_input (outputPage->columnEnd)); 
    if (strcmp(fl_get_input (outputPage->rowStart), "") != 0)
       fprintf(options_file, "start_row=%s\n", 
       fl_get_input (outputPage->rowStart)); 
    if (strcmp(fl_get_input (outputPage->rowEnd), "") != 0)
       fprintf(options_file, "end_row=%s\n", 
       fl_get_input (outputPage->rowEnd)); 

    /* Mosaic preview; parse the input boxes */
    if (strcmp(fl_get_input (outputPage->mosaicStart), "") != 0)
       fprintf(options_file, "preview_start_size=%s\n", 
               fl_get_input (outputPage->mosaicStart)); 
    if (strcmp(fl_get_input (outputPage->mosaicEnd), "") != 0)
       fprintf(options_file, "preview_end_size=%s\n", 
               fl_get_input (outputPage->mosaicEnd)); 
 
  
    fprintf(options_file, "quality=%d\n", pov.quality); 

    fprintf(options_file, "antialias=%d\n", pov.antialias); 
    fprintf(options_file, "antialias_depth=%d\n", pov.aa_depth); 
    fprintf(options_file, "sampling_method=%d\n", pov.aa_sampling_method); 
    fprintf(options_file, "antialias_threshold=%s\n", 
        fl_get_input (tracingPage->antialiasThreshold)); 

    fprintf(options_file, "radiosity=%d\n", pov.radiosity); 

    fprintf(options_file, "display=%d\n", pov.display_during_render);
        
    /* If displaying during a render, pause to view image */    
    if (pov.display_during_render == 1)
       fprintf(options_file, "pause_when_done=%d\n", 1); 

    fprintf(options_file, "output_to_file=%d\n", pov.output_to_file);

    fprintf(options_file, "remove_bounds=%d\n", pov.remove_bounds);

    fprintf(options_file, "split_unions=%d\n", pov.split_unions); 

    fprintf(options_file, "bounding=%d\n", pov.bounding);
    
    if (strcmp(fl_get_input (tracingPage->boundingThreshold), "") != 0)
       pov.bounding_threshold = 
           atoi(fl_get_input (tracingPage->boundingThreshold));
    else 
       pov.bounding_threshold = POV_BOUNDING_THRESHOLD; 
    fprintf(options_file, "bounding_threshold=%d\n", pov.bounding_threshold); 

    fprintf(options_file, "light_buffer=%d\n", pov.light_buffer);

    fprintf(options_file, "vista_buffer=%d\n", pov.vista_buffer);

    fprintf(options_file, "clock=%s\n", 
        fl_get_input (animationPage->clockExternalInput)); 

    fprintf(options_file, "initial_clock=%s\n", 
        fl_get_input (animationPage->clockStartInput)); 

    fprintf(options_file, "final_clock=%s\n", 
        fl_get_input (animationPage->clockEndInput)); 
 
    pov.initial_frame = atoi(fl_get_input (animationPage->frameStartInput));
    fprintf(options_file, "initial_frame=%d\n", pov.initial_frame);
      
    pov.final_frame = atoi(fl_get_input (animationPage->frameEndInput));
    fprintf(options_file, "final_frame=%d\n", pov.final_frame);

    if (strcmp(fl_get_input (animationPage->subsetFrameStartInput), "") != 0) {
       fprintf(options_file, "subset_start_frame=%s\n", 
           fl_get_input (animationPage->subsetFrameStartInput));
    }
    if (strcmp(fl_get_input (animationPage->subsetFrameEndInput), "") != 0) {
       fprintf(options_file, "subset_end_frame=%s\n", 
           fl_get_input (animationPage->subsetFrameEndInput));
    }
       
    fprintf(options_file, "cyclic_animation=%d\n", pov.cyclic_animation);

    fprintf(options_file, "field_render=%d\n", pov.field_rendering);
    if (pov.field_rendering == 1)
       fprintf(options_file, "odd_field=%d\n", pov.odd_field);

    switch (pov.palette) {
           case 1: fprintf(options_file, "palette=T\n");
                   break;
           case 2: fprintf(options_file, "palette=H\n");
                   break;
           case 3: fprintf(options_file, "palette=G\n");
                   break;
    }           

    switch (pov.output_filetype) {
           case 1: fprintf(options_file, "output_file_type=T\n");
                   break;
           case 2: fprintf(options_file, "output_file_type=C\n");
                   break;
           case 3: fprintf(options_file, "output_file_type=N\n");
                   break;
           case 4: fprintf(options_file, "output_file_type=P\n");
                   break;
           case 5: fprintf(options_file, "output_file_type=S\n");
                   break;
    }           

    fprintf(options_file, "jitter=%d\n", pov.jitter); 

    fprintf(options_file, "jitter_amount=%s\n", 
        fl_get_input (tracingPage->jitterInput)); 


    fclose(options_file);

    /*
    ** Update the config folder
    */
    fl_freeze_form (configPage->form);
    fl_clear_browser (configPage->configBrowser);
    fl_load_browser (configPage->configBrowser, ini_file);    
    fl_unfreeze_form (configPage->form);
    
    if (do_free)
       free (ini_file);

 
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* 
** Function to read the POV-Ray configuration file, and update widget values 
*/
int read_config(char *ini_file)
{

    FILE *fh;
    int i, numLibs = 0;
    char buf[80];
    char *key;
    char *value;
    const char delimiter[] = "=";     

   
              
    if ((fh = fopen(ini_file, "r")) == NULL) {
       fprintf (stderr, "xfpovray: read_config\n");
       fprintf (stderr, "  Could not read config file %s\n", ini_file);
       return (1);
    }

    /* Read the header */
    fgets (buf, sizeof(buf), fh);
    if (strstr(buf, "xfpovray") == NULL) {
       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       flx_show_dialog ("error", 
           "File not written by xfpovray.\nCannot load ini file.");
       return (1);
    }
    

    /* Clear library paths */
    for (i = 0; i < 5; i++) 
	fl_set_input (parsingPage->libpathInput[i], ""); 
	
    while (fgets(buf, sizeof(buf), fh) != NULL) {

          /* Get the key/value pair */
          key = strtok (buf, delimiter);
          value = strtok (NULL, delimiter);
          strtruncate(value, '\n');
	         
		 
          if (strcmp(key, "version") == 0) {
	     pov.version = atoi(value);
             fl_set_choice (parsingPage->languageVersion, pov.version);
	     continue;
          }
          if (strcmp(key, "input_file_name") == 0) {
             if (strcmp(value, "NULL") != 0) {
	        fl_set_input (xfData->sceneFileInput, value);
	        flx_justify_input (xfData->sceneFileInput,  value,
	            FL_NORMAL_STYLE, FL_NORMAL_SIZE); 
		pov.input_file_name = 
		    (char *) fl_get_input (xfData->sceneFileInput);
	     }
	     continue;
          }
          if (strcmp(key, "library_path") == 0) {
	     fl_set_input (parsingPage->libpathInput[numLibs], value); 
	     numLibs++;
	     continue;
          }
          if (strcmp(key, "height") == 0) {
             pov.height = atoi(value);
             fl_set_input (outputPage->imageHeightInput, value);
	     continue;
          }
          if (strcmp(key, "width") == 0) {
             pov.width = atoi(value);
             fl_set_input (outputPage->imageWidthInput, value);
	     continue;
          }
          if (strcmp(key, "verbose") == 0) {
             pov.verbose = atoi(value);
	     pov.redirect_on = (pov.verbose > 1 ? 1 : 0);
             fl_set_choice (outputPage->verbosity, pov.verbose + 1);
	     continue;
          }
          if (strcmp(key, "buffer_output") == 0) {
             pov.buffer_output = atoi(value);
             fl_set_choice (outputPage->bufferOutput, pov.buffer_output + 1);
	     continue;
          }
          if (strcmp(key, "buffer_size") == 0) {
             pov.buffer_size = atoi(value);
	     continue;
          }
          if (strcmp(key, "all_console") == 0) {
	     continue;
          }
          if (strcmp(key, "fatal_file") == 0) {
	     continue;
          }
          if (strcmp(key, "all_file") == 0) {
	     continue;
          }
          if (strcmp(key, "start_column") == 0) {
	     fl_set_input (outputPage->columnStart, value);
	     continue;
          }
          if (strcmp(key, "end_column") == 0) {
	     fl_set_input (outputPage->columnEnd, value);
	     continue;
          }
          if (strcmp(key, "start_row") == 0) {
	     fl_set_input (outputPage->rowStart, value);
	     continue;
          }
          if (strcmp(key, "end_row") == 0) {
	     fl_set_input (outputPage->rowEnd, value);
	     continue;
          }
          if (strcmp(key, "preview_start_size") == 0) {
	     fl_set_input (outputPage->mosaicStart, value);
	     continue;
          }
          if (strcmp(key, "preview_end_size") == 0) {
	     fl_set_input (outputPage->mosaicEnd, value);
	     continue;
          }
          if (strcmp(key, "quality") == 0) {
	     pov.quality = atoi(value);
	     fl_set_slider_value (tracingPage->qualitySlider, pov.quality);
	     continue;
          }
          if (strcmp(key, "antialias") == 0) {
             pov.antialias = atoi(value);
             fl_set_choice (tracingPage->antialias, pov.antialias + 1);
	     continue;
          }
          if (strcmp(key, "antialias_depth") == 0) {
             pov.aa_depth = atoi(value);
	     fl_set_slider_value (tracingPage->antialiasSlider, pov.aa_depth);
	     continue;
          }
          if (strcmp(key, "sampling_method") == 0) {
	     pov.aa_sampling_method = atoi(value);
             fl_set_choice (tracingPage->antialiasMethod, 
	         pov.aa_sampling_method);
	     continue;
          }
          if (strcmp(key, "antialias_threshold") == 0) {
	     fl_set_input (tracingPage->antialiasThreshold, value);
	     continue;
          }	  
          if (strcmp(key, "radiosity") == 0) {
	     pov.radiosity = atoi(value);
             fl_set_choice (tracingPage->radiosity, pov.radiosity + 1);
	     continue;
          }
          if (strcmp(key, "display") == 0) {
	     pov.display_during_render = atoi(value);
             fl_set_choice (outputPage->displayDuringRender, 
	         pov.display_during_render + 1);
	     continue;
          }
          if (strcmp(key, "pause_when_done") == 0) {
	     continue;
          }
          if (strcmp(key, "output_to_file") == 0) {
	     pov.output_to_file = atoi(value);
             fl_set_choice (outputPage->imageToFile, pov.output_to_file + 1);
	     continue;
          }
          if (strcmp(key, "remove_bounds") == 0) {
	     pov.remove_bounds = atoi(value);
             fl_set_choice (parsingPage->removeBounds, pov.remove_bounds + 1);
	     continue;
          }
          if (strcmp(key, "split_unions") == 0) {
	     pov.split_unions = atoi(value);
             fl_set_choice (parsingPage->splitUnions, pov.split_unions + 1);
	     continue;
          }
          if (strcmp(key, "bounding") == 0) {
	     pov.bounding = atoi(value);
             fl_set_choice (tracingPage->bounding, pov.bounding + 1);
	     continue;
          }
          if (strcmp(key, "bounding_threshold") == 0) {
	     pov.bounding_threshold = atoi(value);
             fl_set_input (tracingPage->boundingThreshold, value);
	     continue;
          }
          if (strcmp(key, "light_buffer") == 0) {
	     pov.light_buffer = atoi(value);
             fl_set_choice (tracingPage->lightBuffer, pov.light_buffer + 1);
	     continue;
          }
          if (strcmp(key, "vista_buffer") == 0) {
	     pov.vista_buffer = atoi(value);
             fl_set_choice (tracingPage->vistaBuffer, pov.vista_buffer + 1);
	     continue;
          }
          if (strcmp(key, "clock") == 0) {
             fl_set_input (animationPage->clockExternalInput, value);
	     continue;
          }
          if (strcmp(key, "initial_clock") == 0) {
             fl_set_input (animationPage->clockStartInput, value);
	     continue;
          }
          if (strcmp(key, "final_clock") == 0) {
             fl_set_input (animationPage->clockEndInput, value);
	     continue;
          }
          if (strcmp(key, "initial_frame") == 0) {
	     pov.initial_frame = atoi(value);
             fl_set_input (animationPage->frameStartInput, value);
	     continue;
          }
          if (strcmp(key, "final_frame") == 0) {
	     pov.final_frame = atoi(value);
             fl_set_input (animationPage->frameEndInput, value);
	     continue;
          }
          if (strcmp(key, "subset_start_frame") == 0) {
             fl_set_input (animationPage->subsetFrameStartInput, value);
	     continue;
          }
          if (strcmp(key, "subset_end_frame") == 0) {
             fl_set_input (animationPage->subsetFrameEndInput, value);
	     continue;
          }
          if (strcmp(key, "cyclic_animation") == 0) {
	     pov.cyclic_animation = atoi(value);
             fl_set_choice (animationPage->cyclicAnimation, 
	         pov.cyclic_animation + 1);	     
	     continue;
          }
          if (strcmp(key, "field_render") == 0) {
	     pov.field_rendering = atoi(value);
             fl_set_choice (animationPage->fieldRendering, 
	         pov.field_rendering + 1);	     
	     continue;
          }
          if (strcmp(key, "odd_field") == 0) {
	     pov.odd_field = atoi(value);
	     continue;
          }
          if (strcmp(key, "palette") == 0) {
             if (strcmp(value, "T") == 0) 
	        pov.palette = 1;
             else if (strcmp(value, "H") == 0)
	        pov.palette = 2;
             else if (strcmp(value, "G") == 0)
	        pov.palette = 3;
	     fl_set_choice (outputPage->palette, pov.palette);
	     continue;
          }
          if (strcmp(key, "output_file_type") == 0) {
             if (strcmp(value, "T") == 0) 
	        pov.output_filetype = 1;
             else if (strcmp(value, "C") == 0)
	        pov.output_filetype = 2;
             else if (strcmp(value, "N") == 0)
	        pov.output_filetype = 3;
             else if (strcmp(value, "P") == 0)
	        pov.output_filetype = 4;
             else if (strcmp(value, "S") == 0)
	        pov.output_filetype = 5;
             fl_set_choice (outputPage->fileType, pov.output_filetype);
	     continue;
          }
          if (strcmp(key, "jitter") == 0) {
	     pov.jitter = atoi(value);
             fl_set_choice (tracingPage->jitter, pov.jitter + 1);
	     continue;
          }
          if (strcmp(key, "jitter_amount") == 0) {
             fl_set_input (tracingPage->jitterInput, value);
	     continue;
          }
    
    }
    	  
    fclose (fh);
    
    
    return(0);
}
