/* 
** Saves and restores the state of xfpovray.
** State is stored in $HOME/.xfpovray/xfpovray.state
*/


#include "forms.h"
#include <stdlib.h>
#include "xfpovray.h"
#include "pov-defaults.h"

extern char *xfp_version;
extern int auto_view;
extern struct pov_options pov;

extern char *CONFIG_POV_COMMAND;
extern char *CONFIG_POV_LIB_PATH;
extern char *CONFIG_POV_EDITOR;
extern char *CONFIG_DISPLAY_COMMAND;
extern char *CONFIG_POV_DIR;
extern char *CONFIG_SCENE_FILTER;
extern char *CONFIG_IMAGE_FILTER;
extern char *CONFIG_POV_WIDTH; 
extern char *CONFIG_POV_HEIGHT;

extern XFDATA_S *xfData;
extern PARSING_PAGE_S   *parsingPage;
extern OUTPUT_PAGE_S	*outputPage;
extern TRACING_PAGE_S   *tracingPage;
extern ANIMATION_PAGE_S *animationPage;
extern CONFIG_PAGE_S	*configPage;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void save_state (void)
{
    FILE *state_file;
    char filename[256];
    char temp[256];
    int i;
   
       
    /* Get home directory, and set filepath */
    sprintf (filename, "%s/.xfpovray/xfpovray.state", getenv("HOME"));

    if ((state_file = fopen(filename, "w")) == NULL) {
       fprintf (stderr, "Unable to open %s for writing! State not saved.\n",
               filename);
       return;
    }


    /* Write the current state */
    sprintf(temp, "; xfpovray %s, generated by state.c\n", xfp_version); 
    fprintf (state_file, temp);


    /* Parsing */
    
    for (i = 0; i < 5; i++) {
        if (strcmp(fl_get_input (parsingPage->libpathInput[i]), "") != 0)
           fprintf (state_file, "library_path=%s\n", 
                   fl_get_input (parsingPage->libpathInput[i]));    
        else
           fprintf (state_file, "library_path=NULL\n", 
                   fl_get_input (parsingPage->libpathInput[i]));    
    }
    
    fprintf (state_file, "version=%d\n", pov.version);
    fprintf (state_file, "remove_bounds=%d\n", pov.remove_bounds);
    fprintf (state_file, "split_unions=%d\n", pov.split_unions);


    /* Output */

    if (strcmp(fl_get_input(outputPage->imageHeightInput), "") == 0)
       fprintf (state_file, "height=%d\n", POV_HEIGHT);
    else
       fprintf (state_file, "height=%d\n", pov.height);
 
    if (strcmp(fl_get_input(outputPage->imageWidthInput), "") == 0)
       fprintf (state_file, "width=%d\n", POV_WIDTH);
    else
       fprintf (state_file, "width=%d\n", pov.width);

    if (strcmp(fl_get_input(outputPage->mosaicStart), "") == 0)
       fprintf (state_file, "preview_start_size=NULL\n");
    else
       fprintf (state_file, "preview_start_size=%s\n", 
           fl_get_input (outputPage->mosaicStart));

    if (strcmp(fl_get_input(outputPage->mosaicEnd), "") == 0)
       fprintf (state_file, "preview_end_size=NULL\n");
    else
       fprintf (state_file, "preview_end_size=%s\n", 
           fl_get_input (outputPage->mosaicEnd));

    if (strcmp(fl_get_input(outputPage->columnStart), "") == 0)
       fprintf (state_file, "start_column=NULL\n");
    else
       fprintf (state_file, "start_column=%s\n", 
           fl_get_input (outputPage->columnStart));

    if (strcmp(fl_get_input(outputPage->columnEnd), "") == 0)
       fprintf (state_file, "end_column=NULL\n");
    else
       fprintf (state_file, "end_column=%s\n", 
           fl_get_input (outputPage->columnEnd));

    if (strcmp(fl_get_input(outputPage->rowStart), "") == 0)
       fprintf (state_file, "start_row=NULL\n");
    else
       fprintf (state_file, "start_row=%s\n", 
           fl_get_input (outputPage->rowStart));

    if (strcmp(fl_get_input(outputPage->rowEnd), "") == 0)
       fprintf (state_file, "end_row=NULL\n");
    else
       fprintf (state_file, "end_row=%s\n", 
           fl_get_input (outputPage->rowEnd));
  
    fprintf (state_file, "display=%d\n", pov.display_during_render);
    fprintf (state_file, "output_to_file=%d\n", pov.output_to_file);
    fprintf (state_file, "buffer_output=%d\n", pov.buffer_output);
    fprintf (state_file, "buffer_size=%d\n", pov.buffer_size);
    fprintf (state_file, "palette=%d\n", pov.palette);
    fprintf (state_file, "output_file_type=%d\n", pov.output_filetype);
    fprintf (state_file, "verbose=%d\n", pov.verbose);


    /* Tracing */ 

    fprintf (state_file, "quality=%d\n", pov.quality);
    fprintf (state_file, "antialias_depth=%d\n", pov.aa_depth);
    fprintf (state_file, "antialias=%d\n", pov.antialias);
    fprintf (state_file, "sampling_method=%d\n", pov.aa_sampling_method);
    fprintf (state_file, "radiosity=%d\n", pov.radiosity);
    fprintf (state_file, "bounding=%d\n", pov.bounding);
    fprintf (state_file, "light_buffer=%d\n", pov.light_buffer);
    fprintf (state_file, "vista_buffer=%d\n", pov.vista_buffer);
    fprintf (state_file, "jitter=%d\n", pov.jitter);

    if (strcmp(fl_get_input (tracingPage->antialiasThreshold), "") == 0)
       fprintf (state_file, "antialias_threshold=%d\n", 
           POV_ANTIALIAS_THRESHOLD);
    else
       fprintf (state_file, "antialias_threshold=%s\n", 
           fl_get_input (tracingPage->antialiasThreshold));
  
    if (strcmp(fl_get_input (tracingPage->boundingThreshold), "") == 0)
       fprintf (state_file, "bounding_threshold=%d\n", POV_BOUNDING_THRESHOLD);
    else
       fprintf (state_file, "bounding_threshold=%s\n", 
           fl_get_input (tracingPage->boundingThreshold));
  
    if (strcmp(fl_get_input (tracingPage->jitterInput), "") == 0)
       fprintf (state_file, "jitter_amount=%d\n", POV_JITTER_AMOUNT);
    else
       fprintf (state_file, "jitter_amount=%s\n", 
           fl_get_input (tracingPage->jitterInput));
 
    /* Animation */ 

    if (strcmp(fl_get_input (animationPage->frameStartInput), "") == 0)
       fprintf (state_file, "initial_frame=%d\n", POV_INITIAL_FRAME);
    else
       fprintf (state_file, "initial_frame=%s\n",
           fl_get_input (animationPage->frameStartInput));

    if (strcmp(fl_get_input (animationPage->frameEndInput), "") == 0)
       fprintf(state_file, "final_frame=%d\n", POV_FINAL_FRAME);
    else
       fprintf (state_file, "final_frame=%s\n", 
           fl_get_input (animationPage->frameEndInput));

    if (strcmp(fl_get_input (animationPage->subsetFrameStartInput), "") == 0)
       fprintf (state_file, "subset_start_frame=NULL\n");
    else
       fprintf (state_file, "subset_start_frame=%s\n",
           fl_get_input (animationPage->subsetFrameStartInput));

    if (strcmp(fl_get_input (animationPage->subsetFrameEndInput), "") == 0)
       fprintf (state_file, "subset_end_frame=NULL\n");
    else
       fprintf (state_file, "subset_end_frame=%s\n",
           fl_get_input (animationPage->subsetFrameEndInput));


    if (strcmp(fl_get_input (animationPage->clockStartInput), "") == 0)
       fprintf (state_file, "initial_clock=%f\n", POV_INITIAL_CLOCK);
    else
       fprintf (state_file, "initial_clock=%s\n",
           fl_get_input (animationPage->clockStartInput));

    if (strcmp(fl_get_input (animationPage->clockEndInput), "") == 0)
       fprintf (state_file, "final_clock=%f\n", POV_FINAL_CLOCK);
    else
       fprintf (state_file, "final_clock=%s\n", 
           fl_get_input (animationPage->clockEndInput));

    fprintf (state_file, "cyclic_animation=%d\n", pov.cyclic_animation);
    fprintf (state_file, "field_render=%d\n", pov.field_rendering);
    fprintf (state_file, "odd_field=%d\n", pov.odd_field);

    if (strcmp(fl_get_input (animationPage->clockExternalInput), "") == 0)
       fprintf (state_file, "clock=%f\n", POV_EXTERNAL_CLOCK);
    else
       fprintf (state_file, "clock=%s\n", 
           fl_get_input (animationPage->clockExternalInput));


    /* Configuration */ 

    fprintf (state_file, "config_pov_command=%s\n", CONFIG_POV_COMMAND);
    fprintf (state_file, "config_libpath=%s\n", CONFIG_POV_LIB_PATH);
    fprintf (state_file, "config_editor=%s\n", CONFIG_POV_EDITOR);
    fprintf (state_file, "config_display_command=%s\n", CONFIG_DISPLAY_COMMAND);
    fprintf (state_file, "config_pov_dir=%s\n", CONFIG_POV_DIR);
    fprintf (state_file, "config_scene_filter=%s\n", CONFIG_SCENE_FILTER);
    fprintf (state_file, "config_image_filter=%s\n", CONFIG_IMAGE_FILTER);
    fprintf (state_file, "config_width=%s\n", CONFIG_POV_WIDTH); 
    fprintf (state_file, "config_height=%s\n", CONFIG_POV_HEIGHT);

    /* Auto image viewer */
    fprintf (state_file, "auto_view=%d\n", auto_view);

    fclose (state_file);

/*    
    flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
    flx_show_dialog ("message", "xfpovray state saved");
*/
 
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int get_state (int show_warning)
{
    FILE *state_file;
    char filename[512];
    int i, numLibs = 0;
    char buf[80];
    char *key;
    char *value;
    const char delimiter[] = "=";     
    
       
    /* Get home directory, and set filepath */
    sprintf(filename, "%s/.xfpovray/xfpovray.state", getenv("HOME"));

    /* If no state file exists, show a warning, or do nothing */
    if ((state_file = fopen(filename, "r")) == NULL) {
       if (show_warning) {
          flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
          flx_show_dialog ("info", "No state file found!\n");
       }
       return (1);
    }

    /* Read the header.  Check for version consistency. */
    fgets (buf, sizeof(buf), state_file);
    if (strstr(buf, xfp_version) == NULL) {
       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       flx_show_dialog ("warning", 
       "\nObsolete state file.\nYou will have to reconfigure\nthe state of xfpovray\nand resave.");
       save_state();
       return (1);
    }
         
    while (fgets(buf, sizeof(buf), state_file) != NULL) {

          /* Get the key/value pair */
          key = strtok (buf, delimiter);
          value = strtok (NULL, delimiter);
          strtruncate(value, '\n');

          if (strcmp(key, "library_path") == 0) {
	     if (strcmp(value, "NULL") == 0) 
	        fl_set_input (parsingPage->libpathInput[numLibs], "");
	     else	  
	        fl_set_input (parsingPage->libpathInput[numLibs], value); 
	     numLibs++;
	     continue;
          }
          if (strcmp(key, "version") == 0) {
	     pov.version = atoi(value);
             fl_set_choice (parsingPage->languageVersion, pov.version);
	     continue;
          }
          if (strcmp(key, "remove_bounds") == 0) {
	     pov.remove_bounds = atoi(value);
             fl_set_choice (parsingPage->removeBounds, pov.remove_bounds + 1);
	     continue;
          }
          if (strcmp(key, "split_unions") == 0) {
	     pov.split_unions = atoi(value);
             fl_set_choice (parsingPage->splitUnions, pov.split_unions + 1);
	     continue;
          }
          if (strcmp(key, "height") == 0) {
             pov.height = atoi(value);
             fl_set_input (outputPage->imageHeightInput, value);
	     continue;
          }
          if (strcmp(key, "width") == 0) {
             pov.width = atoi(value);
             fl_set_input (outputPage->imageWidthInput, value);
	     continue;
          }
          if (strcmp(key, "preview_start_size") == 0) {
             if (strcmp(value, "NULL") == 0)
	        fl_set_input (outputPage->mosaicStart, "");
             else 
	        fl_set_input (outputPage->mosaicStart, value);
	     continue;
          }
          if (strcmp(key, "preview_end_size") == 0) {
             if (strcmp(value, "NULL") == 0)
  	        fl_set_input (outputPage->mosaicEnd, "");
             else 
  	        fl_set_input (outputPage->mosaicEnd, value);
	     continue;
          }
          if (strcmp(key, "start_column") == 0) {
             if (strcmp(value, "NULL") == 0)
	        fl_set_input (outputPage->columnStart, "");
             else
	        fl_set_input (outputPage->columnStart, value);
	     continue;
          }
          if (strcmp(key, "end_column") == 0) {
             if (strcmp(value, "NULL") == 0)
	        fl_set_input (outputPage->columnEnd, "");
             else
	        fl_set_input (outputPage->columnEnd, value);
	     continue;
          }
          if (strcmp(key, "start_row") == 0) {
             if (strcmp(value, "NULL") == 0)
	        fl_set_input (outputPage->rowStart, "");
             else
	        fl_set_input (outputPage->rowStart, value);
	     continue;
          }
          if (strcmp(key, "end_row") == 0) {
             if (strcmp(value, "NULL") == 0)
  	        fl_set_input (outputPage->rowEnd, "");
             else
  	        fl_set_input (outputPage->rowEnd, value);
	     continue;
          }
          if (strcmp(key, "display") == 0) {
	     pov.display_during_render = atoi(value);
             fl_set_choice (outputPage->displayDuringRender, 
	         pov.display_during_render + 1);
	     continue;
          }
          if (strcmp(key, "output_to_file") == 0) {
	     pov.output_to_file = atoi(value);
             fl_set_choice (outputPage->imageToFile, pov.output_to_file + 1);
	     continue;
          }
          if (strcmp(key, "buffer_output") == 0) {
             pov.buffer_output = atoi(value);
             fl_set_choice (outputPage->bufferOutput, pov.buffer_output + 1);
	     continue;
          }
          if (strcmp(key, "buffer_size") == 0) {
             pov.buffer_size = atoi(value);
	     continue;
          }
          if (strcmp(key, "palette") == 0) {
             pov.palette = atoi(value);
	     fl_set_choice (outputPage->palette, pov.palette);
	     continue;
          }
          if (strcmp(key, "output_file_type") == 0) {
             pov.output_filetype = atoi(value);
             fl_set_choice (outputPage->fileType, pov.output_filetype);
	     continue;
          }
          if (strcmp(key, "verbose") == 0) {
             pov.verbose = atoi(value);
	     pov.redirect_on = (pov.verbose > 1 ? 1 : 0);
             fl_set_choice (outputPage->verbosity, pov.verbose + 1);
	     continue;
          }
          if (strcmp(key, "quality") == 0) {
	     pov.quality = atoi(value);
	     fl_set_slider_value (tracingPage->qualitySlider, pov.quality);
	     continue;
          }
          if (strcmp(key, "antialias") == 0) {
             pov.antialias = atoi(value);
             fl_set_choice (tracingPage->antialias, pov.antialias + 1);
	     continue;
          }
          if (strcmp(key, "antialias_depth") == 0) {
             pov.aa_depth = atoi(value);
	     fl_set_slider_value (tracingPage->antialiasSlider, pov.aa_depth);
	     continue;
          }
          if (strcmp(key, "sampling_method") == 0) {
	     pov.aa_sampling_method = atoi(value);
             fl_set_choice (tracingPage->antialiasMethod, 
	         pov.aa_sampling_method);
	     continue;
          }
          if (strcmp(key, "antialias_threshold") == 0) {
	     fl_set_input (tracingPage->antialiasThreshold, value);
	     continue;
          }	  
          if (strcmp(key, "radiosity") == 0) {
	     pov.radiosity = atoi(value);
             fl_set_choice (tracingPage->radiosity, pov.radiosity + 1);
	     continue;
          }
          if (strcmp(key, "bounding") == 0) {
	     pov.bounding = atoi(value);
             fl_set_choice (tracingPage->bounding, pov.bounding + 1);
	     continue;
          }
          if (strcmp(key, "light_buffer") == 0) {
	     pov.light_buffer = atoi(value);
             fl_set_choice (tracingPage->lightBuffer, pov.light_buffer + 1);
	     continue;
          }
          if (strcmp(key, "vista_buffer") == 0) {
	     pov.vista_buffer = atoi(value);
             fl_set_choice (tracingPage->vistaBuffer, pov.vista_buffer + 1);
	     continue;
          }
          if (strcmp(key, "jitter") == 0) {
	     pov.jitter = atoi(value);
             fl_set_choice (tracingPage->jitter, pov.jitter + 1);
	     continue;
          }
          if (strcmp(key, "antialias_threshold") == 0) {
             fl_set_input (tracingPage->antialiasThreshold, value);
	     continue;
          }	  
          if (strcmp(key, "bounding_threshold") == 0) {
 	     pov.bounding_threshold = atoi(value);
             fl_set_input (tracingPage->boundingThreshold, value);
	     continue;
          }
          if (strcmp(key, "jitter_amount") == 0) {
             fl_set_input (tracingPage->jitterInput, value);
	     continue;
          }
          if (strcmp(key, "initial_frame") == 0) {
	     pov.initial_frame = atoi(value);
             fl_set_input (animationPage->frameStartInput, value);
	     continue;
          }
          if (strcmp(key, "final_frame") == 0) {
	     pov.final_frame = atoi(value);
             fl_set_input (animationPage->frameEndInput, value);
	     continue;
          }
          if (strcmp(key, "subset_start_frame") == 0) {
             if (strcmp(value, "NULL") == 0)
                fl_set_input (animationPage->subsetFrameStartInput, "");
             else 
                fl_set_input (animationPage->subsetFrameStartInput, value);
	     continue;
          }
          if (strcmp(key, "subset_end_frame") == 0) {
             if (strcmp(value, "NULL") == 0)
                fl_set_input (animationPage->subsetFrameEndInput, "");
             else  
                fl_set_input (animationPage->subsetFrameEndInput, value);
	     continue;
          }
          if (strcmp(key, "initial_clock") == 0) {
             fl_set_input (animationPage->clockStartInput, value);
	     continue;
          }
          if (strcmp(key, "final_clock") == 0) {
             fl_set_input (animationPage->clockEndInput, value);
	     continue;
          }
          if (strcmp(key, "cyclic_animation") == 0) {
	     pov.cyclic_animation = atoi(value);
             fl_set_choice (animationPage->cyclicAnimation, 
	         pov.cyclic_animation + 1);	     
	     continue;
          }
          if (strcmp(key, "field_render") == 0) {
	     pov.field_rendering = atoi(value);
             fl_set_choice (animationPage->fieldRendering, 
	         pov.field_rendering + 1);	     
	     continue;
          }
          if (strcmp(key, "odd_field") == 0) {
	     pov.odd_field = atoi(value);
	     continue;
          }
          if (strcmp(key, "clock") == 0) {
             fl_set_input (animationPage->clockExternalInput, value);
	     continue;
          }
          if (strcmp(key, "config_pov_command") == 0) {
             fl_set_input (xfData->configOptions[0], value);
             CONFIG_POV_COMMAND = 
                 (char *) fl_get_input (xfData->configOptions[0]);
	     continue;
          }
          if (strcmp(key, "config_libpath") == 0) {
	     fl_set_input (xfData->configOptions[1], value);
             CONFIG_POV_LIB_PATH = 
                 (char *) fl_get_input (xfData->configOptions[1]);
	     continue;
          }
          if (strcmp(key, "config_editor") == 0) {
             fl_set_input (xfData->configOptions[2], value);
             CONFIG_POV_EDITOR =
                 (char *) fl_get_input (xfData->configOptions[2]);
	     continue;
          }
          if (strcmp(key, "config_display_command") == 0) {
             fl_set_input (xfData->configOptions[3], value); 
             CONFIG_DISPLAY_COMMAND =
                 (char *) fl_get_input (xfData->configOptions[3]);
	     continue;
          }
          if (strcmp(key, "config_pov_dir") == 0) {
             fl_set_input (xfData->configOptions[4], value); 
             CONFIG_POV_DIR =
                (char *) fl_get_input (xfData->configOptions[4]);
	     continue;
          }
          if (strcmp(key, "config_scene_filter") == 0) {
             fl_set_input (xfData->configOptions[5], value); 
             CONFIG_SCENE_FILTER =
        	 (char *) fl_get_input (xfData->configOptions[5]);
	     continue;
          }
          if (strcmp(key, "config_image_filter") == 0) {
             fl_set_input (xfData->configOptions[6], value); 
             CONFIG_IMAGE_FILTER =
                 (char *) fl_get_input (xfData->configOptions[6]);
	     continue;
          }
          if (strcmp(key, "config_width") == 0) {
             fl_set_input (xfData->configOptions[7], value); 
             CONFIG_POV_WIDTH =
                 (char *) fl_get_input (xfData->configOptions[7]);
	     continue;
          }
          if (strcmp(key, "config_height") == 0) {
             fl_set_input (xfData->configOptions[8], value); 
             CONFIG_POV_HEIGHT =
                 (char *) fl_get_input (xfData->configOptions[8]);
	     continue;
          }
          if (strcmp(key, "auto_view") == 0) {
	     auto_view = atoi(value);
	     continue;
          }
          
    }  /* endwhile */     
     
    fclose (state_file);


    return(0);  
}
