/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
**+
** NAME:
**
**	xfpovray
**
** PURPOSE:
**
**      An XForms interface to the Persistence of Vision Raytracer (POV-Ray)
**
**      The basic method is as follows:  Options are set on the widget; options
**      not set are assigned default values.  All options are written out to
**      a file that is read by POV-Ray upon startup.
**
**      User options in xfpovray.h 
**
** USAGE:
**
**      xfpovray [-geometry +x+y | file.pov]
**
**      o Due to an XForms problem on Linux, the geometry flag cannot be
**        specified as -g.  You must type -geometry.
**      o A standard X geometry string is acceptable; however, the size
**        of the widget will be ignored; only the position can be set.
**  
** MODIFICATION HISTORY:
**
**	v0.9: 1996 July, written, mallozzir@cspar.uah.edu
**                       (http://cspar.uah.edu/~mallozzir/)
**
**      Consult the file CHANGES for modification history
**
**-
** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdlib.h>
#include <stdio.h>

#include "forms.h"
#include "xfpovray.h"
#include "pov-defaults.h"
#include "flx.h"

#include "xfpovray_icon.xpm"
#include "xfpovray_info.xpm"
#include "xfpovray.xpm"


#if FL_VERSION < MIN_XFORMS_VER || FL_REVISION < MIN_XFORMS_REV
#error
#error CANNOT COMPILE WITH THIS VERSION OF XFORMS.
#error PLEASE UPGRADE TO THE LATEST VERSION.
#error
#endif


/* xfpovray version and distribution info */
const char *xfp_version = "v1.3.1";
const char *xfp_author  = "Robert S. Mallozzi";
const char *xfp_email   = "mallozzir@cspar.uah.edu";
const char *xfp_url     = "http://cspar.uah.edu/~mallozzir/";

/* Starting position for xfpovray, default center placement */ 
int start_x;
int start_y;
int have_geometry = FALSE;
  
/* POV-Ray sub-process id */ 
int povray_pid = 0;

/* Set options to defaults, defined in pov-defaults.h */
struct pov_options pov = {
       NULL, /* input_file_name */ 
       /* NULL,  output_file_name */ 
       POV_HEIGHT,
       POV_WIDTH,
       POV_DISPLAY_DURING_RENDER,
       POV_OUTPUT_TO_FILE,
       POV_VERBOSE,
       POV_BUFFER_OUTPUT,
       POV_BUFFER_SIZE,
       POV_ANTIALIAS,
       POV_AA_SAMPLING_METHOD,
       POV_AA_DEPTH,
       POV_RADIOSITY,
       POV_QUALITY,
       POV_REDIRECT_ON,
       POV_VERSION, 
       POV_REMOVE_BOUNDS,
       POV_SPLIT_UNIONS,
       POV_BOUNDING,
       POV_BOUNDING_THRESHOLD,
       POV_LIGHT_BUFFER,
       POV_VISTA_BUFFER,
       POV_JITTER,
       POV_INITIAL_FRAME,
       POV_FINAL_FRAME,
       POV_CYCLIC_ANIMATION,
       POV_FIELD_RENDERING,
       POV_ODD_FIELD,
       POV_PALETTE,
       POV_OUTPUT_FILETYPE
};

/* Temporary output files */
char *POV_INIFILE;
char *STREAM_FILE;
char *FATAL_FILE;
char *PREVIEW_FILE;
char *TEMP_FILE;

int auto_view = FALSE;
int show_tips = TRUE;

/* Structures for the include file previewer */
struct preview_options {
       float camera[3];
       float camera_lookat[3];
       float light[3];
};
struct preview_options preview_setup = {
       {2.0, 1.0, -1.5},                     /* camera */
       {0.0, 0.0, 0.0},                      /* camera lookat */
       {2.0, 4.0, -5.0},                     /* light  */
};

/* Save the default values */
struct preview_options preview_setup_def = {
       {2.0, 1.0, -1.5},                     
       {0.0, 0.0, 0.0},                    
       {2.0, 4.0, -5.0},               
};

/* xforms defaults */
FL_IOPT control;       

/* Current configuration */
char *CONFIG_POV_COMMAND;
char *CONFIG_POV_LIB_PATH;
char *CONFIG_POV_EDITOR;
char *CONFIG_DISPLAY_COMMAND;
char *CONFIG_POV_DIR;
char *CONFIG_SCENE_FILTER;
char *CONFIG_IMAGE_FILTER;
char *CONFIG_POV_WIDTH; 
char *CONFIG_POV_HEIGHT;

XFDATA_S *xfData;
PARSING_PAGE_S   *parsingPage;
OUTPUT_PAGE_S	 *outputPage;
TRACING_PAGE_S   *tracingPage;
ANIMATION_PAGE_S *animationPage;
CONFIG_PAGE_S	 *configPage;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int main (int argc, char **argv)
{
 
    FL_OBJECT *obj, *ret;
    int version, revision;
    static char *home;
    FILE *fp;
    char *temp;
    unsigned int w, h;
    int userSceneFile = FALSE;
    Pixmap xfp_pix, xfp_mask, info_pix, info_mask;
    long flags;
    
          
    /* Check distribution version numbers */
    if (FL_INCLUDE_VERSION != fl_library_version (&version, &revision)) {
       fprintf (stderr, "Header/library version mismatch!\n");
       fprintf (stderr, "    Header: %d.%d\n", FL_VERSION, FL_REVISION);
       fprintf (stderr, "    Library: %d.%d\n", version, revision);
       exit(1);
    }

    /* Check for geometry string, w (width) and h (height) are ignored */
    if (argc > 1) {    
       if (strcmp(argv[1], "-geometry") == 0) {          
          have_geometry = TRUE;
          flags = XParseGeometry(argv[2], &start_x, &start_y, &w, &h);
          if ((WidthValue|HeightValue) & flags) {
             fprintf(stderr, 
	         "\nUsage: xfpovray [-geometry +x+y | file.pov]\n\n");
             have_geometry = FALSE;
          }
       } 
       else if (argc == 2) {    
          /* Assume scene file was specified */
	  userSceneFile = TRUE;          
          pov.input_file_name = argv[1];       
       }
       else {  
	  fprintf(stderr, 
	      "\nUsage: xfpovray [-geometry +x+y | file.pov]\n\n");
             have_geometry = FALSE;
       }
    }
    
    /* Get home directory, and set filepaths */
    home = getenv("HOME");
    if (home == NULL) {
       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       flx_show_dialog ("error", "HOME environment not defined!\n");
       exit (1);
    }           

    POV_INIFILE = (char *) fl_calloc (1, strlen(home) + strlen("/.xfpovray/") + 
        strlen(_POV_INIFILE) + 1);
    sprintf(POV_INIFILE, "%s/.xfpovray/%s", home, _POV_INIFILE);

    STREAM_FILE = (char *) fl_calloc (1, strlen(home) + strlen("/.xfpovray/") + 
        strlen(_STREAM_FILE) + 1);
    sprintf(STREAM_FILE, "%s/.xfpovray/%s", home, _STREAM_FILE);

    FATAL_FILE = (char *) fl_calloc (1, strlen(home) + strlen("/.xfpovray/") + 
        strlen(_FATAL_FILE) + 1);
    sprintf(FATAL_FILE, "%s/.xfpovray/%s", home, _FATAL_FILE);

    PREVIEW_FILE = (char *) fl_calloc (1, strlen(home) + strlen("/.xfpovray/") +
        strlen(_PREVIEW_FILE) + 1);
    sprintf(PREVIEW_FILE, "%s/.xfpovray/%s", home, _PREVIEW_FILE);

    TEMP_FILE = (char *) fl_calloc (1, strlen(home) + strlen("/.xfpovray/") + 
        strlen(_TEMP_FILE) + 1);
    sprintf(TEMP_FILE, "%s/.xfpovray/%s", home, _TEMP_FILE);
    
    /* Initialize configuration */    
    CONFIG_POV_COMMAND     = POV_COMMAND;
    CONFIG_POV_LIB_PATH    = POV_LIB_PATH;
    CONFIG_POV_EDITOR      = POV_EDITOR;
    CONFIG_DISPLAY_COMMAND = DISPLAY_COMMAND;
    CONFIG_POV_DIR         = POV_DIR;
    CONFIG_SCENE_FILTER    = SCENE_FILTER;
    CONFIG_IMAGE_FILTER    = IMAGE_FILTER; 
    CONFIG_POV_WIDTH       = POV_WIDTH_S; 
    CONFIG_POV_HEIGHT      = POV_HEIGHT_S; 

    /* Set some defaults and initialize the library */
    control.inputFontSize   = 12;
    control.browserFontSize = 12;
    control.buttonFontSize  = 12;
    control.labelFontSize   = 12;
    control.menuFontSize    = 12;
    control.borderWidth     = -2;
        
    fl_set_defaults (FL_PDInputFontSize|
                     FL_PDBrowserFontSize|
                     FL_PDButtonFontSize|
                     FL_PDLabelFontSize|
                     FL_PDMenuFontSize|
                     FL_PDBorderWidth,
                     &control);
           
    fl_initialize (&argc, argv, "XFpovray", 0, 0);
     
    /* Check for directory $HOME/.xfpovray */
    if ((fp = fopen(TEMP_FILE, "w")) == NULL) {
   
       temp = (char *) fl_calloc (1, strlen(MKDIR) + strlen(" ") + 
           strlen(home) + strlen("/.xfpovray") + 1); 
       sprintf (temp, "%s %s/.xfpovray", MKDIR, home);
       system (temp);
       free (temp);

       temp = (char *) fl_calloc (1, 
	   strlen("\nWelcome to xfpovray!\nCreated directory ") + 
	   strlen(home) + strlen("/.xfpovray") +
	   strlen("\n\nCheck your setup with\nOptions->Configure") + 
	   + 1); 
       sprintf (temp, 
           "\nWelcome to xfpovray!\nCreated directory %s/.xfpovray\n\nCheck your setup with\nOptions->Configure",
	    home);
       flx_set_dialog_font (FL_BOLD_STYLE, FL_NORMAL_SIZE);
       flx_show_dialog ("message", temp);
       free (temp);   
       fclose (fp);
    }

    /* 
    ** Build the various forms 
    */   
    xfData = (XFDATA_S *) fl_calloc (1, sizeof (*xfData));
        
    build_main(); 
    build_folder(); 
    build_text();
    build_fatal();        
    build_about();
    build_preview();     
    build_help();
    build_render_pup();
    build_configurator();
             
   
    /* Scene file given on command line */
    if (userSceneFile) {
       fl_set_input (xfData->sceneFileInput, pov.input_file_name);
       flx_justify_input (xfData->sceneFileInput, pov.input_file_name, 
           FL_NORMAL_STYLE, FL_NORMAL_SIZE);
    }
    
    /* Set the main form icons */
    xfp_pix = fl_create_from_pixmapdata (fl_root, xfpovray_icon,
        &w, &h, &xfp_mask, 0, 0, 0);
    fl_set_form_icon (xfData->mainForm, xfp_pix, xfp_mask);
    fl_set_form_icon (xfData->folderForm, xfp_pix, xfp_mask);

    /* Set the help form icon */
    info_pix = fl_create_from_pixmapdata (fl_root, xfpovray_info,
        &w, &h, &info_mask, 0, 0, 0);
    fl_set_form_icon (xfData->helpForm, info_pix, info_mask);

    /* Scale form based on current font */
    flx_scale_form (xfData->mainForm);

    /* Get saved state */
    get_state(0);
    
    /* Realize the main form */
    if (have_geometry == FALSE) {        
       fl_show_form (xfData->mainForm, FL_PLACE_SIZE, 
                    FL_FULLBORDER, "xfpovray");
    }                   
    else {
       fl_set_form_position (xfData->mainForm, start_x, start_y);
       fl_show_form (xfData->mainForm, FL_PLACE_GEOMETRY, 
                    FL_FULLBORDER, "xfpovray");
    }
    
    /* Idle callback to monitor the POV-Ray child process */
    povray_pid = 0;
    fl_set_idle_callback (PovCheckCB, 0); 

    
    do {
       ret = fl_do_forms();
    } while (ret != NULL);

    fl_hide_form (xfData->mainForm);
    exit(0);

    return (0); 
}


/* These functions build the various forms */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* The main widget */

void build_main(void)
{
    FL_OBJECT *obj;
    FL_OBJECT *oMenu;
    FILE *ext;
    static char *home;
    char *temp;
    char *t;
    char buf[256];
    int bw = abs(control.borderWidth);
    int current_menu = 1;
    int menu, subm[7];
                        

    xfData->mainForm = fl_bgn_form (FL_UP_BOX, 420, 200);

        /* Menubar */
        obj = fl_add_box (FL_UP_BOX, 5, 5, 420-10, 25, "");
        
	obj = fl_add_menu (FL_PULLDOWN_MENU, 
	    10, 5 + bw, 30, 25 - (bw * 2), "File");
	fl_set_object_lstyle (obj, FL_BOLD_STYLE);
	fl_set_menu (obj, "New|Open...|Quit");
	fl_set_object_callback (obj, QuitCB, 0);      

	xfData->optionsMenu = oMenu = fl_add_menu (FL_PULLDOWN_MENU, 
	    55, 5 + bw, 50, 25 - (bw * 2), "Options");
        fl_set_object_lstyle (oMenu, FL_BOLD_STYLE);

        fl_set_menu (oMenu, "Configure");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
        current_menu++;
        
        fl_addto_menu (oMenu, "Save State");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
        current_menu++;

        fl_addto_menu (oMenu, "Restore State");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
        current_menu++;

        fl_addto_menu (oMenu, "Show Render Popup");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
        fl_set_menu_item_mode (oMenu, current_menu, FL_PUP_BOX);
        current_menu++;

        fl_addto_menu (oMenu, "Auto Image Display");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
        fl_set_menu_item_mode (oMenu, current_menu,
	    auto_view ? FL_PUP_CHECK : FL_PUP_BOX);	
	current_menu++;

/*
        fl_addto_menu (oMenu, "Show Tooltips");
        fl_set_object_callback (oMenu, OptionsCB, 0);      
	fl_set_menu_item_mode (oMenu, current_menu, 
	    show_tips ? FL_PUP_CHECK : FL_PUP_BOX);	
	current_menu++;
*/

	/* Check for external routines */
	home = getenv("HOME");
        temp = (char *) fl_calloc (1, 
	    strlen(home) + strlen(".xfpovray/external.routines") + 1); 
	sprintf(temp, "%s/.xfpovray/external.routines", home);
	if ((ext = fopen(temp, "r")) != NULL) {
           
           /* Callback key is the menu item number of the first added
           ** external routine.  This, combined with the information of the 
           ** selected menu item, allows selection of the correct command to
           ** fork().  See OptionsCB().
           */
           fl_set_object_callback (oMenu, OptionsCB, current_menu);
           
           while (fgets(buf, sizeof(buf), ext) != NULL) {
                  t = strtok (buf, " ");
                  t = strtok (NULL, " ");
                  fl_addto_menu (oMenu, t);
                  current_menu++;
           }
           fclose(ext);
	}
        else {
           fl_addto_menu (oMenu, "No External Routines");
           fl_set_menu_item_mode (oMenu, current_menu, FL_PUP_GREY);
        }
        free(temp);
	 
	obj = fl_add_menu (FL_PULLDOWN_MENU, 
	    120, 5 + bw, 70, 25 - (bw * 2), "Templates");
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        menu = fl_newpup (fl_default_win());
        subm[0] = fl_newpup(0);
        subm[1] = fl_newpup(0);
        subm[2] = fl_newpup(0);
        subm[3] = fl_newpup(0);
        subm[4] = fl_newpup(0);
        subm[5] = fl_newpup(0);
        subm[6] = fl_newpup(0);
        
        fl_addtopup (menu, "Finite Solids%m", subm[0]);
        fl_addtopup (subm[0], "Blob%x11");
        fl_addtopup (subm[0], "Box%x12");
        fl_addtopup (subm[0], "Cone%x13");
        fl_addtopup (subm[0], "Cylinder%x14");
        fl_addtopup (subm[0], "Height Field%x15");
        fl_addtopup (subm[0], "Julia Fractal%x16");
        fl_addtopup (subm[0], "Lathe%x17");
        fl_addtopup (subm[0], "Sphere%x18");
        fl_addtopup (subm[0], "Superellipsoid%x19");
        fl_addtopup (subm[0], "Surface of Revolution%x110");
        fl_addtopup (subm[0], "Text%x111");
        fl_addtopup (subm[0], "Torus%x112");

        fl_addtopup (menu, "Infinite Solids%m", subm[1]);
        fl_addtopup (subm[1], "X Plane%x21");
        fl_addtopup (subm[1], "Y Plane%x22");
        fl_addtopup (subm[1], "Z Plane%x23");
        fl_addtopup (subm[1], "Axes%x24");

        fl_addtopup (menu, "Lights%m", subm[2]);
        fl_addtopup (subm[2], "Point%x31");
        fl_addtopup (subm[2], "Spot%x32");
        fl_addtopup (subm[2], "Cylindrical%x33");
        fl_addtopup (subm[2], "Area%x34");

        fl_addtopup (menu, "Cameras%m", subm[3]);
        fl_addtopup (subm[3], "Perspective%x41");
        fl_addtopup (subm[3], "Orthographic%x42");
        fl_addtopup (subm[3], "Fisheye%x43");
        fl_addtopup (subm[3], "Ultra Wide Angle%x44");
        fl_addtopup (subm[3], "Omnimax%x45");
        fl_addtopup (subm[3], "Panoramic%x46");

        fl_addtopup (menu, "Include Files%m", subm[4]);
        fl_addtopup (subm[4], "Chars%x51");
        fl_addtopup (subm[4], "Colors%x52");
        fl_addtopup (subm[4], "Consts%x53");
        fl_addtopup (subm[4], "Finish%x54");
        fl_addtopup (subm[4], "Glass%x55");
        fl_addtopup (subm[4], "Golds%x56");
        fl_addtopup (subm[4], "Metals%x57");
        fl_addtopup (subm[4], "Shapes%x58");
        fl_addtopup (subm[4], "Shapes2%x59");
        fl_addtopup (subm[4], "Shapesq%x510");
        fl_addtopup (subm[4], "Skies%x511");
        fl_addtopup (subm[4], "Stars%x512");
        fl_addtopup (subm[4], "Stones%x513");
        fl_addtopup (subm[4], "Stones1%x514");
        fl_addtopup (subm[4], "Stones2%x515");
        fl_addtopup (subm[4], "Textures%x516");
        fl_addtopup (subm[4], "Woodmaps%x517");
        fl_addtopup (subm[4], "Woods%x518");

        fl_addtopup (menu, "Atmospheric Effects%m", subm[5]);
        fl_addtopup (subm[5], "Atmosphere%x61");
        fl_addtopup (subm[5], "Background%x62");
        fl_addtopup (subm[5], "Fog%x63");
        fl_addtopup (subm[5], "Sky Sphere%x64");
        fl_addtopup (subm[5], "Rainbow%x65");

        fl_addtopup (menu, "Global Settings%m", subm[6]);
        fl_addtopup (subm[6], "ADC Bailout%x71");
        fl_addtopup (subm[6], "Ambient Light%x72");
        fl_addtopup (subm[6], "Assumed Gamma%x73");
        fl_addtopup (subm[6], "Heightfield Gray%x74");
        fl_addtopup (subm[6], "Irid Wavelength%x75");
        fl_addtopup (subm[6], "Max Trace Level%x76");
        fl_addtopup (subm[6], "Max Intersections%x77");
        fl_addtopup (subm[6], "Number of Waves%x78");
        fl_addtopup (subm[6], "Radiosity%x79");

        fl_addtopup (menu, "Scene Template%x8");

        fl_set_menu_popup (obj, menu); 
        fl_set_object_callback (obj, LoadClipboardCB, 0);

        
	obj = fl_add_menu (FL_PULLDOWN_MENU, 
	    380, 5 + bw, 30, 25 - (bw * 2), "Help");
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_menu (obj, "Help|About");        
        fl_set_object_callback (obj, HelpCB, 0);
 	
	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    15, 40, 100, 25, "Scene File"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, GetFileCB, 0);
 
	xfData->sceneFileInput = obj = fl_add_input (FL_NORMAL_INPUT, 
	    125, 40, 280, 25, "");
        fl_set_input (obj, "");

/* *********
	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    15, 70, 100, 25, "Output File"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, GetFileCB, 3);
 
	xfData->outFileInput = obj = fl_add_input (FL_NORMAL_INPUT, 
	    125, 70, 280, 25, "");
        fl_set_input (obj, "");  
********* */

	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    15, 70, 100, 25, "Edit Scene"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, GetFileCB, 1);
 
	obj = fl_add_menu (FL_PULLDOWN_MENU, 
            15, 100, 100, 25, "Config File"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_boxtype (obj, FL_UP_BOX);
        fl_set_menu (obj, "View|Save|Load");        
        fl_set_object_callback (obj, ConfigCB, 0);
 
	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    15, 130, 100, 25, "View Image"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, GetFileCB, 2);

	obj = fl_add_button (FL_NORMAL_BUTTON, 
            15, 160, 100, 25, "Inc Preview"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, PreviewCB, 0);

 	xfData->toggleFolder = obj = fl_add_button (FL_NORMAL_BUTTON, 
            125, 70, 100, 25, "Render Options");
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, ToggleFolderCB, 0);

 	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    125, 100, 100, 55, "Render\nImage"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, PovCB, 0);

	obj = fl_add_button (FL_NORMAL_BUTTON, 
	    125, 160, 100, 25, "Abort Render"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, AbortCB, 0);

        obj = fl_add_box (FL_DOWN_BOX, 235, 70, 170, 115, "");
        obj = fl_add_pixmap(FL_NORMAL_PIXMAP, 235+2, 70+2, 170-4, 115-4, "");
        fl_set_pixmap_data(obj, xfpovray);    
        if (show_tips) showtip (obj, "xfpovray !");
  	
    fl_end_form();

} /* end function build_main() */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
static void build_folder(void)
{    
    FL_OBJECT *obj;
    FL_OBJECT *folder;
    const int margin = 10;


    xfData->folderForm = fl_bgn_form (FL_UP_BOX, 420, 375);

	xfData->mainFolder = folder = fl_add_tabfolder (FL_TOP_TABFOLDER,  
            margin, margin, 400, 320, "");
	fl_set_object_lstyle (folder, FL_BOLD_STYLE);
        fl_set_object_callback (folder, ChangedFolderCB, 0);
        /* 
	** Bug(?) in xforms v0.88: Default gravity for folder is FL_RESIZE_NONE
	*/
        fl_set_object_gravity (folder, FL_NoGravity, FL_NoGravity);
	fl_set_object_resize (folder, FL_RESIZE_ALL);

        obj = fl_add_button (FL_NORMAL_BUTTON, 
            175, 340, 70, 25, "Dismiss"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, DismissFolderCB, 0);

    fl_end_form();

    parsingPage   = build_parsing_page (folder);
    outputPage    = build_output_page (folder);
    tracingPage   = build_tracing_page (folder);
    animationPage = build_animation_page (folder);
    configPage    = build_config_page (folder);

    fl_addto_tabfolder (folder, "Parsing",   parsingPage->form);
    fl_addto_tabfolder (folder, "Output",    outputPage->form);
    fl_addto_tabfolder (folder, "Tracing",   tracingPage->form);
    fl_addto_tabfolder (folder, "Animation", animationPage->form);
    fl_addto_tabfolder (folder, "Config",    configPage->form);

    /* Scale form based on current font */
    flx_scale_form (xfData->folderForm);

} /* end function build_folder() */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Show this form if redirect option is on */

void build_text(void)
{
    FL_OBJECT *obj;


    xfData->messageForm = fl_bgn_form (FL_UP_BOX, 560, 370);
 
        xfData->messageBrowser = fl_add_browser (FL_NORMAL_BROWSER, 
            15, 15, 530, 300, ""); 
            
        obj = flx_add_return_button (FL_RETURN_BUTTON, 
	    245, 330, 70, 25, "Dismiss"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, DismissMessageCB, 0);
        
    fl_end_form(); 

} /* end function build_text() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Show this form if there is a fatal error */

void build_fatal(void)
{
    FL_OBJECT *obj;


    xfData->fatalForm = fl_bgn_form(FL_UP_BOX, 560, 370);

        xfData->fatalBrowser = fl_add_browser (FL_NORMAL_BROWSER, 
            15, 15, 530, 300, ""); 
            
        obj = flx_add_return_button (FL_RETURN_BUTTON, 
	    245, 330, 70, 25, "Dismiss"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, DismissMessageCB, 1);
        
    fl_end_form();

} /* end function build_fatal() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Help->About form */

void build_about(void)
{
    FL_OBJECT *obj;
    int l_width, l_height;
    const int delayTime = 2;
    

    fl_get_string_dimension (FL_BOLDITALIC_STYLE|FL_EMBOSSED_STYLE, 
        FL_HUGE_SIZE, "xfpovray", strlen("xfpovray"), &l_width, &l_height);

    xfData->aboutForm = fl_bgn_form (FL_UP_BOX, 400, 270);

        xfData->aboutText = obj = fl_add_box (FL_NO_BOX, 150, 30,
            l_width + 10, l_height + 10, "xfpovray"); 
        fl_set_object_lsize (obj, FL_HUGE_SIZE);
        fl_set_object_lstyle (obj, FL_BOLDITALIC_STYLE|FL_EMBOSSED_STYLE);
        fl_set_object_lcol (obj, FL_RED);
   
        obj = fl_add_box (FL_NO_BOX, 150, 130, 120, 25, xfp_author); 
        fl_set_object_lsize (obj, FL_MEDIUM_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_lcol (obj, FL_BLACK);

        obj = fl_add_box (FL_NO_BOX, 150, 150, 120, 25, xfp_email); 
        fl_set_object_lsize (obj, FL_MEDIUM_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_lcol (obj, FL_BLACK);

        obj = fl_add_box (FL_NO_BOX, 150, 170, 120, 25, xfp_url); 
        fl_set_object_lsize (obj, FL_MEDIUM_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_lcol (obj, FL_BLACK);
                           
        obj = fl_add_box (FL_NO_BOX, 150, 190, 120, 25, xfp_version); 
        fl_set_object_lsize (obj, FL_MEDIUM_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_lcol (obj, FL_BLACK);

        obj = flx_add_return_button (FL_RETURN_BUTTON, 
	    175, 225, 70, 25, "Dismiss");
        fl_set_object_callback (obj, DismissAboutCB, 0);
        fl_set_object_lsize (obj, FL_NORMAL_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
   
        xfData->aboutTimer = obj = 
	    fl_add_timer (FL_HIDDEN_TIMER, 0, 0, 1, 1, "");
        fl_set_timer (obj, delayTime);
        fl_set_object_callback (obj, ContinueCB, 0);
 
    fl_end_form();


} /* end function build_about() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* This form shows the .inc file previews */
 
void build_preview(void)
{
    FL_OBJECT *obj;
    char temp[10];
    

    xfData->previewForm = fl_bgn_form(FL_UP_BOX, 420, 580);

        /* Option radio buttons */
        fl_bgn_group();
        	     
            xfData->previewOption[0] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                15, 15, 100, 25, "Color");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 0);

            xfData->previewOption[1] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                15, 45, 100, 25, "Stone");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 1);
 
            xfData->previewOption[2] = obj= fl_add_lightbutton(FL_RADIO_BUTTON, 
                15, 75, 100, 25, "Wood");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 2);

            xfData->previewOption[3] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                15, 105, 100, 25, "Glass");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 3);

            xfData->previewOption[4] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                15, 135, 100, 25, "Metal");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 4);

            xfData->previewOption[5] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                120, 15, 100, 25, "Gold");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 5);

            xfData->previewOption[6] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                120, 45, 100, 25, "Finish");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 6);
 
            xfData->previewOption[7] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                120, 75, 100, 25, "Skies/Stars");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 7);

            xfData->previewOption[8] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                120, 105, 100, 25, "Shapes");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 8);

            xfData->previewOption[9] = obj = fl_add_lightbutton(FL_RADIO_BUTTON,
                120, 135, 100, 25, "Platonic");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
            fl_set_object_callback (obj, PreviewRadioCB, 9);
   
        fl_end_group();

        fl_bgn_group();
        
            xfData->previewType[0] = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 
                305, 15, 100, 25, "Pigment");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
 
            xfData->previewType[1] = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 
                305, 45, 100, 25, "Texture");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
  
            xfData->previewType[2] = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 
                305, 75, 100, 25, "Finish");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);

            xfData->previewType[3] = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 
                305, 105, 100, 25, "Geometry");
            fl_set_object_lstyle (obj, FL_BOLD_STYLE);
 
        fl_end_group();


        obj = fl_add_text(FL_NORMAL_TEXT, 
            15, 175, 100, 25, "Camera"); 
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);

        xfData->previewCamera[0] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    120, 175, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera[0]);
        fl_set_input (obj, temp);        

        xfData->previewCamera[1] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    175, 175, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera[1]);
        fl_set_input (obj, temp);        

        xfData->previewCamera[2] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    230, 175, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera[2]);
        fl_set_input (obj, temp);        

        obj = fl_add_button (FL_NORMAL_BUTTON, 
            285, 180, 20, 20, "@square");
            fl_set_object_callback (obj, ResetPreviewCB, 0);
            fl_set_object_lsize(obj, FL_NORMAL_SIZE);

        obj = fl_add_text(FL_NORMAL_TEXT, 
            15, 205, 100, 25, "Camera LookAt"); 
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);

        xfData->previewCameraLookat[0] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    120, 205, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera_lookat[0]);
        fl_set_input (obj, temp);        

        xfData->previewCameraLookat[1] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    175, 205, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera_lookat[1]);
        fl_set_input (obj, temp);        

        xfData->previewCameraLookat[2] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    230, 205, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.camera_lookat[2]);
        fl_set_input (obj, temp);        

        obj = fl_add_button (FL_NORMAL_BUTTON, 
            285, 210, 20, 20, "@square");
        fl_set_object_callback (obj, ResetPreviewCB, 1);
        fl_set_object_lsize(obj, FL_NORMAL_SIZE);


        obj = fl_add_text(FL_NORMAL_TEXT, 
            15, 235, 100, 25, "Light"); 
        fl_set_object_lstyle(obj, FL_BOLD_STYLE);

        xfData->previewLight[0] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    120, 235, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.light[0]);
        fl_set_input (obj, temp);        

        xfData->previewLight[1] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    175, 235, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.light[1]);
        fl_set_input (obj, temp);        

        xfData->previewLight[2] = obj = fl_add_input(FL_NORMAL_INPUT, 
	    230, 235, 50, 25, "");
        sprintf(temp, "%4.1f", preview_setup.light[2]);
        fl_set_input (obj, temp);        

        obj = fl_add_button (FL_NORMAL_BUTTON, 
            285, 240, 20, 20, "@square");
        fl_set_object_callback (obj, ResetPreviewCB, 2);
        fl_set_object_lsize (obj, FL_NORMAL_SIZE);


        xfData->showAxes = obj = fl_add_checkbutton (FL_PUSH_BUTTON, 
            315, 235, 50, 25, "Show Axes");
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_button (obj, RELEASED);      
         

        xfData->previewBrowser = fl_add_browser (FL_HOLD_BROWSER, 
            10, 270, 400, 255, ""); 
        
        obj = fl_add_button (FL_NORMAL_BUTTON, 105, 540, 100, 25, "Render");
        fl_set_object_callback (obj, DoPreviewCB, 0);
        fl_set_object_lsize (obj, FL_NORMAL_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);

        obj = fl_add_button (FL_NORMAL_BUTTON, 215, 540, 100, 25, "Dismiss");
        fl_set_object_callback (obj, PreviewDismissCB, 0);
        fl_set_object_lsize (obj, FL_NORMAL_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
 
    fl_end_form();

    /* Scale form based on current font */
    flx_scale_form (xfData->previewForm);

} /* end function build_preview() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Build the on-line help widget */
 
void build_help(void)
{
    FL_OBJECT *obj;
    char temp[1024];
    

    xfData->helpForm = fl_bgn_form (FL_UP_BOX, 640, 360);
 
        obj = fl_add_browser (FL_HOLD_BROWSER, 10, 10, 120, 300, "");
        fl_set_object_callback (obj, DisplayHelpCB, 0);
        sprintf(temp, "%s/help_topics", HELP_DIR);
        fl_load_browser (obj, temp);
        fl_select_browser_line (obj, 1);     

        xfData->helpBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER, 
            140, 10, 490, 300, ""); 
        sprintf(temp, "%s/Introduction", HELP_DIR);
        fl_load_browser (obj, temp);

        obj = flx_add_return_button (FL_RETURN_BUTTON, 
	    285, 320, 70, 25, "Dismiss");
        fl_set_object_callback (obj, DismissHelpCB, 0);
        fl_set_object_lsize (obj, FL_NORMAL_SIZE);
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);

        fl_bgn_group();
        
            obj = fl_add_button (FL_RADIO_BUTTON, 545, 320, 25, 25, "A");
            fl_set_object_callback (obj, SetHelpFontCB, 0);
            fl_set_object_lsize (obj, FL_NORMAL_SIZE);             
            fl_set_button (obj, PUSHED);

            obj = fl_add_button (FL_RADIO_BUTTON, 575, 320, 25, 25, "A");
            fl_set_object_callback (obj, SetHelpFontCB, 1);
            fl_set_object_lsize (obj, FL_MEDIUM_SIZE);

            obj = fl_add_button (FL_RADIO_BUTTON, 605, 320, 25, 25, "A");
            fl_set_object_callback (obj, SetHelpFontCB, 2);
            fl_set_object_lsize (obj, FL_LARGE_SIZE);

        fl_end_group();
        
        
    fl_end_form();

    /* Scale form based on current font */
    flx_scale_form (xfData->helpForm);

} /* end function build_help() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Build the render pop-up form */
 
void build_render_pup(void)
{
    FL_OBJECT *obj;
    

    xfData->renderPupForm = fl_bgn_form (FL_UP_BOX, 100, 80);

        obj = fl_add_button (FL_NORMAL_BUTTON, 0, 0, 100, 55, "Render\nImage"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, PovCB, 0);
   
	obj = fl_add_button (FL_NORMAL_BUTTON, 0, 55, 100, 25, "Abort Render"); 
        fl_set_object_lstyle (obj, FL_BOLD_STYLE);
        fl_set_object_callback (obj, AbortCB, 0);

    fl_end_form();

    /* Scale form based on current font */
    flx_scale_form (xfData->renderPupForm);
   
   
} /* end function build_render_pup() */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* This form handles xfpovray configuration */
 
void build_configurator(void)
{
    FL_OBJECT *obj;


    xfData->configuratorForm = fl_bgn_form (FL_UP_BOX, 400, 470);

      obj = fl_add_frame (FL_ENGRAVED_FRAME, 10, 10, 380, 450, "");

      obj = fl_add_text (FL_NORMAL_TEXT, 15, 20, 165, 25, "POV-Ray Command   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[0] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 20, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_COMMAND);
      
      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 60, 165, 25, "POV-Ray Include Path   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[1] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 60, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_LIB_PATH);
  
      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 100, 165, 25, "Editor   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[2] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 100, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_EDITOR);

      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 140, 165, 25, "Image Viewer   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[3] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 140, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_DISPLAY_COMMAND);

      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 180, 165, 25, "Scene File Directory   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[4] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 180, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_DIR);

      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 220, 165, 25, "Scene File Filter   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[5] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 220, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_SCENE_FILTER);

      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 260, 165, 25, "Image File Filter   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[6] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 260, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_IMAGE_FILTER);
  
      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 300, 165, 25, "Image Width   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[7] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 300, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_WIDTH);
  
      obj = fl_add_text (FL_NORMAL_TEXT, 
          15, 340, 165, 25, "Image Height   ");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_RIGHT);

      xfData->configOptions[8] = obj = 
          fl_add_input (FL_NORMAL_INPUT, 180, 340, 200, 25, "");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_input (obj, CONFIG_POV_HEIGHT);

      obj = fl_add_text (FL_NORMAL_TEXT, 
          30, 380, 350, 25, "Save State to make changes permanent");
      fl_set_object_lcol (obj, FL_BLUE);
      fl_set_object_lsize (obj, FL_SMALL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_lalign (obj, FL_ALIGN_CENTER);

      obj = fl_add_button (FL_NORMAL_BUTTON, 
          80, 410, 100, 25, "Accept");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_callback (obj, DismissConfiguratorCB, 0);
      
      obj = fl_add_button (FL_NORMAL_BUTTON, 
          230, 410, 100, 25, "Cancel");
      fl_set_object_lsize (obj, FL_NORMAL_SIZE);
      fl_set_object_lstyle (obj, FL_BOLD_STYLE);
      fl_set_object_callback (obj, DismissConfiguratorCB, 1);

  fl_end_form();


} /* end function build_configurator() */
