//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include <qpushbt.h> 
#include <qbitmap.h> 

#include "cal.h"
#include "calendar.h"

#include "calendar.xbm"
#include "left.xbm"
#include "right.xbm"
#include "double_left.xbm"
#include "double_right.xbm"

//*****************************************************************************

QtedCalendar:: 
QtedCalendar(int month, int year, QWidget *parent, const char *name) : 
             QWidget(parent, name), _month(month), _year(year), _lines(0) {
   int i, days[MAX_LINES_PER_MONTH * DAYS_PER_WEEK];

   QBitmap icon(calendar_width, calendar_height, calendar_bits, TRUE);
   setIcon(icon);

   _header_p = new QLabel("Foo", this);
   _header_p->setAlignment(AlignCenter);

   _left_arrow_p = new QPushButton(this);
   QBitmap left_bm(left_width, left_height, left_bits, TRUE);
   _left_arrow_p->setPixmap(left_bm);
   _left_arrow_p->setAutoResize(TRUE);

   _right_arrow_p = new QPushButton(this);
   QBitmap right_bm(right_width, right_height, right_bits, TRUE);
   _right_arrow_p->setPixmap(right_bm);
   _right_arrow_p->setAutoResize(TRUE);

   _dbl_left_p = new QPushButton(this);
   QBitmap dbl_left_bm(double_left_width, double_left_height, 
                       double_left_bits, TRUE);
   _dbl_left_p->setPixmap(dbl_left_bm);
   _dbl_left_p->setAutoResize(TRUE);

   _dbl_right_p = new QPushButton(this);
   QBitmap dbl_right_bm(double_right_width, double_right_height, 
                        double_right_bits, TRUE);
   _dbl_right_p->setPixmap(dbl_right_bm);
   _dbl_right_p->setAutoResize(TRUE);

   _close_p = new QPushButton("Dismiss", this);
   _close_p->setAutoResize(TRUE);

   day_array (_month, _year, days);

   _wdl_p = new QtedWeekdayLabels(QtedWeekdayLabels::Horizontal, this);
   _monthdayButtons = new QButtonGroup(this);

   for (i = 0; i < MAX_LINES_PER_MONTH * DAYS_PER_WEEK; i++) {
      _month_days[i] = new QtedMonthdayButton(_monthdayButtons);
      if (days[i] > 0)
         _month_days[i]->day(days[i]);
      else
         _month_days[i]->setEnabled(FALSE);
   }

   connect(_left_arrow_p,  SIGNAL(clicked()), this, SLOT(leftClicked()));
   connect(_dbl_left_p,    SIGNAL(clicked()), this, SLOT(dblLeftClicked()));
   connect(_right_arrow_p, SIGNAL(clicked()), this, SLOT(rightClicked()));
   connect(_dbl_right_p,   SIGNAL(clicked()), this, SLOT(dblRightClicked()));

   connect(_close_p,       SIGNAL(clicked()), this, SLOT(closeButton()));
   connect(_monthdayButtons, SIGNAL(clicked(int)), 
           this, SLOT(monthdayClicked(int)));

   update();
}

//*****************************************************************************

QtedMonthdayButton *QtedCalendar::
getMonthdayButton(int offset) {
   QtedMonthdayButton *result = NULL;

   if (offset >= 0 && offset < MAX_LINES_PER_MONTH * DAYS_PER_WEEK)
      result = _month_days[offset];

   return (result);
}

//*****************************************************************************

QtedMonthdayButton *QtedCalendar::
getMonthdayButtonByDay(int day) {
   QtedMonthdayButton *result = NULL;

   int offset = day + _first_day - 1;

   if (offset >= 0 && offset < MAX_LINES_PER_MONTH * DAYS_PER_WEEK) {
      result = _month_days[offset];
      if (result && result->day() == -1)  
         result = NULL;
   }

   return (result);
}

//*****************************************************************************

int QtedCalendar::
maxDay(void) {
   return (_max_day);
}

//*****************************************************************************

int QtedCalendar::
dayNameOffset(int day) {
   int offset = day + _first_day - 1;
   int result;
   QtedMonthdayButton *md_p;

   if (offset >= 0 && offset < MAX_LINES_PER_MONTH * DAYS_PER_WEEK) {
      md_p = _month_days[offset];
      if (md_p && md_p->day() != -1)
         result = offset % DAYS_PER_WEEK;   
   }
   return (result);
}

//*****************************************************************************

void QtedCalendar::
nextMonth() {
   _month++;
   if (_month > MAX_MONTH) {
      _year++;
      if (_year > MAX_YEAR) {
         _year--;
         _month--;   
      } else {
         _month = MIN_MONTH;
      }
   }
   update();
}

//*****************************************************************************

void QtedCalendar::
previousMonth() {
   _month--;
   if (_month < MIN_MONTH) {
      _year--;
      if (_year < MIN_YEAR) {
         _month++;
         _year++;
      } else {
         _month = MAX_MONTH;
      }
   }
   update();
}

//*****************************************************************************

void QtedCalendar::
nextYear() {
   if (_year < MAX_YEAR) {
      _year++;
      update();
   }
} 

//*****************************************************************************

void QtedCalendar::
previousYear() {
   if (_year > MIN_YEAR) {
      _year--;
      update();
   }
}

//*****************************************************************************

void QtedCalendar::
update(void) {
   int i, days[MAX_LINES_PER_MONTH * DAYS_PER_WEEK];
   QString str;
   _max_day = 0;

   str.sprintf("%s %d", month_names[_month - 1], _year);
   _header_p->setText(str);

   day_array (_month, _year, days);

   for (i = 0; i < MAX_LINES_PER_MONTH * DAYS_PER_WEEK; i++) {
      if (days[i] == 1)
         _first_day = i;
      if (days[i] > _max_day)
         _max_day = days[i];
      _month_days[i]->day(days[i]);
   }

   if (linesNeeded() != _lines)
      resizeEvent(NULL);

   emit monthChanged();
}

//*****************************************************************************

int QtedCalendar::
linesNeeded(void) {
   int result;

   if (_month_days[(MAX_LINES_PER_MONTH - 2) * DAYS_PER_WEEK]->day() <= 0)
      result = MAX_LINES_PER_MONTH - 2; 
   else if (_month_days[(MAX_LINES_PER_MONTH - 1) * DAYS_PER_WEEK]->day() <= 0)
      result = MAX_LINES_PER_MONTH - 1; 
   else
      result = MAX_LINES_PER_MONTH;

   return (result);
}

//*****************************************************************************

void QtedCalendar::
resizeEvent(QResizeEvent *event) {
   int x, y = 0, next_y = 0, line, day;
   int ht, pad = 5;

   _lines = linesNeeded();
   next_y = y + height() / 20;
   _header_p->setGeometry(0, y, width(), next_y - y);
   y = next_y;
   next_y = y + height() / 20;
   _wdl_p->setGeometry(0, y, width(), next_y - y);
   y = next_y;

   if (double_left_height > fontMetrics().lineSpacing() + 6)
      ht = _dbl_left_p->height();
   else
      ht = _close_p->height();

   _monthdayButtons->setGeometry(0, y, width(), height() - y - ht - pad * 2);
// The geometry of buttons in a group is relative to the button group geometry
   y = 0;
   for (line = 0; line < MAX_LINES_PER_MONTH; line++) {
      if (line >= _lines) {	// Hide buttons that are not displayed
         for (day = 0; day < DAYS_PER_WEEK; day++)
            _month_days[line * DAYS_PER_WEEK + day]->setGeometry(0, 0, 0, 0); 
      } else {
         next_y = y + (_monthdayButtons->height() - y) / (_lines - line);
         int next_x;
         for (day = 0, x = 0; day < DAYS_PER_WEEK; day++) {
            next_x = x + (width() - x) / (DAYS_PER_WEEK - day);
            _month_days[line * DAYS_PER_WEEK + day]->setGeometry(x, y, 
                                       next_x - x, next_y - y);
            x = next_x;
         }
         y = next_y;
      }
   }

   int wid = _dbl_left_p->width();	// all arrow buttons are the same width

   y = height() - ht - pad;
   next_y = height();

   _dbl_left_p->setGeometry(width()/4 - wid, y, wid, ht);
   _left_arrow_p->setGeometry(width()/4, y, wid, ht);
   _right_arrow_p->setGeometry(width() * 3/4 - wid, y, wid, ht);
   _dbl_right_p->setGeometry(width() * 3/4, y, wid, ht);

   x = width() / 2 - _close_p->width() / 2; 
   _close_p->setGeometry(x, y, _close_p->width(), ht);
}

//*****************************************************************************
