//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include <stream.h>
#include <errno.h>
#include <gdbm.h>
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <qdstream.h>

#include "event.h"
#include "weekday_labels.h"
#include "cal.h"
#include "util.h"
#include "month_cache.h"
#include "remind.h"

//*****************************************************************************

Event::
Event(int year, int month, int day, QString comment, int hour, int minute) : 
      _year(year), _month(month), _day(day), _day_name_offset(0), 
      _comment(comment), _hour(hour), _minute(minute), _week_num(0), 
      _type(Individual) {
   init();
}

//*****************************************************************************

Event::
Event(UINT8 day_name_offset, QString comment, int hour, int minute) : 
      _year(0), _month(0), _day(0), _day_name_offset(day_name_offset), 
      _comment(comment), _hour(hour), _minute(minute), _week_num(0), 
      _type(Weekday) {
   init();
}

//*****************************************************************************

Event::
Event(int day, QString comment, int hour, int minute) : 
      _year(0), _month(0), _day(day), _day_name_offset(0), _comment(comment),
      _hour(hour), _minute(minute), 
      _week_num(0), _type(MonthDay) {
   init();
}

//*****************************************************************************

Event::
Event(UINT8 day_name_offset, int week_num, QString comment, int hour, 
      int minute) : 
      _year(0), _month(0), _day(0), _day_name_offset(day_name_offset), 
      _comment(comment), _hour(hour), _minute(minute), _week_num(week_num), 
      _type(MonthWkday) {
   init();
}

//*****************************************************************************

Event::
Event(int month, int day, QString comment, int hour, int minute) : 
      _year(0), _month(month), _day(day), _day_name_offset(0), 
      _comment(comment), _hour(hour), _minute(minute), _week_num(0), 
      _type(Yearly) {
   init();
}

//*****************************************************************************

Event::
Event(const Event& event) : _year(event._year), _month(event._month), 
	_day(event._day), _day_name_offset(event._day_name_offset),
        _week_num(event._week_num), _hour(event._hour), _minute(event._minute),
        _type(event._type), _comment(event._comment) {
   int *ip;
   Event *ep = (Event *)&event;
   for (ip = ep->_notify_list.first(); ip; ip = ep->_notify_list.next())
      _notify_list.append(new int(*ip));  

   this->QtedPersist::QtedPersist(event);
   
}

//*****************************************************************************

Event::
~Event() {
}

//*****************************************************************************

void Event::
init(void) {
   _notify_list.setAutoDelete(TRUE);
}

//*****************************************************************************

QDateTime Event::
dateTime(void) {
   QDateTime result(QDate((int)_year, (int)_month, (int)_day),
                    QTime((int)_hour, (int)_minute));
   return (result);
}

//*****************************************************************************

int Event::
hour(void) {
   return (_hour);
}

//*****************************************************************************

int Event::
hour12(void) {
   int hour = _hour;

   if (hour >= 12)
      hour -= 12;

   if (hour == 0)
      hour = 12;

   return (hour);
}

//*****************************************************************************

int Event::
minute(void) {
   return (_minute);
}

//*****************************************************************************

int Event::
isPm(void) {
   return (_hour >= 12);
}

//*****************************************************************************

QList<int> Event::
notifyList(void) {
   return (_notify_list);   
}

//*****************************************************************************

Event::EventType Event::
type(void) {
   return (_type);
}

//*****************************************************************************

QString Event::
eventText(void) {
   return (_comment);
}

//*****************************************************************************

void Event::
clearNotifyList(void) {
   _notify_list.clear();
}

//*****************************************************************************

void Event::
addNotifyTime(int minutes_before_event) {
   int *ip, found = 0;
   for (ip = _notify_list.first(); ip; ip = _notify_list.next()) {
      if (minutes_before_event == *ip) {
         found = 1;
         break;
      }
   }
   if (!found)
      _notify_list.append(new int(minutes_before_event));
}

//*****************************************************************************

QString Event::
string(int long_version) {
   QString result;
   if (long_version) {
      result.sprintf("%s, %s %02d:%02d %s - %s", month_names[_month - 1], 
                    (const char *)ordinal(_day), hour12(), minute(), 
                    isPm() ? "PM" : "AM", (const char*)_comment); 
   } else {
      result.sprintf("%02d:%02d %s - %s", hour12(), _minute, 
                     isPm() ? "PM" : "AM", (const char*)_comment); 
   }
   return (result);
}

//*****************************************************************************

int Event::
minutes(void) {
   return (_hour * 60 + _minute);
}

//*****************************************************************************

int Event::
matches(int year, int month, int day, UINT8 day_name_offset) {
   int result = 0;

   switch (_type) {
      case Individual:
         if (day == _day && month == _month && year == _year)
            result = 1;
         break;
      case Weekday:
         if (day_name_offset == _day_name_offset) {
            _year = year;
            _month = month;
            _day = day;
            result = 1;
         }
         break;
      case MonthDay:
         if (day == _day) {
            _year = year;
            _month = month;
            result = 1;
         }
         break;
      case MonthWkday:
         if (day_name_offset == _day_name_offset && 
             ((day - 1) / 7) + 1 == _week_num) {
            _year = year;
            _month = month;
            _day = day;
            result = 1;
         }
         break;
      case Yearly:
         if (day == _day && month == _month) {
            _year = year;
            result = 1;
         }
         break;
   }

   return (result);
}

//*****************************************************************************

int Event::
operator < (Event &ev) {
   return (minutes() < ev.minutes());
}

//*****************************************************************************

int Event::
operator == (Event &ev) {
   return (minutes() == ev.minutes());
}

//*****************************************************************************
//
//	This function assumes that the QDate passed in is a day the this event
//	is supposed to be on.
//
//*****************************************************************************

QList<QDateTime> Event::
notifyTimes(QDate a_date) {
   time_t event_seconds;	// I need these time variables 
   QList<QDateTime> result;

   event_seconds = epochSeconds(QDateTime(a_date, QTime(_hour, _minute, 0)));

   for (int *ip = _notify_list.first(); ip; ip = _notify_list.next()) {
      QDateTime *date_time_p = new QDateTime;
      date_time_p->setTime_t(event_seconds - (*ip * 60));
      result.append(date_time_p);
   }

   return (result); 
}

//*****************************************************************************
// DEBUG 
//*****************************************************************************

void Event::
show(void) {
   cerr << "+ Event::show()\n";
   cerr << "   type     = ";
   switch (_type) {
      case Individual:
         cerr << "Individual\n";
         cerr << "   year     = " << _year << "\n";
         cerr << "   month    = " << (int)_month << "\n";
         cerr << "   day      = " << (int)_day << "\n";
         cerr << "   hour     = " << (int)_hour << "\n";
         cerr << "   minute   = " << (int)_minute << "\n";
         cerr << "   comment  = " << _comment << "\n";
         break;
      case Weekday:
         cerr << "Weekday\n";
         cerr << "   day_name = " << weekdayNames[_day_name_offset] << "\n";
         cerr << "   hour     = " << (int)_hour << "\n";
         cerr << "   minute   = " << (int)_minute << "\n";
         cerr << "   comment  = " << _comment << "\n";
         break;
      case MonthDay:
         cerr << "MonthDay\n";
         cerr << "   day      = " << (int)_day << "\n";
         cerr << "   hour     = " << (int)_hour << "\n";
         cerr << "   minute   = " << (int)_minute << "\n";
         cerr << "   comment  = " << _comment << "\n";
         break;
      case MonthWkday:
         cerr << "MonthWkday\n";
         cerr << "   day_name = " << weekdayNames[_day_name_offset] << "\n";
         cerr << "   week_num = " << (int)_week_num << "\n";
         cerr << "   hour     = " << (int)_hour << "\n";
         cerr << "   minute   = " << (int)_minute << "\n";
         cerr << "   comment  = " << _comment << "\n";
         break;
      case Yearly:
         cerr << "Yearly\n";
         cerr << "   month    = " << (int)_month << "\n";
         cerr << "   day      = " << (int)_day << "\n";
         cerr << "   hour     = " << (int)_hour << "\n";
         cerr << "   minute   = " << (int)_minute << "\n";
         cerr << "   comment  = " << _comment << "\n";
         break;
   }
   int *ip;
   cerr << "  notify  = ";
   for (ip = _notify_list.first(); ip; ip = _notify_list.next())
      cerr << *ip << " ";
   cerr << "\n";

   cerr << "- Event::show()\n";
}

//*****************************************************************************
//  These functions are needed by QtedPersist
//*****************************************************************************

Event::
Event(int key, QByteArray data) : QtedPersist(key) {
   dbRestore(data);
}

//*****************************************************************************

const char *Event::
dbClassName(void) { 
   return("Event"); 
}

//*****************************************************************************

int Event::
dbSave(void) {
   int result = 0;
   if (dbIsDirty()) {
      datum key, content;
      QByteArray data;
      QDataStream ds(data, IO_WriteOnly);
      ds << (int)_type << _year << _month << _day << _day_name_offset 
         << _week_num << _hour << _minute << _comment;
      
      int *ip;
      for (ip = _notify_list.first(); ip; ip = _notify_list.next())
         ds << *ip;
      ds << (int)-1;

      result = QtedPersist::dbSave(data);
   }
   return (result);
}

//*****************************************************************************

int Event::
dbRestore(QByteArray data) {
   int result = 0, type;
   QDataStream ds(data, IO_ReadOnly);
   ds >> type >> _year >> _month >> _day >> _day_name_offset >> _week_num 
      >> _hour >> _minute >> _comment;

   int notify;
   for(ds >> notify; notify != -1; ds >> notify)
      _notify_list.append(new int(notify));

   _type = (EventType)type;

   return (result);
}

//*****************************************************************************

Event *Event::
getByKey(int key) {
   Event *result = NULL;
   QByteArray *data = QtedPersist::dbGet("Event", key);
   if (data) {
      result = new Event(key, *data);
      delete data;
   }
   return (result);
}

//*****************************************************************************
//
//	This searches the database for events for a given month
//	and loads a MonthCache class
//
//*****************************************************************************

MonthCache *Event::
getByMonth(QtedCalendar *cal_p) {
   int  key = 0;
   Event *ep;
   MonthCache *result = new MonthCache(cal_p); 

   if (dbOpen("Event", GDBM_READER)) {
      while (key = dbNextRec("Event", key)) {
         ep = getByKey(key);
         if (ep && result->addEvent(ep) == 0)         
            delete (ep);
      }
      dbClose("Event");
   }

   return(result);
}

//*****************************************************************************

void Event::
loadAlarms(Remind *remind_p) {
   int  key = 0;
   Event *ep;
   if (dbOpen("Event", GDBM_READER)) {
      while (key = dbNextRec("Event", key)) {
         if (ep = getByKey(key)) {
            if (!remind_p->addToAlarmList(ep)) {
               delete (ep);
            }
         }
      }
      dbClose("Event");
   }
}

//*****************************************************************************
