//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include <qbitmap.h>

#include "cal.h"
#include "util.h"
#include "event.h"
#include "event_edit.h"
#include "calendar.h"
#include "month_cache.h"
#include "settime.h"
#include "remind.h"

#include "event_edit.xbm"

//*****************************************************************************

EventEdit::
EventEdit(Remind *remind_p, MonthCache *month_cache_p, 
          QtedMonthdayButton *mdb_p, int offset, 
          QWidget *parent, const char *name) : QWidget(parent, name), 
          _remind_p(remind_p), _month_cache_p(month_cache_p), 
          _selected_event(-1) {
   QString str;

   QBitmap icon(event_edit_width, event_edit_height, event_edit_bits, TRUE);
   setIcon(icon);

   _month_cache_p->incRefCount();

   if (mdb_p) {
      _day_name_offset = offset % DAYS_PER_WEEK;
      _week_num = (mdb_p->day() - 1) / DAYS_PER_WEEK + 1;
      _month = month_cache_p->month();
      _day = mdb_p->day();
      _year = month_cache_p->year();
      str.sprintf("%s %s, %s %d", weekdayNames[_day_name_offset],
                              month_names[month_cache_p->month() - 1],
                              (const char *)ordinal(mdb_p->day()), _year);
   }
   setCaption(str);

   setupEventTimeFrame();
   setupTypeButtonGroup(mdb_p, month_cache_p->month(), offset);
   setupNotifyButtonGroup();
   _text_label_p = new QLabel("Text", this);
   _line_edit_p = new QLineEdit(this);

   _list_box_p = new QListBox(this);
   _list_box_p->setFocusPolicy(NoFocus);
   _save_button_p = new QPushButton("Save", this);
   _dismiss_button_p = new QPushButton("Dismiss", this);
   _deselect_button_p = new QPushButton("Deselect", this);
   _deselect_button_p->setEnabled(FALSE);
   _delete_button_p = new QPushButton("Delete", this);
   _delete_button_p->setEnabled(FALSE);

   connect(_save_button_p, SIGNAL(clicked()), this, SLOT(savePressed()));
   connect(_dismiss_button_p, SIGNAL(clicked()), this, SLOT(dismissPressed()));
   connect(_deselect_button_p, SIGNAL(clicked()), this, SLOT(newPressed()));
   connect(_delete_button_p, SIGNAL(clicked()), this, SLOT(deletePressed()));
   connect(_list_box_p, SIGNAL(highlighted(int)),this,SLOT(itemSelected(int)));

   resizeEvent(NULL);
   setMinimumSize(_min_x, _min_y);

   resetForm();
   updateListBox();
}

//*****************************************************************************

EventEdit::
~EventEdit() {
   _month_cache_p->decRefCount();
}

//*****************************************************************************

void EventEdit::
setupEventTimeFrame(void) {
   _event_time_frame_p = new QFrame(this);
   _event_time_frame_p->setFrameStyle(QFrame::Box|QFrame::Sunken);
   _event_time_label_p = new QLabel("Event Time", _event_time_frame_p);
   _event_time_label_p->setAlignment(AlignHCenter);
   _event_time_p = new QtedSetTime(0, 0, 0, QtedSetTime::TimeOfDay, 
                                   _event_time_frame_p);
   int fw, label_space, space = 10;
   fw = _event_time_frame_p->frameWidth();
   label_space = (int)(fontMetrics().lineSpacing() * 1.5);
   _event_time_frame_p->setMinimumSize(_event_time_p->width() + 2*(fw + space),
               _event_time_p->height() + 2 * fw + space + label_space);
   _event_time_label_p->setGeometry(fw + space, fw,
                      _event_time_p->width(), fontMetrics().lineSpacing());
   _event_time_p->setGeometry(fw + space, label_space,
                      _event_time_p->width(), _event_time_p->height()); 
}

//*****************************************************************************

void EventEdit::
setupTypeButtonGroup(QtedMonthdayButton *mdb_p, int month, int offset) {
   int radio_button_spacing = (int)(fontMetrics().lineSpacing() * 1.5); 
   int x_space = 10;

   _event_type_bg_p = new QButtonGroup(this);
   _event_type_label_p = new QLabel("Event Type", _event_type_bg_p);
   _event_type_label_p->setAlignment(AlignHCenter);
   QString str;
   int y = radio_button_spacing;
   str = "Individual";
   _ind_p = new QRadioButton(str, _event_type_bg_p);
   _ind_p->setGeometry(x_space, y, _ind_p->sizeHint().width(), 
                              _ind_p->sizeHint().height()); 
   y += radio_button_spacing;

   str.sprintf("Every %s", weekdayNames[offset % DAYS_PER_WEEK]);
   
   _weekly_p = new QRadioButton(str, _event_type_bg_p);
   _weekly_p->setGeometry(x_space, y, _weekly_p->sizeHint().width(), 
                                 _weekly_p->sizeHint().height()); 
   y += radio_button_spacing;

   str.sprintf("Every %s %s of the month", 
           (const char*) ordinal((mdb_p->day() - 1) / DAYS_PER_WEEK + 1 ), 
           (const char*) weekdayNames[offset % DAYS_PER_WEEK]);
   _ordinal_weekday_p = new QRadioButton(str, _event_type_bg_p);
   _ordinal_weekday_p->setGeometry(x_space, y, 
                                 _ordinal_weekday_p->sizeHint().width(), 
                                 _ordinal_weekday_p->sizeHint().height()); 
   y += radio_button_spacing;
   
   str.sprintf("Every %s of the month", (const char *)ordinal(mdb_p->day()));
   _day_of_month_p = new QRadioButton(str, _event_type_bg_p);
   _day_of_month_p->setGeometry(x_space, y, 
                                 _day_of_month_p->sizeHint().width(), 
                                 _day_of_month_p->sizeHint().height()); 
   y += radio_button_spacing;

   str.sprintf("Every %s %s", month_names[month - 1], 
                              (const char *)ordinal(mdb_p->day()));
   _yearly_p = new QRadioButton(str, _event_type_bg_p);
   _yearly_p->setGeometry(x_space, y, _yearly_p->sizeHint().width(), 
                                 _yearly_p->sizeHint().height()); 
   y += radio_button_spacing;

   int w = x_space * 2 + _ordinal_weekday_p->sizeHint().width();
   _event_type_bg_p->setMinimumSize(w, y);
   _event_type_label_p->setGeometry(_event_type_bg_p->frameWidth(), 
                                    _event_type_bg_p->frameWidth(),
                                    w - (_event_type_bg_p->frameWidth() * 2), 
                                    fontMetrics().lineSpacing());
}

//*****************************************************************************

void EventEdit::
setupNotifyButtonGroup(void) {
   int radio_button_spacing = (int)(fontMetrics().lineSpacing() * 1.5); 
   int x_space = 10;

   _notify_bg_p = new QButtonGroup(this);
   _notify_time_label_p = new QLabel("Notify Time", _notify_bg_p);
   _notify_time_label_p->setAlignment(AlignHCenter);
   _notify_time_button_p = new QCheckBox(_notify_bg_p);
   _notify_time_p = new QtedSetTime(0, 0, 0, QtedSetTime::Interval,
                                             _notify_bg_p); 
   _notify_day_button_p  = new QCheckBox("1 day", _notify_bg_p);
   _notify_week_button_p = new QCheckBox("1 week", _notify_bg_p);
   
   int y = radio_button_spacing;
   _notify_time_button_p->setGeometry(x_space, y, 
          _notify_time_button_p->sizeHint().width(), 
          _notify_time_button_p->sizeHint().height()); 
   _notify_time_p->setGeometry(x_space * 2 + _notify_time_button_p->width(), y,
                               _notify_time_p->width(), 
                               _notify_time_p->height()); 
   y += _notify_time_p->height(); 

   _notify_day_button_p->setGeometry(x_space, y, 
                               _notify_day_button_p->sizeHint().width(),
                               _notify_day_button_p->sizeHint().height());

   y += radio_button_spacing; 

   _notify_week_button_p->setGeometry(x_space, y, 
                               _notify_week_button_p->sizeHint().width(),
                               _notify_week_button_p->sizeHint().height());

   y += radio_button_spacing;
   int w = x_space * 3 + _notify_time_button_p->width()+_notify_time_p->width();
   _notify_bg_p->setMinimumSize(w, y);
   _notify_time_label_p->setGeometry(_notify_bg_p->frameWidth(), 
                                     _notify_bg_p->frameWidth(),
                     w - _notify_bg_p->frameWidth() * 2,
                     fontMetrics().lineSpacing());
}

//*****************************************************************************

void EventEdit::
resizeEvent(QResizeEvent *the_event) {
   int x, x2, y, space = 10;
   x = y = space;

   QSize size = _notify_bg_p->minimumSize();
   _event_time_frame_p->setGeometry(x, y, _event_time_frame_p->width(), 
                                          size.height());
   x += _event_time_frame_p->width();
   _notify_bg_p->setGeometry(x, y, size.width(), size.height());

   x2 = x + size.width() + space;
   x = space;
   y += _notify_bg_p->height();

   size = _event_type_bg_p->minimumSize();
   _event_type_bg_p->setGeometry(x, y, 
        _event_time_frame_p->width() + _notify_bg_p->width(), size.height());
   y += size.height() + 5;
   _text_label_p->setGeometry(x, y, fontMetrics().width(_text_label_p->text()),
                 fontMetrics().lineSpacing() + 8);
   x += fontMetrics().width(_text_label_p->text()) + 5;

   _line_edit_p->setGeometry(x, y, _event_type_bg_p->width() - x + space, 
                 fontMetrics().lineSpacing() + 8);

   y += fontMetrics().lineSpacing() + space;
   _min_y = y;

   int spacing, btn_width, btn_height;
   btn_width = fontMetrics().width("Deselect") + space;
   btn_height = fontMetrics().lineSpacing() + 6; 
   spacing =  width() / 5;
   x = spacing - (btn_width / 2);
   y = height() - space - btn_height;
   _save_button_p->setGeometry(x, y, btn_width, btn_height);
   x += spacing;
   _dismiss_button_p->setGeometry(x, y, btn_width, btn_height);
   x += spacing;
   _delete_button_p->setGeometry(x, y, btn_width, btn_height);
   x += spacing;
   _deselect_button_p->setGeometry(x, y, btn_width, btn_height);
   _list_box_p->setGeometry(x2, space, width() - x2 - space * 2, 
                                   height() - space * 3 - btn_height);
   _min_y += btn_height + space;
   _min_x = width();
}

//*****************************************************************************

void EventEdit::
resetForm(void) {
   _selected_event = -1;
   _event_time_p->setTime(8, 0);
   _notify_time_p->setTime(0, 10);
   _ind_p->setChecked(TRUE);
   _weekly_p->setChecked(FALSE);
   _ordinal_weekday_p->setChecked(FALSE);
   _day_of_month_p->setChecked(FALSE);
   _yearly_p->setChecked(FALSE);
   _notify_time_button_p->setChecked(TRUE);
   _notify_day_button_p->setChecked(FALSE);
   _notify_week_button_p->setChecked(FALSE);
   _line_edit_p->setText("");
}

//*****************************************************************************

void EventEdit::
loadForm(Event *ep) {
   int *ip;
   QList<int> notify_list;

   _event_time_p->setTime(ep->hour12(), ep->minute(), ep->isPm());
   notify_list = ep->notifyList();
   _notify_time_button_p->setChecked(FALSE);
   _notify_day_button_p->setChecked(FALSE);
   _notify_week_button_p->setChecked(FALSE);
   for (ip = notify_list.first(); ip; ip = notify_list.next()) {
      int minutes = *ip;
      if (minutes < (24 * 60)) {
         _notify_time_p->setTime(minutes/60, minutes%60);
         _notify_time_button_p->setChecked(TRUE); 
      } else if (minutes == (24 * 60)) {
         _notify_day_button_p->setChecked(TRUE); 
      } else if (minutes == (7 * 24 * 60)) {
         _notify_week_button_p->setChecked(TRUE); 
      }
   }
   _ind_p->setChecked(FALSE);
   _weekly_p->setChecked(FALSE);
   _ordinal_weekday_p->setChecked(FALSE);
   _day_of_month_p->setChecked(FALSE);
   _yearly_p->setChecked(FALSE);
   switch(ep->type()) {
      case Event::Individual:
         _ind_p->setChecked(TRUE);
         break;
      case Event::Weekday:
         _weekly_p->setChecked(TRUE);
         break;
      case Event::MonthDay:
         _day_of_month_p->setChecked(TRUE);
         break;
      case Event::MonthWkday:
         _ordinal_weekday_p->setChecked(TRUE);
         break;
      case Event::Yearly:
         _yearly_p->setChecked(TRUE);
         break;
   } 
   _line_edit_p->setText(ep->eventText());
}

//*****************************************************************************

void EventEdit::
reset(void) {
   updateListBox();
   resetForm();
   _month_cache_p->updateCalendar();
}

//*****************************************************************************

void EventEdit::
updateListBox(void) {
   _list_box_p->clear();
   QtedSortList<Event> *event_list = _month_cache_p->dayList(_day);

   if (event_list) {
      Event *ep;
      for (ep = event_list->first(); ep; ep = event_list->next())
         _list_box_p->insertItem(ep->string());
   }
   _delete_button_p->setEnabled(FALSE);
   _deselect_button_p->setEnabled(FALSE);
}

//*****************************************************************************

void EventEdit::
savePressed() {
   Event *ep;
   QtedSortList<Event> *event_list_p = _month_cache_p->dayList(_day);

   if (_selected_event >= 0 && event_list_p) {
      ep = event_list_p->at(_selected_event);
      _remind_p->removeEvent(ep->dbKey());
      _month_cache_p->removeEvent(ep);
      ep->dbDelete();
      delete (ep);
   }

   if (_ind_p->isChecked())
      ep = new Event(_year, _month, _day, _line_edit_p->text(), 
                     _event_time_p->getHour24(), _event_time_p->getMinute()); 
   else if (_weekly_p->isChecked())
      ep = new Event(_day_name_offset, _line_edit_p->text(), 
                     _event_time_p->getHour24(), _event_time_p->getMinute());
   else if (_ordinal_weekday_p->isChecked())
      ep = new Event(_day_name_offset, _week_num, _line_edit_p->text(), 
                     _event_time_p->getHour24(), _event_time_p->getMinute());
   else if (_day_of_month_p->isChecked())
      ep = new Event(_day, _line_edit_p->text(), 
                     _event_time_p->getHour24(), _event_time_p->getMinute());
   else if (_yearly_p->isChecked())
      ep = new Event(_month, _day, _line_edit_p->text(), 
                     _event_time_p->getHour24(), _event_time_p->getMinute());
   _month_cache_p->addEvent(ep);
   if (_notify_time_button_p->isChecked())
      ep->addNotifyTime(_notify_time_p->getHour() * 60 + 
                        _notify_time_p->getMinute());
   if (_notify_day_button_p->isChecked())
      ep->addNotifyTime(24*60); 
   if (_notify_week_button_p->isChecked())
      ep->addNotifyTime(7*24*60); 

   ep->dbSave();
   _remind_p->addToAlarmList(ep, TRUE); 
   reset();
}

//*****************************************************************************

void EventEdit::
newPressed() {
   resetForm();
   updateListBox(); // It would be nice to be able to make no item selected
}

//*****************************************************************************

void EventEdit::
deletePressed() {
   int current_item = _list_box_p->currentItem();
   if (current_item >= 0) {
      QtedSortList<Event> *event_list_p = _month_cache_p->dayList(_day);

      if (current_item >= 0 && event_list_p) {
         Event *ep;
         ep = event_list_p->at(current_item);
         _remind_p->removeEvent(ep->dbKey());
         _month_cache_p->removeEvent(ep);
         ep->dbDelete();
         delete(ep);
         _list_box_p->removeItem(current_item);
         _month_cache_p->updateCalendar();
         reset();
      }
   }
}

//*****************************************************************************

void EventEdit::
dismissPressed() {
   delete (this);
}

//*****************************************************************************

void EventEdit::
itemSelected(int i) {
   QtedSortList<Event> *event_list = _month_cache_p->dayList(_day);
   _selected_event = i;
   _deselect_button_p->setEnabled(TRUE);
   _delete_button_p->setEnabled(TRUE);
   if (event_list) {
      loadForm(event_list->at(i));
   } else {
      cerr << "EventEdit::itemSelected(" << i << ") item selected but can't ";
      cerr << "get event list\n";
   }
}

//*****************************************************************************
