//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include "event.h"
#include "month_cache.h"

QList<MonthCache> MonthCache::_full_list;

//*****************************************************************************
//
//	The cal_p is set upon entry but may be set to NULL if the calendar 
//	is changed.
//

MonthCache::
MonthCache(QtedCalendar *cal_p) : _year(cal_p->year()), _month(cal_p->month()),
                                  _max_day(cal_p->maxDay()), 
                                  _first_day_offset(cal_p->dayNameOffset(1)), 
                                  _ref_count(0), _cal_p(cal_p) {
   _full_list.append(this);
}

//*****************************************************************************

MonthCache::
~MonthCache() {
   int day, i;
   QList<Event> event_list, *event_list_p;
   
   event_list.setAutoDelete(TRUE);

   for (day = 1; day <= _max_day; day++) {
      if (event_list_p = _day_list_dict.find(day)) {
         Event *ep;
         for (ep = event_list_p->first(); ep; ep = event_list_p->next()) {
            if (!event_list.findRef(ep))
               event_list.append(ep);
         }
      } 
   }
   _full_list.removeRef(this);
}

//*****************************************************************************

MonthCache *MonthCache::
get(QtedCalendar *cal_p) {
   MonthCache *mc_p, *result = NULL;

   for (mc_p = _full_list.first(); mc_p; mc_p = _full_list.next()) {
      if (mc_p->year() == cal_p->year() && mc_p->month() == cal_p->month()) {
         result = mc_p;
         mc_p->cal(cal_p);
         break;
      }
   }

   if (!result)
      result = Event::getByMonth(cal_p);

   return (result);
}

//*****************************************************************************

int MonthCache::
year(void) {
   return (_year);
}

//*****************************************************************************

int MonthCache::
month(void) {
   return (_month);
}

//*****************************************************************************

void MonthCache::
incRefCount(void) {
   _ref_count++;
}

//*****************************************************************************

void MonthCache::
decRefCount(void) {
   _ref_count--;
   if (_ref_count == 0 && _cal_p == NULL)
      delete (this);
}

//*****************************************************************************

void MonthCache::
cal(QtedCalendar *cal_p) {
   _cal_p = cal_p;

   if (_cal_p == NULL && _ref_count == 0)
      delete (this);
}

//*****************************************************************************

int MonthCache::
dayNameOffset(int day) {
   int offset = day + _first_day_offset - 1;
   return (offset % DAYS_PER_WEEK);
}

//*****************************************************************************

QtedSortList<Event> *MonthCache::
dayList(int day) { 
   return(_day_list_dict.find(day));
}

//*****************************************************************************
//
//	Returns true if event was added to the cache
//
//*****************************************************************************

int MonthCache::
addEvent(Event *ep) {
   int day, result = 0;
   QtedSortList<Event> *event_list_p;

   for (day = 1; day <= _max_day; day++) {
      if (ep->matches(_year, _month, day, dayNameOffset(day))) {
         event_list_p = _day_list_dict.find(day);
         if (event_list_p == NULL) {
            event_list_p = new QtedSortList<Event>;
            _day_list_dict.insert(day, event_list_p);
         }
         result = 1; 
         event_list_p->inSort(ep); 
      }
   }
   return (result);
}

//*****************************************************************************
//
//	Returns true if the event was present and was removed from cache
//
//*****************************************************************************

int MonthCache::
removeEvent(Event *ep) {
   int day, found = 0;
   QtedSortList<Event> *event_list_p;

   for (day = 1; day <= _max_day; day++)
      if (event_list_p = _day_list_dict.find(day))
         if (event_list_p->removeRef(ep))
            found = 1;

   return (found);
}

//*****************************************************************************
//
//	Return true if event is still in cache
//
//*****************************************************************************

int MonthCache::
updateEvent(Event *ep) {
   removeEvent(ep);
   return (addEvent(ep));
}

//*****************************************************************************

void MonthCache::
updateCalendar(void) {
   if (_cal_p) {
      int day;
      QtedSortList<Event> *event_list_p;
      QtedMonthdayButton *mdb_p;

      for (day = 1; day <= _max_day; day++) {
         if (mdb_p = _cal_p->getMonthdayButtonByDay(day)) {
            mdb_p->clearText();
            if (event_list_p = dayList(day)) {
               Event *ep;
               for (ep = event_list_p->first(); ep; ep = event_list_p->next()) {
                  mdb_p->addLine(ep->string(), black);
               }
            }
         }
      }
   }
}

//*****************************************************************************
