//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include <time.h>
#include <qapp.h>
#include <qmsgbox.h> 
#include <qbitmap.h>

#include "util.h"
#include "remind.h"
#include "month_cache.h"
#include "event_edit.h"
#include "event.h"

#include "remind.xbm"

//*****************************************************************************

Remind::
Remind(QApplication *app, QWidget *parent, const char *name) : 
       QWidget(parent, name), _app(app), _month_cache_p(NULL), _cal_p(NULL), 
       _timer_p(NULL) {
   QBitmap icon(remind_width, remind_height, remind_bits, TRUE);
   setIcon(icon);

   _main_button_p = new QPushButton("Main", this);
   _exit_button_p = new QPushButton("Exit", this);

   _update = _start = _end = QDateTime::currentDateTime(); // start time is now
   _end = _end.addDays(7);			 // end time is one week
   _end.setTime(QTime(23, 59));

   Event::loadAlarms(this);

   _start.setTime(QTime(23, 59));
   _start = _start.addDays(6);		// set _start time to _end - 1 day
   _update.setTime(QTime(23, 59));	// update just before midnight
   heartBeat();

   connect(_main_button_p, SIGNAL(clicked()), this, SLOT(mainClicked()));
   connect(_exit_button_p, SIGNAL(clicked()), qApp, SLOT(quit()));

   resizeEvent(NULL);
}

//*****************************************************************************

Remind::
~Remind() {
   if (_month_cache_p)
      _month_cache_p->cal(NULL);
}

//*****************************************************************************
//
//      Return true if added to list
//
//*****************************************************************************

int Remind::
addToAlarmList(Event *ep, int use_current_time) {
   int result = 0;
   QDateTime t, start_time;

   if (use_current_time)
      start_time = QDateTime::currentDateTime();
   else
      start_time = _start;

   for (t = start_time; t < _end; t = t.addDays(1)) {
      QDate date = t.date();
      int day_name_offset = date.dayOfWeek();
      if (day_name_offset == 7)
         day_name_offset = 0;
      if (ep->matches(date.year(), date.month(), date.day(), day_name_offset)) {
         QList<QDateTime> alarm_time_list = ep->notifyTimes(date);
         alarm_time_list.setAutoDelete(TRUE);
         QDateTime *dt_p;
         for (dt_p = alarm_time_list.first(); dt_p;
              dt_p = alarm_time_list.next()) {
            if (*dt_p >= start_time && *dt_p < _end) {
               result = 1;
               _alarm_list.inSort(new AlarmEntry(*dt_p, ep));
            }
         }
      }
   }

   return (result);
}

//*****************************************************************************
//
// The event pointer is not the event that is on this list, it is a copy
// Therefore, we need to find the event on this list by dbKey then remove the
// event that is on this list
//
//*****************************************************************************

void Remind::
removeEvent(unsigned int key) {
   _alarm_list.removeEvent(key);
}

//*****************************************************************************

void Remind::
heartBeat() {
   char time_str[64];
   struct tm *time_p;
   time_t time_sec;

// update main button time
   time_sec = time(NULL);		// get current time
   time_p = localtime(&time_sec);	// convert to local into tm struct
   strftime(time_str, 64, "%a %b %d %Y %I:%M %p", time_p);

   _main_button_p->setText(time_str);
   _main_button_p->update();

// fire any expired notifications
   AlarmEntry *ae_p;
   QDateTime time_now(QDateTime::currentDateTime());
   while (ae_p = _alarm_list.expired(time_now)) {
      notify(ae_p->event());
      delete(ae_p);
   }

// See if we need to update the main list
   if (time_now >= _update) {
      _update = _update.addDays(1);
      _start = _start.addDays(1);
      _end = _end.addDays(1);
      Event::loadAlarms(this);
   }

   if (!_timer_p) {
      _timer_p = new QTimer(this);
      connect(_timer_p, SIGNAL(timeout()), this, SLOT(heartBeat())); 
   }
   _timer_p->changeInterval((60 - time_p->tm_sec) * 1000);
}

//*****************************************************************************

void Remind::
dayClicked(int offset) {
   QtedMonthdayButton *mdb_p;
   EventEdit *ev_edit_p;

   ev_edit_p = new EventEdit(this, _month_cache_p, 
                             _cal_p->getMonthdayButton(offset), offset);
   ev_edit_p->show();
}

//*****************************************************************************

void Remind::
monthChanged() {
   int day;
   QtedSortList<Event> *event_list_p;
   QtedMonthdayButton *mdb_p;

   if (_month_cache_p) {		// if we have a month cache
      _month_cache_p->cal(NULL);	// set the cal ptr to NULL
   }
   _month_cache_p = MonthCache::get(_cal_p);
   _month_cache_p->updateCalendar();
}

//*****************************************************************************

void Remind::
mainClicked() {
   struct tm *time_p;
   time_t time_sec;

   _main_button_p->setEnabled(FALSE);
   time_sec = time(NULL);
   time_p = localtime(&time_sec);

   _cal_p = new QtedCalendar(time_p->tm_mon + 1, 1900 + time_p->tm_year);
   _cal_p->show();
   connect(_cal_p, SIGNAL(dayClicked(int)), this, SLOT(dayClicked(int)));
   connect(_cal_p, SIGNAL(closeClicked()), this, SLOT(calClose()));
   connect(_cal_p, SIGNAL(monthChanged()), this, SLOT(monthChanged()));
   monthChanged();
}

//*****************************************************************************

void Remind::
calClose() {
   _month_cache_p->cal(NULL);
   delete (_cal_p);
   _cal_p = NULL;
   _month_cache_p = NULL;
   _main_button_p->setEnabled(TRUE);
}

//*****************************************************************************

void Remind::
resizeEvent(QResizeEvent *) {
   int space = 10;

   int wid1 = fontMetrics().width("Mon September 30 20000 00:00 PM");
   int wid2 = fontMetrics().width(_exit_button_p->text());
   int ht = fontMetrics().lineSpacing();

   setMinimumSize(wid1 + wid2 + 2 * space, ht + space);
   setMaximumSize(wid1 + wid2 + 2 * space, ht + space);

   _main_button_p->setGeometry(0, 0, wid1 + space, ht + space); 
   _exit_button_p->setGeometry(wid1 + space, 0, wid2 + space, ht + space); 
}

//*****************************************************************************

void Remind::
notify(Event *ep) {
   _app->beep();
   QString text;
   QDateTime time_now = QDateTime::currentDateTime(); 
   int seconds = epochSeconds(ep->dateTime()) - epochSeconds(time_now);
   seconds += 30;

   int minutes = seconds / 60;
   int days = minutes / (60 * 24);

   if (days == 0) {
      if (minutes)
         text.sprintf("%d minute%s advanced notification\n", minutes, 
                     minutes > 1 ? "s" : "");
   } else {
      text.sprintf("%d day%s advanced notification\n", days, 
                   days > 1 ? "s" : "");
   }
   text += ep->string(1);
   QMessageBox::message("Reminder", text);
}

//*****************************************************************************
