/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "komBase.h"
#include "komApplication.h"

int cmpIOR( CORBA::Object_ptr p1, CORBA::Object_ptr p2 )
{
  CORBA::String_var s1 = komapp_orb->object_to_string( p1 );
  CORBA::String_var s2 = komapp_orb->object_to_string( p2 );

  return strcmp( s1, s2 );
}

KOMBase::KOMBase()
{
  m_ulRefCount = 0;
  
  m_dctSignals.setAutoDelete( true );
  m_lstFilters.setAutoDelete( true );
  m_lstClients.setAutoDelete( true );

  m_bIsClean = false;
}
KOMBase::~KOMBase()
{
  if ( m_bIsClean )
    return;

  disconnectAll();

  m_bIsClean = true;
}

void KOMBase::cleanUp()
{
  if ( m_bIsClean )
    return;

  disconnectAll();

  m_bIsClean = true;
}

/***********************************************
 *
 * Signals & Slots
 *
 ***********************************************/

void KOMBase::connect( const char* _signal, KOM::Base_ptr obj, const char* function ) 
{
  // if no exception ....
  ClientList *l = m_dctSignals[ _signal ];
  if ( !l )
    mico_throw( KOM::Base::UnknownSignal() );
 
  // TODO: catch exceptions
  l->append( new Client( obj, function ) );
  obj->connectNotify( this );
}

void KOMBase::connectNotify( KOM::Base_ptr _sig )
{
  KOM::Base_var *v = new KOM::Base_var;
  (*v) = KOM::Base::_duplicate( _sig );
  m_lstClients.append( v );
}

void KOMBase::disconnectNotify( KOM::Base_ptr obj )
{
  disconnectObjectIntern( obj, false );
}

void KOMBase::disconnect( const char *_signal, KOM::Base_ptr _obj, const char* _function )
{
  // Find signal
  ClientList *l = m_dctSignals[ _signal ];
  if ( !l )
    mico_throw( KOM::Base::UnknownSignal() );

  // Disconnect from signal
  QListIterator<Client> it( *l );
  for ( ; it.current(); ++it )
  {
    if ( cmpIOR( _obj, it.current()->obj ) == 0L && it.current()->function == _function )
    {
      it.current()->obj->disconnectSignalNotify( this, _signal, _function );
      l->removeRef( it.current() );
      return;
    }
  }
}

void KOMBase::disconnectSignalNotify( KOM::Base_ptr obj, const char * /* _signal */, const char * /* _function */ )
{
  KOM::Base_var *v;
  for( v = m_lstClients.first(); v != 0L; v = m_lstClients.next() )
    if ( strcmp( (*v)->_ident(), obj->_ident() ) == 0 )
    {
      m_lstClients.remove();
      return;
    }
}

KOM::Base::ConnectionSeq* KOMBase::describeConnections()
{
  KOM::Base::ConnectionSeq* seq = new KOM::Base::ConnectionSeq;
  int i = 0;
  seq->length( i );
  
  // Iterate over all signals
  QDictIterator<ClientList> dit( m_dctSignals );
  for( ; dit.current(); ++dit )
  {
    QList<Client> list;
    
    // Iterate over all clients of this signal
    QListIterator<Client> it( *dit.current() );
    for ( ; it.current(); ++it )
    {
      KOM::Base::Connection c;
      c.sig = CORBA::string_dup( dit.currentKey() );
      c.receiver = it.current()->obj;
      c.function = CORBA::string_dup( it.current()->function );
      seq->length( i + 1 );
      (*seq)[i++] = c;
    }
  }

  return seq;
}

/**********************************************************
 *
 * Mighty functions for closing any kind of connections:
 * signals, slots, filters, relations
 *
 **********************************************************/

void KOMBase::disconnectObject( KOM::Base_ptr obj )
{
  disconnectObjectIntern( obj, true );
}

void KOMBase::disconnectObjectIntern( KOM::Base_ptr obj, bool _send_notify )
{
  /******
   * Disconnect "obj"'s slots from our signals
   ******/
  
  // Iterate over all signals
  QDictIterator<ClientList> dit( m_dctSignals );
  for( ; dit.current(); ++dit )
  {
    QList<Client> list2;
    
    // Iterate over all clients of this signal
    QListIterator<Client> it( *dit.current() );
    for ( ; it.current(); ++it )
    {
      if ( strcmp( it.current()->obj->_ident(), obj->_ident() ) == 0 )
      {
	if ( _send_notify )
	  it.current()->obj->disconnectSignalNotify( this, dit.currentKey(), it.current()->function );
	list2.append( it.current() );
      }
    }
    
    // Remove object from signals list
    QListIterator<Client> it2( list2 );
    for ( ; it2.current(); ++it2 )
      dit.current()->removeRef( it2.current() );
  }
  
  /******
   * "obj" may no longer filter our events
   ******/
  QList<Filter> filters;
  // Iterate over all filters
  QListIterator<Filter> it( m_lstFilters );
  for( ; it.current(); ++it )
  {
    if ( strcmp( it.current()->obj->_ident(), obj->_ident() ) == 0 )
    {
      if ( _send_notify )  
	it.current()->obj->disconnectFilterNotify( this );
      filters.append( it.current() );
    }
  }
  // Remove from filters
  QListIterator<Filter> it2( filters );
  for( ; it2.current(); ++it2 )
    m_lstFilters.removeRef( it2.current() );
  filters.clear();
  
  /******
   * Disconnect our slots from "obj"'s events and
   * do not filter "obj"'s events any more
   ******/
  QList<KOM::Base_var> list2;
  // Iterate over all objects connected to one of our slots
  // and all objects which were adopted.
  QListIterator<KOM::Base_var> it3( m_lstClients );
  for( ; it3.current(); ++it3 )
  {
    if ( strcmp( (*(it3.current()))->_ident(), obj->_ident() ) == 0 )
    {
      if ( _send_notify )
	(*(it3.current()))->disconnectNotify( this );
      list2.append( it3.current() );
    }
  }
  // Remove from list
  QListIterator<KOM::Base_var> it4( list2 );
  for( ; it4.current(); ++it4 )
    m_lstClients.removeRef( it4.current() );
  list2.clear();
  
  /******
   * Disconnect from relatives
   ******/
  list<KOM::Base_var>::iterator stlit = m_lstRelatives.begin();
  for( ; stlit != m_lstRelatives.end(); ++stlit )
  {
    if ( cmpIOR( *stlit, obj ) == 0L )
    {
      list2.append( &*stlit );
      (*stlit)->leaveNotify( this );
    }
  }
  // Remove from list
  QListIterator<KOM::Base_var> it5( list2 );
  for( ; it5.current(); ++it5 )
    m_lstClients.removeRef( it5.current() );
  list2.clear();
}

void KOMBase::disconnectAll()
{
  /*****
   * Disconnect all listeners from our signals
   *****/
  QDictIterator<ClientList> dit( m_dctSignals );
  for( ; dit.current(); ++dit )
  {
    QListIterator<Client> it( *dit.current() );
    for ( ; it.current(); ++it )
      it.current()->obj->disconnectSignalNotify( this, dit.currentKey(), it.current()->function );
  }

  /****
   * Disconnect all event filters
   ****/
  QListIterator<Filter> it( m_lstFilters );
  for ( ; it.current(); ++it )
  {
    it.current()->obj->disconnectFilterNotify( this );    
  }

  /****
   * Disconnect our slots from all signals
   * and do not filter other objects events any more.
   ****/
  QListIterator<KOM::Base_var> it2( m_lstClients );
  for( ; it2.current(); ++it2 )
  {
    (*(it2.current()))->disconnectNotify( this );
  }
  
  /**
   * Disconnect from all relatives
   */
  list<KOM::Base_var>::iterator it3 = m_lstRelatives.begin();
  for( ; it3 != m_lstRelatives.end(); ++it3 )
  {
    (*it3)->leaveNotify( this );
  }
  
  m_lstFilters.clear();
  m_dctSignals.clear();
  m_lstClients.clear();
  m_lstRelatives.clear();
}

/**********************************************************
 *
 * Event filters
 *
 **********************************************************/

void KOMBase::installFilter( KOM::Base_ptr obj, const char* function,
			     const KOM::EventTypeSeq& events,
			     KOM::Base::FilterMode mode )
{
  obj->connectNotify( this );
  // TODO: check for exceptions here
  m_lstFilters.append( new Filter( obj, function, events, mode ) );
}

void KOMBase::disconnectFilterNotify( KOM::Base_ptr obj )
{
  KOM::Base_var *v;
  for( v = m_lstClients.first(); v != 0L; v = m_lstClients.next() )
    if ( strcmp( (*v)->_ident(), obj->_ident() ) == 0 )
    {
      m_lstClients.remove();
      return;
    }
}

void KOMBase::uninstallFilter( KOM::Base_ptr obj )
{
  QList<Filter> filters;
  // Iterate over all filters
  QListIterator<Filter> it( m_lstFilters );
  for( ; it.current(); ++it )
    if ( strcmp( it.current()->obj->_ident(), obj->_ident() ) == 0 )
    {
      filters.append( it.current() );
      it.current()->obj->disconnectFilterNotify( this );
    }
  
  // Remove from filters
  QListIterator<Filter> it2( filters );
  for( ; it2.current(); ++it2 )
    m_lstFilters.removeRef( it2.current() );  
}

KOM::Base::EventFilterSeq* KOMBase::describeEventFilters()
{
  KOM::Base::EventFilterSeq *seq = new KOM::Base::EventFilterSeq;
  int i = 0;
  seq->length( i );
  
  QListIterator<Filter> it( m_lstFilters );
  for( ; it.current(); ++it )
  {
    KOM::Base::EventFilter f;
    f.receiver = it.current()->obj;
    f.events = it.current()->events;
    f.function = CORBA::string_dup( it.current()->function );

    seq->length( i + 1 );
    (*seq)[i++] = f;
  }
  
  return seq;
}

void KOMBase::receiveASync( const char* type, const CORBA::Any& value )
{
  receive( type, value );
}

void KOMBase::receive( const char* type, const CORBA::Any& value )
{
  KOM::Base::FilterMode mode = KOM::Base::FM_WRITE;
  
  QListIterator<Filter> it( m_lstFilters );
  while( it.current() != 0L )
  {
    if ( it.current()->mode == mode )
    {
      bool skip = true;
      CORBA::ULong max = it.current()->events.length();
      for ( CORBA::ULong i = 0; i < max; i++ )
      {
	QString tmp( it.current()->events[i] );
	if ( tmp.right(1) == "*" )
        {
	  tmp = tmp.left( tmp.length() - 1 );
	  if ( strncmp( type, tmp, tmp.length() ) == 0 )
	  {
	    skip = false;
	    break;
	  }
	}
	else
        {
	  if( strcmp( type, tmp ) == 0 )
	  {
	    skip = false;
	    break;
	  }
	}
      }

      if ( max == 0 )
	skip = false;
      
      if ( !skip )
      {  
	CORBA::Request_var _req = it.current()->obj->_request( it.current()->function.data() );
	_req->add_in_arg() <<= CORBA::Any::from_object( this, "Object" );
	_req->add_in_arg() <<= CORBA::Any::from_string( (char *) CORBA::string_dup( type ), 0 );
	_req->add_in_arg() <<= value;
	_req->result()->value()->type( CORBA::_tc_boolean );
	_req->invoke();

	if( !_req->env()->exception() )
        {
	  CORBA::Boolean _res;
	  *_req->result()->value() >>= CORBA::Any::to_boolean( _res );
	  if ( _res )
	    return;
	}
      }
    }
    
    ++it;
    if( it.current() == 0L )
    {
      if ( mode == KOM::Base::FM_WRITE )
      {
	mode = KOM::Base::FM_IMPLEMENT;
	it.toFirst();
      }
      else if ( mode == KOM::Base::FM_IMPLEMENT )
      {
	mode = KOM::Base::FM_READ;
	it.toFirst();
      }
    }
  }
  
  event( type, value );
}

/**************************************************
 *
 * Relatives Management
 *
 **************************************************/

void KOMBase::adopt( KOM::Base_ptr _obj )
{
  _obj->adoptNotify( this );
  // TODO: check for exceptions here

  m_lstRelatives.push_back( KOM::Base::_duplicate( _obj ) );
}

void KOMBase::adoptNotify( KOM::Base_ptr _ptr )
{
  m_lstRelatives.push_back( KOM::Base::_duplicate( _ptr ) );
}

void KOMBase::leave( KOM::Base_ptr _ptr )
{
  list<KOM::Base_var>::iterator it = m_lstRelatives.begin();
  for( ; it != m_lstRelatives.end(); ++it )
  {
    if ( cmpIOR( *it, _ptr ) == 0L )
    {
      (*it)->leaveNotify( this );
      m_lstRelatives.erase( it );
      return;
    }
  }
  
  cerr << "KOMBase::leave # Some relatives wants to leave, but I dont know him ...." << endl;
}

void KOMBase::leaveNotify( KOM::Base_ptr _ptr )
{
  list<KOM::Base_var>::iterator it = m_lstRelatives.begin();
  for( ; it != m_lstRelatives.end(); ++it )
  {
    if ( cmpIOR( *it, _ptr ) == 0L )
    {
      m_lstRelatives.erase( it );
      return;
    }
  }

  cerr << "KOMBase::leaveNotify # Some relatives wants to leave, but I dont know him ...." << endl;
}

KOM::Base::RelativesSeq* KOMBase::describeRelatives()
{
  KOM::Base::RelativesSeq* seq = new KOM::Base::RelativesSeq;
  int i = 0;
  seq->length( i );
  
  list<KOM::Base_var>::iterator it = m_lstRelatives.begin();
  for( ; it != m_lstRelatives.end(); ++it )
  {
    seq->length( i + 1 );
    (*seq)[i++] = *it;
  }
  
  return seq;
}

/********************************************************
 *
 * Reference management
 *
 ********************************************************/

void KOMBase::incRef()
{
  m_ulRefCount++;
}

void KOMBase::decRef()
{
  if ( m_ulRefCount > 0 )
    m_ulRefCount--;

  if ( m_ulRefCount == 0 )
  {
    cleanUp();
    CORBA::release( this );
  }
}

void KOMBase::destroy()
{
  incRef();
  decRef();
}

/********************************************************
 *
 * Signals
 *
 ********************************************************/

void signal_call0( const char *_signal, QDict<KOMBase::ClientList> *_dict )
{
    KOMBase::ClientList *list = (*_dict)[ _signal ];
    if( list == 0L )
    {
      cerr << "Unknown signal " << _signal << endl;
      cerr << "Please insert SIGNAL_IMPL( \"" << _signal << "\" ); in "
	   << "your constructor or choose a valid signal" << endl;
      assert( 0 );
    }
    
    QListIterator<KOMBase::Client> it( *list );
    for ( ; it.current(); ++it )
    {
        CORBA::Request_var _req = it.current()->obj->_request( it.current()->function.data() );
	_req->result()->value()->type( CORBA::_tc_void );
	_req->invoke();
    }
}


#include <unistd.h>
#include <stdio.h>  
#include <stdarg.h>      

int iFdeb = 0;

void mdeb( const char *msg, ... )
{
  /* va_list ap;
    va_start( ap, msg );                        // use variable arg list

    fprintf( stderr, "%i: ",iFdeb);
    
  for( int i = 0; i < iFdeb; i++ )
    fprintf(stderr, " ");
  fprintf(stderr, ">");

  vfprintf( stderr, msg, ap );         
  va_end( ap );  */
}

void sdeb( const char *msg, ... )
{  
  /* va_list ap;
    va_start( ap, msg );                        // use variable arg list

    fprintf( stderr, "%i: ",iFdeb);

  for( int i = 0; i  < iFdeb; i++ )
    fprintf( stderr," ");
  iFdeb++;
 
  vfprintf( stderr, msg, ap );         
    va_end( ap );  */
}

void edeb( const char *msg, ... )
{  
  /* va_list ap;
    va_start( ap, msg );                        // use variable arg list

  iFdeb--;
    fprintf( stderr, "%i: ",iFdeb);
  for( int i = 0; i < iFdeb; i++ )
    fprintf( stderr, " ");
 
  vfprintf( stderr, msg, ap );         
    va_end( ap ); */
}




