/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include <CORBA.h>
#include <mico/template_impl.h>
#include <mico/ir_impl.h>

#include <iostream>
#include <fstream>

#include <string>
#include <list>
#include <string.h>
#include <unistd.h>

CORBA::Repository_var g_vRepo;
string g_strModule;
list<string> g_lstImport;

ostream* pout = &cout;
#define sout (*pout)

const char* keywords[] = {
  "and","assert","break","class","continue","def","del","elif","else",
  "except","exec"," finally","for","from","global","if","import","in","is",
  "lambda","not","or","pass","print","raise","return","try","while", 0L }; 
string g_buffer;
const char* py( const char* _str )
{
  int i = 0;
  while( keywords[i] && strcmp( keywords[i], _str ) != 0L )
    i++;

  if ( keywords[i] )
  {
    g_buffer = _str;
    g_buffer += "_";
    return g_buffer.c_str();
  }

  return _str;
}

void importModule( const char *_id )
{
  string repoid = _id + 4;
  if ( strncmp( repoid.c_str(), "omg.org/" , 8 ) == 0 )
    repoid.assign( repoid, 8 );

  int i = repoid.find( "/" );
  if ( i == -1 )
    return;
  string mod;
  mod.assign( repoid, 0, i );
  if ( mod == g_strModule )
    return;

  // Do not insert dupes
  list<string>::iterator it = g_lstImport.begin();
  for( ; it != g_lstImport.end(); it++ )
    if ( *it == mod )
      return;
  
  g_lstImport.push_back( mod );
}

string repoid2Ident( const char *_id, bool _strip_module = true, bool _strip_pragma = true )
{
  string repoid = _id + 4;
  repoid.erase( repoid.rfind( ":" ) );
  int pos;
  while ( ( pos = repoid.find( "/" ) ) != -1 )
    repoid[pos] = '.';

  if ( _strip_pragma )
  {    
    if ( strncmp( repoid.c_str(), "omg.org." , 8 ) == 0 )
      repoid.assign( repoid, 8 );
  }
  
  if ( !_strip_module )
    return repoid;
 
  
  string cmp = g_strModule;
  cmp += ".";  
  if ( strncmp( repoid.c_str(), cmp.c_str(), cmp.size() ) == 0 )
    repoid.assign( repoid, cmp.size() );
  return repoid;
}

void print_ir( CORBA::Container_ptr repo );
bool print_type( CORBA::IDLType_ptr t );

void print_typedef( CORBA::TypedefDef_ptr t )
{
  sout << "_tc_";
  CORBA::TypeCode_var tc = t->type();
  sout << tc->name();
  string tcstr = tc->stringify();
  sout << " = libpymico.typecode( 23, \"" << tcstr << "\" )" << endl;
}

void print_alias( CORBA::AliasDef_ptr t )
{
  sout << "_tc_";
  CORBA::TypeCode_var tc = t->type();
  sout << tc->name();
  string tcstr = tc->stringify();
  sout << " = libpymico.typecode( 23, \"" << tcstr << "\" )" << endl;
}

bool print_type( CORBA::IDLType_ptr t )
{
  CORBA::String_var tmp;
  string abs_name;
  
  switch( t->def_kind() )
  {
  case CORBA::dk_Struct:
  case CORBA::dk_Union:
  case CORBA::dk_Enum:
  case CORBA::dk_Alias:
  case CORBA::dk_Interface:
  case CORBA::dk_Exception:
    {
      CORBA::TypeCode_var tc = t->type();
      importModule( tc->id() );
      string ident = repoid2Ident( tc->id() );
      int pos = ident.rfind( "." );
      if ( pos == -1 )
	ident.insert( 0, "_tc_", 4 );
      else
	ident.replace( pos, 1, "._tc_" );
      sout << ident;
      break;
    }
  case CORBA::dk_Primitive:
    {
      CORBA::PrimitiveDef_var p = CORBA::PrimitiveDef::_narrow( t );
      assert( !CORBA::is_nil( p ) );
      switch( p->kind() )
	{
	case CORBA::pk_any:
	  sout << "CORBA._tc_any";
	  break;
	case CORBA::pk_void:
	  sout << "CORBA._tc_void";
	  return false;
	  break;
	case CORBA::pk_boolean:
	  sout << "CORBA._tc_boolean";
	  break;
	case CORBA::pk_char:
	  sout << "CORBA._tc_char";
	  break;
	case CORBA::pk_wchar:
	  sout << "CORBA._tc_wchar";
	  break;
	case CORBA::pk_octet:
	  sout << "CORBA._tc_octet";
	  break;
	case CORBA::pk_ushort:
	  sout << "CORBA._tc_ushort";
	  break;
	case CORBA::pk_ulong:
	  sout << "CORBA._tc_ulong";
	  break;
	case CORBA::pk_short:
	  sout << "CORBA._tc_short";
	  break;
	case CORBA::pk_long:
	  sout << "CORBA._tc_long";
	  break;
	case CORBA::pk_longlong:
	  sout << "CORBA._tc_longlong";
	  break;
	case CORBA::pk_ulonglong:
	  sout << "CORBA._tc_ulonglong";
	  break;
	case CORBA::pk_float:
	  sout << "CORBA._tc_float";
	  break;
	case CORBA::pk_double:
	  sout << "CORBA._tc_double";
	  break;
	case CORBA::pk_longdouble:
	  sout << "CORBA._tc_longdouble";
	  break;
	case CORBA::pk_string:
	  sout << "CORBA._tc_string";
	  break;
	case CORBA::pk_wstring:
	  sout << "CORBA._tc_wstring";
	  break;
	case CORBA::pk_objref:
	  sout << "CORBA._tc_Object";
	  break;
	case CORBA::pk_TypeCode:
	  sout << "CORBA._tc_TypeCode";
	  break;
	case CORBA::pk_Principal:
	  sout << "CORBA._tc_Principal";
	  break;
	default:
	  assert( 0 );
	}
      break;
    }
  default:
    assert( 0 );
  }

  return true;
}

void print_union( CORBA::UnionDef_ptr _union, const char *_indent )
{
  /**
   * TypeCode
   */
  CORBA::TypeCode_var tc = _union->type();
  string stc = tc->stringify();
  sout << _indent << "_tc_" << tc->name() << " = CORBA.typecode( 23, \"" << stc << "\" )" << endl;

  /**
   * Class definition
   */
  sout << _indent << "class " << tc->name() << "( CORBA.Union ):" << endl;
  sout << _indent << "\tdef __init__( self ):" << endl;
  sout << _indent << "\t\tCORBA.Union.__init__( self )" << endl;
  sout << endl;
}

void print_struct( CORBA::StructDef_ptr _struct, const char *_indent )
{
  /**
   * TypeCode
   */
  CORBA::TypeCode_var tc = _struct->type();
  string stc = tc->stringify();
  sout << _indent << "_tc_" << tc->name() << " = CORBA.typecode( 23, \"" << stc << "\" )" << endl;

  /**
   * Class definition
   */
  sout << _indent << "class " << tc->name() << "( CORBA.Struct ):" << endl;
  sout << _indent << "\tdef __init__( self";

  CORBA::StructMemberSeq_var members = _struct->members();
 
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << ", " << members[l].name.in();
  
  sout << " ):" << endl;
  sout << _indent << "\t\tCORBA.Struct.__init__( self )" << endl;
  
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << _indent << "\t\tself." << members[l].name.in() << " = " << members[l].name.in() << endl;

  sout << endl;
}

void print_exception( CORBA::ExceptionDef_ptr _exc, const char *_indent )
{
  /**
   * TypeCode
   */
  CORBA::TypeCode_var tc = _exc->type();
  string stc = tc->stringify();
  sout << _indent << "_tc_" << tc->name() << " = CORBA.typecode( 23, \"" << stc << "\" )" << endl;

  /**
   * Class definition
   */
  sout << _indent << "class " << tc->name() << "( CORBA.UserException ):" << endl;
  sout << _indent << "\tdef __init__( self";

  CORBA::StructMemberSeq_var members = _exc->members();
 
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << ", " << members[l].name.in();
  
  sout << " ):" << endl;
  sout << _indent << "\t\tCORBA.UserException.__init__( self )" << endl;
  
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << _indent << "\t\tself." << members[l].name.in() << " = " << members[l].name.in() << endl;

  sout << endl;
}

void print_enum( CORBA::EnumDef_ptr _enum, const char *_indent )
{
  CORBA::TypeCode_var tc = _enum->type();
  string stc = tc->stringify();
  sout << _indent << "_tc_" << tc->name() << " = CORBA.typecode( 23, \"" << stc << "\" )" << endl;

  CORBA::EnumMemberSeq_var members = _enum->members();
 
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << _indent << members[l].in() << " = \"" << members[l].in() << '"' << endl;
  sout << endl;
}

void print_module( CORBA::ModuleDef_ptr _mod )
{
  CORBA::String_var mod = _mod->name();
  sout << "******** MODULE " << mod.in() << "**********" << endl;  
  g_strModule = mod.in();
  g_lstImport.clear();
  string tmp = mod.in();
  tmp += ".tmp";
  ofstream out( tmp.c_str() );
  pout = &out;
  print_ir( _mod );
  pout = &cout;
  out.close();

  tmp = mod.in();
  tmp += ".py";
  ofstream out2( tmp.c_str() );
  out2 << "import libpymico" << endl;
  out2 << "import CORBA" << endl;
  list<string>::iterator it = g_lstImport.begin();
  for( ; it != g_lstImport.end(); ++it )
    out2 << "import " << *it << endl;

  tmp = mod.in();
  tmp += ".tmp";  
  ifstream in( tmp.c_str() );  
  char buffer[ 4096 ];
  while ( !in.eof() && out2.good() )
  {
    in.read( buffer, 4096 );
    out2.write( buffer, in.gcount() );
  }
  in.close();
  out2.close();
  unlink( tmp.c_str() );
}

void print_interface( CORBA::InterfaceDef_ptr _iface )
{
  // Get all operations defined in this interface
  CORBA::InterfaceDef::FullInterfaceDescription_var desc;
  desc = _iface->describe_interface();

  /**
   * TypeCode
   */
  string tc = desc->type->stringify();
  sout << "_tc_" << desc->name.in() << " = CORBA.typecode( 23, \"" << tc << "\" )" << endl;

  CORBA::InterfaceDefSeq_var supertypes = _iface->base_interfaces();

  /**
   * Stub
   */
  list<string> superlist;

  /////////
  // Declare the class
  /////////
  if ( supertypes->length() == 0 )
    sout << "class " << desc->name.in() << "( CORBA.Object ):" << endl;
  else
  {
    sout << "class " << desc->name.in() << "( ";
    for( CORBA::ULong s = 0; s < supertypes->length(); s++ )
    {
      // Get all operations defined in this interface
      CORBA::InterfaceDef::FullInterfaceDescription_var superdesc;
      superdesc = supertypes[s]->describe_interface();

      importModule( superdesc->id.in() );
      string name = repoid2Ident( superdesc->id.in() );
      superlist.push_back( name );
      sout << name;
      if ( s + 1 < supertypes->length() )
	sout << ", ";
    }
    sout << " ):" << endl;
  }

  //////////////
  // Get the contents of the interface: typecodes, structs, ....
  //////////////
  CORBA::ContainedSeq_var c;
  c = _iface->contents( CORBA::dk_all, 1 );
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    if ( desc->kind == CORBA::dk_Typedef )
    {
      CORBA::TypedefDef_var in = CORBA::TypedefDef::_narrow( c[ j ] );
      sout << "\t";
      print_typedef( in );
    }
    else if ( desc->kind == CORBA::dk_Alias )
    {
      CORBA::AliasDef_var in = CORBA::AliasDef::_narrow( c[ j ] );
      sout << "\t";
      print_alias( in );
    }
    else if ( desc->kind == CORBA::dk_Struct )
    {
      CORBA::StructDef_var in = CORBA::StructDef::_narrow( c[ j ] );
      print_struct( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Union )
    {
      CORBA::UnionDef_var in = CORBA::UnionDef::_narrow( c[ j ] );
      print_union( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Exception )
    {
      CORBA::ExceptionDef_var in = CORBA::ExceptionDef::_narrow( c[ j ] );
      print_exception( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Enum )
    {
      CORBA::EnumDef_var in = CORBA::EnumDef::_narrow( c[ j ] );
      print_enum( in, "\t" );
    }
  }

  ///////////////
  // Constructor
  ///////////////
  string global_ident = repoid2Ident( desc->id.in(), false );
  int pos;
  while ( ( pos = global_ident.find( "." ) ) != -1 )
    global_ident[pos] = '_';

  sout << "\tdef __init__( self, obj=\"\", ior=\"\", repoid=\"" << desc->id.in() << "\" ):" << endl;
  sout << "\t\tif \"_init_" << global_ident << "_\" in self.__dict__.keys():" << endl;
  sout << "\t\t\treturn" << endl;
  sout << "\t\tself.__dict__[\"_init_" << global_ident << "_\"] = 1" << endl;
  sout << "\t\tif repoid == \"\":" << endl;
  sout << "\t\t\trepoid = \"" << desc->id.in() << "\"" << endl;
  if ( supertypes->length() == 0 )
    sout << "\t\tCORBA.Object.__init__( self, obj, ior, repoid )" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    for( ; it != superlist.end(); ++it )
    {
      sout << "\t\t" << *it << ".__init__( self, obj, ior, repoid )" << endl;
    }
  }
  // Attributes
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      CORBA::AttributeDef_var adef = CORBA::AttributeDef::_narrow( c[j] );
      sout << "\t\tself.__dict__[\"_attribs_\"][\"" << attr.name.in() << "\"] = ( ";
      CORBA::IDLType_var t = adef->type_def();
      print_type( t );
      sout << ", ";
      if ( adef->mode() == CORBA::ATTR_READONLY )
	sout << "CORBA.ATTR_READONLY";
      else
	sout << "CORBA.ATTR_NORMAL";
      sout << " )" << endl;
    }
  }
  
  /////////////////
  // Method stubs
  ////////////////
  for( CORBA::ULong k = 0; k < desc->operations.length(); k++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ k ];
    sout << "\tdef " << op.name.in() << "( self";

    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      switch( p[l].mode )
	{
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_IN:
	  sout << ", " << py( p[ l ].name.in() );
	  break;
	default:
	  break;
	}
    }
    sout << " ):" << endl;
    sout << "\t\tr = libpymico.request( self.__dict__[\"_ior_\"], \"" << op.name.in() << "\", ";

    CORBA::Contained_var con = g_vRepo->lookup_id( op.id );
    assert( !CORBA::is_nil( con ) );
    CORBA::OperationDef_var opdef = CORBA::OperationDef::_narrow( con );
    assert( !CORBA::is_nil( opdef ) );
    CORBA::IDLType_var idltype = opdef->result_def();
    // tells, wether te return type of the function is void
    bool isvoid = !print_type( idltype );
    sout << " )" << endl;

    // hasout holds the number of out/inout parameters
    int hasout = 0;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      switch( p[l].mode )
	{
	case CORBA::PARAM_INOUT:
	  hasout++;
	  sout << "\t\tlibpymico.add_param( r, \"" << p[l].name.in()
	       << "\", " << py( p[l].name.in() ) << ", ";
	  print_type( p[l].type_def );
	  sout << ", 1 )" << endl;
	  break;
	case CORBA::PARAM_IN:
	  sout << "\t\tlibpymico.add_param( r, \"" << p[l].name.in()
	       << "\", " << py( p[l].name.in() ) << ", ";
	  print_type( p[l].type_def );
	  sout << ", 0 )" << endl;
	  break;
	case CORBA::PARAM_OUT:
	  hasout++;
	  sout << "\t\tlibpymico.add_out_param( r, \"" << p[l].name.in() << "\", ";
	  print_type( p[l].type_def );
	  sout << ", " << l << " )" << endl;
	  break;
	}
    }

    // Invoke the function
    if ( !isvoid )
      sout << "\t\t_result_ = libpymico.invoke( r )" << endl;
    else
      sout << "\t\tlibpymico.invoke( r )" << endl;

    // Check for exceptions
    for ( CORBA::ULong l = 0; l < op.exceptions.length(); l++ )
    {
      sout << "\t\t(t,v) = libpymico.check_exception( r, \"" << op.exceptions[l].id.in() << "\", ";
      string id = op.exceptions[l].type->id();
      importModule( id.c_str() );
      string ident = repoid2Ident( id.c_str() );
      int pos = ident.rfind( "." );
      if ( pos == -1 )
	ident.insert( 0, "_tc_", 4 );
      else
	ident.replace( pos, 1, "._tc_" );
      sout << ident;

      sout << " )" << endl;
      sout << "\t\tif t != 0:" << endl;
      sout << "\t\t\traise t, v" << endl;
    }
    
    // Do we have to deal with multiple return values ?
    if ( (!isvoid && hasout >= 1 ) || hasout >= 2 )
    {
      for( CORBA::ULong l = 0; l < p.length(); l++ )
      {
	switch( p[l].mode )
	  {
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_OUT:
	  sout << "\t\t_result" << l << "_ = libpymico.get_out_param( r, ";
	  print_type( p[l].type_def );
	  sout << ", " << l << " )" << endl;
	  break;
	  }
      }
      sout << "\t\tlibpymico.release_request( r )" << endl;
      sout << "\t\treturn ( ";
      if ( !isvoid )
	sout << "_result_, ";
      int ho = 0;
      for( CORBA::ULong l = 0; l < p.length(); l++ )
      {
	switch( p[l].mode )
	  {
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_OUT:
	  ho++;
	  sout << "_result" << l << "_";
	  if ( ho < hasout )
	    sout << ", ";
	  }
      }
      sout << " )" << endl;
    }
    // Only one return value that is NOT an out/inout parameter ?
    else if ( !isvoid )
    {
      sout << "\t\tlibpymico.release_request( r )" << endl;
      sout << "\t\treturn _result_" << endl;
    }
    // Only one return value that IS an out/inout parameter
    else if ( hasout == 1 )
    {
      for( CORBA::ULong l = 0; l < p.length(); l++ )
      {
	switch( p[l].mode )
	  {
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_OUT:
	  sout << "\t\t_result_ = libpymico.get_out_param( r, ";
	  print_type( p[l].type_def );
	  sout << ", " << l << " )" << endl;
	  }
      }
      sout << "\t\tlibpymico.release_request( r )" << endl;
      sout << "\t\treturn _result_" << endl;
    }
    // No return value at all
    else
    {
      sout << "\t\tlibpymico.release_request( r )" << endl;
    }
  }

  sout << endl;
  
  /**
   * Skeleton
   */
  if ( supertypes->length() == 0 )
    sout << "class " << desc->name.in() << "_skel( CORBA.Server ):" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    sout << "class " << desc->name.in() << "_skel( ";
    while( it != superlist.end() )
    {
      sout << *it++ << "_skel";
      if( it != superlist.end() )
	sout << ", ";
    }
    sout << " ):" << endl;
  }
  
  sout << "\tdef __init__( self, repoid=\"" << desc->id.in() << "\" ):" << endl;

  sout << "\t\tif \"_init_" << global_ident << "_\" in self.__dict__.keys():" << endl;
  sout << "\t\t\treturn" << endl;
  sout << "\t\tself._init_" << global_ident << "_ = 1" << endl;

  if ( supertypes->length() == 0 )
    sout << "\t\tCORBA.Server.__init__( self, repoid )" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    for( ; it != superlist.end(); ++it )
    {
      sout << "\t\t" << *it << "_skel.__init__( self, repoid )" << endl;
    }
  }

  // Setting attribute to None
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      sout << "\t\t" << attr.name.in() << " = None" << endl;
    }
  }
  
  // Register every function
  for( CORBA::ULong x = 0; x < desc->operations.length(); x++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ x ];

    CORBA::Contained_var con = g_vRepo->lookup_id( op.id );
    assert( !CORBA::is_nil( con ) );
    CORBA::OperationDef_var opdef = CORBA::OperationDef::_narrow( con );
    assert( !CORBA::is_nil( opdef ) );

    sout << "\t\tlibpymico.register_callback( self._server_, self, \"" << op.name.in()
	 << "\", ";

    CORBA::IDLType_var idltype = opdef->result_def();
    print_type( idltype );

    sout << ", [ ";

    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      sout << "( ";
      print_type( p[l].type_def );
      switch( p[l].mode )
	{
	case CORBA::PARAM_IN:
	  sout << ", CORBA.ARG_IN )";
	  break;
	case CORBA::PARAM_OUT:
	  sout << ", CORBA.ARG_OUT )";
	  break;
	case CORBA::PARAM_INOUT:
	  sout << ", CORBA.ARG_INOUT )";
	  break;
	}
      
      if ( l + 1 < p.length() )
	sout << ", ";
    }
    sout << " ] )" << endl;
  }

  // Function prototypes
  for( CORBA::ULong k = 0; k < desc->operations.length(); k++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ k ];
    sout << "\tdef " << op.name.in() << "( self";

    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      switch( p[l].mode )
	{
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_IN:
	  sout << ", " << py( p[ l ].name.in() );
	}
    }
    sout << " ):" << endl;
    sout << "\t\traise CORBA.NotImplemented" << endl;
  }

  // Attribute access function prototypes
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      sout << "\tdef _get_" << attr.name.in() << "( self ):" << endl;
      sout << "\t\treturn " << attr.name.in() << endl;
      if ( attr.mode == CORBA::ATTR_NORMAL )
      {
	sout << "\tdef _set_" << attr.name.in() << "( self, value ):" << endl;
	sout << "\t\t" << attr.name.in() << " = value" << endl;
      }
    }
  }

  sout << endl;
}

void print_ir( CORBA::Container_ptr repo )
{
  CORBA::ContainedSeq_var c;

  // Get the contents of the IR, but only objects of type
  // InterfaceDef and only objects directly contained in the IR
  c = repo->contents( CORBA::dk_all, 1 );
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    if ( desc->kind == CORBA::dk_Interface )
    {
      CORBA::InterfaceDef_var in = CORBA::InterfaceDef::_narrow( c[ j ] );
      print_interface( in );
    }
    else if ( desc->kind == CORBA::dk_Module )
    {
      CORBA::ModuleDef_var in = CORBA::ModuleDef::_narrow( c[ j ] );
      print_module( in );
    }
    else if ( desc->kind == CORBA::dk_Typedef )
    {
      CORBA::TypedefDef_var in = CORBA::TypedefDef::_narrow( c[ j ] );
      print_typedef( in );
    }
    else if ( desc->kind == CORBA::dk_Alias )
    {
      CORBA::AliasDef_var in = CORBA::AliasDef::_narrow( c[ j ] );
      print_alias( in );
    }
    else if ( desc->kind == CORBA::dk_Struct )
    {
      CORBA::StructDef_var in = CORBA::StructDef::_narrow( c[ j ] );
      print_struct( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Union )
    {
      CORBA::UnionDef_var in = CORBA::UnionDef::_narrow( c[ j ] );
      print_union( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Exception )
    {
      CORBA::ExceptionDef_var in = CORBA::ExceptionDef::_narrow( c[ j ] );
      print_exception( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Enum )
    {
      CORBA::EnumDef_var in = CORBA::EnumDef::_narrow( c[ j ] );
      print_enum( in, "" );
    }
  }
}

int main( int argc, char *argv[] )
{
  cerr << "1" << endl;
  
  // ORB initialization
  CORBA::ORB_var orb = CORBA::ORB_init( argc, argv, "mico-local-orb" );
  cerr << "2" << endl;
  CORBA::BOA_var boa = orb->BOA_init( argc, argv, "mico-local-boa");
  cerr << "3" << endl;
  // Create a local IR
  CORBA::Object_var o = orb->resolve_initial_references ("InterfaceRepository");
  cerr << "4" << endl;
  assert( !CORBA::is_nil( o ) );
  cerr << "5" << endl;
  g_vRepo = CORBA::Repository::_narrow( o );
  cerr << "6" << endl;
  assert( !CORBA::is_nil( g_vRepo ) );

  cerr << "Printing" << endl;
  
  print_ir( g_vRepo );
}
