/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include <assert.h>
#include <iostream>

#include "pm_python.h"

KPythonModule::KPythonModule( const char *_name )
{
  m_strName = _name;
    
  Py_Initialize();
  m_pModule = PyImport_AddModule( (char*)_name );
  assert( m_pModule );
  
  m_pDict = PyModule_GetDict( m_pModule );
  assert( m_pDict );
  
  PyDict_SetItemString( m_pDict, "__dummy__", Py_None );
  PyDict_SetItemString( m_pDict, "__builtins__", PyEval_GetBuiltins() );
}

void KPythonModule::fetchFunctions()
{
  PyObject* pmod = PyImport_ImportModule( "CORBA" );
  if ( pmod == 0L )
  {    
    PyErr_PrintEx( 0 );
    exit(1);
  }
  
  assert( pmod != 0L );
  m_pNewStruct = PyObject_GetAttrString( pmod, "newStruct" );
  assert( m_pNewStruct != 0L );
  m_pNewUnion = PyObject_GetAttrString( pmod, "newUnion" );
  assert( m_pNewUnion != 0L );
  m_pNewAny = PyObject_GetAttrString( pmod, "newAny" );
  assert( m_pNewAny != 0L );
  m_pNewObject = PyObject_GetAttrString( pmod, "newObject" );
  assert( m_pNewObject != 0L );
  m_pNewException = PyObject_GetAttrString( pmod, "newException" );
  assert( m_pNewException != 0L );
}

PyObject* KPythonModule::newException( const char *_module, const char *_class, PyObject* _args )
{
  PyObject* args = Py_BuildValue( "(ssO)", _module, _class, _args );
  PyObject* exc = PyEval_CallObject( m_pNewException, args );
  Py_DECREF( args );
  
  return exc;
}

PyObject* KPythonModule::newObject( const char *_module, const char *_class)
{
  PyObject* args = Py_BuildValue( "(ss)", _module, _class );
  PyObject* struc = PyEval_CallObject( m_pNewObject, args );
  Py_DECREF( args );
  
  return struc;
}

PyObject* KPythonModule::newObject()
{
  PyObject* args = Py_BuildValue( "(ss)", "", "" );
  PyObject* struc = PyEval_CallObject( m_pNewObject, args );
  Py_DECREF( args );
  
  return struc;
}

PyObject* KPythonModule::newStruct()
{
  PyObject* args = Py_BuildValue( "()" );
  PyObject* struc = PyEval_CallObject( m_pNewStruct, args );
  Py_DECREF( args );
  
  return struc;
}

PyObject* KPythonModule::newUnion()
{
  PyObject* args = Py_BuildValue( "()" );
  PyObject* uni = PyEval_CallObject( m_pNewUnion, args );
  Py_DECREF( args );
  
  return uni;
}

PyObject* KPythonModule::newAny()
{
  PyObject* args = Py_BuildValue( "()" );
  PyObject* any = PyEval_CallObject( m_pNewAny, args );
  Py_DECREF( args );
  
  return any;
}

int KPythonModule::runCodeStr( StringModes mode, char *code, char *resfmt, void *cresult )
{
    PyObject *presult = PyRun_String( code, ( mode == PY_EXPRESSION ? eval_input : file_input ),
				      m_pDict, m_pDict );
    
    if ( mode == PY_STATEMENT )
    {
	int result = ( presult == NULL ? -1 : 0 );
	Py_XDECREF( presult );
	return result;
    }
    return convertResult( presult, resfmt, cresult );
}

int KPythonModule::convertResult( PyObject *presult, char *resFormat, void *resTarget )
{
    if ( presult == NULL )
	return -1;
    if ( resTarget == NULL )
    {
	Py_DECREF( presult );
	return 0;
    }
    if ( !PyArg_Parse( presult, resFormat, resTarget ) )
    {
	Py_DECREF( presult );
	return -1;
    }
    if ( strcmp( resFormat, "0" ) != 0 )
	Py_DECREF( presult );
    return 0;
}

int KPythonModule::runFile( const char *_filename )
{
    FILE *f = fopen( _filename, "r" );
    if ( !f )
	return -1;

    string script;
    char buffer[ 4096 ];
    while ( !feof( f ) )
    {
	int n = fread( buffer, 1, 4095, f );
	if ( n > 0 )
	{
	    buffer[n] = 0;
	    script += buffer;
	}
    }
    fclose( f );

    return runCodeStr( PY_STATEMENT, (char*)script.c_str() );
}

void KPythonModule::registerMethods( struct PyMethodDef* _methods )
{
    Py_InitModule( (char*)(m_strName.c_str()), _methods );
}

PyObject* KPythonModule::getAttr( const char* _name )
{
  return PyObject_GetAttrString( m_pModule, (char*)_name );
}

PyObject* KPythonModule::eval( const char* _cmd )
{
    printf("Running '%s'\n",_cmd);
    
    PyObject *v = PyRun_String( (char*)_cmd, eval_input, m_pDict, m_pDict );
    if ( v == 0L )
    {
	printf("ERROR: Python: Could not exec\n");

	PyObject *e = PyErr_Occurred();
	if ( e == NULL )
	    return 0L;

	PyObject *e1, *e2, *e3;
	PyErr_Fetch( &e1, &e2, &e3 );
	
	char *str1 = "";
	if ( e1 != 0L )
	    PyArg_Parse( e1, "s", &str1 );
	
	char *str2 = "";
	if ( e2 != 0L )
	    PyArg_Parse( e2, "s", &str2 );

	char *str3 = "";
	if ( e2 != 0L )
	    PyArg_Parse( e3, "s", &str3 );

	printf("Traceback:\n%s\n%s\n%s\n",str1,str2,str3);
	
	PyErr_Clear();
	
	return 0L;
    }
    
    return v;
}
