/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "main.h"

#include <opApplication.h>
#include <opMainWindowIf.h>

#include <iostream>

Explorer::Explorer()
{
  cout << "+Explorer" << endl;
  
  // OpenParts need this, since OPPartif
  // is not derived from QWidget.
  setWidget( this );

  SIGNAL_IMPL( "selected" );

  // create the HTML widget as a child
  m_pHTML = new KHTMLView( this );
  m_pHTML->show();

  // Connect Qt slots. Use the scoping operator "::" here,
  // because the name connect collides with the connect
  // method of KOM otherwise.
  QObject::connect( m_pHTML, SIGNAL( URLSelected( const char*, int ) ),
		    this, SLOT( slotSelected( const char*, int ) ) );
}

Explorer::~Explorer()
{
  cout << "-Explorer" << endl;
}
  
void Explorer::resizeEvent( QResizeEvent * )
{
  cerr << "Resizing" << endl;
  // Keep the html widget in shape
  m_pHTML->setGeometry( 0, 0, width(), height() );
}

void Explorer::begin( const char* url )
{
  m_pHTML->begin( url );
}

void Explorer::write( const char* text )
{
  m_pHTML->write( text );
}

void Explorer::end()
{
  m_pHTML->end();
}

void Explorer::parse()
{
  m_pHTML->parse();
}

void Explorer::selected( const char* url, CORBA::Long button )
{
  // Emit a KOM Signal
  // Remember, that CORBA::Any has no operator<<= for strings.
  // So we needs to use this construction here. This is not KOMs
  // fault. Consult your CORBA literature for details and
  // see /mico/include/mico/any.h
  SIGNAL_CALL2( "selected", CORBA::Any::from_string( (char*)url, 0 ), button );
}

void Explorer::slotSelected( const char *_url, int button )
{
  // This function translates the Qt Signal in a KOM signal
  selected( _url, button );
}

MainWindow::MainWindow()
{
  cout << "+MainWindow" << endl;

  // Create a frame that will host the HTML Component
  m_pFrame = new OPFrame( this );
  // The frame is the view. This is not a must. A frame
  // is a widget like every other Qt widget, too.
  setView( m_pFrame );
 
  // Create the HTML component. The component may run in
  // another process, but in this example it does not.
  m_vExplorer = Embed::Explorer::_duplicate( new Explorer );
  // Tell the component about the toplevel window.
  m_vExplorer->setMainWindow( interface() );
  // Display the HTML component in the frame
  // The Frame will care about KOM reference counting. So we
  // do not need to care.
  m_pFrame->attach( m_vExplorer );

  // Show some HTML stuff
  m_vExplorer->begin( "meta:/" );
  m_vExplorer->parse();
  m_vExplorer->write( "<HTML><HEAD><TITLE>OpenParts</TITLE></HEAD>" );
  m_vExplorer->write( "<BODY bgcolor=#FFFFFF><H1>OpenParts ... are cool</H1></BODY></HTML>" );
  m_vExplorer->end();
}

MainWindow::~MainWindow()
{
  cleanUp();

  cout << "-MainWindow" << endl;
}

void MainWindow::cleanUp()
{
  cout << "MainWindow cleanUp" << endl;
  
  // Free all references
  m_vExplorer = 0L;

  // Release the HTML component. This will delete
  // the component.
  if ( m_pFrame )
    m_pFrame->detach();
}

int main( int argc, char **argv )
{
  OPApplication app( argc, argv );

  // Create a top level window
  MainWindow* win = new MainWindow();
  win->show();
  
  app.exec();
  
  return 0;
}

#include "main.moc"
