/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "main.h"

#include <opApplication.h>
#include <opMainWindowIf.h>
#include <opMenuBarManager.h>
#include <opToolBarManager.h>
#include <opUIUtils.h>
#include <opMenu.h>

#include <qkeycode.h>

#include <kapp.h>
#include <kiconloader.h>

#include <iostream>

#include <qmsgbox.h>
#include <qstring.h>

Explorer::Explorer()
{
  cout << "+Explorer" << endl;
  
  // OpenParts need this, since OPPartif
  // is not derived from QWidget.
  setWidget( this );

  // Accept the focus
  OPPartIf::setFocusPolicy( OpenParts::Part::ClickFocus ); 

  // We have one signal
  SIGNAL_IMPL( "selected" );

  // create the HTML widget as a child
  m_pHTML = new KHTMLView( this );
  m_pHTML->show();

  // Connect Qt slots. Use the scoping operator "::" here,
  // because the name connect collides with the connect
  // method of KOM otherwise.
  QObject::connect( m_pHTML, SIGNAL( URLSelected( const char*, int ) ),
		    this, SLOT( slotSelected( const char*, int ) ) );
}

void Explorer::init()
{
  /******************************************************
   * Menu
   ******************************************************/

  // We want to display a menubar
  OpenParts::MenuBarManager_var menu_bar_manager = m_vMainWindow->menuBarManager();
  if ( !CORBA::is_nil( menu_bar_manager ) )
    menu_bar_manager->registerClient( id(), this );
}

Explorer::~Explorer()
{
  cout << "-Explorer" << endl;
}
  
void Explorer::cleanUp()
{
  // We must do that to avoid recursions.
  if ( m_bIsClean )
    return;

  // Say bye to our menu bar
  OpenParts::MenuBarManager_var menu_bar_manager = m_vMainWindow->menuBarManager();
  if ( !CORBA::is_nil( menu_bar_manager ) )
    menu_bar_manager->unregisterClient( id() );
  
  // Very important!!! The base classes must run
  // their cleanup function bevore! the constructor does it.
  // The reason is that CORBA reference counting is not very
  // happy about Qt, since Qt deletes its child widgets and does
  // not care about reference counters.
  OPPartIf::cleanUp();
}

void Explorer::resizeEvent( QResizeEvent * )
{
  cerr << "Resizing" << endl;
  // Keep the html widget in shape
  m_pHTML->setGeometry( 0, 0, width(), height() );
}

void Explorer::begin( const char* url )
{
  m_pHTML->begin( url );
}

void Explorer::write( const char* text )
{
  m_pHTML->write( text );
}

void Explorer::end()
{
  m_pHTML->end();
}

void Explorer::parse()
{
  m_pHTML->parse();
}

void Explorer::selected( const char* url, CORBA::Long button )
{
  // Emit a KOM Signal
  // Remember, that CORBA::Any has no operator<<= for strings.
  // So we needs to use this construction here. This is not KOMs
  // fault. Consult your CORBA literature for details and
  // see /mico/include/mico/any.h
  SIGNAL_CALL2( "selected", CORBA::Any::from_string( (char*)url, 0 ), button );
}

void Explorer::slotSelected( const char *_url, int button )
{
  // This function translates the Qt Signal in a KOM signal
  selected( _url, button );
}

void Explorer::bookmarkSelected( CORBA::Long id )
{
  QString tmp;
  tmp.sprintf( "Selected bookmark\n%i", id );
  QMessageBox::information( 0L, "Bookmark", tmp, "OK" );
}

bool Explorer::event( const char* _event, const CORBA::Any& _value )
{
  // Here we map events to function calls
  EVENT_MAPPER( _event, _value );

  MAPPING( OpenPartsUI::eventCreateMenuBar, OpenPartsUI::typeCreateMenuBar_var, mappingCreateMenubar );

  END_EVENT_MAPPER;
  
  // No, we could not handle this event
  return false;
}

bool Explorer::mappingCreateMenubar( OpenPartsUI::MenuBar_ptr _menubar )
{
  // Do we loose control over the menubar `
  if ( CORBA::is_nil( _menubar ) )
  {
    m_vMenuEdit = 0L;
    return true;
  }

  // Edit  
  _menubar->insertMenu( i18n( "&Bookmarks" ), m_vMenuEdit, -1, -1 );
  
  // We want to be informed about activated items
  m_vMenuEdit->connect( "activated", this, "bookmarkSelected" );
  
  m_vMenuEdit->insertItem( "Bookmark 1", 0L, "", 0 );
  m_vMenuEdit->insertItem( "Bookmark 2", 0L, "", 0 );
  m_vMenuEdit->insertItem( "Bookmark 3", 0L, "", 0 );

  // Yes, we handled this event
  return true;
}

Editor::Editor()
{
  cout << "+Editor" << endl;
  
  // OpenParts need this, since OPPartif
  // is not derived from QWidget.
  setWidget( this );

  // Accept the focus
  OPPartIf::setFocusPolicy( OpenParts::Part::ClickFocus ); 
  
  m_pEdit = new QMultiLineEdit( this );
  m_pEdit->show();
  m_pEdit->setReadOnly( false );
}

void Editor::init()
{
  /******************************************************
   * Menu
   ******************************************************/
  
  OpenParts::MenuBarManager_var menu_bar_manager = m_vMainWindow->menuBarManager();
  if ( !CORBA::is_nil( menu_bar_manager ) )
    menu_bar_manager->registerClient( id(), this );

  /******************************************************
   * Toolbar
   ******************************************************/

  // We want to display a toolbar
  OpenParts::ToolBarManager_var tool_bar_manager = m_vMainWindow->toolBarManager();
  if ( !CORBA::is_nil( tool_bar_manager ) )
    tool_bar_manager->registerClient( id(), this );
}

Editor::~Editor()
{
  cout << "-Editor" << endl;

  cleanUp();
}

void Editor::cleanUp()
{
  // Avoid recursions
  if ( m_bIsClean )
    return;
  
  // Free our menu and toolbar
  OpenParts::MenuBarManager_var menu_bar_manager = m_vMainWindow->menuBarManager();
  if ( !CORBA::is_nil( menu_bar_manager ) )
    menu_bar_manager->unregisterClient( id() );

  OpenParts::ToolBarManager_var tool_bar_manager = m_vMainWindow->toolBarManager();
  if ( !CORBA::is_nil( tool_bar_manager ) )
    tool_bar_manager->unregisterClient( id() );
  
  // Important!
  OPPartIf::cleanUp();
}

void Editor::resizeEvent( QResizeEvent * )
{
  cerr << "Resizing" << endl;
  // Keep the editor in shape
  m_pEdit->setGeometry( 0, 0, width(), height() );
}

void Editor::setText( const char *_txt )
{
  m_pEdit->setText( _txt );
}

void Editor::append( const char *_txt )
{
  m_pEdit->append( _txt );
}

char* Editor::text()
{
  QString tmp = m_pEdit->text();
  
  return CORBA::string_dup( tmp.data() );
}

void Editor::cut()
{
  m_pEdit->cut();
}

void Editor::copy()
{
  m_pEdit->copyText();
}

void Editor::paste()
{
  m_pEdit->paste();
}

bool Editor::event( const char* _event, const CORBA::Any& _value )
{
  // Map events to method calls
  EVENT_MAPPER( _event, _value );

  MAPPING( OpenPartsUI::eventCreateMenuBar, OpenPartsUI::typeCreateMenuBar_var, mappingCreateMenubar );
  MAPPING( OpenPartsUI::eventCreateToolBar, OpenPartsUI::typeCreateToolBar_var, mappingCreateToolbar );

  END_EVENT_MAPPER;
  
  return false;
}

bool Editor::mappingCreateToolbar( OpenPartsUI::ToolBarFactory_ptr _factory )
{
  // Do we loose control over our toolbar ?
  if ( CORBA::is_nil( _factory ) )
  {
    m_vToolBarEdit = 0L;
    return true;
  }

  m_vToolBarEdit = _factory->create( OpenPartsUI::ToolBarFactory::Transient );

  QString tmp = kapp->kde_toolbardir().copy();
  tmp += "/editcopy.xpm";
  OpenPartsUI::Pixmap_var pix = OPUIUtils::loadPixmap( tmp );
  m_idButtonEdit_Copy = m_vToolBarEdit->insertButton2( pix, 1, SIGNAL( clicked() ), this, "copySelection", true, i18n( "Copy" ), -1 );

  tmp = kapp->kde_toolbardir().copy();
  tmp += "/editcut.xpm";
  pix = OPUIUtils::loadPixmap( tmp );
  m_idButtonEdit_Cut = m_vToolBarEdit->insertButton2( pix, 2, SIGNAL( clicked() ), this, "cutSelection", true, i18n( "Cut" ), -1 );

  tmp = kapp->kde_toolbardir().copy();
  tmp += "/editpaste.xpm";
  pix = OPUIUtils::loadPixmap( tmp );
  m_idButtonEdit_Paste = m_vToolBarEdit->insertButton2( pix , 3, SIGNAL( clicked() ), this, "paste", true, i18n( "Paste" ), -1 );

  m_vToolBarEdit->enable( OpenPartsUI::Show );

  return true;
}

bool Editor::mappingCreateMenubar( OpenPartsUI::MenuBar_ptr _menubar )
{
  if ( CORBA::is_nil( _menubar ) )
  {
    m_vMenuEdit = 0L;
    return true;
  }

  // Edit  
  _menubar->insertMenu( i18n( "&Edit" ), m_vMenuEdit, -1, -1 );

  QString path = kapp->kde_toolbardir().copy();
  path += "/editcut.xpm";
  // Convert the pixmap in something that CORBA can transport
  OpenPartsUI::Pixmap_var pix = OPUIUtils::loadPixmap( path );
  m_idMenuEdit_Cut = m_vMenuEdit->insertItem6( pix, i18n("C&ut"), this,
					       "cut", CTRL + Key_X, -1, -1 );
  
  path = kapp->kde_toolbardir().copy();
  path += "/editcopy.xpm";
  pix = OPUIUtils::loadPixmap( path );
  m_idMenuEdit_Copy = m_vMenuEdit->insertItem6( pix, i18n("&Copy"), this,
						"copy", CTRL + Key_C, -1, -1 );
  
  path = kapp->kde_toolbardir().copy();
  path += "/editpaste.xpm";
  pix = OPUIUtils::loadPixmap( path );
  m_idMenuEdit_Paste = m_vMenuEdit->insertItem6( pix, i18n("&Paste"), this,
						 "paste", CTRL + Key_V, -1, -1 );
  // Yes, we handled that event
  return true;
}

enum ids { TOOLBAR_NEW, TOOLBAR_OPEN, TOOLBAR_SAVE, TOOLBAR_PRINT, TOOLBAR_RELOAD };

MainWindow::MainWindow()
{
  cout << "+MainWindow" << endl;

  m_pFileMenu = 0L;
  m_pHelpMenu = 0L;
  
  // We want to know when the focus changes. That means we are informed
  // when the explorer or the editor are clicked.
  connect( this, SIGNAL( activePartChanged( unsigned long, unsigned long ) ),
	   this, SLOT( slotActivePartChanged( unsigned long, unsigned long ) ) );

  // create the menu bar
  (void)menuBarManager();
  
  // create the toolbar manager to handle the toolbars of the embedded parts
  (void)toolBarManager();

  // build a toolbar and insert some buttons
  opToolBar()->insertButton(Icon("filenew.xpm"),TOOLBAR_NEW, SIGNAL( clicked() ),
			    this, SLOT( slotFileNew() ), true, i18n("New"));
  opToolBar()->insertButton(Icon("fileopen.xpm"),TOOLBAR_OPEN, SIGNAL( clicked() ),
			    this, SLOT( slotFileOpen() ), true, i18n("Open File"));
  opToolBar()->insertButton(Icon("filefloppy.xpm"), TOOLBAR_SAVE,
			    SIGNAL( clicked() ), this, SLOT( slotFileSave() ),
			    true, i18n("Save File") );
  opToolBar()->setItemEnabled( TOOLBAR_SAVE, false );
  opToolBar()->insertButton(Icon("fileprint.xpm"), TOOLBAR_PRINT,
			    SIGNAL( clicked() ), this, SLOT( slotFilePrint() ),
			    true, i18n("Print") );
  opToolBar()->setItemEnabled( TOOLBAR_PRINT, false );
  opToolBar()->insertButton(Icon("reload.xpm"),TOOLBAR_RELOAD, SIGNAL( clicked() ),
			    this, SLOT( slotUpdate() ), true, i18n("Update"));

  // Build a default menubar with at least file and help menu
  menuBarManager()->create( 0 );

  // Create the Splitter that holds the editor and the explorer
  m_pSplitter = new QSplitter( QSplitter::Horizontal, this );
  setView( m_pSplitter );
  
  // Create a frame that will host the HTML Component
  m_pExplorerFrame = new OPFrame( m_pSplitter );
  m_pEditorFrame = new OPFrame( m_pSplitter );

  // Create the HTML component. The component may run in
  // another process, but in this example it does not.
  m_vExplorer = Embed::Explorer::_duplicate( new Explorer );
  m_vEditor = Embed::Editor::_duplicate( new Editor );
  
  // Tell the component about the toplevel window.
  m_vExplorer->setMainWindow( interface() );
  m_vEditor->setMainWindow( interface() );
  
  // Display the HTML component in the frame
  // The Frame will care about KOM reference counting. So we
  // do not need to care.
  m_pExplorerFrame->attach( m_vExplorer );
  m_pEditorFrame->attach( m_vEditor );
  
  // Show some HTML code in the editor
  m_vEditor->append( "<HTML><HEAD><TITLE>OpenParts</TITLE></HEAD>" );
  m_vEditor->append( "<BODY bgcolor=#FFFFFF><H1>OpenParts ... are cool</H1></BODY></HTML>" );
  
  // Show that in the HTML component, too.
  slotUpdate();
}

MainWindow::~MainWindow()
{
  cleanUp();

  cout << "-MainWindow" << endl;
}

void MainWindow::cleanUp()
{
  cout << "MainWindow cleanUp" << endl;
  
  // Free all references
  m_vExplorer = 0L;
  m_vEditor = 0L;
  
  // Release the HTML component. This will delete
  // the component.
  if ( m_pEditorFrame )
    m_pEditorFrame->detach();
  if ( m_pExplorerFrame )
    m_pExplorerFrame->detach();
}

void MainWindow::slotActivePartChanged( unsigned long _new_part_id,
					unsigned long _old_part_id )
{
  // Clear menu and toolbars
  menuBarManager()->clear();
  toolBarManager()->clear();
  // Create new menu and toolbars.
  menuBarManager()->create( _new_part_id );
  toolBarManager()->create( _new_part_id );
}

void MainWindow::createFileMenu( OPMenuBar* _menubar )
{
  // Do we loose control over the menubar ?
  if ( _menubar == 0L )
  {
    m_pFileMenu = 0L;
    return;
  }
  
  // Usual Qt way of dealing with menus. We dont have to go the
  // CORBA way here, because the menubar is ours. That means it runs
  // in the same process.
  m_pFileMenu = new OPMenu( _menubar );

  m_idMenuFile_Update = m_pFileMenu->insertItem( Icon( "reload.xpm" ),
						 i18n( "&Update" ), this,
						 SLOT( slotUpdate() ),
						 CTRL + Key_U );
  m_pFileMenu->insertSeparator(-1);
  
  m_idMenuFile_Quit = m_pFileMenu->insertItem( i18n( "&Quit" ), this,
					       SLOT( slotQuit() ),
					       CTRL + Key_Q );

  _menubar->insertItem( i18n( "&File" ), m_pFileMenu );
}

void MainWindow::createHelpMenu( OPMenuBar* _menubar )
{
  // Do we loose control over the menubar ?
  if ( _menubar == 0L )
  {
    m_pHelpMenu = 0L;
    return;
  }
  
  // Ask for the help menu. This is needed since
  // we share the help menu with the embedded components
  m_pHelpMenu = _menubar->helpMenu();
  // No help menu yet ?
  if ( m_pHelpMenu == 0L )
  {    
    m_pHelpMenu = new OPMenu( _menubar );

    _menubar->insertSeparator();
    _menubar->insertItem( i18n( "&Help" ), m_pHelpMenu );
  }
  else
    m_pHelpMenu->insertSeparator();

  // Insert our item
  m_idMenuHelp_About = m_pHelpMenu->insertItem( i18n( "&About KOM" ),
						this, SLOT( slotHelpAbout() ) );
}

void MainWindow::slotHelpAbout()
{
}

void MainWindow::slotQuit()
{
  delete this;
  
  kapp->exit();
}

void MainWindow::slotUpdate()
{
  // Get the text from the editor ...
  CORBA::String_var str = m_vEditor->text();
  // ... and pipe it to the HTML component
  m_vExplorer->begin( "meta:/" );
  m_vExplorer->parse();
  m_vExplorer->write( str );
  m_vExplorer->end();
}

int main( int argc, char **argv )
{
  OPApplication app( argc, argv );

  // Create a top level window
  MainWindow* win = new MainWindow();
  win->show();
  
  app.exec();
  
  return 0;
}

#include "main.moc"
