/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "main.h"

#include <komApplication.h>

#include <iostream>

Sender::Sender()
{
  cout << "+Sender" << endl;

  SIGNAL_IMPL( "clicked" );
  SIGNAL_IMPL( "text" );
  SIGNAL_IMPL( "birthday" );
}

Sender::~Sender()
{
  cout << "-Sender" << endl;
}

void Sender::clicked()
{
  // Please mention the zero in the name.
  // This number indicates the amount of parameters,
  // that the signal uses.
  SIGNAL_CALL0( "clicked" );
}

void Sender::text( const char* message )
{
  // The argumentgs are always the arguments, which you could pipe
  // in a CORBA::Any with the <<= operator.
  // So if you can do
  //   CORBA::Any any
  //   any <<= XXXX;
  // then XXXX is what you must use here, too. CORBA::Any needs
  // special treatments for some types like strings. See
  // /mico/include/mico/any.h for details or consult your
  // prefered CORBA book.
  SIGNAL_CALL1( "text", CORBA::Any::from_string( (char*)message, 0 ) );
}

void Sender::birthday( const Signals::Birthday& bday )
{
  // The IDL compiler did generate an operator<<= function
  // for Signals::Birthday. That means we can just pass
  // it along here.
  SIGNAL_CALL1( "birthday", bday );
}

Receiver::Receiver()
{
  cout << "+Receiver" << endl;
}

Receiver::~Receiver()
{
  cout << "-Receiver" << endl;
}

void Receiver::slotClicked()
{
  cout << "Clicked" << endl;
}

void Receiver::slotText( const char* message )
{
  cout << "Text: '" << message << "'" << endl;
}

void Receiver::slotBirthday( const Signals::Birthday& bday )
{
  cout << "Birthday: " << bday.day << "." << bday.month << "." << bday.year << endl;
}

int main( int argc, char **argv )
{
  KOMApplication app( argc, argv );

  /***********
   * Creae a sender and a receiver
   * and connect their signals & slots
   ************/
  // The right side of the operator= is what usually happens
  // inside a factory. The factory returns a Signals::Signal_ptr
  // that we assign to 'sender'.
  Signals::Sender_var sender = Signals::Sender::_duplicate( new Sender );
  Signals::Receiver_var recv = Signals::Receiver::_duplicate( new Receiver );

  sender->connect( "clicked", recv, "slotClicked" );
  sender->connect( "text", recv, "slotText" );
  sender->connect( "birthday", recv, "slotBirthday" );

  // Print the connections
  KOM::Base::ConnectionSeq_var seq = sender->describeConnections();
  for( CORBA::ULong l = 0; l < seq->length(); ++l )
  {
    CORBA::String_var str = komapp_orb->object_to_string( (*seq)[l].receiver );
    cout << "Signal=" << (*seq)[l].sig << endl
	 << "Slot=" << (*seq)[l].function << endl;
  }
  // Free the sequence. This is important!!
  // This sequence carries references on our objects. As long as
  // these references exist we can not destroy our objects since
  // CORBA uses refernce counting. So free all references to objects
  // at once if you dont need them any more.
  seq->length( 0 );
  
  // Lets fire some signals
  sender->clicked();
  sender->text( "Hi KOM fans :-)" );
  Signals::Birthday bday;
  bday.day = 29;
  bday.month = 11;
  bday.year = 1998;
  sender->birthday( bday );

  cerr << "1 ---------------------------------------------" << endl;
  
  /*************
   * Disconnect one signal
   *************/
  sender->disconnect( "text", recv, "slotText" );

  // Print the connections
  seq = sender->describeConnections();
  for( CORBA::ULong l = 0; l < seq->length(); ++l )
  {
    CORBA::String_var str = komapp_orb->object_to_string( (*seq)[l].receiver );
    cout << "Signal=" << (*seq)[l].sig << endl
	 << "Slot=" << (*seq)[l].function << endl;
  }
  // Free the sequence
  seq->length( 0 );
  
  cerr << "2 ---------------------------------------------" << endl;

  /*************
   * Remove the receiver.
   * Then send something. We should not see anything
   * since no one is listening any more
   *************/

  // Usually we should do
  //   recv = ....
  //   recv->incRef();
  //   ....
  //   recv->decRef();
  // What we have done here is more like 
  //   recv = ....
  //   ....
  //   recv->incRef();
  //   recv->decRef();
  // That is a bit shorter, but almost the same as the other one.
  // That is because destroy() calls incRef() and decRef()
  // just after it. This will activate the KOM reference counting.
  // If the KOM reference counter falls down on 0, the function
  // cleanUp() is invoked. You MUST overload this function if
  // you hold references on other objects. You MUST free them all
  // in cleanUp(). In this example neither Sender nor Receiver holds
  // any references => we did not overload cleanUp().
  // Releasing all CORBA references on some object causes the
  // CORBA reference counter to fall on 0 and that invokes
  // the destructor.
  // Note, that the KOM reference counting works across distributed
  // objects, while the CORBA reference counting does not cross borders.
  // So you have to use the KOM reference counting or you have no real
  // control over the live cycle of your distributed objects.
  recv->destroy();
  // Bring the CORBA reference counter one deeper; hopefully on 0 at this point.
  // If recv would reside in another process, then the object is already
  // destroyed. KOM did that for you. What you would do here is freeing
  // the CORBA proxy object. That is because CORBA reference counting
  // does not cross process borders.
  // But since recv resides in the same process, this line decreases
  // the reference counter of the real Receiver object. That means
  // Receiver is deleted in this line.
  // I know, that is complicated. Just forget it. Call destroy() or
  // incRef()+decRef() and free all references in the cleanUp() function.
  // KOM does the rest for you :-)
  recv = 0L;
  
  sender->clicked();
  
  cerr << "3 ---------------------------------------------" << endl;

  /*************
   * Destroy the sender
   *************/
  sender->destroy();
  sender = 0L;
  
  cerr << "4 ---------------------------------------------" << endl;

  return 0;
}
