/*
**  pcmem.c: Modularised version of Privilege Controlled Memory
**
** Copyright 2004 Rocklyte Systems, Paul Manias
** Redistribution of this file is permitted under the GNU LGPL.
**
*/

#define PCMEM_NAME "pcmem"
#define PCMEM_VERSION "1.0"

#define PCMEM_MAJOR 240 //MEM_MAJOR
#define PCMEM_MINOR 12

#ifdef CONFIG_MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/major.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <asm/uaccess.h>

#include "pcmem.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
EXPORT_NO_SYMBOLS;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
/* New License scheme */
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#ifdef MODULE_AUTHOR
MODULE_AUTHOR("Rocklyte Systems: Paul Manias");
#endif
#ifdef MODULE_DESCRIPTION
MODULE_DESCRIPTION("Privilege Controlled Memory");
#endif
#endif

#define DEVICE_NAME PCMEM_NAME
#define DEVICE_NR(device) (MINOR(device))

struct pcmem_range {
	unsigned long start;	/* Start of address range */
	unsigned long end;	/* End of address range */
	int mode;		/* Permission flags, relevance: S_IRGRP, S_IWGRP, S_IROTH, S_IWOTH. */
	int gid;		/* Group with access to this address range (if zero, everyone is treated as 'others') */
};

static struct pcmem_range *pcmem_ranges = 0;	/* Array of pcmem address ranges */
static int pcmem_total = 0;         /* No of elements in the pcmem_ranges array */

static char *pcmem_name=PCMEM_NAME;

/*
 * Architectures vary in how they handle caching for addresses
 * outside of main memory.
 *
 */
static inline int uncached_access(struct file *file, unsigned long addr)
{
#if defined(__i386__)
	/*
	 * On the PPro and successors, the MTRRs are used to set
	 * memory types for physical addresses outside main memory,
	 * so blindly setting PCD or PWT on those pages is wrong.
	 * For Pentiums and earlier, the surround logic should disable
	 * caching for the high addresses through the KEN pin, but
	 * we maintain the tradition of paranoia in this code.
	 */
	if (file->f_flags & O_SYNC)
		return 1;
 	return !( test_bit(X86_FEATURE_MTRR, boot_cpu_data.x86_capability) ||
		  test_bit(X86_FEATURE_K6_MTRR, boot_cpu_data.x86_capability) ||
		  test_bit(X86_FEATURE_CYRIX_ARR, boot_cpu_data.x86_capability) ||
		  test_bit(X86_FEATURE_CENTAUR_MCR, boot_cpu_data.x86_capability) )
	  && addr >= __pa(high_memory);
#elif defined(CONFIG_IA64)
	/*
	 * On ia64, we ignore O_SYNC because we cannot tolerate memory attribute aliases.
	 */
	return !(efi_mem_attributes(addr) & EFI_MEMORY_WB);
#else
	/*
	 * Accessing memory above the top the kernel knows about or through a file pointer
	 * that was marked O_SYNC will be done non-cached.
	 */
	if (file->f_flags & O_SYNC)
		return 1;
	return addr >= __pa(high_memory);
#endif
}

static int ioctl_pcmem(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct mm_permit permit;
	struct mm_remove remove;
	int i, mode;

	/* Only privileged users may manipulate the user-accessible memory ranges */

	if (!capable(CAP_SYS_ADMIN)) return -EPERM;

	switch (cmd) {
	case MMIO_PERMIT_RANGE:
	/* Purpose: Permit address mapping for the specified range.
	** Args:    start, end, mode
	** Note:    The current gid determines group access.
	*/

	if (copy_from_user(&permit, (void *)arg, sizeof(permit))) return -EFAULT;

	if (permit.end <= permit.start) return -EINVAL;

	permit.end = permit.start + PAGE_ALIGN(permit.end - permit.start);

	mode = (permit.mode & (S_IROTH | S_IWOTH | S_IRGRP | S_IWGRP));

	/* If the address range will be restricted to the root group
	** and root owner, and if 'others' are not granted any access
	** flags, there is no point in mapping this address range.
	*/

	if ((!current->egid) && (!(mode & S_IRWXO))) return 0;

	/* If the range is already registered, do not create a new mapping */

	for (i=0; i < pcmem_total; i++) {
		if ((pcmem_ranges[i].start == permit.start) && (pcmem_ranges[i].end == permit.end) &&
			(pcmem_ranges[i].gid == current->egid)) {
			pcmem_ranges[i].mode = S_IRWXU | mode;
			return 0;
		}
	}
	/* Find an empty array entry.  If full, increase the array size */

	for (i=0; i < pcmem_total; i++) {
	   if ((!pcmem_ranges[i].start) && (!pcmem_ranges[i].end)) break;
	}

	if (i == pcmem_total) {
		struct pcmem_range *new;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
		new = vmalloc(sizeof(struct pcmem_range) * (pcmem_total + 1));
#else
		new = kmalloc(sizeof(struct pcmem_range) * (pcmem_total + 1), GFP_KERNEL);
#endif

		if (!new) return -ENOMEM;

		if (pcmem_ranges) {
			memcpy(new, pcmem_ranges, sizeof(struct pcmem_range) * pcmem_total);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
			vfree(pcmem_ranges);
#else
			kfree(pcmem_ranges);
#endif
		}

		pcmem_ranges = new;
		pcmem_total++;
	}

	pcmem_ranges[i].start = permit.start;
	pcmem_ranges[i].end   = permit.end;
	pcmem_ranges[i].gid   = current->egid;
	pcmem_ranges[i].mode  = S_IRWXU | mode;
	return 0;

	case MMIO_REMOVE_ALL:
	/* Remove all registered address ranges (fairly drastic,
	** provided for debugging only).
	*/

	for (i=0; i < pcmem_total; i++) {
/*
		pcmem_ranges[i].start = 0;
		pcmem_ranges[i].end = 0;
*/
		printk("%d: $%x - $%x, GID: %d, Mode: %o\n", i, (unsigned int)pcmem_ranges[i].start, (unsigned int)pcmem_ranges[i].end, pcmem_ranges[i].gid, pcmem_ranges[i].mode);
	}
	return 0;

	case MMIO_REMOVE_RANGE:
	/* Purpose: Remove user level access from the specified address range.
	** Args:    start, end
	** Note:    You must be using the same gid as in your initial PERMIT_RANGE call.
        **          Partial unmappings are not allowed.  You must match a previous PERMIT_RANGE call exactly.
	*/

	if (copy_from_user(&remove, (void *)arg, sizeof(remove))) return -EFAULT;

	/* It's possible that there may be multiple maps for a specific range, so
	** we have to scan the entire array.
	*/

	for (i=0; i < pcmem_total; i++) {
	   if ((pcmem_ranges[i].start == remove.start) && (pcmem_ranges[i].end == remove.end)) {
	      pcmem_ranges[i].start = 0;
	      pcmem_ranges[i].end = 0;
	   }
	}

	return 0;

	}

	return -EINVAL;
}

static int mmap_pcmem(struct file * file, struct vm_area_struct * vma)
{
	unsigned long offset, end;
	int uncached, i;

	/* If the user is privileged then he can map any region.
	**
	** If the user is not privileged then he is at the mercy of
	** existing memory map ranges that have been marked as user accessible
	** by privileged processes.  We will fail if his request
	** falls outside of the allowable address ranges.
	*/

	offset = vma->vm_pgoff << PAGE_SHIFT;

	if (!capable(CAP_SYS_ADMIN)) {
		/* Check if the user is allowed access to the requested address range */
		if (!pcmem_ranges) return -EACCES;

		end = offset + (vma->vm_end - vma->vm_start);
		for (i=0; i < pcmem_total; i++) {
			/* Requested address page must fall within an acceptable range */

			if (offset < pcmem_ranges[i].start) continue;
			if (end > pcmem_ranges[i].end) continue;

			/* Check mmap() protection flags against granted access flags */

			if (pcmem_ranges[i].mode & S_IWOTH) break;

			if ((pcmem_ranges[i].mode & S_IROTH) &&
                        	(!(vma->vm_flags & (VM_MAYWRITE|VM_WRITE)))) {
				break;
			}

			if (in_group_p(pcmem_ranges[i].gid)) {
				if (pcmem_ranges[i].mode & S_IWGRP) {
					/* Group granted write/read access */
					break;
				}

				if (pcmem_ranges[i].mode & S_IRGRP) {
					/* Group granted read-only access */
					if (!(vma->vm_flags & (VM_MAYWRITE|VM_WRITE))) {
						break;
					}
				}
			}
			else printk("pcmem_fail: Range: $%x - $%x, Req. GID: %d, You: %d\n", (int)offset, (int)end, pcmem_ranges[i].gid, current->egid);
		}

		/* If the entire array was scanned without success, the user is not allowed
		** access to the requested address range.
		*/

		if (i >= pcmem_total) return -EACCES;
	}

        uncached = uncached_access(file, offset);
#ifdef pgprot_noncached
	if (uncached)
		vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#endif

	/* Don't try to swap out physical pages.. */
	vma->vm_flags |= VM_RESERVED;

	/*
	 * Don't dump addresses that are not real memory to a core file.
	 */
	if (uncached)
		vma->vm_flags |= VM_IO;

	if (remap_page_range(vma, vma->vm_start, offset, vma->vm_end-vma->vm_start,
			     vma->vm_page_prot))
		return -EAGAIN;
	return 0;
}

static struct file_operations pcmem_fops =
{
   owner: THIS_MODULE,
   mmap:  mmap_pcmem,
   ioctl: ioctl_pcmem,
};

static int __init pcmem_init(void)
{
   int err;

   printk("%s: Initializing %s v"PCMEM_VERSION"\n", pcmem_name, pcmem_name);

   if (PCMEM_MAJOR != MEM_MAJOR) {
      if ((err = register_chrdev(PCMEM_MAJOR, pcmem_name, &pcmem_fops))) {
         printk(KERN_WARNING "%s: Unable to register the pcmem device.\n", pcmem_name);
         return err;
      }
   }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
   devfs_register(NULL, "pcmem", DEVFS_FL_NONE, PCMEM_MAJOR, PCMEM_MINOR, S_IFCHR | S_IRUGO | S_IWUGO, &pcmem_fops, NULL);
#else
   devfs_mk_cdev(MKDEV(PCMEM_MAJOR, PCMEM_MINOR), S_IFCHR | S_IRUGO | S_IWUGO, "pcmem");
#endif

   printk(KERN_INFO "pcmem: loaded\n");

   return 0;
}

static void __exit pcmem_exit(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
   devfs_remove(pcmem_name);
#endif

   if (PCMEM_MAJOR != MEM_MAJOR) {
      if (unregister_chrdev(MEM_MAJOR, pcmem_name) != 0) {
         printk(KERN_WARNING "%s: cannot unregister pcmem device\n", pcmem_name);
      }
   }

   printk("%s: unloaded.\n", pcmem_name);
}

/* The pcmem init and exit function registration (especially needed for Kernel 2.6) */
module_init(pcmem_init);
module_exit(pcmem_exit);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
#include <linux/vermagic.h>
#include <linux/compiler.h>

MODULE_INFO(vermagic, VERMAGIC_STRING);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,6)
#undef unix
struct module __this_module
__attribute__((section(".gnu.linkonce.this_module"))) = {
 .name = __stringify(KBUILD_MODNAME),
 .init = pcmem_init,
#ifdef CONFIG_MODULE_UNLOAD
 .exit = pcmem_exit,
#endif
};
#endif

static const char __module_depends[]
__attribute_used__
__attribute__((section(".modinfo"))) =
"depends=";
#endif
