/* (C) Marc SCHAEFER, 1998.
 * Parts have been done within DILOG SA
 * Released under the GPL
 */

#include <linux/module.h>

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/random.h>
#include <linux/malloc.h>
#include <linux/wrapper.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/irq.h>

#include "benchio.h"

#undef DEBUG_SCSI
#define  DEBUG_MEMORY_ALLOCATION

#include "/usr/src/linux/include/linux/blkdev.h"
#include "/usr/src/linux/drivers/scsi/scsi.h"
#include "/usr/src/linux/drivers/scsi/hosts.h"

/* BUGS
 *    - Not allocated officially
 */
#define BENCHIO_MINOR 147

typedef signed char BOOL;
#define FALSE 0

/* Kernel entry points implementation */

static void benchio_close(struct inode *inode,
                         struct file *file) {
   MOD_DEC_USE_COUNT;
}

static int benchio_open(struct inode *inode,
                        struct file *file) {
   MOD_INC_USE_COUNT;  
   return 0;
}

static int benchio_write(struct inode *inode,
                         struct file *file,
                         const char *buffer,
                         int count) {
   const char *temp = buffer;
   int orig_count = count;
   char c;

   printk(KERN_INFO "benchio: got %d bytes [", count);

   while (count > 0) {
      c = get_user(temp);
      temp++;
      count--;
      printk("0x%x", c);
      if (count) {
         printk(" ");
      }
   }

   printk("]\n");

   return orig_count;
}

static int benchio_read(struct inode *inode,
                        struct file *file,
                        char *buffer,
                        int count) {
   int r = 0;

   printk(KERN_INFO "benchio: reading %d bytes.\n", count);

   /* Needed ? */
   if ((r = verify_area(VERIFY_WRITE, buffer, count))) {
      return r;
   }

   /* Too complex ... */

   if (count) {
      put_user('D', buffer);
      count--;
      r++;
   }
   if (count) {
      put_user('I', buffer + 1);
      count--;
      r++;
   }
   if (count) {
      put_user('L', buffer + 2);
      count--;
      r++;
   }
   if (count) {
      put_user('O', buffer + 3);
      count--;
      r++;
   }
   if (count) {
      put_user('G', buffer + 4);    
      count--;
      r++;
   }
   if (count) {
      put_user('\n', buffer + 5);    
      count--;
      r++;
   }

   return r;
}

static unsigned char *buffer;

static Scsi_Cmnd *the_cmd;
static struct scatterlist *scatter_list;
static unsigned long int lba;
static struct file *block_device_file;
struct wait_queue *io_wait; /* one at a time */
unsigned long int buffer_size;

static void my_done(Scsi_Cmnd *cmnd) {
   wake_up_interruptible(&io_wait);
#ifdef DEBUG_SCSI
   printk(KERN_INFO "benchio: Did cmd: 0x%x.\n", cmnd->result);
#endif /* DEBUG_SCSI */
}

extern struct Scsi_Host *scsi_hostlist;

static inline int initiate_cmd(unsigned int flags, BOOL is_read) {
   unsigned short block_count = buffer_size / 512;
   unsigned long io_size = buffer_size;

   the_cmd->host = scsi_hostlist;

#ifdef DEBUG_SCSI
   printk("benchio: host: 0x%lx irq 0x%x\n",
          (unsigned long int) the_cmd->host->base,
          the_cmd->host->irq);
#endif /* DEBUG_SCSI */
   the_cmd->device = 0;
   the_cmd->target = 0;
   the_cmd->lun = 0;
   the_cmd->channel = 0;
   the_cmd->next = NULL;
   the_cmd->prev = NULL;
   the_cmd->device_next = NULL;
   the_cmd->reset_chain = NULL;

   the_cmd->cmd_len = 10;

   the_cmd->cmnd[0] = is_read ? 0x28 : 0x2A;
   the_cmd->cmnd[1] = 0;
   the_cmd->cmnd[2] = (unsigned char) (lba >> 24) & 0xff;
   the_cmd->cmnd[3] = (unsigned char) (lba >> 16) & 0xff;
   the_cmd->cmnd[4] = (unsigned char) (lba >> 8) & 0xff;
   the_cmd->cmnd[5] = (unsigned char) lba & 0xff;
   the_cmd->cmnd[6] = 0;
   the_cmd->cmnd[7] = (unsigned char) (block_count >> 8) & 0xff;
   the_cmd->cmnd[8] = (unsigned char) block_count & 0xff;
   the_cmd->cmnd[9] = 0;

   lba += block_count;

#if 0
   the_cmd->use_sg = 1;
   the_cmd->request_buffer = NULL;
   scatter_list->address = (unsigned char *) buffer;
   scatter_list->alt_address = NULL;
   scatter_list->length = io_size;
   the_cmd->buffer = scatter_list;
#else
   the_cmd->use_sg = 0;
   the_cmd->request_bufflen = io_size;
   the_cmd->request_buffer = (unsigned char *) buffer;
   the_cmd->buffer = NULL;
#endif

   the_cmd->bufflen = io_size; /* probably: total len */

#ifdef DEBUG_SCSI
   {
      int i;

      printk("benchio: dumping scsi cmd:");
      for (i = 0; i < the_cmd->cmd_len; i++) printk("%02x ", the_cmd->cmnd[i]);
      printk("\n");
   }
#endif /* DEBUG_SCSI */

   if (the_cmd->host) {
#ifdef BENCHIO_ENABLE_HIGH_DIRECT
      if (flags & BENCHIO_IOCTL_FLAGS_DIRECT_HIGH) {
         /* @@@ call the device's read and write routines, which might
          * @@@ not exist in Linux ...
          * @@@ this could be added in some devices.
          */
      }
      else
#endif /* BENCHIO_ENABLE_HIGH_DIRECT */
      if (flags & BENCHIO_IOCTL_FLAGS_DIRECT_MIDDLE) {
	 /* NOTES
	  *    - Will do an overlapped copy (not a problem).
	  */
	 scsi_do_cmd(the_cmd,
		     (const void *) the_cmd->cmnd,
		     (unsigned char *) buffer,
		     buffer_size,
		     my_done,
		     (14000 * HZ) /* long */,
		     0 /* no retry */);
      }
      else {
         the_cmd->host->hostt->queuecommand(the_cmd, my_done);
      }
#ifdef DEBUG_SCSI
      {
         int i;
      
         printk("benchio: dumping some scsi data:");
         for (i = 0; i < 20; i++) printk("%02x ", buffer[i]);
         printk("\n");
      }
#endif /* DEBUG_SCSI */

#ifdef DEBUG_SCSI
      printk(KERN_INFO "benchio: Sent SCSI cmd.\n");
#endif /* DEBUG_SCSI */
      return 0;
   }
   else {
      return -EINVAL;
   }
}

static int benchio_ioctl(struct inode *inode,
			 struct file * file,
			 unsigned int cmd,
			 unsigned long arg) {
   unsigned int the_cmd = cmd & BENCHIO_IOCTL_MASK_CMD;

   if (the_cmd == BENCHIO_IOCTL_NOPE) {
      return 0;
   }
   else if (the_cmd == BENCHIO_IOCTL_DO_READ) {
      if (buffer) {
	 int errno = initiate_cmd(cmd & BENCHIO_IOCTL_MASK_FLAGS, TRUE);

         if (errno) {
            return errno;
         }
         else {
            /* wait for completion */
            interruptible_sleep_on(&io_wait);
            /* well, was interrupted ? */
            /* no result check */
         }
      }
      else {
         return -ENOMEM;
      }
      return 0;
   }
   else if (the_cmd == BENCHIO_IOCTL_DO_WRITE) {
      if (buffer) {
	 int errno = initiate_cmd(cmd & BENCHIO_IOCTL_MASK_FLAGS, FALSE);

         if (errno) {
            return errno;
         }
         else {
            /* wait for completion */
            interruptible_sleep_on(&io_wait);
            /* well, was interrupted ? */
            /* no result check */
         }
      }
      else {
         return -ENOMEM;
      }
      return 0;
   }
   else if (the_cmd == BENCHIO_IOCTL_DO_READ_MULTIPLE) {
      if (buffer) {
         while (arg--) {
	    int errno = initiate_cmd(cmd & BENCHIO_IOCTL_MASK_FLAGS, TRUE);

            if (errno) {
               return errno;
            }
            else {
	       /* wait for completion */
	       interruptible_sleep_on(&io_wait);
	       /* well, was interrupted ? */
	       /* no result check */
            }
         }
      }
      else {
         return -ENOMEM;
      }
      return 0;
   }
   else if (the_cmd == BENCHIO_IOCTL_DO_WRITE_MULTIPLE) {
      if (buffer) {
         while (arg--) {
	    int errno = initiate_cmd(cmd & BENCHIO_IOCTL_MASK_FLAGS, FALSE);

            if (errno) {
               return errno;
            }
            else {
	       /* wait for completion */
	       interruptible_sleep_on(&io_wait);
	       /* well, was interrupted ? */
	       /* no result check */
            }
         }
      }
      else {
         return -ENOMEM;
      }
      return 0;
   }
#ifdef BENCHIO_ENABLE_HIGH_DIRECT
   else if (the_cmd == BENCHIO_IOCTL_SET_DEVICE) {
      block_device_file = fget(arg); /* @@@ no matching fput @@@ */
      if (block_device_file) {
         return 0;
      }
      else {
         return -ENODEV;
      }
   }
#endif /* BENCHIO_ENABLE_HIGH_DIRECT */
   else if (the_cmd == BENCHIO_IOCTL_SET_LBA) {
      lba = arg;
      return 0;
   }
   else if (the_cmd == BENCHIO_IOCTL_SET_BUFFER_SIZE) {
      if (arg & 511) {
         return -EINVAL;
      }
      else {
         if (buffer) {
            kfree((unsigned char *) buffer);
         }

	 buffer = kmalloc(arg, GFP_KERNEL);
	 if (buffer) {
            buffer_size = arg;
	    return 0;
	 }
	 else {
            buffer_size = 0;
            buffer = NULL;
	    return -ENOMEM;
	 }
         /* NOT REACHED */
      }
      /* NOT REACHED */
   }
   else {
      printk(KERN_INFO "benchio: unknown cmd 0x%x arg 0x%lx\n", cmd, arg);
      return -EINVAL;
   }
   /* NOT REACHED */
}

/* Private constants
 * NOTES
 *    - Must be at this unusual place because of declarations of
 *      functions.
 */

static struct file_operations benchio_fops
   = {  NULL,           /* seek */
	benchio_read,      
	benchio_write,     
	NULL,           /* readdir */
	NULL,           /* select */
	benchio_ioctl,
	NULL,           /* mmap */
	benchio_open,
	benchio_close,
	NULL,           /* fsync */
	NULL            /* fasync */
     };

static struct miscdevice benchio = {
   BENCHIO_MINOR,
   "benchio",
   &benchio_fops
};

/* Private function implementation */

static int benchio_init(void) {
   lba = 0;
   buffer = NULL;
   buffer_size = 0;
   block_device_file = NULL;
   io_wait = NULL;
   the_cmd = kmalloc(sizeof(*the_cmd)
                     + sizeof(*scatter_list), GFP_KERNEL);
#ifdef DEBUG_MEMORY_ALLOCATION
   printk("kmalloc(%d).\n", sizeof(*the_cmd)
			    + sizeof(*scatter_list));
#endif /* DEBUG_MEMORY_ALLOCATION */
   if (the_cmd) {
      scatter_list
         = (struct scatterlist *)
              ((unsigned char *) the_cmd) + sizeof(*the_cmd);
      printk(KERN_INFO "benchio installed.\n");
      printk(KERN_INFO "benchio: $Id: benchio.c,v 1.25 1998/12/02 18:13:36 schaefer Exp $\n");
      misc_register(&benchio);
      return 0;
   }
   else {
      return -ENOMEM;
   }
   /* NOT_REACHED */
}

static void benchio_deinit(void) {
   printk(KERN_INFO "benchio deinstalled.\n");
   misc_deregister(&benchio);
   the_cmd = NULL;
   scatter_list = NULL;
   if (buffer) {
      kfree((unsigned char *) buffer);
   }
   buffer = NULL;
   buffer_size = 0;
   kfree(the_cmd);
}

/* Module configuration functions */

#ifdef MODULE
int init_module(void) {
   return benchio_init();
}

void cleanup_module(void) {
   benchio_deinit();
}
#endif /* MODULE */









