/*
 *    ext2hide - a superblock reserved space file stashing utility
 *      by infi/2006
 *
 *        ext2hide.h
 *      contains global types, defaults, structures and function declarations
 */

#include <ext2fs/ext2fs.h>

#define PROGNAME	"ext2hide"
#define AUTHOR		"Jason McManus"
#define COPYYEAR	"2006"

#define SBOFFSET    1024		// primary superblock offset on ext2
#define MAXFNAMELEN	255			// max filename length in chars
#define MAXSB 		48			// max number of superblocks (ARBITRARY!)
#define DEBUG 		1			// display debugging msgs (bool)
#define USAGESHORT	0			// print short usage information
#define USAGELONG	1			// print long usage information
#define NOFUNC		0			// cfg->function = 0
#define DEFPROMPT	1			// default user prompt setting (bool)

#define MINLIB		135			// minimum safe e2fsprogs library

#define DEFVERBOSITY	VNORMAL	// default verbosity level
#define VQUIET			0
#define VNORMAL			1
#define VERBOSE			2
#define VNOISY			3
#define VDEBUG			4

//typedef unsigned long ulong;
typedef unsigned long long ullong;

struct cfg_type {
	char fsdev[MAXFNAMELEN];	// user-supplied filesystem device
	int libver;			// ext2fs library version
	int verbose;		// print verbose info
	int prompt;			// prompt for confirmation
	char function;		// what are we doing today?
	char fname[MAXFNAMELEN];	// user-supplied filename (if specified)
	ulong total;		// number of free bytes
	ulong sbcount;		// number of superblocks
	ulong sb[MAXSB];	// superblock block group numbers
	ulong usable[MAXSB];	// usable reserved bytes per superblock
};

struct file_entry_type {
	char *f_name;
	ulong f_size;
	char *f_data;
	void *next;
};

typedef struct file_entry_type file_entry;

struct meta_info_type {
	ulong sig;
	ulong total_size;
	file_entry *file;
};

typedef struct cfg_type config;
typedef struct meta_info_type meta_info;

char sig[] = "\x49\x4e\x46\x49";

/******************************************/
/************** PROTOTYPES ****************/
/******************************************/

/* HIGH LEVEL FUNCTIONS (called from main()) */

// print usage information
void usage(int longargs);
// print version string
void version(void);
// validate arguments list (with getopt)
int check_args(int argc,char **argv,config *cfg);
// check library version, and warn user if pre-1.35
int check_lib_ver(int minlib);
// write a new file to the reserved space (ext2hide -w)
int write_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space);
// extract a stored file from reserved space (ext2hide -x)
int extract_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space);
// delete a stored file from reserved space (ext2hide -d)
int delete_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space);
// clear (zero) reserved space (ext2hide -c)
int clear_all(ext2_filsys fs,config *cfg,meta_info *meta,char *space);
// list stored files in reserved space (ext2hide -l)
int list_files(ext2_filsys fs,config *cfg,meta_info *meta,char *space);

/* MID LEVEL FUNCTIONS (called from other procedures) */

// prompt the user with a yes/no question, clear buffer, return true or false
int prompt(char *PS1);
// open the filesystem (using ext2 libs)
int fs_open(ext2_filsys *fs,config *cfg);
// find all of the superblocks (using ext2 libs)
int find_superblocks(ext2_filsys fs,ulong sb[]);
// read the superblock information (using ext2 libs)
ulong get_sbinfo(ext2_filsys fs,config *cfg);
// check if this block has a superblock (using ext2 libs)
blk_t has_superblock(ext2_filsys fs, ulong grpdesc);

// print the filesystem information
void print_fsinfo(ext2_filsys fs,config *cfg);
// read all of the superblocks
char *get_blocks(ext2_filsys fs,config *cfg);

// calculate total usable space
ulong get_usable(ulong bgnum, ulong blocksize, ulong reserved);
// calculate remaining usable space
int get_free_space(config *cfg,meta_info *meta,char *space);
// check if signature exists on filesystem (bool)
int check_sig(char *buf);
// extract a single file from filesystem
int get_file(meta_info *meta,char *buf2,ulong bufsize,ulong bgnum);
// zero and re-fill the buffer from stored meta information
void fill_buffer_from_meta(config *cfg,meta_info *meta,char *space);
// write the buffer back to the filesystem
int writeout_buffer(ext2_filsys fs,config *cfg,meta_info *meta,char *space);
// determine the offset of a particular blockgroup
ullong get_offset(ullong bgnum, ullong blocksize, ullong blockspergroup);

/* END ext2hide.h */
