/***************************************************************************
 * e2glob - glob() interface routines for the ext2 tools
 *
 * Copyright (C) 1995 Claus Tondering, ct@login.dknet.dk
 * This file may be redistributed under the terms of the GNU Public License.
 ***************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>

#include "ext2_fs.h"
#include "ext2fs/ext2fs.h"
#include "ldisk.h"
#include "istat.h"
#include "e2err.h"
#include "globlib/glob.h"

#define NAMELEN 80

struct fileinfo {
	u_char	name[NAMELEN];
};

struct dirst {
	struct fileinfo *list;
	int listix;
	int maxlist;
};

static ext2_filsys fs;

/**********************************************************************
 * myproc is a callback routine which is called once for each entry
 * in the directory
 **********************************************************************/

static int myproc(struct ext2_dir_entry *dirent,
				  int	offset,
				  int	blocksize,
				  char	*buf,
				  void	*private)
{
	struct dirst *di = (struct dirst*)private;

	if (!di->list) {
		di->list = malloc(100*sizeof(struct fileinfo));
		if (!di->list)
			return E2E_BADMEM;
		di->maxlist = 100;
	}

	strncpy(di->list[di->listix].name,dirent->name,dirent->name_len);
	di->list[di->listix].name[dirent->name_len] = 0;

	di->listix++;
	if (di->listix==di->maxlist) {
		di->list = realloc(di->list, (di->maxlist+100) * sizeof(struct fileinfo));
		if (!di->list) {
			fprintf(stderr,"Cannot allocate memory\n");
			return DIRENT_ABORT;
		}
		di->maxlist += 100;
	}
	return 0;
}

/**********************************************************************
 * opendir_hook, readdir_hook, and closedir_hook are routines to
 * open, read, and close an ext2 directory.
 * 
 * opendir_hook allocates and returns a struct dirst pointer, which is
 * used by the two other routines
 **********************************************************************/

static
__ptr_t opendir_hook (const char *directory)
{
	ino_t ino;
	struct ext2_inode e2ino;
	int err;
	struct dirst *di;

	/* Lookup specified name */
	err = ext2fs_namei(fs, 2, cwdino, directory, &ino);
	if (err)
		e2_err2("Cannot find file",directory,err);

	/* Read specified inode */
	err = ext2fs_read_inode(fs, ino, &e2ino);
	if (err)
		e2_err("Cannot read inode information", err);

	/* Is it a directory? */
	if (!S_ISDIR(e2ino.i_mode))
		return 0;
	
	/* Gather file list */
	di = (struct dirst*)malloc(sizeof(struct dirst));
	if (!di) {
		fprintf(stderr,"Cannot allocate memory\n");
		return 0;
	}
	di->listix = 0;
	di->maxlist = 0;
	di->list = NULL;

	if (ext2fs_dir_iterate(fs, ino, 0, 0, myproc, di))
		return 0;
	else {
		di->maxlist = di->listix;
		di->listix = 0;
		return (__ptr_t)di;
	}
}

static
const char * readdir_hook (__ptr_t stream)
{
	struct dirst *di = (struct dirst*)stream;
	
	if (di->listix==di->maxlist)
		return 0;
	else
		return di->list[di->listix++].name;
}

void closedir_hook(__ptr_t stream)
{
	struct dirst *di = (struct dirst*)stream;
	free(di->list);
	free(di);
}

/**********************************************************************
 * e2glob_init sets up the directory open, read, and close routines
 * for glob().
 **********************************************************************/

void
e2glob_init(ext2_filsys efs)
{
	fs = efs;
	__glob_opendir_hook=opendir_hook;
	__glob_readdir_hook=readdir_hook;
	__glob_closedir_hook=closedir_hook;
}
