/* vi: set ts=4 sw=4 expandtab: */
/*
 * linuxrc.c 
 *
 * compile with: cc linuxrc.c -s -Os -static -o linuxrc
 * or:           diet -Os cc linuxrc.c -s -static -o linuxrc
 *
 * Copyright (C) 2003 by Timo Benk <t_benk@web.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <linux/unistd.h>
#include <linux/posix_types.h>
//#undef dev_t
//#define dev_t __kernel_dev_t
#include <linux/loop.h>
//#undef dev_t

#define FLAG_CRAMFS      1
#define FLAG_CLOOP       2
#define FLAG_CD_INTO_RAM 4

char *cds[] = 
{
    "/dev/sr0",  "/dev/sr1",  "/dev/sr2",  "/dev/sr3",
    "/dev/sr4",  "/dev/sr5",  "/dev/sr6",  "/dev/sr7",
    "/dev/sr8",  "/dev/sr9",  "/dev/sr10", "/dev/sr11",
    "/dev/sr12", "/dev/sr13", "/dev/sr14", "/dev/sr15",
    "/dev/sr16", NULL
};

char *cloop_argv[] = 
{ 
    "/sbin/insmod", 
    "/lib/modules/cloop.o", 
    NULL 
};

extern char **environ;
char * tmpfs_size;

/**
 * A little malloc () wrapper.
 */
void * alloc (size_t len)
{
    void *ptr;

    if ((ptr = malloc (len)) == NULL)
    {
        perror ("[E046] malloc");
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }

    return (ptr);
}

/**
 * error () prints an error msg to stdout which is formatted
 * according to the following rules.
 * If errno_msg == NULL:
 * [idx] msg
 * if errno_msg != NULL:
 * [idx] msg: errno_msg
 * idx should be unique for each call to error ().
 */
void error (const char *idx, const char *msg, const char *errno_msg)
{
    char *err_msg;

    err_msg = alloc (strlen (idx) + strlen (msg) + strlen (errno_msg) + 6);
    
    strcpy (err_msg, "[");
    strcat (err_msg, idx);
    strcat (err_msg, "] ");
    strcat (err_msg, msg);

    if (errno_msg != NULL)
    {
        strcat (err_msg, ": ");
        strcat (err_msg, errno_msg);
    }

    puts (err_msg);
    free (err_msg);
}

/**
 * exec () executes filename and waits until the child
 * process exits. If filename was executed successfully
 * 0 is returned and -1 on any error. If exec was unsuccessfull
 * an informal message will be printed on stdout.
 */
int exec (const char *filename, char *const argv [])
{
    int status;
    pid_t pid;

    switch ((pid = fork ()))
    {
        case -1:
            error ("E001", "fork ()", strerror (errno));
            return (-1);
        case 0:
            if (execve (filename, argv, NULL) == -1);
            {
                error ("E002", filename, strerror (errno));
            }
            exit (EXIT_FAILURE);
        default:
            if (waitpid (pid, &status, 0) == -1)
            {
                error ("E003", filename, strerror (errno));
            }

            if (WIFEXITED (status) == 0)
            {
                error ("E032", filename, "Problems executing.");
                return (-1);
            }
            
            if (WIFEXITED (status) != 0 && WEXITSTATUS (status) != EXIT_SUCCESS)
            {
                error ("E033", filename, "Non-Zero Exit Code.");
                return (-1);
            }
    }

    return (0);
}

/**
 * find_cd () returns the index in the array cds[] of the CD-drive 
 * that contains the file RESCUECD.
 * If the CD cannot be found -1 is returned and an informative
 * error msg will be printed.
 */
int find_cd ()
{
    int i;

    for (i=0; cds[i] != NULL; i++)
    {
        if (mount(cds[i], "/mnt/cdrom", "iso9660", MS_RDONLY, NULL) == 0)
        {
            if (access ("/mnt/cdrom/RESCUECD", F_OK) == 0)
            {
                if (umount ("/mnt/cdrom") != 0)
                {
                    error ("E004", "umount /mnt/cdrom", (char*)strerror (errno));
                    return (-1);
                }
                
                return(i);
            }
        }
    }
    
    if (umount ("/mnt/cdrom") != 0)
    {
        error ("E047", "umount /mnt/cdrom", (char*)strerror (errno));
    }
    return(-1);
}

/**
 * _clone_file () copies src to dst and sets the original
 * ownership and permissions. If dst exists the function
 * will fail.
 * On error -1 is returned and an informative
 * error msg will be printed, otherwise 0 is returned.
 */
int _clone_file (char *src, char *dst)
{
    struct stat src_stat;

    if (lstat (src, &src_stat) != 0)
    {
        error ("E007", src, (char*)strerror (errno));
        return (-1);
    }

    if (S_ISDIR (src_stat.st_mode))
    {
        if (mkdir (dst, src_stat.st_mode) != 0)
        {
            error ("E008", dst, (char*)strerror (errno));
            return (-1);
        }

        if (chown (dst, src_stat.st_uid, src_stat.st_gid) != 0)
        {
            error ("E009", dst, (char*)strerror (errno));
            return (-1);
        }
    }
    else if (S_ISBLK  (src_stat.st_mode) ||
             S_ISCHR  (src_stat.st_mode) ||
             S_ISFIFO (src_stat.st_mode))
    {
        if (mknod (dst, src_stat.st_mode, src_stat.st_rdev) != 0)
        {
            error ("E010", dst, (char*)strerror (errno));
            return (-1);
        }

        if (chown (dst, src_stat.st_uid, src_stat.st_gid) != 0)
        {
            error ("E011", dst, (char*)strerror (errno));
            return (-1);
        }
    }
    else if (S_ISLNK (src_stat.st_mode))
    {
        char buf[4097];
        int len;

        if ((len = readlink (src, buf, 4096)) == -1)
        {
            error ("E012", src, (char*)strerror (errno));
            return (-1);
        }

        buf[len] = '\0';

        if (symlink (buf, dst) != 0)
        {
            error ("E013", dst, (char*)strerror (errno));
            return (-1);
        }
    }	
    else if (S_ISREG (src_stat.st_mode))
    {
        int ifd, ofd;
        int bytes;
        char buf[4096];

        if ((ifd = open (src, O_RDONLY)) == -1)
        {
            error ("E014", src, (char*)strerror (errno));
            return (-1);
        }

        if ((ofd = open (dst, O_WRONLY|O_CREAT)) == -1)
        {
            error ("E015", dst, (char*)strerror (errno));
            close (ifd);
            return (-1);
        }

        while ((bytes = read (ifd, buf, 4096)) > 0)
        {
            if (write (ofd, buf, bytes) == -1)
            {
                error ("E051", "write", strerror (errno));
                close (ifd); close (ofd);
                return (-1);
            }
        }
        
        if (bytes == -1)
        {
            error ("E048", "read", (char*)strerror (errno));
            close (ifd); close (ofd);
            return (-1);
        }

        close (ifd);
        close (ofd);

        if (chown (dst, src_stat.st_uid, src_stat.st_gid) != 0)
        {
            error ("E016", dst, (char*)strerror (errno));
            return (-1);
        }

        if (chmod (dst, src_stat.st_mode) != 0)
        {
            error ("E017", dst, (char*)strerror (errno));
            return (-1);
        }
    }

    return (0);
}

/**
 * copy () copies the file pointed to by src to dst. If src is
 * a directory, the whole directory will be copied. 
 * If dst exists and is a directory, all files in src are
 * copied into dst, dst/src will *NOT* be created.
 * On success 0 is returned and -1 on any error and an informative
 * error msg will be printed.
 */
int copy (char *src, char *dst)
{
    struct stat src_stat;

    if (lstat (src, &src_stat) != 0)
    {
        error ("E005", src, (char*)strerror (errno));
        return (-1);
    }

    _clone_file (src, dst);
        
    if (S_ISDIR (src_stat.st_mode))
    {
        struct dirent *d;
        DIR * dp;

        if ((dp = opendir (src)) == NULL)
        {
            error ("E006", src, (char*)strerror (errno));
            return (-1);
        }

        while ((d = readdir(dp)) != NULL) 
        {
            char *new_src, *new_dst;

            if (strcmp(d->d_name, ".")  == 0 || strcmp(d->d_name, "..") == 0)
            {
                continue;
            }

            new_src = (char*)alloc (strlen (src) + NAME_MAX + 2);
            strcpy (new_src, src); strcat (new_src, "/"); strcat (new_src, d->d_name);
            new_dst = (char*)alloc (strlen (dst) + NAME_MAX + 2);
            strcpy (new_dst, dst); strcat (new_dst, "/"); strcat (new_dst, d->d_name);
            
            copy (new_src, new_dst);

            free (new_src);
            free (new_dst);
        }
    }

    return (0);
}

/**
 * parse_cmdline () parses the kernel cmdline and returns
 * an or'ed list of the flags found.
 */
int parse_cmdline (int argc, char **argv)
{
    int idx = 0;
    int flags = 0;

    for (idx; idx < argc; idx++)
    {
        if (strcmp (argv[idx], "cramfs") == 0)
        {
            flags |= FLAG_CRAMFS;
        }
        
        if (strcmp (argv[idx], "cloop") == 0)
        {
            flags |= FLAG_CLOOP;
        }

        if (strcmp (argv[idx], "cd_into_ram") == 0)
        {
            flags |= FLAG_CD_INTO_RAM;
        }
    }

    if (getenv ("tmpfs_size") != 0)
    {
        tmpfs_size = malloc (strlen (getenv ("tmpfs_size") + 6));
        strcpy (tmpfs_size, "size=");
        strcat (tmpfs_size, getenv ("tmpfs_size"));
    }

    return (flags);
}

/**
 * lodetach () detaches the loop device pointed to by device. 
 * On error -1 is returned and an informative
 * error msg will be printed, otherwise 0 is returned.
 */
/*
int lodetach (char *device)
{
    int fd;

    if ((fd = open (device, O_RDONLY)) < 0) 
    {
        error ("E019", device, (char*)strerror (errno));
        return (-1);
    }

    if (ioctl (fd, LOOP_CLR_FD, 0) < 0) 
    {
        error ("E020", "ioctl: LOOP_CLR_FD", (char*)strerror (errno));
        close (fd);
        return (-1);
    }

    close (fd);
    return 0;
}
*/

/**
 * losetup () attaches file to the loop device
 * pointed to by device. 
 * On error -1 is returned and an informative
 * error msg will be printed, otherwise 0 is returned.
 */
int losetup (char *device, char *file)
{
    int ffd, dfd;
    struct loop_info loopinfo;
    
    if ((ffd = open (file, O_RDWR)) == -1)
    {
        if (errno == EROFS)
        {
            ffd = open (file, O_RDONLY);
        }

        if (ffd == -1)
        {
            error ("E021", file, (char*)strerror (errno));
            return (-1);
        }
    }

    if ((dfd = open (device, O_RDWR)) == -1)
    {
        if (errno == EROFS)
        {
            dfd = open (device, O_RDONLY);
        }

        if (dfd == -1)
        {
            error ("E022", device, (char*)strerror (errno));
            close (ffd);
            return (-1);
        }
    }

    memset (&loopinfo, 0, sizeof (loopinfo));

    if (ioctl (dfd, LOOP_SET_FD, ffd) < 0) 
    {
        error ("E023", "ioctl: LOOP_SET_FD", (char*)strerror (errno));
        close (ffd); close (dfd);
        return (-1);
    }

    if (ioctl (dfd, LOOP_SET_STATUS, &loopinfo) < 0) 
    {
        error ("E024", "ioctl: LOOP_SET_STATUS", (char*)strerror (errno));
        close (ffd); close (dfd);
        return (-1);
    }

    close (ffd);
    close (dfd);
    return (0);
}

/**
 * mount_cd () prepares the cd for the final system.
 * It depends on the configuration what is really done, eg
 * if the CD should be loaded into RAM mount_cd () will take
 * care of this. If successfull the final rootfs is mounted
 * on /mnt/cdrom.iso and 0 will be returned. If unsuccessfull
 * -1 will be returned and an informative error msg will be
 *  printed.
 */
int mount_cd (char *cd, int flags)
{
    // first we need to mount the CD on /mnt/cdrom
    if (mount(cd, "/mnt/cdrom", "iso9660", MS_RDONLY, NULL) != 0)
    {
        error ("E025", "mount [CD] /mnt/cdrom", (char*)strerror (errno));
        return (-1);
    }

    if (flags&FLAG_CRAMFS)
    {
        // now its time to decide whether the image should be
        // loaded into RAM or not
        if (flags&FLAG_CD_INTO_RAM)
        {
            if (mount("ramfs", "/mnt/ramfs", "ramfs", 0, NULL) != 0)
            {
                error ("E026", "mount ramfs /mnt/ramfs", (char*)strerror (errno));
                return (-1);
            }

            puts ("Loading cd image into RAM.");
            puts ("That could take some time, please be patient...");

            if (copy ("/mnt/cdrom/boot/rootfs.cramfs.img", "/mnt/ramfs/cd.iso") != 0)
            {
                return (-1); // copy () already printed an error msg
            }

            // we won't need the CD anymore
            if (umount ("/mnt/cdrom") != 0)
            {
                error ("E027", "umount /mnt/cdrom", (char*)strerror (errno));
                return (-1);
            }

            if (losetup ("/dev/loop0", "/mnt/ramfs/cd.iso") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }
        else
        {
            if (losetup ("/dev/loop0", "/mnt/cdrom/boot/rootfs.cramfs.img") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }

        // /dev/loop0 is now attached to the ramfs image file
        // and we can mount it on its final mountpoint.
        if (mount("/dev/loop0", "/mnt/cdrom.iso", "cramfs", MS_RDONLY, NULL) != 0)
        {
            error ("E028", 
                   "mount [rootfs.cramfs.img on /dev/loop0] /mnt/cdrom.iso", 
                   (char*)strerror (errno));

            return (-1);
        }
    }
    else if (flags&FLAG_CLOOP)
    {
        // now its time to decide whether the image should be
        // loaded into RAM or not
        if (flags&FLAG_CD_INTO_RAM)
        {
            if (mount("ramfs", "/mnt/ramfs", "ramfs", 0, NULL) != 0)
            {
                error ("E045", "mount ramfs /mnt/ramfs", (char*)strerror (errno));
                return (-1);
            }

            puts ("Loading cd image into RAM.");
            puts ("That could take some time, please be patient...");

            if (copy ("/mnt/cdrom/boot/rootfs.cloop_iso9660.img", "/mnt/ramfs/cd.iso") != 0)
            {
                return (-1); // copy () already printed an error msg
            }

            // we won't need the CD anymore
            if (umount ("/mnt/cdrom") != 0)
            {
                error ("E050", "umount /mnt/cdrom", (char*)strerror (errno));
                return (-1);
            }

            if (losetup ("/dev/loop0", "/mnt/ramfs/cd.iso") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }
        else
        {
            if (losetup ("/dev/loop0", "/mnt/cdrom/boot/rootfs.cloop_iso9660.img") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }

        // /dev/loop0 is now attached to the rootfs.cloop_iso9660.img image file
        if (mount("/dev/loop0", "/mnt/cloop", "iso9660", MS_RDONLY, NULL) != 0)
        {
            error ("E052", 
                   "mount [rootfs.cloop_iso9660.img on /dev/loop0] /mnt/cloop", 
                   (char*)strerror (errno));

            return (-1);
        }

        // insmod the cloop module
        if (exec (cloop_argv[0], cloop_argv) != 0)
        {
            return (-1); // exec () already printed an error msg
        }

        // setup the cloop device
        if (losetup ("/dev/cloop", "/mnt/cloop/rootfs.cloop.img") == -1)
        {
            return (-1); // losetup () already printed an error msg
        }
        
        // /dev/cloop is now attached to the rootfs.cloop.img image file
        if (mount("/dev/cloop", "/mnt/cdrom.iso", "iso9660", MS_RDONLY, NULL) != 0)
        {
            error ("E053", 
                   "mount [rootfs.cloop.img on /dev/cloop] /mnt/cdrom.iso", 
                   (char*)strerror (errno));

            return (-1);
        }
    }
    else
    {
        // now its time to decide whether the image should be
        // loaded into RAM or not
        if (flags&FLAG_CD_INTO_RAM)
        {
            if (mount("ramfs", "/mnt/ramfs", "ramfs", 0, NULL) != 0)
            {
                error ("E029", "mount ramfs /mnt/ramfs", (char*)strerror (errno));
                return (-1);
            }

            puts ("Loading cd image into RAM.");
            puts ("That could take some time, please be patient...");

            if (copy ("/mnt/cdrom/boot/rootfs.isofs.img", "/mnt/ramfs/cd.iso") != 0)
            {
                return (-1); // copy () already printed an error msg
            }

            // we won't need the CD anymore
            if (umount ("/mnt/cdrom") != 0)
            {
                error ("E030", "umount /mnt/cdrom", (char*)strerror (errno));
                return (-1);
            }

            if (losetup ("/dev/loop0", "/mnt/ramfs/cd.iso") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }
        else
        {
            if (losetup ("/dev/loop0", "/mnt/cdrom/boot/rootfs.isofs.img") == -1)
            {
                return (-1); // losetup () already printed an error msg
            }
        }

        // /dev/loop0 is now attached to the isofs image file
        // and we can mount it on its final mountpoint.
        if (mount("/dev/loop0", "/mnt/cdrom.iso", "iso9660", MS_RDONLY, NULL) != 0)
        {
            error ("E031", 
                   "mount [rootfs.iso9660.img on /dev/loop0] /mnt/cdrom.iso", 
                   (char*)strerror (errno));

            return (-1);
        }
    }

    return (0);
}

/**
 * prepare_system () will do some final preparation of the root fs, in
 * particular it will mount tmpfs  on the proper mountpoint
 * and it will copy the tree /mnt/cdrom.iso/usr/lib/varbase/root to
 * /mnt/cdrom.iso/var.
 * On error -1 is returned and an informative
 * error msg will be printed, otherwise 0 is returned.
 */
int prepare_system ()
{
    if (mount("tmpfs", "/mnt/cdrom.iso/var", "tmpfs", 0, tmpfs_size) != 0)
    {
        error ("E034", "mount tmpfs /mnt/cdrom.iso/var", (char*)strerror (errno));
        return (-1);
    }
    free (tmpfs_size);

    if (copy ("/mnt/cdrom.iso/usr/lib/varbase/root", "/mnt/cdrom.iso/var") == -1)
    {
        return (-1); // copy () already printed an error msg.
    }

    return (0);
}

/**
 * main ()
 */
int main (int argc, char **argv)
{
    int cd, flags, ifd, ofd;

#ifdef DEBUG
    puts ("DEBUG: Stage 1.1");
#endif

    // first search the CD and prompt if we cannot find one that suits
    while ((cd = find_cd ()) == -1)
    {
        char buf;
        puts ("Cannot find a valid cdrom in any drive - Fix that and press Enter");
        read (0, &buf, 1);
    }

#ifdef DEBUG
    puts ("DEBUG: Stage 2.1");
#endif

    // parse cmdline - no error checking necessary
    flags = parse_cmdline (argc, argv);

#ifdef DEBUG
    puts ("DEBUG: Stage 2.2");
#endif

    // mount the cd on /mnt/cdrom.iso
    if (mount_cd (cds[cd], flags) == -1)
    {
        puts ("Failed to setup CD - bailing out.");
        exit (EXIT_FAILURE);
    }

#ifdef DEBUG
    puts ("DEBUG: Stage 3.1");
#endif

    // prepare the system
    if (prepare_system () == -1)
    {
        puts ("Failed to prepare system - bailing out.");
        exit (EXIT_FAILURE);
    }

#ifdef DEBUG
    puts ("DEBUG: Stage 4.1");
#endif

    // cd /mnt/cdrom.iso
    if (chdir ("/mnt/cdrom.iso") == -1)
    {
        error ("E039", "chdir (\"/mnt/cdrom.iso\")", (char*)strerror (errno));
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }

#ifdef DEBUG
    puts ("DEBUG: Stage 4.2");
#endif

    // change the root device
    if (pivot_root(".", "initrd") == -1)
    {
        error ("E040", "pivot_root (\".\", \"initrd\")", (char*)strerror (errno));
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }

#ifdef DEBUG
    puts ("DEBUG: Stage 4.3");
#endif

    // chroot
    if (chroot (".") == -1)
    {
        error ("E041", "chroot (\".\")", (char*)strerror (errno));
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }
    
#ifdef DEBUG
    puts ("DEBUG: Stage 4.4");
#endif

    // and finally execute the real /sbin/init and start the system
    if ((ifd = open ("/dev/console", O_RDONLY)) == -1)
    {
        error ("E042", "/dev/console", (char*)strerror (errno));
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }
    
    if ((ofd = open ("/dev/console", O_WRONLY)) == -1)
    {
        error ("E043", "/dev/console", (char*)strerror (errno));
        puts ("Bailing out.");
        exit (EXIT_FAILURE);
    }

    dup2 (ifd, 0); dup2 (ofd, 1); dup2 (ofd, 2);
    close (ifd); close (ofd);

    if (execve ("/sbin/init", argv, environ) == -1)
    {
        error ("E044", "execve (\"/sbin/init\", argv, environ)", (char*)strerror (errno));
        exit (EXIT_FAILURE);
    }

    exit (EXIT_SUCCESS);
}

