#! /bin/sh
# MVM Project (C) Marc SCHAEFER <schaefer@alphanet.ch>
# mvm_low_level_library.sh -- Low level library functions.
# Creation: 14/02/98
# Update:   07/03/99
# V2.0 PV003
# DISCLAIMER
#    No warranty, either expressed or implied.
# COPYRIGHT
#    Fully protected under Berne Convention, use authorized except
#    if you make money from this use (contact me before).
# DESCRIPTION
#    This shell script (to be included through sourcing via '.' in sh)
#    is to be used by:
#       - the mail to voicemail gateway to create voicemails with notify
#       - the voicemail to WWW gateway software
#       - the daily crontab doing delayed notifications
#       - the main mvm_vgetty_script.sh (formerly known as voice_script.sh)
#         which is the work-horse of the interactive system.
#    It contains low level functions, except vgetty interaction.
# BUGS
#    - Logging and errors are reported through stderr. What about
#      a log() function ?
# TODO
#    - Create many low-level decoupled functions
# NOTES
# BASED-ON
#    voice_script.sh V1.3 PV004 of 08/02/98.
# MODIFICATION-HISTORY
#    14/02/98  schaefer  Created this file.
#    22/03/98  schaefer  Added notify.
#    07/03/99  schaefer  Support for new or old rsynth
# $Id: mvm_low_level_library.sh,v 1.7 1999/03/07 10:54:55 schaefer Exp $

# SYNOPSIS
#    result = make_account(password, namefile, personalfile)
# RESULT
#       - account number
#       - NONE if no accounts available
# DESCRIPTION
#    This function creates a new account with the
#    password indicated and the two description files.
# LOCAL-VARIABLES
#    Variables: RESULT
# NOTES
#    - This function does not use vgetty
function make_account {
   RESULT=1 # Do not mess with account 0, the superuser.
   while :
   do
      if [ -e $MAILBOX_DIR/$RESULT ]; then
         RESULT=`expr $RESULT + 1`
      else
         if [ $RESULT -gt $MVM_MAX_ACCOUNT_NUMBER ]; then
             RESULT=NONE
	     break
         else
	    mkdir $MAILBOX_DIR/$RESULT
	    if [ $? != 0 ]; then
	       RESULT=NONE # collision or fatal error ? we don't know, so ...
	       break
	    fi
	    # no tests from here
	    echo $1 > $MAILBOX_DIR/$RESULT/.password
	    cat $2 >  $MAILBOX_DIR/$RESULT/.name
	    cat $3 >  $MAILBOX_DIR/$RESULT/.personalinfo
	    # ev. a level file.
	    break
         fi
      fi
   done

   echo "$RESULT"
}

# SYNOPSIS
#    result = check_password(accountnumber, password)
# RESULT
#       - OK
#       - WRONG
#       - INEXISTANT
# DESCRIPTION
#    This function checks a password. Note that
#    it gives more information than the standard
#    UNIX login gives. However, this is not UNIX and
#    BTW, account numbers are linear so easy to guess.
# LOCAL-VARIABLES
#    Variables: RESULT
# NOTES
#    - This function does not use vgetty
function check_password {
   RESULT=INEXISTANT

   if [ -e $MAILBOX_DIR/$1/.password ]; then
      RESULT=`cat $MAILBOX_DIR/$1/.password`
      if [ "$RESULT" = "$2" ]; then
         RESULT=OK
      else
         RESULT=WRONG
      fi
   fi
   echo "$RESULT"
}

# SYNOPSIS
#    result = message_date(file)
# RESULT
#       - datestring
# DESCRIPTION
#    This functions outputs a date from a filename
# LOCAL-VARIABLES
#    Variables RESULT
# BUGS
#    - In some cases won't produce good output.
# NOTES
#    - This function does not use vgetty
#    - based on voice/script/speakdate.sh. Old implementation is commented
#      out. Uses GNU find.
function message_date {
   find $1 -printf '%TA, %TB %Td, at %Tk %TM\n'
#   RESULT="`ls -la $1`"
#   echo `echo "$RESULT" | cut -c43-48` at `echo "$RESULT" | cut -c50-51` `echo "$RESULT" | cut -c53-54`
}

# SYNOPSIS
#    result = random_file(dir)
# RESULT
#       - a random file name
# DESCRIPTION
#    This functions outputs a random file from a directory
# LOCAL-VARIABLES
# NOTES
#    - This function does not use vgetty
#    - it does not quite what it says, but it works.
function random_file {
   $SELECT_RANDOM_FILE `find $LIBRARY_DIR -type f -print` | awk '{print $1}'
}

# SYNOPSIS
#    result = english_text_to_audio_file(text, file)
#             [ was: say_something(text, file) ]
# RESULT
#       - OK
#       - ERROR
# DESCRIPTION
#    This function uses rsynth and the vgetty pvf utilities
#    to convert a text to its audio file representation.
#    This is slow and should only be used for
#    texts which must be generated and spoken.
# LOCAL-VARIABLES
# BUGS
#    - use of tmp file not unique. patch to rsynth would be needed. We
#      thus create a tmp dir.
#    - we should really do that fully in a pipe.
#    - we should support other generation methods.
# NOTES
#    - This function does not use vgetty
#    - Outputs to stderr removed (caused problems)
#    - pvfcut to allow hearing of last part of message with ZyXEL.
function english_text_to_audio_file {
   if [ "$SAY_RSYNTH_EXECUTABLE_VERSION" = "1" ]; then
      mkdir tmp_$$
      cd tmp_$$
      touch tmp
      $SAY_RSYNTH_EXECUTABLE -r $SPEED_FREQUENCY_VALUE -L "$1" 1>&2
      $PVF_TOOLS_DIR/lintopvf < tmp -s $SPEED_FREQUENCY_VALUE | $PVF_CUT | $PVF_AMP | $PVF_TOOLS_DIR/$PVF_MODEM_FILTER > $2
      cd ..
      rm -rf tmp_$$
      echo "OK"
   else
      # We can't set the speed to $SPEED_FREQUENCY, this
      # doesn't work, we must convert, bizarrely.
      $SAY_RSYNTH_EXECUTABLE -a $SAY_RSYNTH_EXECUTABLE_ARGUMENTS -o - "$1" | $PVF_TOOLS_DIR/autopvf | $PVF_SPEED | $PVF_CUT | $PVF_TOOLS_DIR/$PVF_MODEM_FILTER > $2
      echo "OK"
   fi
}

# SYNOPSIS
#    result = generic_numbers_to_audio_file(file, ...)
# RESULT
#    OK if OK, ERROR if file could not be generated.
# DESCRIPTION
#    Generate numbers language independantly.
# LOCAL-VARIABLES
#    $TMPFILE.gntaf
#    $FILE_NAME
# NOTES
#    - This function does not use vgetty
# BUGS
#    - Too slow with all those conversions. 
#    - cat should support multilanguage.
# TODO
function generic_numbers_to_audio_file {
   if [ $# -gt 1 ]; then
      FILE_NAME=$1
      shift
      echo "generic_numbers_to_audio_file: $SELECTED_LANGUAGE / $FRENCH_GENERATE_ENABLE. Args: $*" >&2

      (if [ "$SELECTED_LANGUAGE" = "$FRENCH_GENERATE_ENABLE" ]; then
	  export SELECTED_LANGUAGE
	  $FRENCH_GENERATE_MESSAGE $* | $PVF_TOOLS_DIR/rmdtopvf | $PVF_TOOLS_DIR/pvftoau
       else
	  # Loop through arguments

	  while [ $# != 0 ];
	  do
	     case $1 in
		--pre-formatted) shift
				 if [ $# != 0 ]; then
				    if [ -e "$1.$SELECTED_LANGUAGE" ]; then
				       cat $1.$SELECTED_LANGUAGE | $PVF_TOOLS_DIR/rmdtopvf | $PVF_TOOLS_DIR/pvftoau
				    else 
				       cat $1 | $PVF_TOOLS_DIR/rmdtopvf | $PVF_TOOLS_DIR/pvftoau
				    fi
				    shift
				 else
				    echo "$0: no argument to --pre-formatted" >&2
				 fi;;
		--say-number)    shift
				 if [ $# != 0 ]; then
				    RESULT=`english_text_to_audio_file $1 $TMPFILE.gntaf`
				    cat $TMPFILE.gntaf | $PVF_TOOLS_DIR/rmdtopvf | $PVF_TOOLS_DIR/pvftoau
				    rm $TMPFILE.gntaf
				    shift
				 else
				    echo "$0: no argument to --say-number" >&2
				 fi;;
		*)               echo "$0: unknown argument $1" >&2
				 break;;
	     esac
	  done
       fi) | $PVF_TOOLS_DIR/autopvf | $PVF_TOOLS_DIR/$PVF_MODEM_FILTER > $FILE_NAME
      echo "OK"
   else
      echo "ERROR"
   fi
}

# SYNOPSIS
#    result = send_voice_mail(toaccountID, fromaccountID, data_file,
#                             checkonly)
# RESULT
#    will be in the form 
#       - OK
#       - INEXISTANT
# DESCRIPTION
#    This function allows to send a voice mail message.
# NOTES
#    - This function does not use vgetty
#    - The data_file will be copied. The calling program must delete it.
#    - If checkonly == YES, we will only check if the user exists, this
#      is useful for interactive recording.
# BUGS
#    - No way to say error.
# LOCAL-VARIABLES
#    The RESULT variable.
# TODO
function send_voice_mail {
   if [ -e $MAILBOX_DIR/$1/.name ]; then
      if [ "$4" != "YES" ]; then
         cp $3 "$MAILBOX_DIR/$1/msg_`date '+%y%m%d%H%M%S'`_from_$2_$$"
         if [ "$NOTIFY_ENABLE" = "yes" ]; then
            $NOTIFY_SCRIPT $1 $2 "$MAILBOX_DIR/$1/msg_`date '+%y%m%d%H%M%S'`_from_$2_$$"
         fi
         # No error checking.
      fi
      RESULT=OK
   else
      RESULT=INEXISTANT
   fi
   echo "$RESULT"
}

