/* ydir - list files, search on wildcards, sort on time, size.

   Copyright (C) 1999-12-26 Donald Axel, dax@superusers.dk
   Saxenkolvej 20, DK3210, Vejby, Denmark. Phone +45 48 70 69 13

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

/* ydir -- search and list files like Digital Research's xdir      */
/* xdir.exe from dri does a good job but is not available on unix. */
/* This is a linux version which should be easy to port. Versions  */
/* for EP/IX, HP-UX, AIX and djgpp are running, but version 1.01   */
/* has not been tested on any platform but Linux 2.++ (RH5.2++)    */
/* Originally, however, the program was designed as an exercise in */
/* multiplatform programming.                                      */
/* On some systems you may need to adapt some macros from mode.h,  */
/* on others you may need fnmatch.[ch].                            */
/* hexdump() may be a help but can easily be out-commented.        */


/* history: */
/* 970110: version #defs for EP/IX, gcc-solaris and djgcc  */
/* 970208: sort options, skip "." and "..", split_and_list */
/* 970227: fnmatch() */
/* 971025: defamode.h */
/* 991210: -c crc option (cyclic redundancy check) */
/* the crc calculation algorithm written by        */
/* Q. Frank Xia, qx@math.columbia.edu.             */
/* CRC method is POSIX compliant.                  */
/* The code is taken from RH textutils-1.22 cksum  */
/* This method uses a crctab, is slow but robust.  */
/* 991214: Sort on size                            */

/* PERFORMANCE CONSIDERATIONS:
 * There are some libraries for treewalking which probably can
 * do the same job faster, however they are not portable.
 * isadir() is not needed on some systems, costs an extra stat()!
 */
/* todo listed by priority: */
/* todo: split source in ydir.h ydir.c, ydirlist.c & ydirutil.c */
/* todo: follow symbolic links */
/* todo: NT long filename support w/dj2xx */
/* todo: NT - shortcuts like symbolic links? */
/* todo: long options. */
/* todo: inodesort, search on time interval */
/* todo: intelligent higlighting of files with same CRC. */
/* todo: dir size often differs, count as zero size when doing tree-walk*/
/* todo: options: ex/include dirs and other file types */
/* todo: optimize so that stat is called only once, see isadir() */
/* todo: ls -l output format as option */

/* 
   examples below must use /'*' instead of ...hmm because some
   compilers are noisy about comment delimiters inside comments!
*/
/* todo: search operation could use alternate wildcard '@'
   (hp-mpe-ix - style); @ would be replaced internally by /'*'
   At present, use "set -f", quotes, "\*" or " sh -f ".
*/
/* NB: prevbase handles wildcard expansion 'bug': do not list 
   same basedir many times given e.g. dn/'*' becomes
      ydir dn/. dn/.. dn/fn 
   prevbase&isdirdots excludes dn/. and dn/.. from being listed/recursed.
   (where dn is dirname, fn is filename.)
*/
/* NB prevbase does not eliminate all inconveniences with DOS/NT 
   cmdline wildcards, which are expanded (use -e option to see why);
   thus you should not issue a command like
      ydir -s dn\*.*
   which will be expanded to: ydir dn dn/fn1 dn/fn2 ... dn/fn{n}
==================================================================
   1st.workaround: do not use wildcards when listing whole paths.
   Instead of `ydir -sne *.*' use `ydir -sne'.
   some ls versions, by the way, suffers from same "bug".
==================================================================
*/

#if __MSDOS__
static int msdos = 1;
#else
static int msdos;
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#if __GNUC__ && __MSDOS__
#include <std.h>
#else
#include <unistd.h>
#endif

#if __host_mips
#include </usr/share/include/sys/types.h>	/* EP/IX location */
#else
#include <sys/types.h>
#endif

#include <sys/param.h>
#include <time.h>
#include <sys/stat.h>
#include "dirent.h"
#include "fnmatch.h"
#include <errno.h>

#if ! defined S_IFIFO
#define S_IFIFO 0
#endif

typedef unsigned long long _uint64;


/* local prototypes: */

char *i2aformat(char *, long long int);
char *i2hex(_uint64 crc);
int  isadir(char *nm);
int  isdirdots(char *);
int  split_and_list(char *name);
void listfile(char *dname, char *fname);
int  listdir(char *dname);
int  listargs(int n, char *v[]);


/* notes for file access rights and time-stamps:

   The ydir program uses a reduced UNIX access rights display, because
   sometimes the long ls format contains much irrelevant information.

   Most often we are not interested in WHO owns the file, but only 
   whether or not we can read or maybe modify the file. 
   "No read" access is displayed as % in middle column. 

   File access rights are not really portable between DOS and UNIX, but
   that does not present a problem in this program.
   
   The 'archive' bit of (C)DOS would be an improvement in UNIX 
   environment. Also, DOS has -h (hidden) -s (system) and -v (volume) 
   attributes. 
   
   We could translate system and hidden to noread, even if it is not 
   quite true that a system file is "no read".

   The DJ109 and DJ201 stat() gives information in the st_mode byte 
   about some attributes, so that is what we use. DJ109 even has a
   pseudo I-node number even if DOS doesnot deal with inode-numbers, 
   links, pipes, sockets, doors (solaris), character devices etc.
   Please note, that support for dj109 has been removed. Support for
   djgpp is not tested with version 1.01
   
   TIME STAMPS:
   
   Also, please note: When a filestamp time is 0, (C)DOS has 
   (date==0 && time==0) whereas unix has (st_mtime==0), and
            DOS  file is 1980-00-00 0:00
            UNIX file is 1970-01-01 0:00

 */


/***********************************************************************
 *                       YDIR DECLARATIONS                             *
 ***********************************************************************/

/* data structure for saving fileinfo in a linked list;
   after savfsb variablesize path+filename is stored.
*/

struct savfsb {
  struct stat fs;
  struct savfsb *flnkl[2];           /* assigned but not used */
};

#define FNAME_MAX     600            /* arbitrary yet meaningful value */
#define PATTERN_MAX   256            /* arbitrary - large enough value */
#define MAXENT      96000            /* number of direntries in ptrarr */
struct savfsb *ap_fi[MAXENT];        /* array of ptrs to filestatbuffers */
                                     /* used for sorting names/sizes etc.*/
static long long int gcnt;           /* glob ndx for ap_fi */
static long long int dircnt;         /* directory count */

struct savfsb linkedbuf;
struct savfsb *p_fsb1 = &linkedbuf;     /* used for starting sort */
struct savfsb *p_fsbnext;
struct savfsb *p_fsbprev = &linkedbuf;
char pattern[PATTERN_MAX+1];            /* room for search - wildcard */

int printfsb(struct stat *,char *);     /* prototypes using savfsb */
void sort_and_print(struct savfsb *);  


/* global vars */

_uint64 totcrc;
_uint64 totsize;
_uint64 spaceused;
int access_errcnt;
int fopen_errcnt;
int save_rval;

/* mutually exclusive commands */

int gotcmd;
int sortontime;
int sortonname;
int sortonsize;


/* options and flags */

int ddt, ddt4, ddt6, ddt8;
int markft; 		/* mark file type appending "/" for dirs */
			/* todo: rsn:@ for links, and * for exes */
int usewild;		/* use a pattern to match filenames */
int reverse;            /* reverse sort order */
int brief, inodshow, onlyminutes;
int subdirs; 
int rq_sort;
int print_totals;
int docrc;
int dohelp;


int ourid;		/* our user id for checking against fileattr */
int ourgid;		/* our group id */
int namecnt;		/* count of filename arguments to ydir.      */


/* --------------  version and platform information  --------------- */

char *thisprog = __FILE__ ; /* for global argv[0] */

char *ydirNAME = "ydir: sort and search in subdirs";
char *ydirCOPY = "Copyright (C) 1999 Donald Axel, dax@superusers.dk";
char *ydirADDR = "Saxenkolvej 20, DK3210 Vejby, Denmark,+45 4870 6913";

#define VERSION_STRLEN 120
char build_id[VERSION_STRLEN] = __FILE__ " 1.01 of " __DATE__ " - " __TIME__
;
/* end of version constant str, do NOT remove semicolon */

/* function version_string(char*) adds compiler version info, if possible
   - Is there any nicer way?
 */

void version_string()
{
#if defined __GNUC__
    char *new_s;
    new_s = malloc(800);
    sprintf(new_s,"%s, GCC version %d.%d",build_id,__GNUC__,__GNUC_MINOR__);
    strncpy(build_id,new_s,VERSION_STRLEN);
    free(new_s);
#else
    strcat(build_id,".\n Compiler version not available.\n");
#endif
}

void printversion(int verbocity)
{
    printf("*****************************************************************\n");
    printf("* Called as: %-50s *\n* %-61s *\n",thisprog,build_id);
    printf("* %-61s *\n",ydirNAME);
    printf("*****************************************************************\n");
    printf("* %-61s *\n",ydirCOPY);
    if (verbocity >=2)
      printf("* %-61s *\n",ydirADDR);
    else
      printf("* %-61s *\n","use -W for details");
    printf("*****************************************************************\n");
}

int version(int talkmode)
{
    version_string();
    printversion(talkmode);
    if (talkmode >= 2) {
            printf("ydir comes with ABSOLUTELY NO WARRANTY; "
               "for details use ydir -W.\n"
               "This is free software, and you are welcome "
               "to redistribute it \n"
               "under certain conditions; use ydir -W for details.\n");
    }
    exit(0);
}

void Warranty()
{
    printversion(2);
    printf("%s\n","This program is distributed in the hope that it will be useful,");
    printf("%s\n","but WITHOUT ANY WARRANTY; without even the implied warranty of");
    printf("%s\n","MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
    printf("%s\n","GNU General Public License for more details.");
    printf("%s\n","");
    printf("%s\n","This program is free software; you can redistribute it and/or modify");
    printf("%s\n","it under the terms of the GNU General Public License as published by");
    printf("%s\n","the Free Software Foundation; either version 2 of the License, or");
    printf("%s\n","(at your option) any later version.");
    printf("%s\n","");
    printf("%s\n","You should have received a copy of the GNU General Public License");
    printf("%s\n","along with this program; if not, write to the Free Software");
    printf("%s\n","Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.");
    exit(0);
}


int usage(int forced) 
{
  FILE *output;
  if (forced)
        output = stderr;
  else
        output = stdout;
  fprintf(output,"%s usage: ydir -[n|r|t]s\n",thisprog);
  fprintf(output," -n : no sort\n");
  fprintf(output," -r : reversed sort (default on filename)\n");
  fprintf(output," -t : sort on timestamp\n");
  fprintf(output," -z : sort on size\n");
  fprintf(output," -s : recurse into subdirs\n");
  fprintf(output," -w \'wildcard\' :  match (remember the quotes)\n");
  fprintf(output,"Minor options: -Fbceimv\n");
  fprintf(output," -F : Add '/' for directories\n");
  fprintf(output," -b : brief listing, only filenames\n");
  fprintf(output," -c : print crc as cksum in col.1\n");
  fprintf(output," -e : print numbered list of command line arguments\n");
  fprintf(output," -i : print i-node number\n");
  fprintf(output," -m : print only minutes in timefield\n");
  fprintf(output," -q : quiet, no totals.\n");
  fprintf(output," -vVW  : print version, copyright and Warranty notices.\n");
  fprintf(output," -468d : different debug printouts, -dh help on exitcode\n");
  fprintf(output," -h : Show help screen (you are looking at it)\n");

  if (ddt){
     printf("\nExit codes:\n");
     printf("after a usage (this) message  ..........   0\n");
     printf("command line error ............... 101 - 103\n");
     printf("persmission denied for dir search ......   1\n");
     printf("error doing stat(2) (-d needed) ........ 226\n");
     printf("memory allocation error ................ 227\n");
     printf("internal list overflow ................. 229\n");
     printf("permission denied for stat on file ..... 255\n");
     printf("max number of files %d\n",MAXENT);
  }
  exit(forced);
}



/***********************************************************************
 *                       YDIR MAIN FUNCTION                            *
 ***********************************************************************/

int main(int argc, char **argv)
{
   int userscmd,rval;
   int c,jj,jk,ll;

   thisprog = argv[0];
   ourid = getuid();
   ourgid = getgid();
   rval = 0;
   /* default options */
   print_totals = rq_sort = sortonname = 1;

   jj=0;
   while (++jj < argc) {
     if ((c = argv[jj][0]) == 0) continue;
     if (c != '-') ++namecnt;
     else {
       jk=0;ll=strlen(argv[jj]);
       while (++jk<ll) {
         userscmd = argv[jj][jk];
         switch (userscmd) {
           case '4': ++ddt4;             break;
           case '6': ++ddt6;             break;
           case '8': ++ddt8;             break;
           case 'F': ++markft;           break;
           case 'b': ++brief;
                     print_totals=0;     break;
           case 'c': ++docrc;            break;
           case 'd': ++ddt;              break;
           case 'e': listargs(argc,argv);break;
           case 'i': ++inodshow;         break;
           case 'h': ++dohelp;           break;
           case 'm': ++onlyminutes;      break;
           case 'n': sortonname=0;rq_sort=0;++gotcmd;
                                         break;
           case 'q': print_totals=0;     break;
           case 'r': ++reverse;          break;
           case 's': ++subdirs;          break;
           case 't': sortonname=0;++sortontime;++rq_sort;++gotcmd;
                                         break;
           case 'V': version(2);         break;
           case 'v': version(1);         break;
           case 'W': Warranty();         break;

                     /* todo: multip.patterns? other OPTARG? */
           case 'w': if (jj+1>=argc) usage(103);
                     ++usewild;
                     strncpy(pattern,argv[jj+1],PATTERN_MAX);
                     // why?  if (argv[jj])
                     *argv[jj+1]=0;
                     break;
           case 'z': sortonname = 0;
                     ++sortonsize;
                     ++gotcmd;
                     break;
           default:  fprintf(stderr,"Command line error: %c (0x%x)\n",
                             userscmd,userscmd);
                     usage(101);
                     break;
         } /* endswitch */
         if (gotcmd>1) {
            fprintf(stderr,"Only one command may be specified (%c)\n",argv[jj][jk]);
            usage(102);
            break;
         }
       } /* endwhile jk single arg: -string_of_argletters */
     } /* endif argv[][0]=='-' option parser */
   } /* endwhile jj all argvs */

   /* by now we have command */
   if (dohelp) usage(0);

   if (msdos) inodshow = 0;
   
   if (namecnt==0)
      rval = listdir("."); 
   else {
      jj=0;
      while (++jj < argc) {
         if ((c = argv[jj][0]) == 0 || c == '-') continue;
         if ((rval=split_and_list(argv[jj])) != 0) {
            //char ss[512];
            //sprintf(ss,"%s: arg %d, file %s ", thisprog, jj, argv[jj]);
            //perror(ss);
            save_rval=rval;
         }
      }
   } /* end_else run thru argv */

   if (ddt && errno) {
      perror("Main error detected");
      errno = 0;
      sleep(2);
   }
   if (rq_sort) {
      sort_and_print(p_fsb1->flnkl[1]);
   }
   if (print_totals){
      char s1[44], s2[44], s3[44], s4[44];
      i2aformat(s1,gcnt); i2aformat(s2,totsize); i2aformat(s3,spaceused);
      printf("%s files; %s bytes; %s kb diskspace used (searched %s directories)\n",
      s1,s2,s3,i2aformat(s4,dircnt));
      if (docrc)
          printf("Crc total: %s\n",i2hex(totcrc)+3);
   }
   if (access_errcnt || fopen_errcnt) {
      if (print_totals) {
         if (access_errcnt)
            fprintf(stderr,"access denied %d times\n",access_errcnt);
         if (fopen_errcnt)
            fprintf(stderr,"fopen failed  %d times\n",fopen_errcnt);
      }
      return save_rval?save_rval:1;
   }
   return save_rval;
} /* end main */



static char prevbase[PATH_MAX];

int split_and_list(char *name)
{
  int c;
  int isdir;
  char *fspec;

   /* opendir, some implementations will open any file (non-dir)*/
   if (!(isdir=isadir(name))) {
      listfile("",name);
      return 0;
   }
   if (isdir == -1) {
      return 1;
   }
   if (!name[0]) return 0; /* should never happen, defensive programming */
                     /* leave c:/ but cut trailing "/..","/.","\..","\." */
   if (!(name[1]==':' && (((c = name[2]) == '\\') || c == '/') && !name[3]))
   if (strlen(name)>1)
     while ((c = *(fspec=(name+strlen(name)-1))) == '.' || c=='\\' || c=='/')
       *fspec = 0;

   if (strcmp(prevbase,name)) {
     strcpy(prevbase,name);
     return listdir(name);
   }
   else return 0;
} /* end split_and_list() */


int listdir(char *dname)
{
   DIR *dirp;
   struct dirent *direntp;
   char dir4print[PATH_MAX];

   strcpy(dir4print,dname);
   if (subdirs && namecnt > 1 && isdirdots(dname)) {     /* CAVEAT!namecnt*/
      return 0; /* probably expanded 'dirname/&' - skip */
   }
   /* common sense listing of current dir excludes ./ prefix */
   if (strcmp(dname,".")==0) {
      strcpy(dir4print,"");
   }
   
   if ((dirp = opendir(dname))==0) {
      ++access_errcnt;
      return -1;
   }
   while ( (direntp = readdir( dirp )) != NULL ) {
      if (!subdirs || !isdirdots(direntp->d_name)) {
         if (ddt4)
             printf("d4:%s:   d_name:%s:\n",
                     dir4print,direntp->d_name);
         listfile(dir4print,direntp->d_name);
      }
   }
   errno = 0;		 /* errno set at EndOfDir */
   closedir( dirp );
   return 0;
} /* end listdir() */


/* listfile() gets basedir and filename to list.
   to avoid spinning if there are circular symlinks
   we do NOT follow symlinks in this version;
   todo: add option to follow symbolic links and detect circular links.
 */

void listfile(char *dname, char *fname)
{
   struct stat fstatbuf;
   char pathfnam[FNAME_MAX], erms[FNAME_MAX];
   struct tm;
   int not_slashed(char *);
   int savfstat(struct stat *fstb, char *finame);

   strcpy(pathfnam,dname);
   if (not_slashed(dname) && dname[0])
      strcat(pathfnam,"/");
   strcat(pathfnam,fname);

   /* need lstat to detect symlinks. readlink() will dereference */
   if (lstat(pathfnam,&fstatbuf) != 0) {
      sprintf(erms,"Error %d reading stat on file \"%s\"",errno,pathfnam);
      perror(erms);
      if (ddt)
          exit(226);
      ++access_errcnt;
      return;
   }

/* EP/IX stat() always sets errno to 1 */
/* GO32 always sets errno to 2 */
/* Note: linux 2.0.36/glibc-2 also sets errno to 1 
   if file attributes (st_mode 9 lower bits) are all 0.
   We can safely ignore errno and do not increase access_errcnt,
   but we must reset errno to zero.
 */
   if (errno) {
      if (ddt) {
         sprintf(erms,"lstat(2) on %s given, but",pathfnam);
         perror(erms);
      }
      errno = 0;
   }

   /* recurse, do NOT expand "." or ".." and do NOT follow symlinks */
   if (fstatbuf.st_mode&S_IFDIR) {
      ++dircnt;
      if (subdirs) {
         if (!isdirdots(pathfnam) && !(fstatbuf.st_mode&S_IFLNK))
            if (listdir(pathfnam) < 0){
               sprintf(erms,"%s, listdir error",pathfnam);
               perror(erms); // ++errcnt? kkk
            }
      } 
   } 
   if ( !usewild || fnmatch(pattern,fname,0)==0 ) /* 0 if match */ {
      if (rq_sort)
         savfstat(&fstatbuf,pathfnam);
      else {
         printfsb(&fstatbuf,pathfnam);
         ++gcnt;
      }
   }
   return;
} /* end listfile() */



int savfstat(struct stat *fstb, char *finame)
{
  int slen, mrq;
  char erms[FNAME_MAX];
  struct savfsb *where;

   mrq = (slen = strlen(finame)) + sizeof(linkedbuf) + 4 ;
#if defined _AIX || defined _HPUX
   where = malloc(mrq); /* aix cc error if cast, proto is void *malloc() */
#else
   (char*) where = malloc(mrq);
#endif
   if (!where) {
      sprintf(erms,"%s malloc() line %d ",thisprog,__LINE__);
      perror(erms);
      exit(227);
   }
   where->flnkl[0]=p_fsbprev;
   p_fsbprev->flnkl[1]=where;
   p_fsbprev=where;			/* caveat! */
   
   memcpy(&(where->fs),fstb,sizeof(linkedbuf.fs));
   memcpy(where+1,finame,slen+1);
   ap_fi[gcnt] = where;		/* store address in static array */
   if (++gcnt>=MAXENT) {
      fprintf(stderr,"too many files (%d) to sort, use -n option\n",MAXENT);
      exit(229);
   }
   return 0;
}


#include "hexdumm.c"

static int cmpfsb(void *p1, void *p2)
{
   char *c1,*c2;

   c1 = (char*)(*(char**)p1 + sizeof(linkedbuf));
   c2 = (char*)(*(char**)p2 + sizeof(linkedbuf));
   if (reverse) return strcmp(c2,c1);
   return strcmp(c1,c2);
}

static int cmpfsbtime(void *t1, void *t2)
{
   struct savfsb *ps_s;
   time_t i1,i2; /* time_t, time type, is assumed, p.t. int32 */

   ps_s = (*(struct savfsb **)t1);
   i1 = ps_s->fs.st_mtime;
   ps_s = (*(struct savfsb **)t2);
   i2 = ps_s->fs.st_mtime;

   if (reverse) return (i2-i1);
   return (i1-i2);
}

static int cmpfsbsize(void *t1, void *t2)
{
   struct savfsb *ps_s;
   off_t i1,i2; /* time_t, time type, is assumed, p.t. int32 */

   ps_s = (*(struct savfsb **)t1);
   i1 = ps_s->fs.st_size;
   ps_s = (*(struct savfsb **)t2);
   i2 = ps_s->fs.st_size;

   if (reverse)
      return (i2-i1);
   return (i1-i2);
}


void sort_and_print(struct savfsb *fsb1)
{
   int jo;
   struct savfsb *p;

   if (!fsb1) {
      if (!gcnt) return;
      printf("invalid internal filetable\n");
      exit(228);
   }
   p = fsb1;
   if (ddt6) {
      printf("ddt6: ptr1 %p, namep %p, filename \"%s\"\n",p,p+1,(char *)(p+1));
      hexdump((char*)p,0x100,(int)p);
      printf("\npress <cr>\n");getchar();
   }
   if (sortonname)
     qsort(ap_fi,gcnt,sizeof(ap_fi[0]),(int(*)())(cmpfsb));
   else if (sortontime)
     qsort(ap_fi,gcnt,sizeof(ap_fi[0]),(int(*)())(cmpfsbtime));
   else if (sortonsize)
     qsort(ap_fi,gcnt,sizeof(ap_fi[0]),(int(*)())(cmpfsbsize));

   for(jo=0;ap_fi[jo];++jo){
      printfsb(&(ap_fi[jo]->fs),(char*)(ap_fi[jo]+1));
   }
}



/* crctab and crccalc are candidates for separate modules */
/* but have been "inlined" in the print function for speed*/

static unsigned long crctab[256] = {
  0x0,
  0x04C11DB7, 0x09823B6E, 0x0D4326D9, 0x130476DC, 0x17C56B6B,
  0x1A864DB2, 0x1E475005, 0x2608EDB8, 0x22C9F00F, 0x2F8AD6D6,
  0x2B4BCB61, 0x350C9B64, 0x31CD86D3, 0x3C8EA00A, 0x384FBDBD,
  0x4C11DB70, 0x48D0C6C7, 0x4593E01E, 0x4152FDA9, 0x5F15ADAC,
  0x5BD4B01B, 0x569796C2, 0x52568B75, 0x6A1936C8, 0x6ED82B7F,
  0x639B0DA6, 0x675A1011, 0x791D4014, 0x7DDC5DA3, 0x709F7B7A,
  0x745E66CD, 0x9823B6E0, 0x9CE2AB57, 0x91A18D8E, 0x95609039,
  0x8B27C03C, 0x8FE6DD8B, 0x82A5FB52, 0x8664E6E5, 0xBE2B5B58,
  0xBAEA46EF, 0xB7A96036, 0xB3687D81, 0xAD2F2D84, 0xA9EE3033,
  0xA4AD16EA, 0xA06C0B5D, 0xD4326D90, 0xD0F37027, 0xDDB056FE,
  0xD9714B49, 0xC7361B4C, 0xC3F706FB, 0xCEB42022, 0xCA753D95,
  0xF23A8028, 0xF6FB9D9F, 0xFBB8BB46, 0xFF79A6F1, 0xE13EF6F4,
  0xE5FFEB43, 0xE8BCCD9A, 0xEC7DD02D, 0x34867077, 0x30476DC0,
  0x3D044B19, 0x39C556AE, 0x278206AB, 0x23431B1C, 0x2E003DC5,
  0x2AC12072, 0x128E9DCF, 0x164F8078, 0x1B0CA6A1, 0x1FCDBB16,
  0x018AEB13, 0x054BF6A4, 0x0808D07D, 0x0CC9CDCA, 0x7897AB07,
  0x7C56B6B0, 0x71159069, 0x75D48DDE, 0x6B93DDDB, 0x6F52C06C,
  0x6211E6B5, 0x66D0FB02, 0x5E9F46BF, 0x5A5E5B08, 0x571D7DD1,
  0x53DC6066, 0x4D9B3063, 0x495A2DD4, 0x44190B0D, 0x40D816BA,
  0xACA5C697, 0xA864DB20, 0xA527FDF9, 0xA1E6E04E, 0xBFA1B04B,
  0xBB60ADFC, 0xB6238B25, 0xB2E29692, 0x8AAD2B2F, 0x8E6C3698,
  0x832F1041, 0x87EE0DF6, 0x99A95DF3, 0x9D684044, 0x902B669D,
  0x94EA7B2A, 0xE0B41DE7, 0xE4750050, 0xE9362689, 0xEDF73B3E,
  0xF3B06B3B, 0xF771768C, 0xFA325055, 0xFEF34DE2, 0xC6BCF05F,
  0xC27DEDE8, 0xCF3ECB31, 0xCBFFD686, 0xD5B88683, 0xD1799B34,
  0xDC3ABDED, 0xD8FBA05A, 0x690CE0EE, 0x6DCDFD59, 0x608EDB80,
  0x644FC637, 0x7A089632, 0x7EC98B85, 0x738AAD5C, 0x774BB0EB,
  0x4F040D56, 0x4BC510E1, 0x46863638, 0x42472B8F, 0x5C007B8A,
  0x58C1663D, 0x558240E4, 0x51435D53, 0x251D3B9E, 0x21DC2629,
  0x2C9F00F0, 0x285E1D47, 0x36194D42, 0x32D850F5, 0x3F9B762C,
  0x3B5A6B9B, 0x0315D626, 0x07D4CB91, 0x0A97ED48, 0x0E56F0FF,
  0x1011A0FA, 0x14D0BD4D, 0x19939B94, 0x1D528623, 0xF12F560E,
  0xF5EE4BB9, 0xF8AD6D60, 0xFC6C70D7, 0xE22B20D2, 0xE6EA3D65,
  0xEBA91BBC, 0xEF68060B, 0xD727BBB6, 0xD3E6A601, 0xDEA580D8,
  0xDA649D6F, 0xC423CD6A, 0xC0E2D0DD, 0xCDA1F604, 0xC960EBB3,
  0xBD3E8D7E, 0xB9FF90C9, 0xB4BCB610, 0xB07DABA7, 0xAE3AFBA2,
  0xAAFBE615, 0xA7B8C0CC, 0xA379DD7B, 0x9B3660C6, 0x9FF77D71,
  0x92B45BA8, 0x9675461F, 0x8832161A, 0x8CF30BAD, 0x81B02D74,
  0x857130C3, 0x5D8A9099, 0x594B8D2E, 0x5408ABF7, 0x50C9B640,
  0x4E8EE645, 0x4A4FFBF2, 0x470CDD2B, 0x43CDC09C, 0x7B827D21,
  0x7F436096, 0x7200464F, 0x76C15BF8, 0x68860BFD, 0x6C47164A,
  0x61043093, 0x65C52D24, 0x119B4BE9, 0x155A565E, 0x18197087,
  0x1CD86D30, 0x029F3D35, 0x065E2082, 0x0B1D065B, 0x0FDC1BEC,
  0x3793A651, 0x3352BBE6, 0x3E119D3F, 0x3AD08088, 0x2497D08D,
  0x2056CD3A, 0x2D15EBE3, 0x29D4F654, 0xC5A92679, 0xC1683BCE,
  0xCC2B1D17, 0xC8EA00A0, 0xD6AD50A5, 0xD26C4D12, 0xDF2F6BCB,
  0xDBEE767C, 0xE3A1CBC1, 0xE760D676, 0xEA23F0AF, 0xEEE2ED18,
  0xF0A5BD1D, 0xF464A0AA, 0xF9278673, 0xFDE69BC4, 0x89B8FD09,
  0x8D79E0BE, 0x803AC667, 0x84FBDBD0, 0x9ABC8BD5, 0x9E7D9662,
  0x933EB0BB, 0x97FFAD0C, 0xAFB010B1, 0xAB710D06, 0xA6322BDF,
  0xA2F33668, 0xBCB4666D, 0xB8757BDA, 0xB5365D03, 0xB1F740B4
};

/* because printfsb is also used for non-sorted listing,
   arg2 is needed.
 */
#define BUFLEN 65535
static char buf[BUFLEN+1];

int printfsb(struct stat *pstatb,char *fn)
{
  register unsigned char *cp;
  register unsigned int crc;
  unsigned int bytes_read;
  FILE *fp;
  char sbuf[44];
  char errmsg[FNAME_MAX];
  struct tm *ts;
  int printmod(struct stat *s);

   if (inodshow) printf("%8x ",(unsigned int) pstatb->st_ino);

   /* this is the POSIX compliant crc from Q. Frank Xia's cksum.c */
   if (docrc) {
      crc = 0;
      if (pstatb->st_mode&S_IFREG) {
          if ( (fp = fopen(fn,"r")) == NULL) {
              sprintf(errmsg,"Cannot open %s",fn);
              perror(errmsg);
              ++fopen_errcnt;
          }
          else {
              while ((bytes_read = fread (buf, 1, BUFLEN, fp)) > 0) {
                  cp = buf;
                  while (bytes_read--)
                      crc = (crc << 8) ^ crctab[((crc >> 24) ^ *(cp++)) & 0xFF];
              }
              fclose(fp);
              /* now, let the size influence the CRC, too */
              bytes_read = pstatb->st_size; 
              while (bytes_read > 0) {
                  crc = (crc << 8) ^ crctab[((crc >> 24) ^ bytes_read) & 0xFF];
              bytes_read >>= 8;
              }
              crc = ~crc & 0xFFFFFFFF;
          }
      }
      if (ddt8)
          printf("%10u ",crc);   /* compare with cksum need decimal output*/
      else
          printf("%08x ",crc);
      totcrc += crc;
   }
   if (brief) printf("%s\n",fn);
   else {
      printmod(pstatb);
      ts = localtime(&(pstatb->st_mtime)); /* + atime, ctime */
      if (markft && (pstatb->st_mode & S_IFDIR)) strcat(fn,"/");
      i2aformat(sbuf,pstatb->st_size);
      if (onlyminutes) 
         printf("%13s  %3d-%02d-%02d  %2d:%02d    %s\n",
             sbuf,ts->tm_year,ts->tm_mon+1,ts->tm_mday,
             ts->tm_hour,ts->tm_min, fn);
      else 
         printf("%13s  %3d-%02d-%02d  %2d:%02d:%02d    %s\n",
             sbuf,ts->tm_year,ts->tm_mon+1,ts->tm_mday,
             ts->tm_hour,ts->tm_min, ts->tm_sec, fn);
   };
   totsize += pstatb->st_size;
   spaceused += (pstatb->st_blocks + 1) / 2;
   return 0;
} /* end printfsb(); */



/* the following fcns are candidates for separate modules */


/* with Cdos printinodinfo() is dead code */

int printinodinfo(struct dirent *dp) 
{
   char s[44];
   i2aformat(s,dp->d_ino);
   printf("%15s ",s);
   return 0;
}


/* this printmod is only concerned about filetype and user-mode */
/* mode chrs (d<rw>x) of unix ls merged, <w> w/o read is absurd */

int printmod(struct stat *s)
{
   int m1,m2,m3;
   int fmod;

   fmod = s->st_mode;      /* first ck for device attributes: */

   m1 = m2 = m3 = '-';
   if (fmod & S_IFIFO)     /* named pipe */
      m1 = 'p';
   else if (S_ISLNK(fmod))
      m1 = 'l';
   else if (S_ISCHR(fmod))
      m1 = 'c';
   else if (S_ISDIR(fmod))
      m1 = 'd';
   else if (S_ISSOCK(fmod))
      m1 = 's';
   else if (fmod & S_IFBLK)
      m1 = 'b';
   else if (fmod & S_IFREG) /* regular file */
      m1 = '-';
   /* else if (fmod & S_IFDOOR) m1 = 'D'; only gcc, solaris */

   m2 = '-';
   if (s->st_uid == ourid) {
      if (fmod & S_IWUSR)
         m2 = 'w';
      if (fmod & S_IXUSR)
         m3 = 'x';
      if (!(fmod & S_IRUSR))     /* neither read/exe */
         m3 = '!';
   } else if (s->st_gid == ourgid) {
      if (fmod & S_IWGRP)
         m2 = 'w';
      if (fmod & S_IXGRP)
         m3 = 'x';
      if (!(fmod & S_IRGRP))     /* neither read/exe */
         m3 = '!';
   } else if (s->st_uid != ourid) {
      if (fmod & S_IWOTH)
         m2 = 'w';
      if (fmod & S_IXOTH)
         m3 = 'x';
      if (!(fmod & S_IROTH))
         m3 = '!';
   }
   if (fmod & S_ISVTX)           /* sticky bit */
      m3 = 's';
   else if (fmod & S_ISUID)      /* set user id */
      m3 = 'u';
   else if (fmod & S_ISGID)      /* set group id */
      m3 = 'g';
   return printf("%c%c%c",m1,m2,m3);
}


char *i2aformat(char *t, long long int ii) 
{
  char *o,*p;
  int c,a;
  
   a=0;
   o=p=t;
   if (!ii) {
     *p='0'; *(++p)=0; return o;
   }
   while (ii) {
      if (++a > 3) {
         *p++ = ',';
         a=1;
      }
      *p++ = (ii % 10) + '0';
      ii /= 10;
   }
   *p = 0;
   while (--p > t) {
      c = *p;
      *p = *t;
      *t++ = c;
   }
   return o;
}


char *i2hex(_uint64 x)
{
   register int i,j;
   static char hexbuf[32];
   sprintf(hexbuf,"%016Lx",x);
   for (i = 3; i > 0; --i) {
      for (j = 3; j >= 0; --j) {
         hexbuf[i*5 + j] = hexbuf[i*4 + j];
      }
   }
   hexbuf[4] = hexbuf[9] = hexbuf[14] = ',';
   return hexbuf;
}






/* for educational purposes: 
   option -e lists the expanded cmd_line args.
*/

int listargs(int n, char *v[])
{
  int i;
   
   i = 0;
   while (++i < n)
      printf("%3d: %s\n",i,v[i]);
   return 0;
}


int isdirdots(char *s)
{
   if(*s)
     s = s + strlen(s) - 1;
   return (*s == '.');
}

int not_slashed(char *s)
{
   if (*s)
     s = s + strlen(s) - 1;
   if (*s == '/' || *s == '\\')
     return 0;
   return 1;
}


/* isadir() calls stat() which is one time too many, but         */
/* some implementations of opendir opens any file.               */
/* However, we _can_ stat any file except e.g. stale NFS handles */

static struct stat sbuf;

int isadir(char *nm)
{
   if (stat(nm,&sbuf) == 0) {
      return (S_IFDIR & sbuf.st_mode);
   }
   perror(nm);
   return -1;
}


struct stat *getfstatbuf(char *fname) /* not used */
{
   return &sbuf;
}

