/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2011 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <glib.h>
#include <string.h>
#include <sys/mman.h>
#include "cache.h"
#include "status.h"
#include "mem.h"
#include "pwmd_error.h"

extern void log_write(const gchar *fmt, ...);

static gboolean valid_key(const guchar *key, gsize len)
{
    gsize b;

    for (b = 0; b < len; b++) {
	if (key[b])
	    return TRUE;
    }

    return FALSE;
}

static file_cache_t *get_entry(const guchar *md5file)
{
    gint t = g_slist_length(key_cache);
    gint i;

    if (!md5file)
	return NULL;

    for (i = 0; i < t; i++) {
	file_cache_t *p = g_slist_nth_data(key_cache, i);

	if (!memcmp(p->filename, md5file, sizeof(p->filename)))
	    return p;
    }

    return NULL;
}

gboolean cache_get_mutex(const guchar *md5file, pth_mutex_t **result)
{
    file_cache_t *p = get_entry(md5file);

    if (p)
	*result = p->mutex;

    return p ? TRUE : FALSE;
}

gint cache_file_count()
{
    gint t = g_slist_length(key_cache);
    gint i;
    gint n = 0;

    for (i = 0; i < t; i++) {
	file_cache_t *p = g_slist_nth_data(key_cache, i);

	if (valid_key(p->key, sizeof(p->key)) == TRUE)
	    n++;
    }

    return n;
}

void cache_adjust_timer()
{
    gint t = g_slist_length(key_cache);
    gint i;

    for (i = 0; i < t; i++) {
	file_cache_t *p = g_slist_nth_data(key_cache, i);

	if (p->reset >= 0) {
	    if (p->timeout > 0)
		p->timeout--;

	    if (p->timeout == 0) {
		if (valid_key(p->key, sizeof(p->key)) == TRUE) {
		    memset(&p->key, 0, sizeof(p->key));
		    send_status_all(STATUS_CACHE);
		}

		p->timeout = -1;
	    }
	}
    }
}

gboolean cache_add_file(const guchar *md5file, const guchar *shakey)
{
    file_cache_t *p = g_malloc0(sizeof(file_cache_t));
    GSList *new;

    if (!p) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return FALSE;
    }

    p->mutex = g_malloc(sizeof(pth_mutex_t));

    if (!p->mutex) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	g_free(p);
	return FALSE;
    }

    pth_mutex_init(p->mutex);
    memcpy(p->filename, md5file, sizeof(p->filename));

    if (shakey)
	memcpy(p->key, shakey, sizeof(p->key));

    p->reset = p->timeout = -2;

    if (mlock(p, sysconf(_SC_PAGESIZE)))
	log_write("mlock(): %s", pwmd_strerror(gpg_error_from_syserror()));

    new = g_slist_append(key_cache, p);

    if (!new) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	g_free(p->mutex);
	memset(p, 0, sizeof(file_cache_t));
	g_free(p);
	return FALSE;
    }

    key_cache = new;
    return TRUE;
}

static void free_entry(file_cache_t *p)
{
    if (!p)
	return;

    if (p->mutex)
	g_free(p->mutex);

    memset(p, 0, sizeof(file_cache_t));
    munlock(p, sysconf(_SC_PAGESIZE));
    key_cache = g_slist_remove(key_cache, p);
    g_free(p);
}

void cache_free()
{
    for (;;) {
	file_cache_t *p = g_slist_nth_data(key_cache, 0);

	if (!p)
	    break;

	free_entry(p);
    }
}

static gboolean reset_entry(file_cache_t *p)
{
    if (p && !p->refcount) {
	free_entry(p);
	return TRUE;
    }
    else if (p)
	memset(p->key, 0, sizeof(p->key));

    return FALSE;
}

gboolean cache_clear(const guchar *md5file, gint which)
{
    file_cache_t *p = md5file ? get_entry(md5file) : NULL;
    gint t, i;

    if (which == 1) {
	if (!p)
	    return FALSE;

	reset_entry(p);
	send_status_all(STATUS_CACHE);
	return TRUE;
    }

again:
    for (i = 0, t = g_slist_length(key_cache); i < t; i++) {
	p = g_slist_nth_data(key_cache, i);

	if (reset_entry(p))
	    goto again;
    }

    send_status_all(STATUS_CACHE);
    return TRUE;
}

gboolean cache_iscached(const guchar *md5file)
{
    file_cache_t *p = get_entry(md5file); 

    return p ? valid_key(p->key, sizeof(p->key)) : FALSE;
}

gboolean cache_reset_timeout(const guchar *md5file, glong timeout)
{
    file_cache_t *p = get_entry(md5file);

    if (!p)
	return FALSE;

    if (p->reset == -2)
	return cache_set_timeout(md5file, timeout);

    return cache_set_timeout(md5file, p->reset);
}

gboolean cache_set_timeout(const guchar *md5file, glong timeout)
{
    file_cache_t *p = get_entry(md5file);

    if (!p)
	return FALSE;

    if (timeout == -2)
	p->timeout = p->reset;
    else
	p->reset = p->timeout = timeout;

    return TRUE;
}

gboolean cache_update_key(const guchar *md5file, const guchar *shakey)
{
    file_cache_t *p = get_entry(md5file);

    if (!p)
	return cache_add_file(md5file, shakey);

    if (p->reset)
	memcpy(p->key, shakey, sizeof(p->key));

    return TRUE;
}

gboolean cache_get_key(const guchar *md5file, guchar *shakey)
{
    file_cache_t *p = get_entry(md5file);

    if (!p || valid_key(p->key, sizeof(p->key)) == FALSE)
	return FALSE;

    memcpy(shakey, p->key, sizeof(p->key));
    return TRUE;
}

gboolean cache_has_file(const guchar *md5file)
{
    file_cache_t *p = get_entry(md5file);

    return p ? TRUE : FALSE;
}

gboolean cache_decr_refcount(const guchar *md5file)
{
    file_cache_t *p = get_entry(md5file);

    if (!p)
	return FALSE;

    p->refcount--;

    if (p->refcount < 0)
	p->refcount = 0;

    return TRUE;
}

gboolean cache_incr_refcount(const guchar *md5file)
{
    file_cache_t *p = get_entry(md5file);

    if (!p)
	return FALSE;

    p->refcount++;
    return TRUE;
}
