/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2011 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include "common.h"
#include "mutex.h"
#include "misc.h"
#include "status.h"
#include "pwmd_error.h"

static void *write_status_cb(void *arg)
{
    struct assuan_cmd_s *data = arg;
    gpg_error_t *rc;
    gint old;

    pth_cancel_state(PTH_CANCEL_ENABLE|PTH_CANCEL_ASYNCHRONOUS, &old);
    rc = g_malloc(sizeof(gpg_error_t));
    *rc = assuan_write_status(data->ctx, data->line, data->line2);
    pth_cancel_state(old, NULL);
    pth_exit(rc);
    return NULL;
}

gpg_error_t send_status(assuan_context_t ctx, status_msg_t which,
	const gchar *fmt, ...)
{
    const gchar *line = NULL;
    gchar buf[ASSUAN_LINELENGTH+1];
    gchar *status = NULL;
    va_list ap;
    gpg_error_t rc;
    struct assuan_cmd_s data;

    if (fmt) {
	va_start(ap, fmt);
	g_vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);
	line = buf;
    }

    switch (which) {
	case STATUS_XFER:
	    status = "XFER";
	    break;
	case STATUS_CACHE:
	    CACHE_LOCK(ctx);
	    line = print_fmt(buf, sizeof(buf), "%i", cache_file_count());
	    CACHE_UNLOCK;
	    status = "CACHE";
	    break;
	case STATUS_CLIENTS:
	    MUTEX_LOCK(&cn_mutex);
	    line = print_fmt(buf, sizeof(buf), "%i", g_slist_length(cn_thread_list));
	    MUTEX_UNLOCK(&cn_mutex);
	    status = "CLIENTS";
	    break;
	case STATUS_CONFIG:
	    status = "CONFIG";
	    break;
	case STATUS_KEEPALIVE:
	    status = "KEEPALIVE";
	    break;
	case STATUS_LOCKED:
	    status = "LOCKED";
	    line = N_("Waiting for lock");
	    break;
	case STATUS_ENCRYPT:
	    status = "ENCRYPT";
	    break;
	case STATUS_DECRYPT:
	    status = "DECRYPT";
	    break;
	case STATUS_DECOMPRESS:
	    status = "DECOMPRESS";
	    break;
	case STATUS_COMPRESS:
	    status = "COMPRESS";
	    break;
    }

    if (!ctx) {
	log_write("%s %s", status, line);
	return 0;
    }

    data.ctx = ctx;
    data.line = status;
    data.line2 = line;
    rc = do_assuan_command(ctx, write_status_cb, &data);
    return rc;
}

void *client_msg_thread(void *arg)
{
    struct client_thread_s *thd = arg;
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);

    for (;;) {
	pth_event_t ev = pth_event(PTH_EVENT_MSG, thd->mp);

	pth_cleanup_push(cleanup_ev_cb, ev);
	pth_wait(ev);
	MUTEX_LOCK(&thd->mp_mutex);

	do {
	    pth_message_t *msg = pth_msgport_get(thd->mp);
	    status_msg_t *s = msg->m_data;
	    gpg_error_t rc;

	    MUTEX_UNLOCK(&thd->mp_mutex);
	    g_free(msg);
	    pth_cleanup_push(g_free, s);
	    rc = send_status(thd->cl->ctx, *s, NULL);
	    pth_cleanup_pop(1);

	    if (rc) {
		log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(rc));
		pth_cancel(thd->tid);
		break;
	    }

	    MUTEX_LOCK(&thd->mp_mutex);
	} while (pth_msgport_pending(thd->mp));

	MUTEX_UNLOCK(&thd->mp_mutex);
	pth_cleanup_pop(1);
    }

    return NULL;
}

void send_status_all(status_msg_t which)
{
    guint i, t;

    MUTEX_LOCK(&cn_mutex);
    pth_cleanup_push(cleanup_mutex_cb, &cn_mutex);

    for (t = g_slist_length(cn_thread_list), i = 0; i < t; i++) {
	struct client_thread_s *cn = g_slist_nth_data(cn_thread_list, i);
	pth_message_t *msg;
	status_msg_t *m;

	msg = g_malloc0(sizeof(pth_message_t));

	if (!msg) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    pth_cancel(cn->tid);
	    continue;
	}

	m = g_malloc(sizeof(status_msg_t));

	if (!m) {
	    g_free(msg);
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    pth_cancel(cn->tid);
	    continue;
	}

	*m = which;
	msg->m_data = m;

	if (cn->mp) {
	    MUTEX_LOCK(&cn->mp_mutex);
	    pth_msgport_put(cn->mp, msg);
	    MUTEX_UNLOCK(&cn->mp_mutex);
	}
    }

    pth_cleanup_pop(1);
}
