/*-------------------------------------------------------------------------
 *  Rocket Center of Pressure (cp) calculator - version 1.2
 *-------------------------------------------------------------------------
 * program cp calculates the center of pressure for a subsonic model
 * rocket. The equations listed here are taken from the Barrowman
 * equations as published in Harry Stien's "Handbook of Model Rocketry"
 * 5th edition.
 *
 * Usage: cp
 *
 * To compile: cc -O -o cp cp.c -lm
 *
 * Author: Mark C. Spiegl
 *
 * Copyright:
 * This program may be freely distributed, copied or modified provided
 * it is not used for profit. No warranty provided.
 *
 * Revisions:
 * 06/93   v1.0   Original
 * 06/93   v1.1   Mark Spiegl - Add support for multiple fin sets
 * 10/94   v1.2   Mark Spiegl - Add help screens
 *
 *------------------------------------------------------------------------
 */

/* Comment out or delete the following line to eliminate help screens */
#define OUTPUT_HELP_STRINGS

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "help.h"


#define VERSION         "1.2"   /* Version of cp */
#define MAXDATA         10      /* Maximum number of moments */
#define MAXSTR          20      /* Maximum size of input string */
#define MAXTR           3       /* Maximum number of transitions */
#define MAXFIN          3       /* Maximum fin sets (used for multi-stage) */
#define YES_NO_DEFAULT  'n'     /* Default answer to yes/no question */
#define DOUBLE_UNDEF    (-1.0)  /* Undefined double */
#define INT_UNDEF       (-1)    /* Undefined int */
#define CHAR_UNDEF      (-1)    /* Undefined char */
#define HELP_KEY        '?'     /* Character which invokes help */
#define NO_HELP         (char *)0       /* no help screen available */
#define TRUE            1
#define FALSE           0


 /*
  * Function declarations
  */
void nose();
void transitions();
void fins();
void calculate();
double pow2();
void get_double();
void get_char();
void get_int();
void initialize_vars();
void print_help_string();


 /*
  * Declaration for a single moment
  */
typedef struct {
   double ddist;                /* Directed distance */
   double mag;                  /* Magnitude */
}      MOMENT;

/*************************************************************************
* Global Variables
*************************************************************************/
MOMENT data[MAXDATA];           /* Array storing individual data points */
int index;                      /* index into "data" */

 /* Nose data */
char nose_shape;                /* nosecone shape - Ogive Cone Parabola (OCP) */
double nose_base_diameter;      /* diameter at base of nose (d) */
double nose_length;             /* length of nose cone (Ln) */

 /* Fin data */
int fin_set_count;              /* number of fin sets */
int fin_count[MAXFIN];          /* number of fins in fin set */
double fin_semispan[MAXFIN];    /* Dist from body tube to tip of fin (S) */
double fin_midchord[MAXFIN];    /* Length of fin itself (Lf) */
double fin_rootchord[MAXFIN];   /* Fin chord along body tube (Cr) */
double fin_tipchord[MAXFIN];    /* Fin chord along tip of fin (Ct) */
double fin_le_tip[MAXFIN];      /* Xr */
double fin_dd[MAXFIN];          /* Dist from top of nose to top of fin (Xb) */
double tail_diameter[MAXFIN];   /* Diameter of rocket at tail (R*2) */

 /* Transition data */
int transition_count;           /* number of transitions */
double transition_dd[MAXTR];    /* Dist from top of nose to top of trans (Xp) */
double front_diameter[MAXTR];   /* Diameter of top of transition (df) */
double rear_diameter[MAXTR];    /* Diameter of rear of transition (dr) */
double transition_length[MAXTR];/* Length of transition (Lt) */


/*************************************************************************
* Main program
*************************************************************************/
main()
{
   char more;                   /* flag asking for another calculation */

   (void) printf("\nCENTER OF PRESSURE CALCULATOR - Version %s\n\n", VERSION);
#ifdef OUTPUT_HELP_STRINGS
   (void) printf("Type '%c' at any prompt for help\n\n", HELP_KEY);
#endif
   initialize_vars();

   /*
    * Until user says no more get data and calculate CP
    */
   for (more = 'Y'; more == 'Y';
        more = 'N', get_char("\nDo another", "YN", no_help, &more)) {
      index = 0;
      nose();
      transitions();
      fins();
      calculate();
   }
}

/*************************************************************************
* Initialize computational variables
*************************************************************************/
void
initialize_vars()
{
   int i;

   /* Init nose data */
   nose_shape = CHAR_UNDEF;
   nose_base_diameter = DOUBLE_UNDEF;
   nose_length = DOUBLE_UNDEF;
   /* Init transition data */
   transition_count = 0;
   for (i = 0; i < MAXTR; i++) {
      transition_dd[i] = DOUBLE_UNDEF;
      front_diameter[i] = DOUBLE_UNDEF;
      rear_diameter[i] = DOUBLE_UNDEF;
      transition_length[i] = DOUBLE_UNDEF;
   }
   /* Init fin data */
   fin_set_count = 0;
   for (i = 0; i < MAXFIN; i++) {
      fin_count[i] = INT_UNDEF;
      fin_semispan[i] = DOUBLE_UNDEF;
      fin_midchord[i] = DOUBLE_UNDEF;
      fin_rootchord[i] = DOUBLE_UNDEF;
      fin_tipchord[i] = DOUBLE_UNDEF;
      fin_le_tip[i] = DOUBLE_UNDEF;
      fin_dd[i] = DOUBLE_UNDEF;
      tail_diameter[i] = DOUBLE_UNDEF;
   }
}

/*************************************************************************
* Process nose cone data
*************************************************************************/
void
nose()
{
   (void) printf("\nNose parameters\n");

   get_double("Diameter at base of nose cone (d)", nose_help, &nose_base_diameter);
   get_double("Length of nose cone (Ln)", nose_help, &nose_length);

   get_char("Nose shape: Ogive Cone Parabola [O,C,P]", "OCP", ocp_help, &nose_shape);
   switch (nose_shape) {
   case 'O':
      data[index].ddist = 0.466 * nose_length;
      data[index++].mag = 2.0;
      break;
   case 'C':
      data[index].ddist = 0.666 * nose_length;
      data[index++].mag = 2.0;
      break;
   case 'P':
      data[index].ddist = 0.5 * nose_length;
      data[index++].mag = 2.0;
      break;
   }
}

/*************************************************************************
* Process transition data
*************************************************************************/
void
transitions()
{
   int i = 0;                   /* index into transitions array */
   char yes_no = 'N';           /* do another flag */

   /*
    * Do we want to do this?
    */
   if (transition_count)
      yes_no = 'Y';
   else
      yes_no = 'N';
   get_char("\nComputate a transition", "YN", tran_help2, &yes_no);
   if (yes_no == 'N')
      return;

   /*
    * Get transition parameters
    */
   for (;;) {
      get_double("Distance from tip of nose to start of transition (Xp)",
                 xp_help, transition_dd + i);
      get_double("Diameter at front of transition (df)",
                 tran_help, front_diameter + i);
      get_double("Diameter at rear of transition (dr)",
                 tran_help, rear_diameter + i);
      get_double("Length of transition (Lt)",
                 tran_help, transition_length + i);

      /*
       * Compute moment
       */
      data[index].mag = 2.0 * (pow2(rear_diameter[i] / nose_base_diameter) -
                               pow2(front_diameter[i] / nose_base_diameter));
      data[index++].ddist = transition_dd[i] + transition_length[i] / 3.0 *
         (1.0 + (1.0 - front_diameter[i] / rear_diameter[i]) /
          (1.0 - pow2(front_diameter[i] / rear_diameter[i])));

      /*
       * Do another one?
       */
      if (++i >= MAXTR) {
         transition_count = MAXTR;
         return;
      }
      if (--transition_count > 0)
         yes_no = 'Y';
      else
         yes_no = 'N';
      get_char("\nComputate another transition", "YN", no_help, &yes_no);
      if (yes_no == 'N') {
         transition_count = i;
         return;
      }
   }
}

/*************************************************************************
* Process fin data
*************************************************************************/
void
fins()
{
   int i = 0;                   /* index into fin section array */
   char yes_no = 'N';           /* do another flag */

   (void) printf("\nFIN parameters\n");

   for (;;) {
      /*
       * Get fin parameters
       */
      get_int("Number of fins", no_help, fin_count + i);
      get_double("Diameter of bodytube at rear (R)",
                 r_help, tail_diameter + i);
      get_double("Fin semispan (S)", fin_help, fin_semispan + i);
      get_double("Fin midchord (Lf)", fin_help, fin_midchord + i);
      get_double("Fin rootchord (Cr)", fin_help, fin_rootchord + i);
      get_double("Fin tipchord (Ct)", fin_help, fin_tipchord + i);
      get_double("Fin sweepback distance (Xr)", fin_help, fin_le_tip + i);
      get_double("Distance from tip of nose to fin leading edge (Xb)",
                 xb_help, fin_dd + i);

      /*
       * Calculate magnitude of moment
       */
      data[index].mag = (1.0 + (tail_diameter[i] / 2.0) /
                         (fin_semispan[i] + (tail_diameter[i] / 2.0))) *
         ((4.0 * fin_count[i]) * pow2(fin_semispan[i] / nose_base_diameter)) /
         (1.0 + sqrt(1.0 + pow2(2.0 * fin_midchord[i] /
                                (fin_rootchord[i] + fin_tipchord[i]))));

      /*
       * Calculate directed distance of moment
       */
      data[index++].ddist = fin_dd[i] + fin_le_tip[i] / 3.0 *
         (fin_rootchord[i] + 2.0 * fin_tipchord[i]) /
         (fin_rootchord[i] + fin_tipchord[i]) +
         1.0 / 6.0 * (fin_rootchord[i] + fin_tipchord[i] -
                      (fin_rootchord[i] * fin_tipchord[i] /
                       (fin_rootchord[i] + fin_tipchord[i])));

      /*
       * Do another one?
       */
      if (++i >= MAXFIN) {
         fin_set_count = i;
         return;
      }
      if (--fin_set_count > 0)
         yes_no = 'Y';
      else
         yes_no = 'N';
      get_char("\nComputate another set of fins", "YN", no_help, &yes_no);
      if (yes_no == 'N') {
         fin_set_count = i;
         return;
      }
   }
}

/*************************************************************************
* Calculate CP from nose data, transition data and fin data
*************************************************************************/
void
calculate()
{
   int i;                       /* array index */
   double mag = 0.0;            /* magnitude of moment */
   double moment = 0.0;         /* magnitude * directed dist */

   (void) printf("-----\nResults\n");
   for (i = 0; i < index; i++) {
      mag += data[i].mag;
      moment += data[i].mag * data[i].ddist;
      (void) printf("   %d: %s=%-9.2f   %s=%-9.2f\n", i,
                    "magnitude of moment", data[i].mag,
                    "acting at directed dist", data[i].ddist);
   }
   (void) printf("\n   Tip of nose to Center-of-Pressure distance = %0.2f\n",
                 moment / mag);
}

/*************************************************************************
* Query for a char
*************************************************************************/
void
get_char(prompt, valid, help_string, ptr)
char *prompt;                   /* Prompt to output */
char *valid;                    /* Valid character set */
char **help_string;             /* pointer to help info (if exists) */
char *ptr;                      /* pointer to return data */
{
   char string[MAXSTR];
   char *tmp;

   for (;;) {
      (void) printf("   %s", prompt);
      if (*ptr != CHAR_UNDEF)
         (void) printf(" (default %c)", *ptr);
      (void) printf(": ");
      (void) fgets(string, MAXSTR, stdin);
      if ((*string == '\n') && (*ptr != CHAR_UNDEF))
         return;

      if ((help_string != (char **) 0) && (*string == HELP_KEY)) {
         print_help_string(help_string);
         continue;
      }
      if (string[1] == '\n') {
         if ((*string <= 'z') && (*string >= 'a'))
            *string += 'A' - 'a';
         for (tmp = valid; *tmp; tmp++) {
            if (*string == *tmp) {
               *ptr = *string;
               return;
            }
         }
      }
   }
}

/*************************************************************************
* Query for an int
*************************************************************************/
void
get_int(prompt, help_string, ptr)
char *prompt;                   /* prompt to output */
char **help_string;             /* pointer to help info (if exists) */
int *ptr;                       /* pointer to return data */
{
   char string[MAXSTR];

   do {
      (void) printf("   %s", prompt);
      if (*ptr != INT_UNDEF)
         (void) printf(" (default %d)", *ptr);
      (void) printf(": ");
      (void) fgets(string, MAXSTR, stdin);
      if ((*string == '\n') && (*ptr != INT_UNDEF))
         return;
      if ((help_string != (char **) 0) && (*string == HELP_KEY))
         print_help_string(help_string);
   } while (sscanf(string, "%d", ptr) != 1);
}

/*************************************************************************
* Query for a double
*************************************************************************/
void
get_double(prompt, help_string, ptr)
char *prompt;                   /* prompt to output */
char **help_string;             /* pointer to help info (if exists) */
double *ptr;                    /* pointer to return data */
{
   char string[MAXSTR];

   do {
      (void) printf("   %s", prompt);
      if (*ptr != DOUBLE_UNDEF)
         (void) printf(" (default %0.2f)", *ptr);
      (void) printf(": ");
      (void) fgets(string, MAXSTR, stdin);
      if ((*string == '\n') && (*ptr != DOUBLE_UNDEF))
         return;
      if ((help_string != (char **) 0) && (*string == HELP_KEY))
         print_help_string(help_string);
   } while (sscanf(string, "%lf", ptr) != 1);
}

/*************************************************************************
* print the help text
*************************************************************************/
void
print_help_string(help_string)
char **help_string;
{
   while (*help_string != (char *) 0)
      (void) printf(*(help_string++));
}

/*************************************************************************
* Square a number (some math libs have this some do not)
*************************************************************************/
double
pow2(x)
double x;
{
   return (x * x);
}
