/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [te_screen.cpp]            Screen Data Type                                */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

/* TODO
   - force some assertions on tmargin, bmargin and the cursor location.
   - make operations failsave.
*/

/*! \class TEScreen

    This class implements the operations of the terminal emulation framework.
    It is a complete passive device, driven by the emulation decoder 
    (AnsiEmulation). By this it forms infact an ADT, that mainly defines
    operations on a rectangular image.

    It does neither know how to display its image nor about escape sequences.
    It is further independent of the underlying toolkit. By this, one can even
    use this module for an ordinary text surface.

    Since the operations are called by a specific emulation decoder, one may
    collect their different operations here.

    The state manipulated by the operations is mainly kept in `image'.

    \sa TEWidget \sa AnsiEmulation
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <assert.h>
#include <strings.h>

#include "TEScreen.h"

#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

//FIXME: this is emulation specific. Use FALSE for xterm, TRUE for ANSI.
//FIXME: see if we can get this from terminfo.
#define BS_CLEARS FALSE

#define loc(X,Y) ((Y)*columns+(X))

/*! creates a `TEScreen' of `lines' lines and `columns' columns.
*/

TEScreen::TEScreen(int lines, int columns)
{
  this->lines   = lines;
  this->columns = columns;

  image  = (ca*) malloc(lines*columns*sizeof(ca));

  reset();
}

/*! Destructor
*/

TEScreen::~TEScreen()
{
  free(image);
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/* Normalized                    Screen Operations                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

// Cursor Setting --------------------------------------------------------------

/*! \section Cursor

    The `cursor' is a location within the screen that is implicitely used in
    many operations. The operations within this section allow to manipulate
    the cursor explicitly and to obtain it's value.

    The position of the cursor is guarantied to be between (including) 0 and
    `columns-1' and `lines-1'.
*/

void TEScreen::cursorUp(int n)
//=CUU
{
  if (n == 0) n = 1; // Default
  int stop = cuY < tmargin ? 0 : tmargin;
  cuX = MIN(columns-1,cuX); // nowrap!
  cuY = MAX(stop,cuY-n);
}

void TEScreen::cursorDown(int n)
//=CUD
{
  if (n == 0) n = 1; // Default
  int stop = cuY > bmargin ? lines-1 : bmargin;
  cuX = MIN(columns-1,cuX); // nowrap!
  cuY = MIN(stop,cuY+n);
}

void TEScreen::cursorLeft(int n)
//=CUB
{
  if (n == 0) n = 1; // Default
  cuX = MIN(columns-1,cuX); // nowrap!
  cuX = MAX(0,cuX-n);
}

void TEScreen::cursorRight(int n)
//=CUF
{
  if (n == 0) n = 1; // Default
  cuX = MIN(columns-1,cuX+n);
}

/*!
*/

void TEScreen::setMargins(int top, int bot)
//=STBM
{
  if (top == 0) top = 1;      // Default
  if (bot == 0) bot = lines;  // Default
  top = top - 1;              // Adjust to internal lineno
  bot = bot - 1;              // Adjust to internal lineno
  if ( !( 0 <= top && top < bot && bot < lines ) )
  { fprintf(stderr,"%s(%d) : setRegion(%d,%d) : bad range.\n",
                   __FILE__,__LINE__,top,bot);
    return;                   // Default error action: ignore
  }
  tmargin = top;
  bmargin = bot;
  cuX = 0;
  cuY = getMode(MODE_Origin) ? top : 0;
}

void TEScreen::index()
//=IND
{
  //FIXME: below bmargin?
  if (cuY >= bmargin) scrollUp(tmargin,1); else cuY += 1;
}

void TEScreen::reverseIndex()
//=RI
{
  //FIXME: above tmargin?
  if (cuY <= tmargin) scrollDown(tmargin,1); else cuY -= 1;
}

void TEScreen::NextLine()
//=NEL
{
  Return(); index();
}

// Line Editing ----------------------------------------------------------------

/*! \section inserting / deleting characters
*/

/*! delete `n' characters starting from (including) the cursor position.
   
    The line is filled in from the right with spaces.
*/

void TEScreen::deleteChars(int n)
{ 
  if (n == 0) n = 1; // Default
  int p = MAX(0,MIN(cuX+n,columns-1));
  moveImage(loc(cuX,cuY),loc(p,cuY),loc(columns-1,cuY));
  clearImage(loc(columns-n,cuY),loc(columns-1,cuY),' ');
}

/*! insert `n' spaces at the cursor position.
   
    The cursor is not moved by the operation.
*/

void TEScreen::insertChars(int n)
{ 
  if (n == 0) n = 1; // Default
  int p = MAX(0,MIN(columns-1-n,columns-1));
  int q = MAX(0,MIN(cuX+n,columns-1));
  moveImage(loc(q,cuY),loc(cuX,cuY),loc(p,cuY));
  clearImage(loc(cuX,cuY),loc(q-1,cuY),' ');
}

/*! delete `n' lines starting from (including) the cursor position.

    The cursor is not moved by the operation.
*/

void TEScreen::deleteLines(int n)
{ 
  if (n == 0) n = 1; // Default
  scrollUp(cuY,n);
}

/*! insert `n' lines at the cursor position.
   
    The cursor is not moved by the operation.
*/

void TEScreen::insertLines(int n)
{ 
  if (n == 0) n = 1; // Default
  scrollDown(cuY,n);
}

// Mode Operations -----------------------------------------------------------

void TEScreen::setMode(int m)
{
  currParm.mode[m] = TRUE;
  switch(m)
  {
    case MODE_Origin : cuX = 0; cuY = tmargin; break; //FIXME: home
  }
}

void TEScreen::resetMode(int m)
{
  currParm.mode[m] = FALSE;
  switch(m)
  {
    case MODE_Origin : cuX = 0; cuY = 0; break; //FIXME: home
  }
}

void TEScreen::saveMode(int m)
{
  saveParm.mode[m] = currParm.mode[m];
}

void TEScreen::restoreMode(int m)
{
  currParm.mode[m] = saveParm.mode[m];
}

//NOTE: this is a helper function
BOOL TEScreen::getMode(int m)
{
  return currParm.mode[m];
}

void TEScreen::saveCursor()
{
  sa_cuX     = cuX;
  sa_cuY     = cuY;
  sa_graphic = graphic;
  sa_cu_bold = cu_bold;
  sa_cu_fg   = cu_fg;
  sa_cu_bg   = cu_bg;
  // FIXME: Character set info: sa_charset = charsets[cScreen->charset];
  //                            sa_charset_num = cScreen->charset;
}

void TEScreen::restoreCursor()
{
  cuX     = sa_cuX;
  cuY     = sa_cuY;
  graphic = sa_graphic;
  cu_bold = sa_cu_bold;
  cu_fg   = sa_cu_fg;
  cu_bg   = sa_cu_bg;
  // FIXME: Character set info: sa_charset = charsets[cScreen->charset];
  //                            sa_charset_num = cScreen->charset;
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                             Screen Operations                             */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/*!  swap fore- and background at cursor position.

    This routine is intended for internal use while refreshing the display.
*/

void TEScreen::swapCursor()
{
  UINT8 f = image[loc(cuX,cuY)].f;
  UINT8 b = image[loc(cuX,cuY)].b;
  image[loc(cuX,cuY)].f = b;
  image[loc(cuX,cuY)].b = f;
}

/*!
   FIXME: experimental
*/

void TEScreen::setFixSize(int lines, int columns, bool fixed)
{
printf("setFixSize(%d,%d,%d)\n", lines, columns, fixed);
  sizeFixed = FALSE;
  resizeImage(lines,columns);
  sizeFixed = fixed;
}

/*!
*/

void TEScreen::resizeImage(int new_lines, int new_columns)
{
  if (sizeFixed) return; //FIXME: experimental

  // make new image
  ca* newimg = (ca*)malloc(new_lines*new_columns*sizeof(ca));

  // clear new image
  for (int y = 0; y < new_lines; y++)
  for (int x = 0; x < new_columns; x++)
  {
    newimg[y*new_columns+x].c = ' ';
    newimg[y*new_columns+x].f = DEFAULT_FORE_COLOR;
    newimg[y*new_columns+x].b = DEFAULT_BACK_COLOR;
  }
  int cpy_lines   = MIN(new_lines,  lines);
  int cpy_columns = MIN(new_columns,columns);
  // copy to new image
  for (int y = 0; y < cpy_lines; y++)
  for (int x = 0; x < cpy_columns; x++)
  {
    newimg[y*new_columns+x].c = image[loc(x,y)].c;
    newimg[y*new_columns+x].f = image[loc(x,y)].f;
    newimg[y*new_columns+x].b = image[loc(x,y)].b;
  }
  free(image);
  image = newimg;
  lines = new_lines;
  columns = new_columns;
  cuX = MIN(cuX,columns-1);
  cuY = MIN(cuY,lines-1);

  // FIXME: evtl. keep values
  tmargin=0;
  bmargin=lines-1;
}

/*!
*/

ca* const TEScreen::getImage() const
{
  return image;
}

/*!
*/

void TEScreen::reset()
{
    setMode(MODE_Wrap  ); saveMode(MODE_Wrap  );
  resetMode(MODE_Origin); saveMode(MODE_Origin);

  sizeFixed = FALSE;

  tmargin=0;
  bmargin=lines-1;

  strncpy(charset,"BBBB",4);

  setDefaultRendition();
  saveCursor();

  clear();
}

/*!
*/

void TEScreen::clear()
{ 
  setDefaultRendition();
  clearEntireScreen();
  home();
}

/*!
*/

void TEScreen::BackSpace()
{
  cuX = MAX(0,cuX-1);
  if (BS_CLEARS) image[loc(cuX,cuY)].c = ' ';
}

/*!
*/

void TEScreen::Tabulate()
{
//FIXME: that's not yet implemented
  // note that TAB is a format effector (does not write ' ');
  cursorRight(1); while(cuX < columns-1 && cuX%8 != 0) cursorRight(1);
}

/*!
*/

void TEScreen::NewLine()
{
//Note: as good as I understand the operation right now,
//      it behaves either as IND (index) or as NEL (NextLine)
//      depending on the NewLine Mode (LNM). This mode also
//      affects the key sequence returned for newline ([CR]LF).

//Return(); //FIXME: depends on newline mode
  index();
}

/*! put `c' literally onto the screen at the current cursor position.

    VT100 uses the convention to produce an automatic newline (am)
    with the *first* character that would fall onto the next line (xenl).
*/

void TEScreen::ShowCharacter(unsigned char c)
{

  // if (xenl)
  if (cuX >= columns)
  {
    if (getMode(MODE_Wrap)) { Return(); NewLine(); } else cuX = columns-1;
  }

  if (graphic) // handle graphical character set here
  {
    if (c == '0') c = ' '; else if (c == 'h') c = ' ';
    if (c >= 0x5f && c <= 0x7e) c = (c == 0x5f) ? 0x7f : c - 0x5f;
  }

  if (getMode(MODE_Insert)) insertChars(1);

  image[loc(cuX,cuY)].c = c;
  image[loc(cuX,cuY)].f = cu_fg;
  image[loc(cuX,cuY)].b = cu_bg;

  cuX += 1;

  //FIXME: VT100 wrapping is more complicated then i first thought.
  //       It does not happen after, but before the character is put.
  //       This has impact on the assumption of valid cursor positions.

  // if (!xenl)
  // if (cuX >= columns) NewLine(); //FIXME: am_wrap = TRUE;

}

/*!
*/

void TEScreen::setCharset(int n, int cs)
{
  charset[n&3] = cs;
}

/*!
*/

void TEScreen::useCharset(int n)
{
  graphic = (charset[n&3] == '0');
}

// Region commands -------------------------------------------------------------


/*! scroll up `n' lines within current region.
    The `n' new lines are cleared.
    \sa setRegion \sa scrollDown
*/

void TEScreen::scrollUp(int from, int n)
{
  if (n <= 0 || from + n > bmargin) return;
  //FIXME: make sure `tmargin', `bmargin', `from', `n' is in bounds.
  moveImage(loc(0,from),loc(0,from+n),loc(columns-1,bmargin));
  clearImage(loc(0,bmargin-n+1),loc(columns-1,bmargin),' ');
}

/*! scroll down `n' lines within current region.
    The `n' new lines are cleared.
    \sa setRegion \sa scrollUp
*/

void TEScreen::scrollDown(int from, int n)
{
  if (n <= 0) return;
  if (from > bmargin) return;
  if (from + n > bmargin) n = bmargin - from;
  //FIXME: make sure `tmargin', `bmargin', `from', `n' is in bounds.
  moveImage(loc(0,from+n),loc(0,from),loc(columns-1,bmargin-n));
  clearImage(loc(0,from),loc(columns-1,from+n-1),' ');
}

void TEScreen::setCursorYX(int y, int x)
{
  setCursorY(y); setCursorX(x);
}

void TEScreen::setCursorX(int x)
{
  if (x == 0) x = 1; // Default
  x -= 1; // Adjust
  cuX = MIN(columns-1, x);
}

void TEScreen::setCursorY(int y)
{
  if (y == 0) y = 1; // Default
  y -= 1; // Adjust
  cuY = MIN(lines  -1, y + (getMode(MODE_Origin) ? tmargin : 0) );
}

/*! set cursor to the `left upper' corner of the screen.
*/

void TEScreen::home()
{
  cuX = 0;
  cuY = 0;
}

/*! set cursor to the begin of the current line.
*/

void TEScreen::Return()
{
  cuX = 0;
}

/*! returns the current cursor columns.
*/

int TEScreen::getCursorX()
{
  return cuX;
}

/*! returns the current cursor line.
*/

int TEScreen::getCursorY()
{
  return cuY;
}

// Erasing ---------------------------------------------------------------------

/*! \section Erasing

    This group of operations erase parts of the screen contents by filling
    it with spaces colored due to the current rendition settings.

    Althought the cursor position is involved in most of these operations,
    it is never modified by them.
*/

/*! fill screen between (including) `loca' and `loce' with spaces.

    This is an internal helper functions. The parameter types are internal
    addresses of within the screen image and make use of the way how the
    screen matrix is mapped to the image vector.
*/

//FIXME: don't know where to set this
#define clearFillsWithDefault 1

void TEScreen::clearImage(int loca, int loce, char c)
{ int i;
  //FIXME: check positions
  for (i = loca; i <= loce; i++)
  {
    image[i].c = c;
    image[i].f = clearFillsWithDefault ? DEFAULT_FORE_COLOR : cu_fg;
    image[i].b = clearFillsWithDefault ? DEFAULT_BACK_COLOR : cu_bg;
  }
}

/*! move image between (including) `loca' and `loce' to 'dst'.

    This is an internal helper functions. The parameter types are internal
    addresses of within the screen image and make use of the way how the
    screen matrix is mapped to the image vector.
*/

void TEScreen::moveImage(int dst, int loca, int loce)
{
  //FIXME: check positions
  memmove(&image[dst],&image[loca],(loce-loca+1)*sizeof(ca));
}

/*! clear from (including) current cursor position to end of screen.
*/

void TEScreen::clearToEndOfScreen()
{
  clearImage(loc(cuX,cuY),loc(columns-1,lines-1),' ');
}

/*! clear from begin of screen to (including) current cursor position.
*/

void TEScreen::clearToBeginOfScreen()
{
  clearImage(loc(0,0),loc(cuX,cuY),' ');
}

/*! clear the entire screen.
*/

void TEScreen::clearEntireScreen()
{
  clearImage(loc(0,0),loc(columns-1,lines-1),' ');
}

/*! fill screen with 'E'
    This is to aid screen alignment
*/

void TEScreen::helpAlign()
{
  clearImage(loc(0,0),loc(columns-1,lines-1),'E');
}

/*! clear from (including) current cursor position to end of current cursor line.
*/

void TEScreen::clearToEndOfLine()
{
  clearImage(loc(cuX,cuY),loc(columns-1,cuY),' ');
}

/*! clear from begin of current cursor line to (including) current cursor position.
*/

void TEScreen::clearToBeginOfLine()
{
  clearImage(loc(0,cuY),loc(cuX,cuY),' ');
}
      
/*! clears entire current cursor line
*/

void TEScreen::clearEntireLine()
{
  clearImage(loc(0,cuY),loc(columns-1,cuY),' ');
}

// Rendition ------------------------------------------------------------------

/*! exchange foreground and background colors.

    This operation is typically used to display `inverted' in monochrome mode.
*/
void TEScreen::setInvert()
{ 
  UINT8 t = cu_fg; cu_fg = cu_bg; cu_bg = t;
}

/*! set `bold' rendition mode

    This selects the highlighted color foreground color group.
*/

void TEScreen::setBold()
{
  cu_bold = 1;
}

/*!
*/

void TEScreen::setDefaultRendition()
{
  setForeColorToDefault();
  setBackColorToDefault();
  cu_bold = 0;
  graphic = FALSE;
}

/*!
*/

void TEScreen::setForeColor(int fgcolor)
{
  cu_fg = 2+8*(cu_bold&1)+(fgcolor&7);
}

/*!
*/

void TEScreen::setBackColor(int bgcolor)
{
  cu_bg = 2+(bgcolor&7);
}

/*!
*/

void TEScreen::setBackColorToDefault()
{
  cu_bg = DEFAULT_BACK_COLOR;
}


/*!
*/

void TEScreen::setForeColorToDefault()
{
  cu_fg = DEFAULT_FORE_COLOR;
}


