/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [emulation.cpp]         Terminal Emulation Decoder                         */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU General Public Licence.       */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

/* \class

   FIXME: explain the role of this module as base class. 
          It's the mother of all emulations herself.

   This class is responsible to scan the escapes sequences of the terminal
   emulation and to map it to their corresponding semantic complements.
   Thus this module knows mainly about decoding escapes sequences and
   is a stateless device w.r.t. the semantics.

   It is also responsible to refresh the TEWidget by certain rules.

   \sa TEWidget \sa TEScreen
*/

/* FIXME
   - evtl. the bulk operations could be made more transparent.
*/

#include "TEmulation.h"
#include "TEWidget.h"
#include "TEScreen.h"

#include <stdio.h>
#include <unistd.h>
#include <qkeycode.h>

#include <assert.h>

#include "TEmulation.moc"

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                Emulation                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */

#define CNTL(c) ((c)-'@')

/*!
*/

Emulation::Emulation(TEWidget* gui)
{
  this->gui = gui;

  this->scr = new TEScreen(gui->Lines(),gui->Columns());

  bulk_nlcnt = 0; // reset bulk newline counter

  QObject::connect(&bulk_timer, SIGNAL(timeout()), this, SLOT(showBulk()) );
  QObject::connect(gui,SIGNAL(changedImageSizeSignal(int,int)),
                   this,SLOT(onImageSizeChange(int,int)));
  QObject::connect(gui,SIGNAL(keyPressedSignal(QKeyEvent*)), 
                   this,SLOT(onKeyPress(QKeyEvent*)));
}

/*!
*/

Emulation::~Emulation()
{
  delete scr;
}

// Interpreting Codes ---------------------------------------------------------

/*
   This section deals with decoding the incoming character stream.
   Decoding means here, that the stream is first seperated into `tokens'
   which are then mapped to a `meaning' provided as operations by the
   `Screen' class.
*/

/*!
*/

void Emulation::onRcvByte(int c)
// process application input to terminal
// this is a trivial scanner
{ 
  c &= 0xff;
  switch (c)
  {
    case '\b'      : scr->BackSpace();                 break;
    case '\t'      : scr->Tabulate();                  break;
    case '\n'      : scr->NewLine(); bulkNewline();    break;
    case '\r'      : scr->Return();                    break;
    case 0x07      : gui->Bell();                      break;
    default        : scr->ShowCharacter(c);            break;
  };
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                             Keyboard Handling                             */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/*!
*/

void Emulation::onKeyPress( QKeyEvent* ev )
{
  if (ev->ascii()>0)
  { unsigned char c[1]; 
    c[0] = ev->ascii(); 
    emit sndBlock((char*)c,1);
  }
}

// helpers  -------------------------------------------------------------------

/*!
*/

void Emulation::onRcvBlock(char *s, int len)
//FIXME: obsolete?
//FIXME: make sure we do not mess up the event loop here.
//       see the 'ripple' test for some strange results.
{
  bulkStart();
  for (int i = 0; i < len; i++)
    onRcvByte(s[i]);
  bulkEnd();
}

/* ------------------------------------------------------------------------- */

/*! A note on refreshing

   Allthough the modifications to the current screen image could immediately
   be propagated via `TEWidget' to the graphical surface, we've choosen
   another way here.

   The reason for doing so is twofold.
   First, experiments show that directly displaying the operation results
   in slowing down the overall performance of the emulations. Appearently
   displaing individual characters using X11 creates a lot of overhead.
   Second, by using the following refreshing method, the screen operations
   can be completely separated from the displaying. This greately simplifies
   the programmer's task of coding and maintaining the screen operations,
   since one has not to bother with differential modifications on the
   display affected the the operation of concern.

   We use a refreshing algorithm here that has been adoped from rxvt/kvt.

   By this, refreshing is driven by a timer, which is (re)started whenever
   a new bunch of data to be interpreted by the emulation arives at `onRcvBlock'.
   As soon as no more data arrive for `BULK_TIMEOUT' milliseconds, we trigger
   refresh. This rule suits both bulk display operation as done by curses as
   well as individual characters typed.
   (BULK_TIMEOUT < 1000 / max characters received from keyboard per second).

   Additionally, we trigger refreshing by newlines comming in to make visual
   snapshots of lists as produced by `cat', `ls' and likely programs, thereby
   producing the illusion of a permanent and immediate display operation.

   Following is not implemented!

   As a last rule, we also refresh after a period, when characters came in
   but none of the above rules have triggered. This is the catch all rule,
   which should only apply to programs which busily update the screen without
   ever pausing at least for BULK_TIMEOUT nor producing newlines - an unlikely
   but possible behavior most typical for programs that attempt to make a sort
   of animation.

   Since the timeout uses for this case should be pretty large to make sure that
   it does not get into the way of the BULK_TIMEOUT refreshing (1s, perhaps), we
   certainly negate the intended behavior of those programs. Because i've not seen
   such things, i don't care about this.
*/

#define BULK_TIMEOUT 10

/*!
*/

void Emulation::bulkNewline()
// called when \n comes in. Evtl. triggers showbulk
{
  bulk_nlcnt += 1;
  if (bulk_nlcnt > gui->Lines()) //FIXME: lines*somefactor
  {
    bulkStart();
    showBulk(); // resets bulk_nlcnt to 0, too.
    bulkEnd();
  }
}

/*!
*/

void Emulation::showBulk()
{
  bulk_nlcnt = 0;                       // reset bulk newline counter
  scr->swapCursor();                    // show cursor
  gui->setImage(scr->getImage(),
                scr->getLines(),
                scr->getColumns());     // actual refresh
  scr->swapCursor();                    // hide cursor
}

void Emulation::bulkStart()
{
  if (bulk_timer.isActive()) bulk_timer.stop();
}

void Emulation::bulkEnd()
{
  bulk_timer.start(BULK_TIMEOUT,TRUE);
}

// ---------------------------------------------------------------------------

/*!  triggered by image size change of the TEWidget `gui'.

    This event is simply propagated to the attached screens
    and to the related serial line.
*/

void Emulation::onImageSizeChange(int lines, int columns)
//FIXME: make sure this is properly overloaded!
{
  scr->resizeImage(lines,columns);        //FIXME: evtl. only `scr'
  emit ImageSizeChanged(lines,columns);   // propagate event to serial line
  gui->setImage(scr->getImage(),scr->getLines(),scr->getColumns());
}
