/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [te_demo.cpp]            Testbed for TE shell                              */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* copyright (c) 1997 by Lars Doelle <lars.doelle@on-line.de>                 */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is copyleft under the GNU General Public Licence.        */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

/* TODO
   - evtl. add pixmap to the custom color menu
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <kfm.h>
#include <kapp.h>
#include <ktopwidget.h>
#include <kcolordlg.h>

#include <qpainter.h>

#include "Shell.h"
#include "TEWidget.h"
#include "vt102emu.h"

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

#define localized(X) klocale->translate(X)

#define ProgId "konsole"

#define VersMajor 0
#define VersMinor 8
#define VersPatch 2

#define MIN(A,B) ((A)>(B)?(B):(A))

class TEDemo : public KTopLevelWidget // QWidget
{ Q_OBJECT

public:

  TEDemo(const char *name=0);
  ~TEDemo();

private slots:
  void configureRequest(TEWidget*,int,int);

  void color_menu_activated(int item);
  void ccolor_menu_activated(int item);
  void opt_menu_activated(int item);
  void size_menu_activated(int item);

private:
  void makeMenu();
  void setColorPixmaps();
  void loadOptions();
  void saveOptions();

private:
  Shell*         sh;
  TEWidget*      te;
  Emulation*     em;
//
  KMenuBar*   menubar;
  QPopupMenu* m_options;
  QPopupMenu* m_font;
  QPopupMenu* m_color;
  QPopupMenu* colors;
//
  QColor      fg; //FIXME: te->getDefaultForeColor
  QColor      bg; //FIXME: te->getDefaultBackColor
  QPixmap*    fg_pm;
  QPixmap*    bg_pm;
//
  KConfig*    config;
  bool        b_menuvis;
  bool        b_framevis;
  int         n_font;
  int         n_color;
};

char *fonts[] = {"6x13", "5x7", "6x10", "7x13", "9x15", "10x20", "vga"};

TEDemo::TEDemo(const char *name=0) : KTopLevelWidget(name)
{
  loadOptions();

  // create terminal emulation framework ////////////////////////////////////

  sh = new Shell();
  te = new TEWidget(this); te->changeFont(fonts[n_font]);
  em = new VT102Emulation(te);

  sh->setSize(te->Lines(),te->Columns()); // not absolutely nessesary
  QObject::connect( sh,SIGNAL(block_in(char*,int)), 
                    em,SLOT(onRcvBlock(char*,int)) );
  QObject::connect( em,SIGNAL(ImageSizeChanged(int,int)),
                    sh,SLOT(setSize(int,int)));
  QObject::connect( em,SIGNAL(sndBlock(char*,int)), 
                    sh,SLOT(send_bytes(char*,int)) );
  QObject::connect( sh,SIGNAL(done()), 
                    kapp,SLOT(quit()) );
  QObject::connect( te,SIGNAL(configureRequest(TEWidget*,int,int)),
                    this,SLOT(configureRequest(TEWidget*,int,int)) );
  sh->run();

  // create applications ////////////////////////////////////////////////////

  makeMenu();
  setMenu(menubar);
  
  char HeadLine[40]; 
  sprintf(HeadLine,"%s %d.%d.%d",ProgId,VersMajor,VersMinor,VersPatch);

  setCaption(HeadLine);
  setView(te,FALSE);
}

TEDemo::~TEDemo()
{
  delete em;
  delete sh;
//FIXME: delete at?
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEDemo::configureRequest(TEWidget* te, int x, int y)
{
//printf("TEDemo::configureRequest(_,%d,%d)\n",x,y);
  m_options->move(te->mapToGlobal(QPoint(x,y)));
  m_options->show();
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEDemo::setColorPixmaps()
{
  QPainter pt;
  pt.begin(fg_pm);
  pt.setBackgroundColor( fg );
  pt.eraseRect(fg_pm->rect());
  pt.end();
  colors->changeItem(*fg_pm,localized("&Foreground Color"),1);
  pt.begin(bg_pm);
  pt.setBackgroundColor( bg );
  pt.eraseRect(bg_pm->rect());
  pt.end();
  colors->changeItem(*bg_pm,localized("&Background Color"),2);
}

void TEDemo::makeMenu()
{
  // options (taken from kvt) //////////////////////////////////////

  QPopupMenu* m_file = new QPopupMenu;
  m_file->insertItem( localized("&Exit"), kapp, SLOT(quit()));

  m_font = new QPopupMenu;
  m_font->setCheckable(TRUE);
  m_font->insertItem( localized("&Normal"), 0);
  m_font->insertSeparator();
  m_font->insertItem( localized("&Tiny"),   1);
  m_font->insertItem( localized("&Small"),  2);
  m_font->insertItem( localized("&Medium"), 3);
  m_font->insertItem( localized("&Large"),  4);
  m_font->insertItem( localized("&Huge"),   5);
  m_font->insertSeparator();
  m_font->insertItem( localized("&VGA"),    6);
  connect(m_font, SIGNAL(activated(int)), SLOT(size_menu_activated(int)));

//m_dimen = new QPopupMenu;
//CHECK_PTR( m_dimen );
//for (int i=0; kvt_dimens[i].text; i++) {
//  m_dimen->insertItem(kvt_dimens[i].text);
//}
//connect(m_dimen, SIGNAL(activated(int)), SLOT(dimen_menu_activated(int)));

  colors = new QPopupMenu;
  colors->insertItem("",1);
  colors->insertItem("",2);
  connect(colors, SIGNAL(activated(int)), SLOT(ccolor_menu_activated(int)));
  
  m_color = new QPopupMenu;
  m_color->setCheckable(TRUE);
  m_color->insertItem( localized("&black/white"),       0);
  m_color->insertItem( localized("&white/black"),       1);
  m_color->insertItem( localized("&green/black"),       2);
  m_color->insertItem( localized("black/light&yellow"), 3);
  m_color->insertSeparator();
  m_color->insertItem( localized("&System colors"),     4);
  m_color->insertItem(localized("&Custom Colors"), colors, 5);
  connect(m_color, SIGNAL(activated(int)), SLOT(color_menu_activated(int)));

  fg_pm = new QPixmap(10,10);
  bg_pm = new QPixmap(10,10);
    
  m_options = new QPopupMenu;
  m_options->setCheckable(TRUE);
  m_options->insertItem( localized("&Menubar"), 1 );
  m_options->setItemChecked(1,b_menuvis);
  m_options->insertItem( localized("&Frame"), 2 );
  m_options->setItemChecked(2,b_framevis);
  m_options->insertSeparator();
//m_options->insertItem( localized("Scroll&bar"), m_scrollbar);
  m_options->insertItem( localized("&Font"), m_font);
  m_options->insertItem( localized("&Color"), m_color);
//m_options->insertItem( localized("&Size"), m_dimen);
//m_options->insertItem( localized("Terminal...") );
  m_options->insertSeparator();
  m_options->insertItem( localized("&Save options"), 8);
//m_options->setItemEnabled(8,FALSE);
  connect(m_options, SIGNAL(activated(int)), SLOT(opt_menu_activated(int)));

  QPopupMenu* m_help = new QPopupMenu;
  m_help->insertItem( localized("&About ..."));

  m_options->installEventFilter( this );
//connect(m_options, SIGNAL(activated(int)), SLOT(options_menu_activated(int)));

  menubar = new KMenuBar( this );
  menubar->insertItem("&File" , m_file);
  menubar->insertItem("&Options", m_options);
  menubar->insertItem("&Help", m_help);

  // set options //////////////////////////////////////////////////////////////

  if (!b_menuvis) menubar->hide();
  te->setFrameStyle( b_framevis
                     ? QFrame::WinPanel | QFrame::Sunken 
                     : QFrame::NoFrame );
  m_font->setItemChecked(n_font,TRUE); //Note: font set in ::TEDemo
  color_menu_activated(n_color);
  
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEDemo::loadOptions()
{
  config = KApplication::getKApplication()->getConfig();
  config->setGroup("options");
  b_menuvis  = config->readBoolEntry("menubar visible",TRUE);
  b_framevis = config->readBoolEntry("has frame",TRUE);
  n_font     = MIN(config->readUnsignedNumEntry("font",0),6);
  n_color    = MIN(config->readUnsignedNumEntry("color",0),5);
  fg         = config->readColorEntry("foreground",&kapp->textColor);
  bg         = config->readColorEntry("background",&kapp->backgroundColor);
}

void TEDemo::saveOptions()
{
  config->setGroup("options");
  config->writeEntry("menubar visible",b_menuvis);
  config->writeEntry("has frame",b_framevis);
  config->writeEntry("font",n_font);
  config->writeEntry("color",n_color);
  config->writeEntry("foreground",fg);
  config->writeEntry("background",bg);
  config->sync();
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEDemo::size_menu_activated(int item)
{
  m_font->setItemChecked(n_font,FALSE);
  n_font = item;
  m_font->setItemChecked(n_font,TRUE);
  te->changeFont(fonts[n_font]);
}

void TEDemo::opt_menu_activated(int item)
{
  switch( item )
  {
    case 1: b_menuvis = !b_menuvis;
            m_options->setItemChecked(1,b_menuvis);
            if (b_menuvis) menubar->show(); else menubar->hide();
            updateRects();
            break;
    case 2: b_framevis = !b_framevis;
            m_options->setItemChecked(2,b_framevis);
            te->setFrameStyle( b_framevis
                               ? QFrame::WinPanel | QFrame::Sunken 
                               : QFrame::NoFrame );
            break;
    case 8: saveOptions();
            break;
  }
}

// --| color selection |-------------------------------------------------------

void TEDemo::color_menu_activated(int item)
{
  switch (item)
  {
    case 0: fg = QColor("black"  ); bg = QColor("white"  );     break;
    case 1: fg = QColor("#b2b2b2"); bg = QColor("black"  );     break;
    case 2: fg = QColor("#18b218"); bg = QColor("black"  );     break;
    case 3: fg = QColor("black"  ); bg = QColor("#ffffdd");     break;
    case 4: fg = kapp->textColor;   bg = kapp->backgroundColor; break;
  }
  te->setDefaultForeColor(fg);
  te->setDefaultBackColor(bg);
  m_color->setItemChecked(n_color,FALSE);
  n_color = item;
  m_color->setItemChecked(n_color,TRUE);
  setColorPixmaps();
}

void TEDemo::ccolor_menu_activated(int item)
{ QColor color;
  switch (item)
  { 
    case 1: color = te->getDefaultForeColor();
            if(KColorDialog::getColor(color) != QDialog::Accepted) return;
            fg = color;
            te->setDefaultForeColor(fg);
            m_color->setItemChecked(n_color,FALSE);
            n_color = 5;
            m_color->setItemChecked(n_color,TRUE);
            setColorPixmaps();
            break;
    case 2: color = te->getDefaultBackColor();
            if(KColorDialog::getColor(color) != QDialog::Accepted) return;
            bg = color;
            te->setDefaultBackColor(bg);
            m_color->setItemChecked(n_color,FALSE);
            n_color = 5;
            m_color->setItemChecked(n_color,TRUE);
            setColorPixmaps();
            break;
  }
}

/* --| main |---------------------------------------------------------------- */

int main( int argc, char ** argv )
{
  KApplication a(argc, argv, ProgId);
  putenv("TERM=xterm"); //FIXME: for mc
  putenv("COLORTERM="); //FIXME: for mc
  QString p = QString("TERMINFO=") + getenv("HOME") + "/.terminfo"; //FIXME: loc
  putenv(p);
  TEDemo m;
  a.setMainWidget( &m );
  m.show();
  return a.exec();
}

#include "main.moc"
