/****************************************************************************
*               EIS2PGM: A KITSAT EIS File Conversion Utility               *
*               Copyright John A. Magliacane, KD2BD 1998-2001               *
*                        Last update: 09-Jul-2001                           *
*****************************************************************************
*                                                                           *
* This program is free software; you can redistribute it and/or modify it   *
* under the terms of the GNU General Public License as published by the     *
* Free Software Foundation; either version 2 of the License or any later    *
* version.                                                                  *
*                                                                           *
* This program is distributed in the hope that it will useful, but WITHOUT  *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
* for more details.                                                         *
*                                                                           *
*****************************************************************************
*               To compile: gcc -O6 -s  eis2pgm.c -o eis2pgm                *
*****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

unsigned short height=576, width=611;
unsigned char eis_data[360000];
unsigned long size;

int read_eis (char *eis_name)
{
	unsigned short x, y, different, non_uniform, dupes=0, packet_size=109;
	unsigned char old_buffer[110], new_buffer[110];
	FILE *infile;

	infile=fopen(eis_name,"rb");

	if (infile!=NULL)
	{
		/* Read EIS Data Into "eis_data" Buffer
		   Without Duplicate DASH Packets */

		printf("\nReading %c%s%c... ",34,eis_name,34);
		fflush(stdout);
		size=0L;

		for (x=0; (x<packet_size)&&(feof(infile)==0); x++)
			old_buffer[x]=fgetc(infile);
		do
		{
			for (x=0; (x<packet_size)&&(feof(infile)==0); x++)
				new_buffer[x]=fgetc(infile);

			different=0;
			non_uniform=0;
			x--;

			for (y=0; y<x; y++)
			{
				different+=old_buffer[y]^new_buffer[y];
				non_uniform+=old_buffer[x]^old_buffer[y];
			}

			if (different || non_uniform==0)
			{
				for (x=0; x<packet_size; x++)
				{
					eis_data[size]=old_buffer[x];
					size++;
				}
			}
			else
				if (non_uniform)
					dupes++;

			for (x=0; x<packet_size; x++)
			{
				eis_data[size]=new_buffer[x];
				size++;
			}
	
			for (x=0; (x<packet_size)&&(feof(infile)==0); x++)
				old_buffer[x]=fgetc(infile);

		} while (feof(infile)==0);

		fclose(infile);

		printf("Done!\n");

		if (eis_data[255]==0x81)
		{
			height=eis_data[253]+(eis_data[254]<<8);
			width=eis_data[251]+(eis_data[252]<<8);
		}

		if (dupes)
		{
			if (dupes==1)
				printf("\nOne Duplicated DASH Packet Was Detected And Discarded.\n");
			else
				printf("\nA Total of %u Duplicated DASH Packets Were Detected And Discarded.\n",dupes);
		}

		return (0);
	}
	
	else
	{
		fprintf(stderr,"*** %s: File Not Found!\n",eis_name);
		return (-1);
	}
}

void display_headers(char *eis_name)
{
	time_t t;
	unsigned long lbyte;
	unsigned char byte, low_nibble, high_nibble, txt_name[100];
	FILE *outfile;

	strcpy (txt_name, eis_name);

	for (byte=0; txt_name[byte]!='.' && txt_name[byte]!=0; byte++);

	txt_name[byte]='.';
	txt_name[byte+1]='t';
	txt_name[byte+2]='x';
	txt_name[byte+3]='t';
	txt_name[byte+4]=0;

	outfile=fopen(txt_name,"w");

	printf("\n\tEIS Image Header Items\n-------------------------------------------\n");
	fprintf(outfile,"\tEIS Image Header Items\n-------------------------------------------\n");

	t=(eis_data[0])|(eis_data[1]<<8)|(eis_data[2]<<16)|(eis_data[3]<<24);
	printf("Image Start Time : %s",asctime(gmtime(&t)));
	fprintf(outfile,"Image Start Time : %s",asctime(gmtime(&t)));

	t=(eis_data[4])|(eis_data[5]<<8)|(eis_data[6]<<16)|(eis_data[7]<<24);
	printf("Image Taken At   : %s",asctime(gmtime(&t)));
	fprintf(outfile,"Image Taken At   : %s",asctime(gmtime(&t)));

	lbyte=(eis_data[8])|(eis_data[9]<<8)|(eis_data[10]<<16)|(eis_data[11]<<24);
	printf("Image Number     : 0x%lx\n",lbyte);
	fprintf(outfile,"Image Number     : 0x%lx\n",lbyte);

	lbyte=(eis_data[12])|(eis_data[13]<<8)|(eis_data[14]<<16)|(eis_data[15]<<24);
	printf("Retries          : 0x%02lx\n",lbyte);
	fprintf(outfile,"Retries          : 0x%02lx\n",lbyte);

	lbyte=(eis_data[16])|(eis_data[17]<<8)|(eis_data[18]<<16)|(eis_data[19]<<24);
	printf("Set Image Number : 0x%02lx\n",lbyte);
	fprintf(outfile,"Set Image Number : 0x%02lx\n",lbyte);

	byte=eis_data[20];
	printf("CCD Integration  : 0x%02x\n",byte);
	fprintf(outfile,"CCD Integration  : 0x%02x\n",byte);

	byte=eis_data[21];
	low_nibble=byte&15;
	high_nibble=(byte&240)>>4;
	printf("CCD Camera Gain  : %u:%u\n",low_nibble,high_nibble);
	fprintf(outfile,"CCD Camera Gain  : %u:%u\n",low_nibble,high_nibble);

	byte=eis_data[255];
	printf("EIS Version      : 0x%0x\n",byte);
	fprintf(outfile,"EIS Version      : 0x%0x\n",byte);

	if (eis_data[255]==0x81)
	{
		printf("Image Height     : %u\n",height);
		fprintf(outfile,"Image Height     : %u\n",height);
		printf("Image Width      : %u\n",width);
		fprintf(outfile,"Image Width      : %u\n",width);
	}

	printf("File Size        : %lu\n",size);

	fclose (outfile);
}

void write_pgm (char *eis_name)
{
	unsigned short x, y;
	unsigned char pgm_name[100], even_pixel, odd_pixel;
	unsigned long even_offset, odd_offset;
	FILE *outfile;

	/* Generate output filename ("eis_name".pgm) */

	strcpy (pgm_name, eis_name);

	for (x=0; pgm_name[x]!='.' && pgm_name[x]!=0; x++);

	pgm_name[x]='.';
	pgm_name[x+1]='p';
	pgm_name[x+2]='g';
	pgm_name[x+3]='m';
	pgm_name[x+4]=0;

	printf("\nWriting %c%s%c... ",34,pgm_name,34);

	outfile=fopen(pgm_name,"wb");
	
	/* Build PGM Header */

	fprintf(outfile,"P5\n%u %u\n255\n",width, height);

	/* Write Interlaced Raster Data To PGM Output File */

	even_offset=280L;
	odd_offset=even_offset+355+((width*height)/2);

	for (x=0; x<(height/2); x++)
	{
		/* Write An Even Line Of Pixel Data */

		for (y=0; y<width; y++)
		{
			/* If a pixel from an even line is missing,
			   use a pixel from an adjacent odd line */

			even_pixel=eis_data[y+(x*width)+even_offset];
			odd_pixel=eis_data[y+(x*width)+odd_offset];

			if (even_pixel==0 && odd_pixel!=0)
				fprintf(outfile,"%c",255^odd_pixel);

			if (even_pixel!=0 && odd_pixel==0)
				fprintf(outfile,"%c",255^even_pixel);

			if (even_pixel==0 && odd_pixel==0)
				fprintf(outfile,"%c",255);

			if (even_pixel!=0 && odd_pixel!=0)
				fprintf(outfile,"%c",255^even_pixel);
		}

		/* Write An Odd Line Of Pixel Data */

		for (y=0; y<width; y++)
		{
			/* If a pixel from an odd line is missing,
			   use a pixel from an adjacent even line */ 

			even_pixel=eis_data[y+(x*width)+even_offset];
			odd_pixel=eis_data[y+(x*width)+odd_offset];

			if (even_pixel==0 && odd_pixel!=0)
				fprintf(outfile,"%c",255^odd_pixel);

			if (even_pixel!=0 && odd_pixel==0)
				fprintf(outfile,"%c",255^even_pixel);

			if (even_pixel==0 && odd_pixel==0)
				fprintf(outfile,"%c",255);

			if (even_pixel!=0 && odd_pixel!=0)
				fprintf(outfile,"%c",255^odd_pixel);
		}
	}

	fclose(outfile);

	printf("Done!\n\n");
}

int main (int argc, char *argv[])
{
	unsigned char x;

	for (x=1; x<argc; x++)
	{
		if (read_eis(argv[x])==0)
		{
			display_headers(argv[x]);	
			write_pgm(argv[x]);
		}
	}
	
	exit(0);
}	
