/***************************************************************************\
*                  PBDIR: A Pacsat Directory Display Program                *
*         Copyright John A. Magliacane, KD2BD 1996, 2001, 2003, 2004        *
*                          Last update: 20-Mar-2004                         *
*****************************************************************************
*                                                                           *
* This program is free software; you can redistribute it and/or modify it   *
* under the terms of the GNU General Public License as published by the     *
* Free Software Foundation; either version 2 of the License or any later    *
* version.                                                                  *
*                                                                           *
* This program is distributed in the hope that it will useful, but WITHOUT  *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
* for more details.                                                         *
*                                                                           *
*****************************************************************************
*            To compile: gcc -O6 -s -lncurses  pbdir.c  -o pbdir            *
\****************************************************************************/
 
#include <curses.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#define MAXHEADERS 700

struct	{  unsigned long file_number;
	   char file_name[256];
	   unsigned long file_size;
	   unsigned long create_time;
	   unsigned long upload_time;	
	   char source[256];
	   char destination[256];
	   char title[256];
	   char keywords[256];
	   unsigned char status;	
	}  header[MAXHEADERS];
	
	
char result[256], filenum[10], *version={"pbdir v1.0"};
unsigned id, indx=0;
unsigned long files[1000];
unsigned char id_lsb, id_msb, length, data[256];
FILE *infile, *outfile;

unsigned int getitem()
{
	int x;

	id_lsb=fgetc(infile);
	id_msb=fgetc(infile);
	length=fgetc(infile);

	for (x=1; x<=(int)length; x++)
		data[x-1]=fgetc(infile);

	return (((int)id_lsb)|((int)id_msb<<8));
}

unsigned int data2int()
{
	unsigned int byte0, byte1;
	
	if (length==2)
	{
		byte0=(unsigned int)data[0];
		byte1=(unsigned int)data[1];

		return ((byte0)|(byte1<<8));
	}
	else
		return 0;
}

unsigned long data2long()
{
	unsigned long byte0, byte1, byte2, byte3;

	if (length==4)
	{
		byte0=(unsigned long)data[0];
		byte1=(unsigned long)data[1];
		byte2=(unsigned long)data[2];
		byte3=(unsigned long)data[3];

		return (byte0|(byte1<<8)|(byte2<<16)|(byte3<<24));
	}
	else
		return 0L;
}

char *data2str()
{
	int x;

	for (x=1; x<=(int)length; x++)
		result[x-1]=(char)data[x-1];

	result[length]=0;

	return result;
}

char *data2time(x)
unsigned long x;
{
	time_t t;

	if (x!=0)
	{
		t=(time_t)x;
		strcpy(result,asctime(gmtime(&t)));
		result[strlen(result)-1]=0;
	
		if (result[8]==' ')
			result[8]='0';
	}
	else
		strcpy(result,"not initialized");

	return result;
}

char *leftstr(str,n)
char *str;
int n;
{
	/* Returns a pointer to the left-most 'n' characters of str */

	strncpy(result,str,n);
	result[n]=0;
	return result;
}

void read_headers()
{
	/* Initialize structure members */

	header[indx].file_number=0L;
	header[indx].file_name[0]=0;
	header[indx].file_size=0L;
	header[indx].create_time=0L;
	header[indx].upload_time=0L;
	header[indx].source[0]=0;
	header[indx].destination[0]=0;
	header[indx].title[0]=0;
	header[indx].keywords[0]=0;
	header[indx].status=0;

	id=getitem();
	
	/* Continue this loop until the end of the
	   Pacsat File Header is reached (id==0) */

	while (id!=0)
	{
		switch(id)
		{
			case 0x01:
			header[indx].file_number=data2long();
			break;

			case 0x02:
			strncpy(header[indx].file_name,data2str(),255);
			break;

			case 0x04:
			header[indx].file_size=data2long();
			break;

			case 0x05:
			header[indx].create_time=data2long();
			break;

			/* Extended Header Information */

			case 0x10:
			strncpy(header[indx].source,data2str(),255);
			break;

			case 0x12:
			header[indx].upload_time=data2long();
			break;

			case 0x14:
			strncpy(header[indx].destination,data2str(),255);
			break;

			/* Optional Header Fields */

			case 0x22:
			strncpy(header[indx].title,data2str(),255);
			break;

			case 0x23:
			strncpy(header[indx].keywords,data2str(),255);
			break;
		}
		id=getitem();
	}
}

void sort_headers()
{
	int a, b;

	for (a=indx-1; a>=0; a--)
		for (b=0; b<a; b++)
		{
			if (header[b].file_number > header[b+1].file_number)
			{
				header[MAXHEADERS-1]=header[b];
				header[b]=header[b+1];
				header[b+1]=header[MAXHEADERS-1];
			}
		}
}

void display_headers(start)
unsigned start;
{
	char ts[30], null[1];
	unsigned x, y, lines_printed=0;

	move(1,0);

	bkgdset(COLOR_PAIR(1));
	refresh();

	for (x=start; x<(23+start) && x<indx; x++)
	{
		if (header[x].status==1)
			attrset(COLOR_PAIR(5)|A_BOLD);
		else
			attrset(COLOR_PAIR(1)|A_NORMAL);

		refresh();

		printw("%06lx\t",header[x].file_number);
		sprintf(ts,"%s",data2time(header[x].upload_time));
		printw("%c%c%c%c%c%c%c ",ts[8],ts[9],ts[4],ts[5],ts[6],ts[22],ts[23]);
		printw("%c%c:%c%c ",ts[11],ts[12],ts[14],ts[15]);
		printw("%7lu  ",header[x].file_size);
		printw("%-7s  ",leftstr(header[x].destination,7));
		printw("%-7s  ",leftstr(header[x].source,7));

		if (header[x].title[0]!=0)
			printw("%-30s\n",leftstr(header[x].title,30));
		else
			printw("%-30s\n",leftstr(header[x].file_name,30));

		lines_printed++;
	}

	null[0]=0;

	for (y=lines_printed; y<23; y++)
		printw("%-80s",null);

	attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);
	mvprintw(0,72,"%3d/%3d",start+lines_printed,indx);

	refresh();
}

void write_headers()
{
	char ts[30];
	unsigned x;

	printf(" File     Date/Time      Size  To       From     Title\n");
	printf("--------------------------------------------------------------------------------");

	for (x=0; x<indx; x++)
	{
		printf("\n%06lx\t",header[x].file_number);
		sprintf(ts,"%s",data2time(header[x].upload_time));
		printf("%c%c%c%c%c%c%c ",ts[8],ts[9],ts[4],ts[5],ts[6],ts[22],ts[23]);
		printf("%c%c:%c%c ",ts[11],ts[12],ts[14],ts[15]);
		printf("%7lu  ",header[x].file_size);
		printf("%-7s  ",leftstr(header[x].destination,7));
		printf("%-7s  ",leftstr(header[x].source,7));

		if (header[x].title[0]!=0)
			printf("%s",leftstr(header[x].title,30));
		else
			printf("%s",leftstr(header[x].file_name,30));
	}

	printf("\n");
}

int read_req()
{
	/* This function reads the download.req file into memory. */

	FILE *fd;
	int x, filecount=0;

	fd=fopen("download.req","rb");

	if (fd==NULL)
		return -1;

	for (x=0; x<1000; x++)
	{
		files[x]=fgetc(fd)|(fgetc(fd)<<8)|(fgetc(fd)<<16)|(fgetc(fd)<<24);

		if (files[x]!=0xffffffff)
			filecount++;
	}

	fclose(fd);
	return filecount;
}

void wash_memory()
{
	/* This function cleans the files[] array of stray 0xffffffffs */

	unsigned x, y, cleanfile[1000];

	for (x=0, y=0; x<1000; x++)
	{
		if (files[x]!=0xffffffff)
		{
			cleanfile[y]=files[x];
			y++;
		}
	}

	for (x=y; x<1000; x++)
		cleanfile[x]=0xffffffff;

	for (x=0; x<1000; x++)
		files[x]=cleanfile[x];
}

int remove_file(filenum)
unsigned filenum;
{
	/* This function removes a file number from the download.req list */

	int x, success=0;

	for (x=0; x<1000 && success==0; x++)
	{
		if (filenum==files[x])
		{
			files[x]=0xffffffff;
			success=1;
		}
	}

	return success;
}

int addfile(filenum)
unsigned filenum;
{
	/* This function adds a message number to download.req
	   provided it is within the range of valid file numbers */

	int x, success=0;

	if (filenum>=header[0].file_number && filenum<=header[indx-1].file_number)
		for (x=0; x<1000 && success==0; x++)
		{
			if (files[x]==0xffffffff)
			{
				files[x]=filenum;
				success=1;
			}
		}

	return success;
}

void savefile()
{
	unsigned char nib1, nib2, nib3, nib4;
	unsigned filecount=0, x;
	FILE *outfile;

	outfile=fopen("download.req","wb");

	for (x=0; x<1000; x++)
	{
		if (files[x]!=0xffffffff)
		{
			nib1=0x000000ff & files[x];
			nib2=(0x0000ff00 & files[x])>>8;
			nib3=(0x00ff0000 & files[x])>>16;
			nib4=(0xff000000 & files[x])>>24;
			fprintf(outfile,"%c%c%c%c",nib1,nib2,nib3,nib4);
			filecount++;
		}

	}

	for (x=filecount; x<1000; x++)
		fprintf(outfile,"%c%c%c%c",0xff,0xff,0xff,0xff);

	fclose(outfile);
}

void display_queue(filecount)
int filecount;
{
	attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);

	if (filecount!=-1 && filecount!=1)
		mvprintw(24,1,"%d files queued     ",filecount);
	else if (filecount!=-1)
		mvprintw(24,1,"%d file queued      ",filecount);

	refresh();
}

void display_holes(holes)
int holes;
{
	attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);

	if (holes==0)
		mvprintw(24,26,"Directory is Up-To-Date!");

	if (holes==1)
		mvprintw(24,28,"Directory has 1 hole");

	if (holes>1)
		mvprintw(24,27,"Directory has %u holes",holes);

	refresh();
}

unsigned readfiles()
{
	char string[81], pfh_file[80];
	unsigned count, holes;
	FILE *fd;

	fd=fopen("pfhdir.hol","rb");

	if (fd==NULL)
	{
		endwin();
		fprintf(stderr, "*** ERROR: \"pfhdir.hol\" file not found!\n");
		exit(-1);
	}

	fgets(string,80,fd);
	fgets(string,80,fd);
	sscanf(string,"%u",&count);
	fgets(string,80,fd);
	sscanf(string,"%u",&holes);
	holes--;
	fclose(fd);

	if (count>=4)
		count-=3;
	else
		count=1;

	sprintf(pfh_file,"pb__%04u.pfh",count);
	infile=fopen(pfh_file,"rb");

	if (infile==NULL)
	{
		endwin();
		fprintf(stderr,"*** ERROR: directory files not found!\n");
		exit(-1);
	}

	indx=0;

	while (infile!=NULL)
	{
		while (feof(infile)==0)
		{
			if (fgetc(infile)==0xaa && fgetc(infile)==0x55)
			{
				read_headers();
				indx++;
			}
		}
	
		fclose(infile);
		count++;
		sprintf(pfh_file,"pb__%04u.pfh",count);
		infile=fopen(pfh_file,"rb");
	}

	return holes;
}

int main(argc,argv)
int argc;
char *argv[];
{
	int filecount, xx, yy, oldxx=0, ans;
	char refresh=1;
	unsigned holes;
	unsigned long hexnumber;

	if ((argc==2) && (strcmp(argv[1],"-o")==0))
	{
		holes=readfiles();
		sort_headers();
		write_headers();
		exit(0);
	}

	initscr();
	start_color();
	noecho();
	scrollok(stdscr,FALSE);
	curs_set(0);
	keypad(stdscr,TRUE);

	init_pair(1,COLOR_WHITE,COLOR_BLUE);
	init_pair(2,COLOR_RED,COLOR_BLUE);
	init_pair(3,COLOR_CYAN,COLOR_BLUE);
	init_pair(4,COLOR_GREEN,COLOR_BLUE);
	init_pair(5,COLOR_YELLOW,COLOR_BLUE);
	init_pair(6,COLOR_RED,COLOR_WHITE);

	while (refresh)
	{
		refresh=0;
		holes=readfiles();
		sort_headers();
		filecount=read_req();

		if (filecount!=-1)
		{
			wash_memory();

			for (yy=0; yy<indx; yy++)
				for (xx=0; xx<filecount; xx++)
					if (files[xx]==header[yy].file_number)
						header[yy].status=1;
		}

		bkgdset(COLOR_PAIR(5));
		clear();
		attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);
		printw(" File     Date/Time      Size  To       From     Title                          ");
		mvprintw(24,0,"%79s ",version);

		display_queue(filecount);
		display_holes(holes);

		xx=oldxx;

		display_headers(xx,23);

		ans=getch();

		do
		{
			switch (ans)
			{
				case KEY_PPAGE:
					if ((xx-23)>=0)
						xx-=23;
					else
						xx=0;

					display_headers(xx);
					break;

				case KEY_NPAGE:
					if ((xx+46)<indx)
						xx+=23;
					else
						xx=indx-23;

					display_headers(xx);
					break;

				case KEY_DOWN:
					if ((xx+23)<indx)
					{
						xx++;
						display_headers(xx);
					}
					break;

				case KEY_UP:
					if ((xx)>=1)
					{
						xx--;
						display_headers(xx);
					}
					break;

				case KEY_HOME:
					xx=0;
					display_headers(xx);
					break;

				case KEY_END:
					if ((indx-23)>0 && (xx+23)<indx)
					{
						xx=indx-23;
						display_headers(xx);
					}
					break;

				case KEY_IC:
					keypad(stdscr,FALSE);
					curs_set(1);
					echo();
					attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);
					mvprintw(24,0,"                   ");
					mvprintw(24,0," Add file: ");
					refresh();
					wgetnstr(stdscr,filenum,6);
					sscanf(filenum,"%lx",&hexnumber);

					if (addfile(hexnumber))
					{
						wash_memory();
						savefile();
					}

					for (yy=0; yy<indx; yy++)
						if (header[yy].file_number==hexnumber)
							header[yy].status=1;

					filecount=read_req();
					display_queue(filecount);

					keypad(stdscr,TRUE);
					curs_set(0);
					noecho();
					display_headers(xx);
					refresh();
					break;

				case KEY_DC:
					keypad(stdscr,FALSE);
					curs_set(1);
					echo();
					attrset(COLOR_PAIR(6)|A_BOLD|A_REVERSE);

					mvprintw(24,0,"                   ");
					mvprintw(24,0," Remove file: ");
					refresh();
					wgetnstr(stdscr,filenum,6);
					sscanf(filenum,"%lx",&hexnumber);

					if (remove_file(hexnumber))
					{
						wash_memory();
						savefile();
					}

					for (yy=0; yy<indx; yy++)
						if (header[yy].file_number==hexnumber)
							header[yy].status=0;

					filecount=read_req();
					display_queue(filecount);

					keypad(stdscr,TRUE);
					curs_set(0);
					noecho();
					display_headers(xx);
					refresh();
					break;

				case KEY_F(1):
					refresh=1;
					oldxx=xx;
					break;
			}

			ans=getch();

		} while (ans!=KEY_F(10) && ans!='q' && ans!='Q' && refresh==0);
	}

	curs_set(1);
	endwin();

	exit(0);
}
