/*
 *  Appindex browser - search
 *  Copyright (C) 1999 Martin Hinner <mhi@penguin.cz>
 *  $Id: search.c,v 0.6 2000/02/26 15:46:57 mhi Exp root $
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif

#include "appindex.h"

#define STRING(x) ((x)?(x):"")

int search_no_mem;

#define S_ALL        0
#define S_AUTHOR     1
#define S_HOMEPAGE   2
#define S_DOWNLOAD   3
#define S_URL        4
#define S_RPM        5
#define S_DEB        6
#define S_DEPENDANCY 7
#define S_DESC       8
#define S_CATEGORY   9
#define S_NAME       10

#define OP_OR	     0
#define OP_AND       1

struct search_exp {
	int what;
	char *str;
	int operator;
        struct search_exp *next;
};

struct app *
dup_app (struct app *app)
{
  struct app *a;

  a = (struct app *) malloc (sizeof (struct app));
  if (!a)
    return 0;

  a->name = app->name;
  a->stable = app->stable;
  a->devel = app->devel;
  a->license = app->license;
  a->homepage = app->homepage;
  a->download = app->download;
  a->changelog = app->changelog;
  a->deb = app->deb;
  a->rpm = app->rpm;
  a->link = app->link;
  a->oneliner = app->oneliner;
  a->description = app->description;
  a->dependancy = app->dependancy;
  a->author = app->author;
  a->coauthor = app->coauthor;
  a->day = app->day;
  a->month = app->month;
  a->year = app->year;
  a->cat = app->cat;
  a->priority = 0;
  a->next = 0;
  a->prev = 0;
  return a;
}

#ifndef __HAVE_STRCASESTR
char *
strcasestr (const char *s1, const char *s2)
{
  int l1, l2;

  l2 = strlen (s2);
  if (!l2)
    return (char *) s1;
  l1 = strlen (s1);
  while (l1 >= l2)
    {
      l1--;
      if (!strncasecmp (s1, s2, l2))
	return (char *) s1;
      s1++;
    }
  return NULL;
}
#endif

void
search_app (struct app *app, const struct search_exp *sexp, struct category *scat)
{
  struct search_exp *exp;
  int match;
  int prio;
  struct app *a;

  exp = sexp;
  prio = 1;
  while (exp) {
    //printf("%u[%s]%c,",exp->what,exp->str,exp->operator==OP_OR?'|':'&');
    match = 0;
    switch (exp->what) {
       case S_ALL:
        if (strcasestr (STRING (app->description), exp->str))
          match = 1;
        if (strcasestr (STRING (app->name), exp->str))
          match = 2;
        if (!strcasecmp (STRING (app->name), exp->str))
          match = 3;
        break;
       case S_AUTHOR:
        if (strcasestr (STRING (app->author), exp->str))
          match = 1;
        if (strcasestr (STRING (app->coauthor), exp->str))
          match = 1;
        break;
       case S_HOMEPAGE:
        if (strcasestr (STRING (app->homepage), exp->str))
          match = 1;
        break;
       case S_DOWNLOAD:
        if (strcasestr (STRING (app->download), exp->str))
          match = 1;
        break;
       case S_URL:
        if (strcasestr (STRING (app->homepage), exp->str))
          match = 1;
        if (strcasestr (STRING (app->download), exp->str))
          match = 1;
        break;
       case S_RPM:
        if (strcasestr (STRING (app->rpm), exp->str))
          match = 1;
        break;
       case S_DEB:
        if (strcasestr (STRING (app->deb), exp->str))
          match = 1;
        break;
       case S_DEPENDANCY:
        if (strcasestr (STRING (app->dependancy), exp->str))
          match = 1;
        break;
       case S_DESC:
        if (strcasestr (STRING (app->description), exp->str))
          match = 1;
        break;
       case S_CATEGORY:
        if (strcasestr (STRING (app->cat->name), exp->str))
          match = 1;
        break;
       case S_NAME:
        if (strcasestr (STRING (app->name), exp->str))
          match = 2;
        if (!strcasecmp (STRING (app->name), exp->str))
          match = 3;
        break;
    }
    if (match == 0)
      prio = 0;
    if (prio && prio < match)
      prio = match;
    if (exp->operator == OP_OR)
     if (prio) {
      a = dup_app (app);
      if (a)
	{
	  a->priority = prio;
	  add_app2cat (scat, a);
	  a->cat = app->cat;
	}
      else
	search_no_mem = 1;
      //printf("!!!!!!!!!!!\n");
      return ;
    }else
     prio = 1;
    exp = exp->next;
  }
 //printf("\r\n");
 //fflush(stdout);
}

void
search_apps (struct app *apps, struct search_exp *sexp, struct category *scat)
{
  struct app *a;

  a = apps;
  while (a)
    {
      search_app (a, sexp, scat);
      a = a->next;
    }
}

void
search_categories (struct search_exp *sexp, struct category *scat)
{
  struct category *c;

  c = categories;
  while (c)
    {
      search_apps (c->apps, sexp, scat);
      c = c->next;
    }
}

void free_search_exps(struct search_exp *exp)
{
  struct search_exp *next;

  while (exp) {
    next = exp->next;
    free(exp);
    exp = next;
  }
}

struct search_exp *parse_search_string(char *sstring)
{
   struct search_exp *first, *last, *exp;
   int i,j;
   int len;
   char *str;

   first = last = 0;
   len = strlen(sstring);
   for (i=0;i<len;i++)
     sstring[i] = toupper(sstring[i]);

   j = 0;
   for (i=0;i<=len;i++) {
    if (len-i>=4)
      if (!strncasecmp(sstring+i," or ",4)) {
        sstring[j++] = '|';
        i+=3;
        continue;
      }
    if (len-i>=5)
      if (!strncasecmp(sstring+i," and ",5)) {
        sstring[j++] = '&';
        i+=4;
        continue;
      }
    sstring[j++] = sstring[i];
  }
  strcat(sstring,"|");

  len = strlen(sstring);
  str = sstring;
  for (i=0;i<len;i++)
    if (sstring[i] == '|' || sstring[i] == '&') {
     exp = (struct search_exp *)malloc(sizeof(struct search_exp));
     if (!exp) {
       free_search_exps(first);
       error("Not enough memory!");
       return 0;
     }
     exp->next = 0;
     exp->operator = sstring[i] == '|' ? OP_OR : OP_AND;

     if (!strncasecmp(str,"Author:",7)) {
       str += 7;
       exp->what = S_AUTHOR;
     }else
     if (!strncasecmp(str,"Homepage:",9)) {
       str += 9;
       exp->what = S_HOMEPAGE;
     }else
     if (!strncasecmp(str,"Download:",9)) {
       str += 9;
       exp->what = S_DOWNLOAD;
     }else
     if (!strncasecmp(str,"URL:",4)) {
       str += 4;
       exp->what = S_URL;
     }else
     if (!strncasecmp(str,"RPM:",4)) {
       str += 4;
       exp->what = S_RPM;
     }else
     if (!strncasecmp(str,"DEB:",4)) {
       str += 4;
       exp->what = S_DEB;
     }else
     if (!strncasecmp(str,"Dependancy:",11)) {
       str += 11;
       exp->what = S_DEPENDANCY;
     }else
     if (!strncasecmp(str,"Dep:",4)) {
       str += 4;
       exp->what = S_DEPENDANCY;
     }else
     if (!strncasecmp(str,"Desc:",5)) {
       str += 5;
       exp->what = S_DESC;
     }else
     if (!strncasecmp(str,"Description:",12)) {
       str += 12;
       exp->what = S_DESC;
     }else
     if (!strncasecmp(str,"Category:",9)) {
       str += 9;
       exp->what = S_CATEGORY;
     }else
     if (!strncasecmp(str,"Name:",5)) {
       str += 5;
       exp->what = S_NAME;
     }else
     if (!strncasecmp(str,"Title:",6)) {
       str += 6;
       exp->what = S_NAME;
     }else
      exp->what = S_ALL;
     str = compress(str);

     sstring[i] = 0;
     exp->str = compress(str);

     if (first)
       last->next = exp;
     else first = exp;
     last = exp;

     str = sstring + i + 1;
    }

  return first;
}

void
do_search (char *substr)
{
  struct app *a, *an;
  struct category *scat;
  struct search_exp *sexp;

  search_no_mem = 0;
  scat = (struct category *) malloc (sizeof (struct category));
  scat->name = "Search";
  scat->apps = 0;
  scat->prev = 0;
  scat->next = 0;
  scat->appno = 0;

  sexp = parse_search_string(substr);
  if (!sexp)
   return ;

  search_categories (sexp, scat);

  free_search_exps(sexp);

  if (search_no_mem)
    error ("Not enough memory, list may be incomplete.");
  if (!scat->apps)
    {
      error ("Pattern not found (press any key)");
      return;
    }

  scat->appno = relink_app (scat->apps);
  app_selection (scat->apps, scat->appno);

  a = scat->apps;
  free (scat);
  while (a)			/* Free all applications */
    {
      an = a->next;
      free (a);
      a = an;
    }
  scat = 0;
}

void
search ()
{
  unsigned char sstr[0x100];
  int c;
  int len;

  sstr[0] = 0;

  while (1)
    {
      len = strlen (sstr);
      attrset (T_COLOR (CP_KEY));
      mvprintw (LINES - 1, 0, "/");
      attrset (T_COLOR (CP_NORMAL));
      printw ("%-*.*s", COLS - 1, COLS - 1, sstr);
      move (LINES - 1, len + 1);
      refresh ();

      c = getch ();
      switch (c)
	{
	case KEY_BACKSPACE:
	case 8:
	  if (len == 0)
	    return;
	  sstr[len - 1] = 0;
	  break;
        case 7:
          return;            /* ^G cancels input */
        case 21:             /* ^U clears line */
          len = 0;
          sstr[0] = 0;
          break;
	case '\n':
	  if (len == 0)
	    return;
	  do_search (sstr);
	  return;
	default:
	  if (len < COLS - 1)
	    {
	      sstr[len] = c;
	      sstr[len + 1] = 0;
	    }
	  break;
	}
    }
}
