/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#ifndef __kom_base_h__
#define __kom_base_h__

#include <qstring.h>
#include <qlist.h>
#include <qdict.h>
#include <list>

#include "kom.h"
#include "komVar.h"

#define SIGNAL_IMPL( name ) { ClientList *l = new QList<Client>; l->setAutoDelete( true ); m_dctSignals.insert( name, l ); }

#define EVENT_MAPPER( event, value ) { string __ev = event; const CORBA::Any& __v = value; 
#define MAPPING( event, type, func ) if ( __ev == event ) { type __t; if ( __v >>= __t ) return func( __t ); }
#define MAPPING_BOOLEAN( event, func ) if ( __ev == event ) { CORBA::Boolean __x; if ( __v >>= CORBA::Any::to_boolean( __x ) ) return func( __x ); }
#define MAPPING_OCTET( event, func ) if ( __ev == event ) { CORBA::Octet __x; if ( __v >>= CORBA::Any::to_octet( __x ) ) return func( __x ); }
#define MAPPING_CHAR( event, func ) if ( __ev == event ) { CORBA::Char __x; if ( __v >>= CORBA::Any::to_char( __x ) ) return func( __x ); }
#define MAPPING_WCHAR( event, func ) if ( __ev == event ) { CORBA::Wchar __x; if ( __v >>= CORBA::Any::to_wchar( __x ) ) return func( __x ); }
#define MAPPING_STRING( event, func ) if ( __ev == event ) { CORBA::String_var __x; if ( __v >>= CORBA::Any::to_string( __x, 0 ) ) return func( __x.in() ); }
#define MAPPING_WSTRING( event, func ) if ( __ev == event ) { CORBA::WString_var __x; if ( __v >>= CORBA::Any::to_wstring( __x, 0 ) ) return func( __x.in() ); }
#define MAPPING_OBJECT( event, func ) if ( __ev == event ) { CORBA::Object_var __x; if ( __v >>= CORBA::Any::to_object( __x ) ) return func( __x ); }
#define END_EVENT_MAPPER return false; }

#define EMIT_EVENT( dest, event, value ) { CORBA::Any __a; __a <<= value; dest->receive( event, __a ); }
#define EMIT_EVENT_BOOLEAN( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_boolean( value ); dest->receive( event, __a ); }
#define EMIT_EVENT_OCTET( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_octet( value ); dest->receive( event, __a ); }
#define EMIT_EVENT_CHAR( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_char( value ); dest->receive( event, __a ); }
#define EMIT_EVENT_WCHAR( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_wchar( value ); dest->receive( event, __a ); }
#define EMIT_EVENT_STRING( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_string( (char*)value, 0 ); dest->receive( event, __a ); }
#define EMIT_EVENT_WSTRING( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_wstring( (char*)value, 0 ); dest->receive( event, __a ); }
#define EMIT_EVENT_OBJECT( dest, event, value ) { CORBA::Any __a; __a <<= CORBA::Any::from_object( value, "Object" ); dest->receive( event, __a ); }

int cmpIOR( CORBA::Object_ptr p1, CORBA::Object_ptr p2 );

class KOMBase : public virtual KOM::Base_skel
{  
public:
  struct Client
  {
    Client( KOM::Base_ptr _obj, const char *_func ) 
    {
      obj = KOM::Base::_duplicate( _obj ); function = _func;
    }
    
    KOM::Base_var obj;
    QString function;
  };

  struct Filter
  {
    Filter( KOM::Base_ptr _o, const char *_func,
	    const KOM::EventTypeSeq& _ev, KOM::Base::FilterMode _mode )
    {
      obj = KOM::Base::_duplicate( _o );
      function = _func;
      events = _ev;
      mode = _mode;
    }
    
    KOM::EventTypeSeq events;
    KOM::Base_var obj;
    QString function;
    KOM::Base::FilterMode mode;
  };

  typedef QList<Client> ClientList;

protected:
  QDict<ClientList> m_dctSignals;
  QList<KOM::Base_var> m_lstClients;
  QList<Filter> m_lstFilters;
  list<KOM::Base_var> m_lstRelatives;
  
  bool m_bIsClean;

  CORBA::ULong m_ulRefCount;
  
  /**
   * Overload this function to be informed about events.
   */
  virtual bool event( const char*, const CORBA::Any& ) { return false; }

public:
  KOMBase();
  virtual ~KOMBase();

  // IDL
  virtual void connect( const char* _signal, KOM::Base_ptr obj, const char* function );
  virtual void disconnect( const char *_signal, KOM::Base_ptr obj, const char* _function );
  virtual void disconnectObject( KOM::Base_ptr obj );
  virtual void disconnectAll();
  virtual void disconnectNotify( KOM::Base_ptr obj );
  virtual void disconnectSignalNotify( KOM::Base_ptr obj, const char * /* _signal */, const char * /* _function */ );
  virtual void disconnectFilterNotify( KOM::Base_ptr obj );
  virtual void connectNotify( KOM::Base_ptr sig );

  KOM::Base::ConnectionSeq* describeConnections();
  KOM::Base::EventFilterSeq* describeEventFilters();
  KOM::Base::RelativesSeq* describeRelatives();
  
  /**
   * Call this function to send an event to this object. The event will pass all installed
   * event filters. If the event is not deleted by a filter, then @ref #event will be called
   * afterwards to finally handle the event.
   */
  virtual void receive( const char* type, const CORBA::Any& value );
  virtual void receiveASync( const char* type, const CORBA::Any& value );
  virtual void installFilter( KOM::Base_ptr obj, const char* function,
			      const KOM::EventTypeSeq& events,
			      KOM::Base::FilterMode mode );
  virtual void uninstallFilter( KOM::Base_ptr obj );
  virtual CORBA::Boolean eventFilter( KOM::Base_ptr, const char*, const CORBA::Any& ) { return false; }

  virtual void incRef();
  virtual void decRef();
  virtual CORBA::ULong refCount() { return m_ulRefCount; }
  virtual void destroy();
  
  virtual void adopt( KOM::Base_ptr _ptr );
  virtual void adoptNotify( KOM::Base_ptr _ptr );
  virtual void leave( KOM::Base_ptr _ptr );
  virtual void leaveNotify( KOM::Base_ptr _ptr );
  
protected:
  // C++
  virtual void cleanUp();

  virtual void disconnectObjectIntern( KOM::Base_ptr obj, bool _send_notify );
};

void signal_call0( const char *_signal, QDict<KOMBase::ClientList> *_dict );

template <class P1>
void signal_call1( const char *_signal, QDict<KOMBase::ClientList> *_dict, P1 _p1 )
{
    KOMBase::ClientList *list = (*_dict)[ _signal ];
    if( list == 0L )
    {
      cerr << "Unknown signal " << _signal << endl;
      cerr << "Please insert SIGNAL_IMPL( \"" << _signal << "\" ); in "
	   << "your constructor or choose a valid signal" << endl;
      assert( 0 );
    }
    
    QListIterator<KOMBase::Client> it( *list );
    for ( ; it.current(); ++it )
    {
        CORBA::Request_var _req = it.current()->obj->_request( it.current()->function.data() );
	_req->add_in_arg() <<= (P1)_p1;
	_req->result()->value()->type( CORBA::_tc_void );
	_req->invoke();
	// if( CORBA::Exception *_ex = _req->env()->exception() )
	// mico_throw( *_ex );
    }
}

template <class P1, class P2>
void signal_call2( const char *_signal, QDict<KOMBase::ClientList> *_dict, P1 _p1, P2 _p2 )
{
    KOMBase::ClientList *list = (*_dict)[ _signal ];
    if( list == 0L )
    {
      cerr << "Unknown signal " << _signal << endl;
      cerr << "Please insert SIGNAL_IMPL( \"" << _signal << "\" ); in "
	   << "your constructor or choose a valid signal" << endl;
      assert( 0 );
    }

    QListIterator<KOMBase::Client> it( *list );
    for ( ; it.current(); ++it )
    {
        CORBA::Request_var _req = it.current()->obj->_request( it.current()->function.data() );
	_req->add_in_arg() <<= (P1)_p1;
	_req->add_in_arg() <<= (P2)_p2;
	_req->result()->value()->type( CORBA::_tc_void );
	_req->invoke();
	// if( CORBA::Exception *_ex = _req->env()->exception() )
	// mico_throw( *_ex );
    }
}

#define SIGNAL_CALL0( _sig ) signal_call0( _sig, &m_dctSignals )
#define SIGNAL_CALL1( _sig, _p1 ) signal_call1( _sig, &m_dctSignals, _p1 )
#define SIGNAL_CALL2( _sig, _p1, _p2 ) signal_call2( _sig, &m_dctSignals, _p1, _p2 )

typedef KOMVar<KOM::Base> Base_ref;

void sdeb( const char *msg, ... );
void edeb( const char *msg, ... );
void mdeb( const char *msg, ... );

#endif
