/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "komComponent.h"

#include <string.h>

CORBA::Boolean operator== ( CORBA::String_var& _a, const char *_b )
{
  if ( strcmp( (const char*)_a, _b ) == 0 )
    return true;
  return false;
}

KOMComponent::KOMComponent() : KOMBase(), KOM::Component_skel()
{ 
}

KOMComponent::~KOMComponent()
{
  cleanUp();
}

void KOMComponent::cleanUp()
{
  if ( m_bIsClean )
    return;

  m_lstDerivedInterfaces.clear();
  m_lstBuiltinAggregates.clear();
  m_lstDynamicAggregates.clear();
  m_lstBuiltinPlugins.clear();
  m_lstDynamicPlugins.clear();

  KOMBase::cleanUp();
}

CORBA::Boolean KOMComponent::supportsInterface( const char* _iface )
{
  list<KOMDerivedInterface>::iterator it = m_lstDerivedInterfaces.begin();
  for ( ; it != m_lstDerivedInterfaces.end(); ++it )
  {
    if ( it->iface == _iface )
      return true;
  }

  list<KOMBuiltinAggregate>::iterator it2 = m_lstBuiltinAggregates.begin();
  for ( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      if ( it2->providedInterfaces[l] == _iface )
	return true;
  }

  list<KOMDynamicAggregate>::iterator it3 = m_lstDynamicAggregates.begin();
  for ( ; it3 != m_lstDynamicAggregates.end(); ++it3 )
  {
    for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
      if ( it3->providedInterfaces[l] == _iface )
	return true;
  }

  return false;
}

CORBA::Object_ptr KOMComponent::getInterface( const char* _iface )
{
  list<KOMDerivedInterface>::iterator it = m_lstDerivedInterfaces.begin();
  for ( ; it != m_lstDerivedInterfaces.end(); ++it )
  {
    if ( it->iface == _iface )
      return CORBA::Object::_duplicate( this );
  }

  list<KOMBuiltinAggregate>::iterator it2 = m_lstBuiltinAggregates.begin();
  for ( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      if ( it2->providedInterfaces[l] == _iface )
	return it2->ref( this );
  }

  list<KOMDynamicAggregate>::iterator it3 = m_lstDynamicAggregates.begin();
  for ( ; it3 != m_lstDynamicAggregates.end(); ++it3 )
  {
    for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
      if ( it3->providedInterfaces[l] == _iface )
	return it3->ref( this );
  }

  return CORBA::Object::_nil();
}

KOM::InterfaceSeq* KOMComponent::interfaces()
{
  return interfaces( false );
}

KOM::InterfaceSeq* KOMComponent::interfaces( bool _include_precomposed )
{
  list<string> lst;
  
  list<KOMDerivedInterface>::iterator it = m_lstDerivedInterfaces.begin();
  for ( ; it != m_lstDerivedInterfaces.end(); ++it )
    lst.push_back( it->iface.in() );

  list<KOMBuiltinAggregate>::iterator it2 = m_lstBuiltinAggregates.begin();
  for ( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      lst.push_back( it2->providedInterfaces[l].in() );
  }

  list<KOMDynamicAggregate>::iterator it3 = m_lstDynamicAggregates.begin();
  for ( ; it3 != m_lstDynamicAggregates.end(); ++it3 )
  {
    if ( it3->composed || ( _include_precomposed && it3->precomposed ) )
    {
      for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
	lst.push_back( it3->providedInterfaces[l].in() );
    }
  }

  KOM::InterfaceSeq *s = new KOM::InterfaceSeq;
  s->length( lst.size() );
  unsigned int l = 0;
  list<string>::iterator it4 = lst.begin();
  for( ; it4 != lst.end(); ++it4 )
    (*s)[l++] = CORBA::string_dup( it4->c_str() );

  return s;
}

/*--------------------------------------------------------------*/

CORBA::Boolean KOMComponent::supportsPluginInterface( const char* _iface )
{
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for ( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      if ( it2->providedInterfaces[l] == _iface )
	return true;
  }

  list<KOMDynamicPlugin>::iterator it3 = m_lstDynamicPlugins.begin();
  for ( ; it3 != m_lstDynamicPlugins.end(); ++it3 )
  {
    for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
      if ( it3->providedInterfaces[l] == _iface )
	return true;
  }

  return false;
}

CORBA::Object_ptr KOMComponent::getPluginInterface( const char* _iface )
{
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for ( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      if ( it2->providedInterfaces[l] == _iface )
	return it2->ref( this );
  }

  list<KOMDynamicPlugin>::iterator it3 = m_lstDynamicPlugins.begin();
  for ( ; it3 != m_lstDynamicPlugins.end(); ++it3 )
  {
    for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
      if ( it3->providedInterfaces[l] == _iface )
	return it3->ref( this );
  }

  return CORBA::Object::_nil();
}

KOM::InterfaceSeq* KOMComponent::pluginInterfaces()
{
  return pluginInterfaces( false );
}

KOM::InterfaceSeq* KOMComponent::pluginInterfaces( bool _include_precomposed )
{
  list<string> lst;
  
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for ( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    for( unsigned long l = 0; l < it2->providedInterfaces.length(); ++l )
      lst.push_back( it2->providedInterfaces[l].in() );
  }

  list<KOMDynamicPlugin>::iterator it3 = m_lstDynamicPlugins.begin();
  for ( ; it3 != m_lstDynamicPlugins.end(); ++it3 )
  {
    if ( it3->composed || ( _include_precomposed && it3->precomposed ) )
    {
      for( unsigned long l = 0; l < it3->providedInterfaces.length(); ++l )
	lst.push_back( it3->providedInterfaces[l].in() );
    }
  }

  KOM::InterfaceSeq *s = new KOM::InterfaceSeq;
  s->length( lst.size() );
  unsigned int l = 0;
  list<string>::iterator it4 = lst.begin();
  for( ; it4 != lst.end(); ++it4 )
    (*s)[l++] = CORBA::string_dup( it4->c_str() );
  
  return s;
}

KOM::Component::PluginInfoSeq* KOMComponent::describePlugins()
{ 
  list<KOM::Component::PluginInfo> lst;
  
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for ( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    KOM::Component::PluginInfo info;
    info.interfaces = it2->providedInterfaces;
    info.id = it2->id;
    info.composed = true;
    lst.push_back( info );
  }

  list<KOMDynamicPlugin>::iterator it3 = m_lstDynamicPlugins.begin();
  for ( ; it3 != m_lstDynamicPlugins.end(); ++it3 )
  {
    KOM::Component::PluginInfo info;
    info.interfaces = it3->providedInterfaces;
    info.id = it3->id;
    info.composed = it3->composed;
    lst.push_back( info );
  }

  KOM::Component::PluginInfoSeq *s = new KOM::Component::PluginInfoSeq;
  s->length( lst.size() );
  unsigned int l = 0;
  list<KOM::Component::PluginInfo>::iterator it4 = lst.begin();
  for( ; it4 != lst.end(); ++it4 )
    (*s)[l++] = *it4;
  
  return s;
}

KOM::Plugin_ptr KOMComponent::getPlugin( KOM::ID id )
{
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for ( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    if ( it2->id == id )
      return it2->ref( this );
  }

  list<KOMDynamicPlugin>::iterator it3 = m_lstDynamicPlugins.begin();
  for ( ; it3 != m_lstDynamicPlugins.end(); ++it3 )
  {
    if ( it3->id == id )
      return it3->ref( this );
  }

  return KOM::Plugin::_nil();
}


KOM::ID KOMComponent::addPlugin( KOM::PluginFactory_ptr _factory,
				 const KOM::InterfaceSeq& _required,
				 const KOM::InterfaceSeq& _required_plugins,
				 const KOM::InterfaceSeq& _provided,
				 CORBA::Boolean _activate )
{
  KOMDynamicPlugin p( _factory, _required, _required_plugins,
		      _provided, _activate, m_id );
  m_lstDynamicPlugins.push_back( p );
  
  // Try to compose all plugins
  composePlugins();
   
  return m_id++;
}

void KOMComponent::removePlugin( KOM::ID _id )
{
  list<KOMDynamicPlugin>::iterator it = m_lstDynamicPlugins.begin();
  for( ; it != m_lstDynamicPlugins.end(); ++it )
    if ( it->id == _id )
    {
      m_lstDynamicPlugins.erase( it );
      return;
    }
  
  // TODO: decompose plugins if needed

  KOM::UnknownID exc;
  exc.id = _id;
  mico_throw( exc );
}

void KOMComponent::composePlugins()
{  
  // Try to compose all yet uncomposed builtin plugins
  list<KOMBuiltinPlugin>::iterator it2 = m_lstBuiltinPlugins.begin();
  for( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    if ( !it2->composed )
      // Ok, lets make it precomposed
      it2->precomposed = true;
  }
  // Try to compose all yet uncomposed dynamic plugins
  list<KOMDynamicPlugin>::iterator it = m_lstDynamicPlugins.begin();
  for( ; it != m_lstDynamicPlugins.end(); ++it )
  {
    if ( !it->composed )
      // Ok, lets make it precomposed
      it->precomposed = true;
  }
  
  // Loop until we did not need to exclude any more plugin
  bool decomposed;
  do
  {    
    decomposed = false;

    // Get a list of all composed and precomposed interfaces
    KOM::InterfaceSeq_var af = interfaces( true );
    KOM::InterfaceSeq_var pf = pluginInterfaces( true );
  
    // Try to compose all yet uncomposed builtin plugins
    it2 = m_lstBuiltinPlugins.begin();
    for( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
    {
      if ( !it2->composed && it2->precomposed )
      {
	// Try to match required interfaces
	for( CORBA::ULong l = 0; l < it2->requiredInterfaces.length(); ++l )
	{
	  CORBA::ULong k;
	  for( k = 0; k < af->length() &&
	       strcmp( it2->requiredInterfaces[l].in(), (*af)[k].in() ) != 0L; ++k );
	  if ( k == af->length() )
	  {
	    it2->precomposed = false;
	    decomposed = true;
	    continue;
	  }
	}

	// Try to match required plugin interfaces
	for( CORBA::ULong l = 0; l < it2->requiredPluginInterfaces.length(); ++l )
        {
	  CORBA::ULong k;
	  for( k = 0; k < pf->length() &&
	       strcmp( it2->requiredPluginInterfaces[l].in(), (*pf)[k].in() ) != 0L;
	       ++k );
	  if ( k == pf->length() )
	  {
	    it2->precomposed = false;
	    decomposed = true;
	    continue;
	  }
	}
      }
    }

    // Try to compose all yet uncomposed dynamic plugins
    it = m_lstDynamicPlugins.begin();
    for( ; it != m_lstDynamicPlugins.end(); ++it )
    {
      if ( !it->composed && it->precomposed )
      {
	// Try to match required interfaces
	for( CORBA::ULong l = 0; l < it->requiredInterfaces.length(); ++l )
	{
	  CORBA::ULong k;
	  for( k = 0; k < af->length() &&
	       strcmp( it->requiredInterfaces[l].in(), (*af)[k].in() ) != 0L; ++k );
	  if ( k == af->length() )
	  {
	    it->precomposed = false;
	    decomposed = true;
	    continue;
	  }
	}

	// Try to match required plugin interfaces
	for( CORBA::ULong l = 0; l < it->requiredPluginInterfaces.length(); ++l )
        {
	  CORBA::ULong k;
	  for( k = 0; k < pf->length() &&
	       strcmp( it->requiredPluginInterfaces[l].in(), (*pf)[k].in() ) != 0L;
	       ++k );
	  if ( k == pf->length() )
	  {
	    it->precomposed = false;
	    decomposed = true;
	    continue;
	  }
	}
      }
    }
  } while( decomposed );

  // All remaining precomposed builtin plugins are now composed
  it2 = m_lstBuiltinPlugins.begin();
  for( ; it2 != m_lstBuiltinPlugins.end(); ++it2 )
  {
    if ( !it2->composed && it2->precomposed )
    {
      it2->composed = true;
    }
  }

  // All remaining precomposed dynamic plugins are now composed
  it = m_lstDynamicPlugins.begin();
  for( ; it != m_lstDynamicPlugins.end(); ++it )
  {
    if ( !it->composed && it->precomposed )
    {
      it->composed = true;
    }
  }

  // Start all autostart dynamic plugins which we composed right now
  it = m_lstDynamicPlugins.begin();
  for( ; it != m_lstDynamicPlugins.end(); ++it )
  {
    if ( it->composed && it->precomposed )
    {
      it->precomposed = false;
      
      // Does only work if the plugin is composed
      if ( it->activate )
	it->start( this );
    }
  }
}

/*-----------------------------------------------------------*/

KOM::ID KOMComponent::addAggregate( KOM::AggregateFactory_ptr _factory,
				    const KOM::InterfaceSeq& _required,
				    const KOM::InterfaceSeq& _provided,
				    CORBA::Boolean _activate )
{
  KOMDynamicAggregate p( _factory, _required, _provided, _activate, m_id );
  m_lstDynamicAggregates.push_back( p );

  composeAggregates();
  composePlugins();
  
  return m_id++;
}

void KOMComponent::removeAggregate( KOM::ID _id )
{
  list<KOMDynamicAggregate>::iterator it = m_lstDynamicAggregates.begin();
  for( ; it != m_lstDynamicAggregates.end(); ++it )
    if ( it->id == _id )
    {
      m_lstDynamicAggregates.erase( it );
      return;
    }

  // TODO: decompose aggregates if needed.
  
  KOM::UnknownID exc;
  exc.id = _id;
  mico_throw( exc );
}

void KOMComponent::composeAggregates()
{  
  // Try to compose all yet uncomposed builtin Aggregates
  list<KOMBuiltinAggregate>::iterator it2 = m_lstBuiltinAggregates.begin();
  for( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
  {
    if ( !it2->composed )
      // Ok, lets make it precomposed
      it2->precomposed = true;
  }

  // Try to compose all yet uncomposed dynamic Aggregates
  list<KOMDynamicAggregate>::iterator it = m_lstDynamicAggregates.begin();
  for( ; it != m_lstDynamicAggregates.end(); ++it )
  {
    if ( !it->composed )
      // Ok, lets make it precomposed
      it->precomposed = true;
  }
  
  // Loop until we did not need to exclude any more Aggregate
  bool decomposed;
  do
  {    
    decomposed = false;

    // Get a list of all composed and precomposed interfaces
    KOM::InterfaceSeq_var af = interfaces( true );

    // Try to compose all yet uncomposed builtin Aggregates
    it2 = m_lstBuiltinAggregates.begin();
    for( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
    {
      if ( !it2->composed && it2->precomposed )
      {
	// Try to match required interfaces
	for( CORBA::ULong l = 0; l < it2->requiredInterfaces.length(); ++l )
	{
	  CORBA::ULong k;
	  for( k = 0; k < af->length() &&
	       strcmp( it2->requiredInterfaces[l].in(), (*af)[k].in() ) != 0L; ++k );
	  if ( k == af->length() )
	  {
	    it2->precomposed = false;
	    decomposed = true;
	    break;
	  }
	}
      }
    }
    
    // Try to compose all yet uncomposed dynamic Aggregates
    it = m_lstDynamicAggregates.begin();
    for( ; it != m_lstDynamicAggregates.end(); ++it )
    {
      if ( !it->composed && it->precomposed )
      {
	// Try to match required interfaces
	for( CORBA::ULong l = 0; l < it->requiredInterfaces.length(); ++l )
	{
	  CORBA::ULong k;
	  for( k = 0; k < af->length() &&
	       strcmp( it->requiredInterfaces[l].in(), (*af)[k].in() ) != 0L; ++k );
	  if ( k == af->length() )
	  {
	    it->precomposed = false;
	    decomposed = true;
	    break;
	  }
	}
      }
    }
  } while( decomposed );

  // All remaining precomposed builtin Aggregates are now composed
  it2 = m_lstBuiltinAggregates.begin();
  for( ; it2 != m_lstBuiltinAggregates.end(); ++it2 )
  {
    if ( !it2->composed && it2->precomposed )
    {
      it2->composed = true;
    }
  }

  // All remaining precomposed dynamic Aggregates are now composed
  it = m_lstDynamicAggregates.begin();
  for( ; it != m_lstDynamicAggregates.end(); ++it )
  {
    if ( !it->composed && it->precomposed )
    {
      it->composed = true;
    }
  }

  // Start all autostart dynamic Aggregates which we composed right now
  it = m_lstDynamicAggregates.begin();
  for( ; it != m_lstDynamicAggregates.end(); ++it )
  {
    if ( it->composed && it->precomposed )
    {
      it->precomposed = false;
      
      // Does only work if the Aggregate is composed
      if ( it->activate )
	it->start( this );
    }
  }
}

KOM::ID KOMComponent::addAggregate( KOMAggregateFunc _func,
				    const KOM::InterfaceSeq& _required,
				    const KOM::InterfaceSeq& _provided )
{
  KOMBuiltinAggregate p( _func, _required, _provided, m_id );
  m_lstBuiltinAggregates.push_back( p );

  composeAggregates();
  composePlugins();
  
  return m_id++;
}

KOM::ID KOMComponent::addAggregate( KOMAggregateFunc _func,
				    const char *_provided_iface )
{
  assert( _provided_iface != 0L );
 
  // One interface provided
  KOM::InterfaceSeq s; 
  s.length(1);
  s[0] = CORBA::string_dup( _provided_iface );
  // No interfaces required
  KOM::InterfaceSeq s2; 
  s2.length(0);
  
  return addAggregate( _func, s2, s );
}

KOM::ID KOMComponent::addAggregate( CORBA::Object_ptr _obj,
				    const KOM::InterfaceSeq& _required,
				    const KOM::InterfaceSeq& _provided )
{
  KOMBuiltinAggregate p( _obj, _required, _provided, m_id );
  m_lstBuiltinAggregates.push_back( p );

  composeAggregates();
  composePlugins();
  
  return m_id++;
}

KOM::ID KOMComponent::addAggregate( CORBA::Object_ptr _obj,
				    const char *_provided_interface )
{
  assert( _provided_interface != 0L );
 
  // One interface provided
  KOM::InterfaceSeq s; 
  s.length(1);
  s[0] = CORBA::string_dup( _provided_interface );

  // No interfaces required
  KOM::InterfaceSeq s2; 
  s2.length(0);
  
  return addAggregate( _obj, s2, s );
}







