/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#ifndef __kom_component_h__
#define __kom_component_h__

#include <list.h>

#include "kom.h"
#include "komBase.h"
#include "komVar.h"
#include "komContainer.h"

class KOMComponent;

typedef KOMVar<KOM::Aggregate> Aggregate_ref;
typedef KOMVar<KOM::Plugin> Plugin_ref;

CORBA::Boolean operator== ( CORBA::String_var&, const char * );

typedef CORBA::Object_ptr (*KOMAggregateFunc)( KOMComponent* );
typedef KOM::Plugin_ptr (*KOMPluginFunc)( KOMComponent* );

struct KOMBuiltinAggregate
{
  KOMBuiltinAggregate( KOMAggregateFunc _func,
		       const KOM::InterfaceSeq& _required,
		       const KOM::InterfaceSeq& _provided, KOM::ID _id )
    { activator = _func; requiredInterfaces = _required; id = _id;
      providedInterfaces = _provided; }
  KOMBuiltinAggregate( CORBA::Object_ptr _obj,
		       const KOM::InterfaceSeq& _required,
		       const KOM::InterfaceSeq& _provided,
		       KOM::ID _id )
    { activator = 0L; aggregate = CORBA::Object::_duplicate( _obj );
      requiredInterfaces = _required; id = _id;
      providedInterfaces = _provided; }
  
  CORBA::Object_ptr ref( KOMComponent* _comp ) {
    if ( CORBA::is_nil( aggregate ) && activator != 0L )
      aggregate = activator( _comp );
    return CORBA::Object::_duplicate( aggregate ); }
  
  KOMAggregateFunc activator;
  CORBA::Object_var aggregate;
  KOM::InterfaceSeq requiredInterfaces;
  KOM::InterfaceSeq providedInterfaces;
  KOM::ID id;
  bool composed;
  bool precomposed;
};

struct KOMDynamicAggregate
{
  KOMDynamicAggregate( KOM::AggregateFactory_ptr _factory, 
		       const KOM::InterfaceSeq& _required, 
		       const KOM::InterfaceSeq& _provided, 
		       CORBA::Boolean _activate, KOM::ID _id )
    { factory = KOM::AggregateFactory::_duplicate( _factory ); id = _id;
      composed = false; requiredInterfaces = _required;
      activate = _activate; precomposed = false;
      providedInterfaces = _provided; }
  
  KOM::Aggregate_ptr ref( KOM::Component_ptr _comp ) {
    if ( !composed )
      return KOM::Aggregate::_nil();
    if ( CORBA::is_nil( aggregate ) )
      aggregate = factory->create( _comp );
    return KOM::Aggregate::_duplicate( aggregate ); }

  void start( KOM::Component_ptr _comp ) {
    if ( !composed )
      return;
    if ( CORBA::is_nil( aggregate ) )
      aggregate = factory->create( _comp ); }
  
  KOM::AggregateFactory_var factory;
  Aggregate_ref aggregate;
  KOM::ID id;
  bool composed;
  bool precomposed;
  KOM::InterfaceSeq requiredInterfaces;
  KOM::InterfaceSeq providedInterfaces;
  CORBA::Boolean activate;
};

struct KOMDerivedInterface
{
  KOMDerivedInterface( const char *_iface ) { iface = CORBA::string_dup( _iface ); }
  
  CORBA::String_var iface;
};

struct KOMBuiltinPlugin
{
  KOMBuiltinPlugin( KOMPluginFunc _func, const KOM::InterfaceSeq& _required,
		    const KOM::InterfaceSeq& _required_plugins, 
		    const KOM::InterfaceSeq& _provided, 
		    KOM::ID _id )
    { activator = _func; requiredInterfaces = _required;
      requiredPluginInterfaces = _required_plugins;
      providedInterfaces = _provided; id = _id; }

  KOM::Plugin_ptr ref( KOMComponent* _comp ) {
    if ( CORBA::is_nil( plugin ) )
      plugin = activator( _comp );
    return KOM::Plugin::_duplicate( plugin ); }
  
  KOMPluginFunc activator;
  KOM::Plugin_var plugin;
  KOM::InterfaceSeq requiredInterfaces;
  KOM::InterfaceSeq requiredPluginInterfaces;
  KOM::InterfaceSeq providedInterfaces;
  KOM::ID id;
  bool composed;
  bool precomposed;
};

struct KOMDynamicPlugin
{
  KOMDynamicPlugin( KOM::PluginFactory_ptr _factory,
		    const KOM::InterfaceSeq& _required,
		    const KOM::InterfaceSeq& _required_plugins, 
		    const KOM::InterfaceSeq& _provided, 
		    CORBA::Boolean _activate, KOM::ID _id )
    { factory = KOM::PluginFactory::_duplicate( _factory ); id = _id;
      composed = false; requiredInterfaces = _required;
      activate = _activate; precomposed = false;
      requiredPluginInterfaces = _required_plugins;
      providedInterfaces = _provided; }

  KOM::Plugin_ptr ref( KOM::Component_ptr _comp ) {
    if ( !composed )
      return KOM::Plugin::_nil();
    if ( CORBA::is_nil( plugin ) )
      plugin = factory->create( _comp );
    return KOM::Plugin::_duplicate( plugin ); }

  void start( KOM::Component_ptr _comp ) {
    if ( !composed )
      return;
    if ( CORBA::is_nil( plugin ) )
      plugin = factory->create( _comp ); }

  KOM::PluginFactory_var factory;
  Plugin_ref plugin;
  KOM::ID id;
  bool composed;
  bool precomposed;
  KOM::InterfaceSeq requiredInterfaces;
  KOM::InterfaceSeq requiredPluginInterfaces;
  KOM::InterfaceSeq providedInterfaces;
  CORBA::Boolean activate;
};

#define ADD_INTERFACE( name ) m_lstDerivedInterfaces.push_back( name );

class KOMComponent : virtual public KOMBase, virtual public KOM::Component_skel
{
public:
  KOMComponent();
  ~KOMComponent();
  
  virtual KOM::ID addAggregate( KOM::AggregateFactory_ptr _factory,
				const KOM::InterfaceSeq& _required,
				const KOM::InterfaceSeq& _provided,
				CORBA::Boolean _activate);
  virtual void removeAggregate( KOM::ID _id );

  virtual KOM::ID addPlugin( KOM::PluginFactory_ptr _factory,
			     const KOM::InterfaceSeq& _required,
			     const KOM::InterfaceSeq& _required_plugins,
			     const KOM::InterfaceSeq& _provided,
			     CORBA::Boolean _activate);
  virtual void removePlugin( KOM::ID _id );
  
  virtual CORBA::Object_ptr getInterface( const char* name );
  virtual KOM::InterfaceSeq* interfaces();
  virtual CORBA::Boolean supportsInterface( const char* name );

  virtual CORBA::Object_ptr getPluginInterface( const char* name );
  virtual KOM::InterfaceSeq* pluginInterfaces();
  virtual CORBA::Boolean supportsPluginInterface( const char* name );

  virtual KOM::Component::PluginInfoSeq* describePlugins();
  virtual KOM::Plugin_ptr getPlugin( KOM::ID id );
  
  // C++
  virtual void cleanUp();

  virtual KOM::ID addAggregate( KOMAggregateFunc _func, 
				const KOM::InterfaceSeq& _required,
				const KOM::InterfaceSeq& _provided );
  virtual KOM::ID addAggregate( KOMAggregateFunc _func, const char *_provided_iface );
  virtual KOM::ID addAggregate( CORBA::Object_ptr _obj, 
				const KOM::InterfaceSeq& _required,
				const KOM::InterfaceSeq& _provided );
  virtual KOM::ID addAggregate( CORBA::Object_ptr _obj, const char *_provided_iface );
  
protected:
  virtual KOM::InterfaceSeq* pluginInterfaces( bool _include_precomposed );
  virtual KOM::InterfaceSeq* interfaces( bool _include_precomposed );

  virtual void composeAggregates();
  virtual void composePlugins();
  
  list<KOMDerivedInterface> m_lstDerivedInterfaces;
  list<KOMBuiltinAggregate> m_lstBuiltinAggregates;
  list<KOMDynamicAggregate> m_lstDynamicAggregates;
  list<KOMBuiltinPlugin> m_lstBuiltinPlugins;
  list<KOMDynamicPlugin> m_lstDynamicPlugins;

  KOM::ID m_id;
};

typedef KOMVar<KOM::Component> Component_ref;

class KOMComponentFactory : virtual public KOM::ComponentFactory_skel
{
public:
};

#endif
