/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "main.h"
#include "pm_error.h"
#include "pm_python.h"
#include "pm_object.h"
#include "pm_skeleton.h"

#include <map>
#include <iostream>
#include <list>
#include <string>

typedef list<string> StructInfo;

int g_id = 1;
map<int,CORBA::Request_var> g_mapRequests;
map<int,CORBA::Any*> g_mapAny;
map<int,PyMicoSkeleton*> g_mapSkels;
map<int,CORBA::TypeCode_var> g_mapTypeCodes;
map<int,CORBA::Object_var> g_mapObjects;

#define FIND_REQ( it ) \
  map<int,CORBA::Request_var>::iterator it = g_mapRequests.find( id ); \
  if ( it == g_mapRequests.end() ) \
  { \
      cerr << "ERROR: Unknown request id" << endl; \
    return Py_BuildValue( "i", PMERR_UNKNOWN_REQUEST_ID ); \
  }

#define FIND_SKEL( id, skel ) \
  PyMicoSkeleton* skel; \
  map<int,PyMicoSkeleton*>::iterator it = g_mapSkels.find( id ); \
  if ( it == g_mapSkels.end() ) \
  { \
      cerr << "ERROR: Unknown request id" << endl; \
      return Py_BuildValue( "i", PMERR_UNKNOWN_REQUEST_ID ); \
  } \
  skel = it->second;

CORBA::TypeCode_ptr createTypeCode( int _id, const char* _str )
{
  if ( _id == pm_tc_null ) return CORBA::TypeCode::_duplicate( CORBA::_tc_null );
  if ( _id == pm_tc_void ) return CORBA::TypeCode::_duplicate( CORBA::_tc_void );
  if ( _id == pm_tc_short ) return CORBA::TypeCode::_duplicate( CORBA::_tc_short );
  if ( _id == pm_tc_long ) return CORBA::TypeCode::_duplicate( CORBA::_tc_long );
  if ( _id == pm_tc_longlong ) return CORBA::TypeCode::_duplicate( CORBA::_tc_longlong );
  if ( _id == pm_tc_ushort ) return CORBA::TypeCode::_duplicate( CORBA::_tc_ushort );
  if ( _id == pm_tc_ulong ) return CORBA::TypeCode::_duplicate( CORBA::_tc_ulong );
  if ( _id == pm_tc_ulonglong ) return CORBA::TypeCode::_duplicate( CORBA::_tc_ulonglong );
  if ( _id == pm_tc_float ) return CORBA::TypeCode::_duplicate( CORBA::_tc_float );
  if ( _id == pm_tc_double ) return CORBA::TypeCode::_duplicate( CORBA::_tc_double );
  if ( _id == pm_tc_longdouble ) return CORBA::TypeCode::_duplicate( CORBA::_tc_longdouble );
  if ( _id == pm_tc_boolean ) return CORBA::TypeCode::_duplicate( CORBA::_tc_boolean );
  if ( _id == pm_tc_char ) return CORBA::TypeCode::_duplicate( CORBA::_tc_char );
  if ( _id == pm_tc_wchar ) return CORBA::TypeCode::_duplicate( CORBA::_tc_wchar );
  if ( _id == pm_tc_octet ) return CORBA::TypeCode::_duplicate( CORBA::_tc_octet );
  if ( _id == pm_tc_any ) return CORBA::TypeCode::_duplicate( CORBA::_tc_any );
  if ( _id == pm_tc_TypeCode ) return CORBA::TypeCode::_duplicate( CORBA::_tc_TypeCode );
  if ( _id == pm_tc_Principal ) return CORBA::TypeCode::_duplicate( CORBA::_tc_Principal );
  if ( _id == pm_tc_Context ) return CORBA::TypeCode::_duplicate( CORBA::_tc_Context ); // extension
  if ( _id == pm_tc_Object ) return CORBA::TypeCode::_duplicate( CORBA::_tc_Object );
  if ( _id == pm_tc_string ) return CORBA::TypeCode::_duplicate( CORBA::_tc_string );
  if ( _id == pm_tc_wstring ) return CORBA::TypeCode::_duplicate( CORBA::_tc_wstring );

  if ( _id == pm_tc_custom )
  {
    string str = _str;
    CORBA::TypeCode_ptr tc = new CORBA::TypeCode( str );
    return tc;
  }

  assert( 0 );
}

CORBA::TypeCode_ptr findTypeCode( int _id )
{
  map<int,CORBA::TypeCode_var>::iterator it = g_mapTypeCodes.find( _id );
  if ( it == g_mapTypeCodes.end() )
  {
    cerr << "ERROR: Unknown typecode" << endl;
    return 0L;
  }
  
  return it->second;
}

/**
 * Packing
 */
bool parseResult( CORBA::Any& _any, PyObject* _arg, CORBA::TypeCode_ptr _tc )
{
  switch( _tc->kind() )
  {
  case CORBA::tk_void:
    return true; 
  case CORBA::tk_any:
    {
      PyObject *oid = PyObject_GetAttrString( _arg, "_any_id_" );
      if ( oid == 0L )
      {
	cerr << "ERROR: Expected any, but got something else" << endl;
	return false;
      }
      int id;
      if ( !PyArg_Parse( oid, "i", &id ) )
      {
	cerr << "ERROR: Any member _any_id_ is not an integer" << endl;
	return false;
      }
      map<int,CORBA::Any*>::iterator it = g_mapAny.find( id );
      if ( it == g_mapAny.end() )
      {
	cerr << "ERROR: Any of id " << id << " is not registered" << endl;
	return 0L;
      }
      _any <<= *(it->second);
      return true;
    }
  case CORBA::tk_enum:
    {
      const char *value;
      if ( !PyArg_Parse( _arg, "s", &value ) )
      {
	cerr << "ERROR: Expected an enum string but got something else" << endl;
	return false;
      }
      for ( CORBA::ULong i = 0; i < _tc->member_count(); i++ )
      {
	if ( strcmp( value, (char *) _tc->member_name (i) ) == 0 )
	{
	  assert( _any.enum_put( i ) );
	  return true;
	}
      }
      
      cerr << "ERROR: Unknown enumartion item " << value << endl;
      cerr << "       Possible values are: ";
      for ( CORBA::ULong i = 0; i < _tc->member_count(); i++ )
	cerr << _tc->member_name(i) << " ";
      cerr << endl;
      
      return false;
    }
  case CORBA::tk_string:
  {
    const char *value;
    if ( !PyArg_Parse( _arg, "s", &value ) )
      return false;
    _any <<= CORBA::Any::from_string( (char *) value, 0 );
    return true;
  }
  case CORBA::tk_short:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::Short x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_ushort:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::UShort x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_long:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::Long x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_longlong:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::LongLong x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_ulong:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::ULong x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_double:
  {
    double i;
    if ( !PyArg_Parse( _arg, "d", &i ) )
      return false;
    CORBA::Double x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_float:
  {
    float i;
    if ( !PyArg_Parse( _arg, "f", &i ) )
      return false;
    CORBA::Float x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_longdouble:
  {
    double i;
    if ( !PyArg_Parse( _arg, "d", &i ) )
      return false;
    CORBA::LongDouble x = i;
    _any <<= x;
    return true;
  }
  case CORBA::tk_boolean:
  {
    int i;
    if ( !PyArg_Parse( _arg, "i", &i ) )
      return false;
    CORBA::Boolean x = true;
    if ( i == 0 )
      x = false;
    _any <<= CORBA::Any::from_boolean( x );
    return true;
  }
  case CORBA::tk_char:
  {
    int i;
    if ( !PyArg_Parse( _arg, "c", &i ) )
      return false;
    CORBA::Char x = true;
    _any <<= CORBA::Any::from_char( x );
    return true;
  }
  case CORBA::tk_octet:
  {
    int i;
    if ( !PyArg_Parse( _arg, "c", &i ) )
      return false;
    CORBA::Octet x = true;
    _any <<= CORBA::Any::from_octet( x );
    return true;
  }
  case CORBA::tk_TypeCode:
  {
    int id;
    if ( !PyArg_Parse( _arg, "i", &id ) )
    {
      cerr << "ERROR: Passes argument is not a typecode" << endl;
      return false;
    }
    CORBA::TypeCode_ptr tc = findTypeCode( id );
    if ( CORBA::is_nil( tc ) )
    {
      cerr << "ERROR: Unknow Typecode id " << id << endl;
      return false;
    }
    _any <<= tc;
    return true;
  }
  case CORBA::tk_objref:
  {
    CORBA::Object_var obj;
    if ( _arg == Py_None )
    {
      _any <<= CORBA::Any::from_object( obj, _tc->name() );
      return true;
    }
    
    // Assume first it is a CORBA.Object object
    PyObject* o = PyObject_GetAttrString( _arg, "_id_" );
    if ( o == 0L )
    {
      // Perhaps it is a server ?
      PyErr_Clear();
      PyObject* o = PyObject_GetAttrString( _arg, "_server_" );
      if ( o != 0L )
      {
	int id;
	if ( !PyArg_Parse( o, "i", &id ) )
        {
	  cerr << "ERROR: Passed argument is not of type CORBA.Object" << endl;
	  Py_DECREF( o );
	  return false;
	}
	Py_DECREF( o );
	FIND_SKEL( id, skel );
	obj = CORBA::Object::_duplicate( skel );
      }
      else
      {  
	// Is it perhaps an IOR
	PyErr_Clear();
	const char* ior;
	if ( !PyArg_Parse( _arg, "s", &ior ) )
        {  
	  cerr << "ERROR: Passed argument is not of type CORBA.Object and not an IOR" << endl;
	  return false;
	}
	obj = pm_orb->string_to_object( ior );
      }
    }
    else
    {
      int id;
      if ( !PyArg_Parse( o, "i", &id ) )
      {
	cerr << "ERROR: Passed argument is not of type CORBA.Object" << endl;
	return false;
      }
      map<int,CORBA::Object_var>::iterator it = g_mapObjects.find( id );
      if ( it == g_mapObjects.end() )
      {
	cerr << "ERROR: Could not find Object of id=" << id << endl;
	return false;
      }
      obj = it->second;
    }
    if ( !obj->_is_a( _tc->id() ) )
    {
      cerr << "ERROR: Object is not of required type " << _tc->id() << " but of type " << obj->_repoid() << " instead" << endl;
      return false;
    }
    cerr << "Packing Object of type " << _tc->name() << endl;
    _any <<= CORBA::Any::from_object( obj, _tc->name() );
    return true;
  }
  case CORBA::tk_sequence:
  {
    CORBA::ULong len = PyList_Size( _arg );
    const CORBA::TypeCode_var ctc = _tc->content_type();
    cerr << "Putting sequence of len " << len << endl;
    assert( _any.seq_put_begin( len ) );
    for ( CORBA::ULong i = 0; i < len; i++ )
    {
      PyObject* p = PyList_GetItem( _arg, i );
      if ( p == 0L )
      {
	cerr << "ERROR: Too few items in list." << endl;
	return false;
      }
      if ( !parseResult( _any, p, ctc ) )
	return false;
    }
    assert( _any.seq_put_end() );
    return true;
  }
  case CORBA::tk_struct:
  {
    cerr << "Packing " << _tc->id() << endl;
    assert( _any.struct_put_begin() );
    CORBA::ULong len = _tc->member_count();
    for( CORBA::ULong i = 0; i < len; i++ )
    {
      const CORBA::TypeCode_var ntc = _tc->member_type( i );
      PyObject* p = PyObject_GetAttrString( _arg, (char*)_tc->member_name( i ) );
      if ( p == 0L )
      {
	cerr << "ERROR: Did not find member " << (char*)_tc->member_name( i ) << " in struct" << endl;
	return false;
      }
      if ( !parseResult( _any, p, ntc ) )
	return false;
    }
    assert( _any.struct_put_end() );
    return true;
  }
  case CORBA::tk_union:
  {
    cerr << "Packing Union " << _tc->id() << endl;

    CORBA::TypeCode_var dt = _tc->discriminator_type();
    CORBA::Long defidx = _tc->default_index();
    assert( _any.union_put_begin() );

    bool found = false;
    CORBA::ULong idx;
    PyObject* disc = PyObject_GetAttrString( _arg, "d" );
    if ( disc == 0L || disc == Py_None )
    {
      cerr << "ERROR: Can not pack union " << _tc->id() << " because the descriminator is not set" << endl;
      return 0L;
    }
    for ( idx = 0; idx < _tc->member_count(); idx++ )
    {
      if ( defidx != -1 && (CORBA::ULong) defidx == idx )
	continue;
      
      CORBA::Any_var lany = _tc->member_label( idx );
      PyObject* label = parseResult( *lany, dt );
      if ( label == 0L )
      {
	cerr << "ERROR: Could not extract label of union " << _tc->id() << endl;
	return 0L;
      }
      if ( PyObject_Compare( label, disc ) == 0 )
      {
	found = true;
	Py_DECREF( label );
	break;
      }
      Py_DECREF( label );
    }

    if ( !parseResult( _any, disc, dt ) )
    {
      cerr << "ERROR: Could not pack discriminator for union " << _tc->id() << endl;
      return 0L;
    }
  
    PyObject* value = PyObject_GetAttrString( _arg, "v" );

    if ( found )
    {
      CORBA::TypeCode_var mtype = _tc->member_type( idx );
      assert( _any.union_put_selection( idx ) );
      if ( !parseResult( _any, value, mtype ) )
      {
	cerr << "ERROR: Could not pack value for union " << _tc->id() << endl;
	return 0L;
      }
    }
    else if ( defidx == -1 )
    {
      // Do not pack any value at all
    }
    else
    {
      CORBA::TypeCode_var mtype = _tc->member_type( idx );
      assert( _any.union_put_selection( idx ) );
      if ( !parseResult( _any, value, mtype ) )
      {
	cerr << "ERROR: Could not pack value for union " << _tc->id() << endl;
	return 0L;
      }
    }
    
    assert( _any.union_put_end() );
    
    return true;
  }
  case CORBA::tk_alias:
  {
    const CORBA::TypeCode_var ntc = _tc->content_type();
    return parseResult( _any, _arg, ntc );
  }
  /**
   * Not handled here ...
   * tk_sequence, tk_array, tk_alias, tk_except,
   */
  /* if ( _t == CORBA::_tc_wchar )
  if ( _t == CORBA::_tc_wstring )
  if ( _t == CORBA::_tc_Principal )
  if ( _t == CORBA::_tc_Context ) */
  default:
    assert( 0 );
  }
}

/**
 * Extract from Anys and return a python objects
 */
PyObject* parseResult( CORBA::Any& _any, CORBA::TypeCode_ptr _tc )
{
  if ( CORBA::is_nil( _tc ) )
  { 
    _tc = _any.type();
    assert( !CORBA::is_nil( _tc ) );
  }
  
  switch( _tc->kind() )
  {
  case CORBA::tk_void:
    return Py_None;
  case CORBA::tk_any:
    {
      PyObject* pany = pm_module->newAny();
      if ( pany == 0L )
      {
	cerr << "ERROR: Could not create a new any" << endl;
	return 0L;
      }
      int id = g_id++;
      CORBA::Any* any = new CORBA::Any();
      _any >>= *any;
      g_mapAny[ id ] = any;
      PyObject* p_id = Py_BuildValue( "i", id );
      PyObject_SetAttrString( pany, "_any_id_", p_id );
      Py_DECREF( p_id );
      return pany;
    }
  case CORBA::tk_enum:
    {
      CORBA::ULong i;
      assert( _any.enum_get( i ) );
      return Py_BuildValue( "s", (char *)_tc->member_name( i ) );
    }
  case CORBA::tk_string:
    {
      char* text;
      if ( !( _any >>= CORBA::Any::to_string( text, 0 ) ) )
	return 0L;
      PyObject* obj = Py_BuildValue( "s", text );
      CORBA::string_free( text );
      return obj;
    }
  case CORBA::tk_short:
    {
      CORBA::Short x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "i", (int)x );
    }
  case CORBA::tk_ushort:
    {
      CORBA::UShort x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "i", (int)x );
    }
  case CORBA::tk_long:
    {
      CORBA::Long x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "i", (int)x );
    }
  case CORBA::tk_ulong:
    {
      CORBA::ULong x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "i", (int)x );
    }
  case CORBA::tk_float:
    {
      CORBA::Float x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "f", x );
    }
  case CORBA::tk_double:
    {
      CORBA::Double x;
      if ( !( _any >>= x ) )
	return 0L;
      return Py_BuildValue( "d", x );
    }
  case CORBA::tk_objref:
    {
      CORBA::Object_var x;
      if ( !( _any >>= CORBA::Any::to_object( x ) ) )
	return 0L;

      if ( CORBA::is_nil( x ) )
      {
	cerr << "Extracting zero" << endl;
	return Py_None;
      }
      
      cerr << "Extracting type " << _tc->id() << endl;
      
      if ( !x->_is_a( _tc->id() ) )
      {  
	cerr << "ERROR: Object received is not of required type " << _tc->id() << " but of type "
	     << x->_repoid() << " instead" << endl;
	return 0L;
      }

      // Create a python object of the correct type
      string repoid = _tc->id() + 4;
      repoid.erase( repoid.rfind( ":" ) );
      int pos;
      while ( ( pos = repoid.find( "/" ) ) != -1 )
	repoid[pos] = '.';
      if ( strncmp( repoid.c_str(), "omg.org." , 8 ) == 0 )
	repoid.assign( repoid, 8 );
      int i = repoid.find( "." );
      if ( i == -1 )
      {
	cerr << "ERROR: Can not handle interfaces outside modules, sorry." << endl
	     << "       " << _tc->id() << endl;
	return 0L;
      }
      string module;
      module.assign( repoid, 0, i );
      string cl;
      cl.assign( repoid, i + 1 );
      cerr << "Creating Python class from module " << module << " and class " << cl << endl;
      PyObject* obj = pm_module->newObject( module.c_str(), cl.c_str() );
      if ( obj == 0L )
      {
	cerr << "ERROR: Could not create object of class " << cl << " from module " << module << endl;
	return 0L;
      }
      
      int id = g_id++;
      g_mapObjects[ id ] = x;
      PyObject* p_id = Py_BuildValue( "i", id );
      PyObject_SetAttrString( obj, "_id_", p_id );
      Py_DECREF( p_id );
      CORBA::String_var ior = pm_orb->object_to_string( x );
      PyObject* p_ior = Py_BuildValue( "s", ior.in() );
      PyObject_SetAttrString( obj, "_ior_", p_ior );
      Py_DECREF( p_ior );
      return obj;
    }
  case CORBA::tk_TypeCode:
    {
      CORBA::TypeCode_var x;
      if ( !( _any >>= x ) )
	return 0L;
      int id = g_id++;
      g_mapTypeCodes[ id ] = x;
      return Py_BuildValue( "i", id );
    }
  case CORBA::tk_boolean:
    {
      CORBA::Boolean x;
      if ( !( _any >>= CORBA::Any::to_boolean( x ) ) )
      {
	cerr << "ERROR: Could not extract boolean" << endl;
	return 0L;
      }
      return Py_BuildValue( "i", ( x ? 1 : 0 ) );
    }
  case CORBA::tk_octet:
    {
      CORBA::Octet x;
      if ( !( _any >>= CORBA::Any::to_octet( x ) ) )
	return 0L;
      return Py_BuildValue( "i", (int)x );
    }
  case CORBA::tk_char:
    {
      CORBA::Char x;
      if ( !( _any >>= CORBA::Any::to_char( x ) ) )
	return 0L;
      char buf[ 2 ];
      buf[0] = x;
      buf[1] = 0;
      return Py_BuildValue( "s", buf );
    }
  case CORBA::tk_sequence:
    {
      CORBA::ULong len;
      assert( _any.seq_get_begin( len ) );
      PyObject* seq = PyList_New( len );
      assert( seq != 0L );
      CORBA::TypeCode_var ctc = _tc->content_type();
      // HACK
      cerr << "content type=" << ctc->id() << endl;
      for( CORBA::ULong i = 0; i < len; i++ )
      {
	PyObject* p = parseResult( _any, ctc );
	if ( p == 0L )
	{
	  cerr << "Could not extract sequence item " << i << endl;
	  return 0L;
	}
	PyList_SetItem( seq, i, p );
      }
      assert( _any.seq_get_end() );
      return seq;
    }
  case CORBA::tk_struct:
    {
      cerr << "Extracting struct " << _tc->id() << endl;
      CORBA::ULong len = _tc->member_count();
      assert( _any.struct_get_begin() );
      PyObject* struc = pm_module->newStruct();
      assert( struc != 0L );
      for( CORBA::ULong i = 0; i < len; i++ )
      {
	CORBA::TypeCode_var ntc = _tc->member_type( i );
	PyObject* p = parseResult( _any, ntc );
	if ( p == 0L )
	  return 0L;
	PyObject_SetAttrString( struc, (char*)_tc->member_name( i ), p );
      }
      assert( _any.struct_get_end() );
      return struc;
    }
  case CORBA::tk_except:
    {
      cerr << "Extracting except " << _tc->id() << endl;
      CORBA::ULong len = _tc->member_count();
      CORBA::String_var repoid;
      assert( _any.except_get_begin( repoid ) );
      PyObject* args = PyTuple_New( len );
      assert( args != 0L );
      for( CORBA::ULong i = 0; i < len; i++ )
      {
	CORBA::TypeCode_var ntc = _tc->member_type( i );
	PyObject* p = parseResult( _any, ntc );
	if ( p == 0L )
	  return 0L;
	PyTuple_SetItem( args, i, p );
      }
      assert( _any.except_get_end() );

      // Create a python exception object of the correct type
      string repoid2 = _tc->id() + 4;
      repoid2.erase( repoid2.rfind( ":" ) );
      int pos;
      while ( ( pos = repoid2.find( "/" ) ) != -1 )
	repoid2[pos] = '.';
      if ( strncmp( repoid2.c_str(), "omg.org." , 8 ) == 0 )
	repoid2.assign( repoid2, 8 );
      int i = repoid2.find( "." );
      if ( i == -1 )
      {
	cerr << "ERROR: Can not handle interfaces outside modules, sorry." << endl
	     << "       " << _tc->id() << endl;
	return 0L;
      }
      string module;
      module.assign( repoid2, 0, i );
      string cl;
      cl.assign( repoid2, i + 1 );
      cerr << "Creating Python exception from module " << module << " and class " << cl << endl;
      PyObject* exc = pm_module->newException( module.c_str(), cl.c_str(), args );
      if ( exc == 0L )
      {
	cerr << "ERROR: Could not create exception of class " << cl << " from module " << module << endl;
	return 0L;
      }
      return exc;
    }
  case CORBA::tk_union:
    {
      cerr << "Extracting Union " << _tc->id() << endl;
      CORBA::TypeCode_var dt = _tc->discriminator_type();
      CORBA::Long defidx = _tc->default_index();
      
      assert( _any.union_get_begin() );

      PyObject* disc = parseResult( _any, dt );
      bool found = false;
      CORBA::ULong idx;
      if ( disc == 0L )
      {
	cerr << "ERROR: Could not extract discriminator of union " << _tc->id() << endl;
	return 0L;
      }
      for ( idx = 0; idx < _tc->member_count(); idx++ )
      {
	if ( defidx != -1 && (CORBA::ULong) defidx == idx )
	  continue;

	CORBA::Any_var lany = _tc->member_label( idx );
	PyObject* label = parseResult( *lany, dt );
	if ( label == 0L )
	{
	  cerr << "ERROR: Could not extract label of union " << _tc->id() << endl;
	  return 0L;
	}
	if ( PyObject_Compare( label, disc ) == 0 )
	{
	  found = true;
	  Py_DECREF( label );
	  break;
	}
	Py_DECREF( label );
      }
      
      PyObject* uni = 0L;

      if ( found )
      {
	CORBA::TypeCode_var mtype = _tc->member_type( idx );
	assert( _any.union_get_selection( idx ) );
	PyObject* memb = parseResult( _any, mtype );
	if ( memb == 0L )
	{
	  cerr << "ERROR: Could not extract value of union " << _tc->id() << endl;
	  return 0L;
	}
	uni = pm_module->newUnion();
	assert( uni != 0L );
	PyObject_SetAttrString( uni, "d", disc );
	PyObject_SetAttrString( uni, "v", memb );
      }
      else if ( defidx == -1 )
      {
	uni = pm_module->newUnion();
	assert( uni != 0L );
	PyObject_SetAttrString( uni, "d", Py_None );
	PyObject_SetAttrString( uni, "v", Py_None );
      }
      else
      {
	assert( defidx != -1 );
	CORBA::TypeCode_var deftype = _tc->member_type( (CORBA::ULong) defidx);
	assert( _any.union_get_selection( defidx ) );
	uni = pm_module->newUnion();
	assert( uni != 0L );
	PyObject* memb = parseResult( _any, deftype );
	if ( memb == 0L )
	{
	  cerr << "ERROR: Could not extract value of union " << _tc->id() << " using default discriminator" << endl;
	  return 0L;
	}
	PyObject_SetAttrString( uni, "d", Py_None );
	PyObject_SetAttrString( uni, "v", memb );
      }

      assert( _any.union_get_end() );

      return uni;
    }
  case CORBA::tk_alias:
    {
      const CORBA::TypeCode_var ntc = _tc->content_type();
      return parseResult( _any, ntc );
    }
  default:
    cerr << "INTERNAL ERROR: Did not handle tk_XXXX=" << (int)_tc->kind() << endl;
    assert( 0 );
  }
}

PyObject* pm_typecode( PyObject*, PyObject *args)
{
  const char* typecode;
  int typecodeid;
  
  if ( !PyArg_ParseTuple( args, "is", &typecodeid, &typecode ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  int id = g_id++;
  g_mapTypeCodes[ id ] = createTypeCode( typecodeid, typecode );
  
  return Py_BuildValue( "i", id );
}

PyObject* pm_request( PyObject*, PyObject *args)
{
  const char* objref;
  const char* method;
  int typecodeid;
  
  if ( !PyArg_ParseTuple( args, "ssi", &objref, &method, &typecodeid ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;

  CORBA::Object_var obj = pm_orb->string_to_object( objref );
  if ( CORBA::is_nil( obj ) )
  {
    cerr << "ERROR: Can not make call to NULL object reference" << endl;
    return NULL;
  }
  
  CORBA::Request_ptr req = obj->_request( method );
  int id = g_id++;
  g_mapRequests[ id ] = req;

  req->result()->value()->type( findTypeCode( typecodeid ) );
  
  return Py_BuildValue( "i", id );
}

PyObject* pm_add_param( PyObject*, PyObject *args)
{
  PyObject* value;
  const char* name;
  int id;
  int typecodeid;
  int inout;
  
  if ( !PyArg_ParseTuple( args, "isOii", &id, &name, &value, &typecodeid, &inout ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_REQ( it );

  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;
  
  CORBA::Any any( tc, (void*)NULL );
  if ( !parseResult( any, value, tc ) )
  {
    cerr << "ERROR: Could not marshall" << endl;
    return NULL;
  }

  if ( inout )
    (it->second)->add_inout_arg( name ) = any;
  else
    (it->second)->add_in_arg( name ) = any;

  return Py_None;
}

PyObject* pm_add_out_param( PyObject*, PyObject *args)
{
  const char* name;
  int id;
  int typecodeid;
  int no;
  
  if ( !PyArg_ParseTuple( args, "isii", &id, &name, &typecodeid, &no ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_REQ( it );

  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;
  
  (it->second)->add_out_arg( name );
  (it->second)->arguments()->item( no )->value()->type( tc );

  return Py_None;
}

PyObject* pm_invoke( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_REQ( it );

  cerr << "Invoking" << endl;
  (it->second)->invoke();
  cerr << "Invoked" << endl;

  // Exception? Just return a zero to avoid python errors
  // The stub will ask for exceptions
  if( (it->second)->env()->exception() )
    return Py_BuildValue( "i", 0 );
    
  PyObject* o = parseResult( *(it->second)->result()->value() );

  return o;
}

PyObject* pm_check_exception( PyObject*, PyObject *args)
{
  int id;
  int typecodeid;
  const char* repoid;
  
  if ( !PyArg_ParseTuple( args, "isi", &id, &repoid, &typecodeid ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }
  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;
  
  FIND_REQ( it );

  if( (it->second)->env()->exception() )
  {
    CORBA::Exception *_ex = (it->second)->env()->exception();
    CORBA::UnknownUserException *_uuex = CORBA::UnknownUserException::_narrow( _ex );
    if( _uuex )
    {
      if( strcmp( _uuex->_except_repoid(), repoid ) == 0L )
      {
	PyObject* o = parseResult( _uuex->exception( tc ), tc );
	return o;
      }
    }
  }
  
  // No exception of this kind
  return Py_BuildValue( "(i,i)", 0, 0 );
}

PyObject* pm_get_out_param( PyObject*, PyObject *args)
{
  int id;
  int typecodeid;
  int no;
  
  if ( !PyArg_ParseTuple( args, "iii", &id, &typecodeid, &no ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_REQ( it );

  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;
 
  cerr << "Out parameter " << no << " in progress" << endl;

  PyObject* o = parseResult( *((it->second)->arguments()->item( no )->value()), tc );

  return o;
}

PyObject* pm_release_request( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_REQ( it );

  g_mapRequests.erase( it );

  return Py_None;
}

PyObject* pm_server( PyObject*, PyObject *args)
{
  const char* repoid;
  
  if ( !PyArg_ParseTuple( args, "s", &repoid ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  CORBA::BOA::ReferenceData_var boaid = new CORBA::BOA::ReferenceData();
  
  int id = g_id++;
  
  PyMicoSkeleton* skel = new PyMicoSkeleton( repoid, boaid );
  g_mapSkels[ id ] = skel;

  return Py_BuildValue( "i", id );  
}

PyObject* pm_register_callback( PyObject*, PyObject *args)
{
  PyObject* obj;
  const char* func;
  int id;
  int typecodeid;
  PyObject* lst;
  
  if ( !PyArg_ParseTuple( args, "iOsiO", &id, &obj, &func, &typecodeid, &lst ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }
  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;
  
  FIND_SKEL( id, skel );
  
  skel->registerCallback( obj, func, tc, lst );

  return Py_None;
}

PyObject* pm_server_ior( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return Py_BuildValue( "i", PMERR_COULD_NOT_PARSE );
  }

  FIND_SKEL( id, skel );

  CORBA::String_var str;
  str = pm_orb->object_to_string( skel );
  
  return Py_BuildValue( "s", str.in() );
}

PyObject* pm_release_server( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return Py_BuildValue( "i", PMERR_COULD_NOT_PARSE );
  }

  FIND_SKEL( id, skel );

  CORBA::release( skel );
  
  return Py_None;
}

PyObject* pm_narrow( PyObject*, PyObject *args)
{
  const char* ior;
  const char* repoid;
  
  if ( !PyArg_ParseTuple( args, "ss", &ior, &repoid ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return Py_BuildValue( "i", PMERR_COULD_NOT_PARSE );
  }

  CORBA::Object_var obj = pm_orb->string_to_object( ior );
  if ( CORBA::is_nil( obj ) )
    return Py_BuildValue( "i", 0 );

  int res = 0;
  if ( obj->_is_a( repoid ) )
    res = 1;
  
  return Py_BuildValue( "i", res );
}

PyObject* pm_object_repoid( PyObject*, PyObject *args)
{
  const char* ior;
  
  if ( !PyArg_ParseTuple( args, "s", &ior ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  CORBA::Object_var obj = pm_orb->string_to_object( ior );
  if ( CORBA::is_nil( obj ) )
  {
    cerr << "ERROR: Could not turn string in object" << endl;
    return NULL;
  }
  
  return Py_BuildValue( "s", obj->_repoid() );
}

PyObject* pm_raise_exception( PyObject*, PyObject *args)
{
  int id;
  int completed;
  CORBA::ULong minor;
  const char* repoid;
  int anyid;
  
  if ( !PyArg_ParseTuple( args, "iiisi", &id, &completed, &minor, &repoid, &anyid ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  FIND_SKEL( id, skel );

  CORBA::CompletionStatus cs;
  if ( completed == 1 )
    cs = CORBA::COMPLETED_YES;
  else if ( completed == 2 )
    cs = CORBA::COMPLETED_NO;
  else if ( completed == 3 )
    cs = CORBA::COMPLETED_MAYBE;

  // TODO: Get any
  // HACK
  CORBA::Any any;
  any <<= (CORBA::Long)123;
  
  skel->raiseException( repoid, any, cs, minor );
  
  return Py_None;
}

PyObject* pm_object( PyObject*, PyObject *args)
{
  const char* ior;
  
  if ( !PyArg_ParseTuple( args, "s", &ior ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  CORBA::Object_var obj = pm_orb->string_to_object( ior );
  if ( CORBA::is_nil( obj ) )
    return Py_BuildValue( "i", 0 );

  int id = g_id++;
  g_mapObjects[ id ] = obj;
    
  return Py_BuildValue( "i", id );
}

PyObject* pm_release_object( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  map<int,CORBA::Object_var>::iterator it = g_mapObjects.find( id );
  if ( it == g_mapObjects.end() )
  {
    cerr << "ERROR: Unknown object id " << id << endl;
    return NULL;
  }
  
  g_mapObjects.erase( it );
  
  return Py_None;
}

PyObject* pm_bind( PyObject*, PyObject *args)
{
  const char* repoid;
  const char* tag;
  const char* address;
  
  if ( !PyArg_ParseTuple( args, "sss", &repoid, &tag, &address ) )
  {
    cerr << "ERROR: Could not parse" << endl;
    return NULL;
  }

  if ( strlen( address ) == 0 )
    address = 0L;

  CORBA::Object_var obj;
  
  CORBA::ORB::ObjectTag_var vtag;
  if ( strlen( tag ) > 0 )
  {    
    vtag = pm_orb->string_to_tag( tag );
    obj = pm_orb->bind( repoid, vtag, address );
  }
  else
    obj = pm_orb->bind( repoid, address );

  PyObject* o = pm_module->newObject();
  assert( o != 0L );

  if ( CORBA::is_nil( obj ) )
  {
    cerr << "Could not bind" << endl;
    PyObject* p_id = Py_BuildValue( "i", 0 );
    PyObject_SetAttrString( o, "_id_", p_id );
    Py_DECREF( p_id );
    return o;
  }
  
  int id = g_id++;
  g_mapObjects[ id ] = obj;
  PyObject* p_id = Py_BuildValue( "i", id );
  PyObject_SetAttrString( o, "_id_", p_id );
  Py_DECREF( p_id );
  CORBA::String_var ior = pm_orb->object_to_string( obj );
  PyObject* p_ior = Py_BuildValue( "s", ior.in() );
  PyObject_SetAttrString( o, "_ior_", p_ior );
  Py_DECREF( p_ior );
  
  return o;
}

PyObject* pm_any( PyObject*, PyObject *args)
{
  int typecodeid;
  PyObject* value;
  
  if ( !PyArg_ParseTuple( args, "iO", &typecodeid, &value ) )
  {
    cerr << "ERROR: Could not parse pm_any" << endl;
    return NULL;
  }

  CORBA::TypeCode_ptr tc = findTypeCode( typecodeid );
  if ( CORBA::is_nil( tc ) )
    return NULL;

  CORBA::Any* any = new CORBA::Any;
  any->type( tc );
  
  if ( !parseResult( *any, value, tc ) )
  {
    Py_DECREF( value );
    delete any;
    return 0L;
  }
  
  int id = g_id++;
  g_mapAny[ id ] = any;
  
  return Py_BuildValue( "i", id );
}

PyObject* pm_release_any( PyObject*, PyObject *args)
{
  int id;
  
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse pm_release_any" << endl;
    return NULL;
  }

  map<int,CORBA::Any*>::iterator it = g_mapAny.find( id );
  if ( it == g_mapAny.end() )
  {
    cerr << "ERROR: Any of id " << id << " is not registered" << endl;
    return 0L;
  }
  
  delete it->second;
  g_mapAny.erase( it );
  
  return Py_None;
}

PyObject* pm_any_value( PyObject*, PyObject *args)
{
  int id;
 
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse pm_any_value" << endl;
    return NULL;
  }

  map<int,CORBA::Any*>::iterator it = g_mapAny.find( id );
  if ( it == g_mapAny.end() )
  {
    cerr << "ERROR: Any of id " << id << " is not registered" << endl;
    return 0L;
  }

  CORBA::TypeCode_var tc = it->second->type();
  PyObject* value = parseResult( *(it->second), tc );
  return value;
}

PyObject* pm_any_typecode( PyObject*, PyObject *args)
{
  int id;
 
  if ( !PyArg_ParseTuple( args, "i", &id ) )
  {
    cerr << "ERROR: Could not parse pm_any_value" << endl;
    return NULL;
  }

  map<int,CORBA::Any*>::iterator it = g_mapAny.find( id );
  if ( it == g_mapAny.end() )
  {
    cerr << "ERROR: Any of id " << id << " is not registered" << endl;
    return 0L;
  }

  CORBA::TypeCode_var tc = it->second->type();
  string stc = tc->stringify();
  return Py_BuildValue( "s", stc.c_str() );
}
