
//**************************************************************************
//**
//** i_cdmus.c
//**
//**************************************************************************

// HEADER FILES ------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <linux/cdrom.h>
#include "h2def.h"
#include "i_sound.h"
#include "i_cdmus.h"

// MACROS ------------------------------------------------------------------

#define CDROM_DEVICE "/dev/cdrom"


// EXTERNAL FUNCTION PROTOTYPES --------------------------------------------

// PUBLIC FUNCTION PROTOTYPES ----------------------------------------------

// PRIVATE FUNCTION PROTOTYPES ---------------------------------------------


// EXTERNAL DATA DECLARATIONS ----------------------------------------------

// PUBLIC DATA DEFINITIONS -------------------------------------------------
char *cd_device;
int cd_fd;
int cd_Error;

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static int cd_FirstDrive;
static int cd_CurDrive;
static int cd_DiscStatus;
static int cd_DeviceStatus;
static int cd_FirstTrack;
static int cd_LastTrack;
static int cd_TrackCount;

static int OkInit = 0;



static boolean cd_MedChng;
// CODE --------------------------------------------------------------------

//==========================================================================
//
// I_CDMusInit
//
// Initializes the CD audio system.  Must be called before using any
// other I_CDMus functions.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusInit(void)
{
	int i=0;
	struct cdrom_tochdr tochdr;

	boolean retry=true;
	
//	struct cdrom_subchnl readsub;
	
	if(OkInit != 1)
	{ // Only execute if uninitialized

		cd_device=(char*)malloc
(sizeof(char*)*(strlen(CDROM_DEVICE)+1));
		sprintf(cd_device,"%s",CDROM_DEVICE);
		cd_fd=open(cd_device,O_NONBLOCK);
		if(cd_fd == -1)
		{
			cd_Error = CDERR_NOTINSTALLED;
			return -1;
		}
		cd_FirstDrive = cd_fd;
		cd_CurDrive = cd_FirstDrive;
		
		while ((retry && i<2)) 
		{
		    cd_DeviceStatus = ioctl(cd_fd,CDROM_DRIVE_STATUS,
							CDSL_CURRENT);
		    if (cd_DeviceStatus != CDS_DISC_OK)
		    {
			switch (cd_DeviceStatus)
			{
			    case CDS_NO_DISC: 
				cd_Error = CDERR_NOTINSTALLED;
				return -1;
			    case CDS_TRAY_OPEN:
				ioctl(cd_fd,CDROMCLOSETRAY);
				retry = true;
				i++;
				break;
			    case CDS_DRIVE_NOT_READY:
				retry = true;
				i++;
				break;
			    default: 
				i++;
				retry = true; // we have no info or ioctl error
			    
			}
		    } 
		    else retry = false;
		}
		if (retry) 
		{
		    	cd_Error = CDERR_NOTINSTALLED;
			return -1;
		}
		cd_DiscStatus = ioctl(cd_fd,CDROM_DISC_STATUS);
		if(cd_DiscStatus==CDS_DATA_1 || cd_DiscStatus==CDS_DATA_2)
		{
			cd_Error = CDERR_NOAUDIOTRACKS;
			return -1;
		}
		
		I_CDMusStopMotor();

		ioctl(cd_fd,CDROMSTART);
	
	}

	// Force audio to stop playing
	// Make sure we have the current TOC
	cd_MedChng = ioctl(cd_fd,CDROM_MEDIA_CHANGED, CDSL_CURRENT);

	// Set track variables
	ioctl(cd_fd,CDROMREADTOCHDR,&tochdr);
	cd_FirstTrack = tochdr.cdth_trk0;
	cd_LastTrack = tochdr.cdth_trk1;
	if(cd_FirstTrack == 0 && cd_FirstTrack == cd_LastTrack)
	{
		cd_Error = CDERR_NOAUDIOTRACKS;
		return -1;
	}
	cd_TrackCount = cd_LastTrack-cd_FirstTrack+1;

	OkInit = 1;
	return 0;
}

//==========================================================================
//
// I_CDMusPlay
//
// Play an audio CD track.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusPlay(int track)
{
    struct cdrom_ti trks;
    if (I_CDMusLastTrack() == -1) return -1;
    if(track < cd_FirstTrack || track > cd_LastTrack)
    {
	cd_Error = CDERR_BADTRACK;
	return(-1);
    }
    trks.cdti_trk0 = trks.cdti_trk1 = track;
    trks.cdti_ind0 = trks.cdti_ind1 = 1;
    ioctl(cd_fd,CDROMPLAYTRKIND,&trks);
    return(0);
}

//==========================================================================
//
// I_CDMusStop
//
// Stops the playing of an audio CD.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusStop(void)
{
    return ioctl(cd_fd,CDROMPAUSE);
}

int I_CDMusStopMotor(void)
{
    return ioctl(cd_fd,CDROMSTOP);
}

//==========================================================================
//
// I_CDMusResume
//
// Resumes the playing of an audio CD.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusResume(void)
{
    return ioctl(cd_fd,CDROMRESUME);
}

//==========================================================================
//
// I_CDMusSetVolume
//
// Sets the CD audio volume (0 - 255).
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusSetVolume(int volume)
{

    struct cdrom_volctrl vol;
    if(!OkInit)
    {
	cd_Error = CDERR_NOTINSTALLED;
	return -1;
    }
    vol.channel0 =
    vol.channel1 =
    vol.channel2 = 
    vol.channel3 = volume;
    ioctl(cd_fd,CDROMVOLCTRL,&vol);
    return 0;
}

//==========================================================================
//
// I_CDMusFirstTrack
//
// Returns: the number of the first track.
//
//==========================================================================

int I_CDMusFirstTrack(void)
{
    struct cdrom_tochdr tochdr;
    if (ioctl(cd_fd,CDROM_MEDIA_CHANGED, CDSL_CURRENT))
    {
	ioctl(cd_fd,CDROMREADTOCHDR,&tochdr);
	cd_FirstTrack = tochdr.cdth_trk0;
	cd_LastTrack = tochdr.cdth_trk1;
	if(cd_FirstTrack == 0 && cd_FirstTrack == cd_LastTrack)
	{
		cd_Error = CDERR_NOAUDIOTRACKS;
		return -1;
	}
	cd_TrackCount = cd_LastTrack-cd_FirstTrack+1;
    }
    return cd_FirstTrack;
}

//==========================================================================
//
// I_CDMusLastTrack
//
// Returns: the number of the last track.
//
//==========================================================================

int I_CDMusLastTrack(void)
{
    struct cdrom_tochdr tochdr;
    if (ioctl(cd_fd,CDROM_MEDIA_CHANGED, CDSL_CURRENT))
    {
	ioctl(cd_fd,CDROMREADTOCHDR,&tochdr);
	cd_FirstTrack = tochdr.cdth_trk0;
	cd_LastTrack = tochdr.cdth_trk1;
	if(cd_FirstTrack == 0 && cd_FirstTrack == cd_LastTrack)
	{
		cd_Error = CDERR_NOAUDIOTRACKS;
		return -1;
	}
	cd_TrackCount = cd_LastTrack-cd_FirstTrack+1;
    }
    return cd_LastTrack;
}

//==========================================================================
//
// I_CDMusTrackLength
//
// Returns: Length of the given track in seconds, or -1 (error, in
// cd_Error).
//
//==========================================================================

int I_CDMusTrackLength(int track)
{
	struct cdrom_msf0 prev_te;
	struct cdrom_tocentry te;
	int mins,secs;
	if (I_CDMusFirstTrack() == -1) return 0; // Check if media changed
	if(track < cd_FirstTrack || track > cd_LastTrack)
	{
		cd_Error = CDERR_BADTRACK;
		return -1;
	}
	te.cdte_track = track;
	te.cdte_format = CDROM_MSF;
	ioctl(cd_fd,CDROMREADTOCENTRY,&te);
	prev_te = te.cdte_addr.msf;
	if (track == cd_LastTrack) 
	    te.cdte_track = CDROM_LEADOUT;
	else
	    te.cdte_track = track + 1;    
	te.cdte_format = CDROM_MSF;   // Do we really need this?
	ioctl(cd_fd,CDROMREADTOCENTRY,&te);
	mins = te.cdte_addr.msf.minute - prev_te.minute;
	secs = te.cdte_addr.msf.second - prev_te.second;
	if (secs < 0) 
	{
	    mins--;
	    secs +=60;
	}
	return mins*60
 + secs-2;
}
