/*
******************************************************************************

   LibGGI display code for Doom and Heretic

   Copyright (C) 1999	Marcus Sundberg [marcus@ggi-project.org]
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/



#include <stdlib.h>
#include <ctype.h>
#include <ggi/ggi.h>

#define SHOW_FPS

#define I_GGI_HERETIC 
#ifdef I_GGI_HERETIC
void R_InitBuffer (int width, int height);
#endif

#include "h2def.h"
#include "am_map.h"
#if defined(I_GGI_HERETIC)
#define SCREEN0_PTR	(screen)
#endif

/* Needed by reset_framecounter() */
static ggi_visual_t ggivis=NULL;
static byte* pcscreen;
extern int screenwidth,screenheight;
extern int ticcount;
extern int DisplayTicker;
#ifdef SHOW_FPS
#include <unistd.h>
#include <sys/time.h>

static struct timeval	starttime;
static long		totalframes;
static int		showfps = 0;

static void reset_framecounter(void)
{
	ggi_color black = { 0x0, 0x0, 0x0 };
	ggi_color white = { 0xffff, 0xffff, 0xffff };

	/* Set text colors */
	ggiSetGCForeground(ggivis, ggiMapColor(ggivis, &white));
	ggiSetGCBackground(ggivis, ggiMapColor(ggivis, &black));

	totalframes = 0;
	gettimeofday(&starttime, NULL);
}
#endif

static int     lastmousex = 0;
static int     lastmousey = 0;
boolean        mousemoved = false;
int            buttonstate=0;

static int     realwidth, realheight;
static int     doublebuffer;
static int     scale;
static int     stride;
static int     pixelsize;

static const ggi_directbuffer *dbuf1, *dbuf2;
static int     usedbuf, havedbuf;
static void   *frameptr[2] = { NULL, NULL };
static void   *oneline = NULL;
static void   *palette = NULL;
static int     curframe = 0;
static int     modexrefresh=0;


static inline void
do_scale8(int xsize, int ysize, uint8 *dest, uint8 *src)
{
	int i, j, destinc = stride*2-xsize*2;
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; /* i is incremented below */) {
			register uint32 pix1 = src[i++], pix2 = src[i++];
#ifdef GGI_LITTLE_ENDIAN
			*((uint32 *) (dest + stride))
				= *((uint32 *) dest)
				= (pix1 | (pix1 << 8)
				   | (pix2 << 16) | (pix2 << 24));
#else
			*((uint32 *) (dest + stride))
				= *((uint32 *) dest)
				= (pix2 | (pix2 << 8)
				   | (pix1 << 16) | (pix1 << 24));
#endif
			dest += 4;
		}
		dest += destinc;
		src += xsize;
	}
}

static inline void
do_scale16(int xsize, int ysize, uint8 *dest, uint8 *src)
{
	int i, j, destinc = stride*2-xsize*4;
	uint16 *palptr = palette;
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; /* i is incremented below */) {
			register uint32 pixel = palptr[src[i++]];
			*((uint32 *) (dest + stride))
				= *((uint32 *) dest)
				= pixel | (pixel << 16);
			dest += 4;
		}
		dest += destinc;
		src += xsize;
	}
}

static inline void
do_scale32(int xsize, int ysize, uint8 *dest, uint8 *src)
{
	int i, j, destinc = stride*2-xsize*8;
	uint32 *palptr = palette;
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; /* i is incremented below */) {
			register uint32 pixel = palptr[src[i++]];
			*((uint32 *) (dest + stride))
				= *((uint32 *) (dest)) = pixel;
			dest += 4;
			*((uint32 *) (dest + stride))
				= *((uint32 *) (dest)) = pixel;
			dest += 4;
		}
		dest += destinc;
		src += xsize;
	}
}


static inline void
do_copy8(int xsize, int ysize, uint8 *dest, uint8 *src)
{
	int i, j;
	uint8 *palptr = palette;
	
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; i++) {
			dest[i] = palptr[src[i]];
		}
		dest += stride;
		src += xsize;
	}
}

static inline void
do_copy16(int xsize, int ysize, uint16 *dest, uint8 *src)
{
	int i, j, destinc = stride/2;
	uint16 *palptr = palette;
	
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; i++) {
			dest[i] = palptr[src[i]];
		}
		dest += destinc;
		src += xsize;
	}
}

static inline void
do_copy32(int xsize, int ysize, uint32 *dest, uint8 *src)
{
	int i, j, destinc = stride/4;
	uint32 *palptr = palette;
	
	for (j = 0; j < ysize; j++) {
		for (i = 0; i < xsize; i++) {
			dest[i] = palptr[src[i]];
		}
		dest += destinc;
		src += xsize;
	}
}


int key(int label, int sym)
{
	int rc=0;
	switch(label) {
	case GIIK_CtrlL:  case GIIK_CtrlR:  rc=KEY_RCTRL; 	break;
	case GIIK_ShiftL: case GIIK_ShiftR: rc=KEY_RSHIFT;	break;
	case GIIK_MetaL:  case GIIK_MetaR:
	case GIIK_AltL:   case GIIK_AltR:   rc=KEY_RALT;	break;
		
	case GIIUC_BackSpace:	rc = KEY_BACKSPACE;	break;
	case GIIUC_Escape:	rc = KEY_ESCAPE;	break;
#ifdef I_GGI_HERETIC
	case GIIK_Delete:
	case GIIK_PDecimal:	rc = KEY_DEL;	break;
	case GIIK_Insert:
	case GIIK_P0:		rc = KEY_INS;	break;
	case GIIK_PageUp:
	case GIIK_P9:		rc = KEY_PGUP;	break;
	case GIIK_PageDown:
	case GIIK_P3:		rc = KEY_PGDN;	break;
	case GIIK_Home:	
	case GIIK_P7:		rc = KEY_HOME;	break;
	case GIIK_End:
	case GIIK_P1:		rc = KEY_END;	break;
#endif /* I_GGI_HERETIC */
	case GIIUC_Tab:	rc = KEY_TAB;		break;
	case GIIK_Up:	
	case GIIK_P8:	rc = KEY_UPARROW;	break;
	case GIIK_Down:
	case GIIK_P2:	rc = KEY_DOWNARROW;	break;
	case GIIK_Left:	
	case GIIK_P4:	rc = KEY_LEFTARROW;	break;
	case GIIK_Right:
	case GIIK_P6:	rc = KEY_RIGHTARROW;	break;
	case GIIK_Enter:rc = KEY_ENTER;		break;
	case GIIK_F1:	rc = KEY_F1;		break;
	case GIIK_F2:	rc = KEY_F2;		break;
	case GIIK_F3:	rc = KEY_F3;		break;
	case GIIK_F4:	rc = KEY_F4;		break;
	case GIIK_F5:	rc = KEY_F5;		break;
	case GIIK_F6:	rc = KEY_F6;		break;
	case GIIK_F7:	rc = KEY_F7;		break;
	case GIIK_F8:	rc = KEY_F8;		break;
	case GIIK_F9:	rc = KEY_F9;		break;
	case GIIK_F10:	rc = KEY_F10;		break;
	case GIIK_F11:	rc = KEY_F11;		break;
	case GIIK_F12:	rc = KEY_F12;		break;
	case GIIK_Pause:rc = KEY_PAUSE;		break;
	case GIIK_PPlus: rc = (int)'+';		break;
	case GIIK_PMinus: rc = (int)'-';	break;
	default:
		/* Must use label here, or it won't work when shift is down */
		if ((label >= '0' && label <= '9') ||
		    (label >= 'A' && label <= 'Z') ||
		    label == '.' ||
		    label == ',') {
			/* We want lowercase */
			rc = tolower(label);
		} else if (sym < 256) {
			/* ASCII key - we want those */
			rc = sym;
			switch (sym) {
				/* Some special cases */
			case '+': rc = KEY_EQUALS;	break;
			case '-': rc = KEY_MINUS;
			default:			break;
			}
		}
	}
	return rc;
}

void I_ShutdownGraphics(void)
{
	if (ggivis != NULL) {
		if (!usedbuf) {
			free(SCREEN0_PTR);
		}
		if (oneline) {
			free(oneline);
			oneline = NULL;
		}
		if (palette) {
			free(palette);
			palette = NULL;
		}
		ggiClose(ggivis);
		ggivis = NULL;
	}
	ggiExit();
}


void I_StartFrame (void)
{
}

void I_GetEvent(void)
{
	event_t event;
	ggi_event ev;
	ggi_event_mask mask;
		
	while (1) {
		struct timeval t={0,0};
		mask = ggiEventPoll(ggivis,emAll,&t);
		if (!mask) return;
		
		ggiEventRead(ggivis,&ev,emAll);
		switch (ev.any.type)
			{
			case evKeyPress:
				event.type = ev_keydown;
				event.data1 = key(ev.key.label,ev.key.sym);
#ifdef SHOW_FPS
				if (event.data1 == KEY_BACKSPACE &&
				    gamestate == GS_LEVEL) {
					/* Toggle and reset the FPS counter */
					showfps = !showfps;
					reset_framecounter();
				}
#endif
				H2_PostEvent(&event);
				break;
			case evKeyRelease:
				event.type = ev_keyup;
				event.data1 = key(ev.key.label,ev.key.sym);
				H2_PostEvent(&event);
				break;
			case evPtrButtonPress:
				event.type = ev_mouse;
				buttonstate = event.data1 = 
					buttonstate | 1<<(ev.pbutton.button-1);
				event.data2 = event.data3 = 0;
				H2_PostEvent(&event);
				break;
			case evPtrButtonRelease:
				event.type = ev_mouse;
				buttonstate = event.data1 = 
					buttonstate ^ 1<<(ev.pbutton.button-1);
				event.data2 = event.data3 = 0;
				H2_PostEvent(&event);
				break;
			case evPtrAbsolute:
				event.type = ev_mouse;
				event.data1 = buttonstate;
				event.data2 = (ev.pmove.x - lastmousex) << 2;
				event.data3 = (lastmousey - ev.pmove.y) << 2;
				
				if (event.data2 || event.data3) {
					lastmousex = ev.pmove.x;
					lastmousey = ev.pmove.y;
					if (ev.pmove.x != screenwidth/2 &&
					    ev.pmove.y != screenheight/2) {
						H2_PostEvent(&event);
						mousemoved = false;
					} else {
						mousemoved = true;
					}
				}
				break;
			case evPtrRelative:
				event.type = ev_mouse;
				event.data1 = buttonstate;
				event.data2 = ev.pmove.x << 2;
				event.data3 = -ev.pmove.y << 2;
				
				if (event.data2 || event.data3) {
					lastmousex += ev.pmove.x;
					lastmousey += ev.pmove.y;
					if (ev.pmove.x != screenwidth/2 &&
					    ev.pmove.y != screenheight/2) {
						H2_PostEvent(&event);
						mousemoved = false;
					} else {
						mousemoved = true;
					}
				}
				break;
			}
	}
}

void I_StartTic (void)
{
	if (ggivis) I_GetEvent();
}

void I_UpdateNoBlit (void)
{

}
void FinishUpdate(void)
{

	if (!usedbuf) {
		int	i;

		if (scale) {
			switch (pixelsize) {
			case 1:	if (havedbuf) {
				do_scale8(screenwidth, screenheight,
					  frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_scale8(screenwidth, 1, oneline,buf);
					ggiPutBox(ggivis, 0, i*2, realwidth,
						  2, oneline);
					buf += screenwidth;
				}
			}
			break;
			case 2: if (havedbuf) {
				do_scale16(screenwidth, screenheight,
					   frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_scale16(screenwidth, 1,
						   oneline, buf);
					ggiPutBox(ggivis, 0, i*2, realwidth,
						  2, oneline);
					buf += screenwidth;
				}
			}
			break;
			case 4: if (havedbuf) {
				do_scale32(screenwidth, screenheight,
					   frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_scale32(screenwidth, 1,
						   oneline, buf);
					ggiPutBox(ggivis, 0, i*2, realwidth,
						  2, oneline);
					buf += screenwidth;
				}
			}
			break;
			}
		} else if (palette) {
			switch (pixelsize) {
			case 1:	if (havedbuf) {
				do_copy8(screenwidth, screenheight,
					 frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_copy8(screenwidth, 1, oneline,buf);
					ggiPutBox(ggivis, 0, i, realwidth,
						  1, oneline);
					buf += screenwidth;
				}
			}
			break;
			case 2: if (havedbuf) {
				do_copy16(screenwidth, screenheight,
					  frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_copy16(screenwidth, 1,
						  oneline, buf);
					ggiPutBox(ggivis, 0, i, realwidth,
						  1, oneline);
					buf += screenwidth;
				}
			}
			break;
			case 4: if (havedbuf) {
				do_copy32(screenwidth, screenheight,
					  frameptr[curframe], SCREEN0_PTR);
			} else {
				uint8 *buf = SCREEN0_PTR;
				for (i=0; i < screenheight; i++) {
					do_copy32(screenwidth, 1,
						  oneline, buf);
					ggiPutBox(ggivis, 0, i, realwidth,
						  1, oneline);
					buf += screenwidth;
				}
			}
			break;
			}
		} else if (!modexrefresh) {
			/* faster, but ugly in ModeX modes */
			ggiPutBox(ggivis, 0, 0, screenwidth, screenheight,
				  SCREEN0_PTR);
		} else {
			/* slower, but nicer in ModeX modes */
			uint8 *buf = SCREEN0_PTR;
			for (i = 0; i < screenheight; i++) {
				ggiPutHLine(ggivis, 0, i, screenwidth, buf);
				buf += screenwidth;
			}
		}
	} else if (!doublebuffer)
	{
	    memcpy(pcscreen,screen,screenwidth*screenheight);
	}
#ifdef SHOW_FPS
	if (showfps) {
		struct timeval curtime;
		double diff;
		char str[64];
		
		totalframes++;
		gettimeofday(&curtime, NULL);
		diff = (curtime.tv_sec - starttime.tv_sec);
		diff += ((double)curtime.tv_usec - starttime.tv_usec)/1000000;
		if (diff != 0) {
			sprintf(str, "FPS: %.1f", totalframes/diff);
			ggiPuts(ggivis, 1, 1, str);
		}
	}
#endif
	if (doublebuffer) {
		ggiSetDisplayFrame(ggivis, curframe);
		curframe = !curframe;
		if (usedbuf) {
			SCREEN0_PTR = frameptr[curframe];
			/* Ouch, we need to recalculate the line offsets */
			R_InitBuffer(scaledviewwidth, viewheight);
		}
		ggiSetWriteFrame(ggivis, curframe);
	}
	
	ggiFlush(ggivis);

}


extern int screenblocks;
extern int UpdateState;
static int lasttic;
void I_Update (void)
{

	int i;
	byte *dest;
	int tics;
	

	if (!pcscreen) pcscreen = (byte *)SCREEN0_PTR;
//
// blit screen to video
//
	if(DisplayTicker)
	{
		if(screenblocks > 9 || UpdateState&(I_FULLSCRN|I_MESSAGES))
		{
			dest = (byte *)screen;
		}
		else
		{
			dest = (byte *)pcscreen;
		}
		ticcount = I_GetTime();
		tics = ticcount-lasttic;
		lasttic = ticcount;
		if(tics > 20)
		{
			tics = 20;
		}
		for(i = 0; i < tics; i++)
		{
			*dest = 0xff;
			dest += 2;
		}
		for(i = tics; i < 20; i++)
		{
			*dest = 0x00;
			dest += 2;
		}
	}

	//memset(pcscreen, 255, SCREENHEIGHT*SCREENWIDTH);

	if(UpdateState == I_NOUPDATE)
	{
		return;
	}
//	if(UpdateState&I_FULLSCRN)
//	{
		FinishUpdate();
		UpdateState = I_NOUPDATE; // clear out all draw types
//	}
}

void I_SetPalette (byte* pal)
{
	int i;
	ggi_color col[256];
	byte *gamma = gammatable[usegamma];
	
	for (i = 0; i < 256; i++) {
		col[i].r = gamma[(*pal++)] << (GGI_COLOR_PRECISION-8);
		col[i].g = gamma[(*pal++)] << (GGI_COLOR_PRECISION-8);
		col[i].b = gamma[(*pal++)] << (GGI_COLOR_PRECISION-8);
	}
	if (palette) {
		ggiPackColors(ggivis, palette, col, 256);
	} else {
		ggiSetPalette(ggivis, 0, 256, col);
	}
}

void I_CheckRes()
{
}

void InitGraphLib(void)
{
}

void I_InitGraphics(void)
{
	ggi_mode mode;

	fprintf(stderr, "I_InitGraphics: Init GGI-visual.\n");

	if (ggiInit() < 0) I_Error("Unable to init LibGGI!\n");
	if ((ggivis = ggiOpen(NULL)) == NULL) {
		I_Error("Unable to open default visual!\n");
	}
	ggiSetFlags(ggivis, GGIFLAG_ASYNC);
	ggiSetEventMask(ggivis, emKey | emPointer);

	modexrefresh = M_CheckParm("-modex") > 0;
	doublebuffer = M_CheckParm("-doublebuffer") > 0;
	scale = M_CheckParm("-scale") > 0;

	realwidth = screenwidth;
	realheight = screenheight;
	if (scale) {
		realwidth *= 2;
		realheight *= 2;
	}

	if ((!doublebuffer || ggiCheckSimpleMode(ggivis, realwidth, realheight,
						 2, GT_8BIT, &mode) < 0) &&
	    (ggiCheckSimpleMode(ggivis, realwidth, realheight, GGI_AUTO,
				GT_8BIT, &mode) < 0) &&
#if defined(I_GGI_HERETIC)
/* Heretic still only works reliably in 320x200 (640x400 if scaled) */
	    (ggiCheckSimpleMode(ggivis, realwidth, realheight, GGI_AUTO,
				GT_AUTO, &mode) < 0 )) {
		I_Error("Can't set %ix%i mode\n",
			realwidth, realheight);
#elif defined(I_GGI_DOOM)
		(ggiCheckSimpleMode(ggivis, realwidth, realheight, GGI_AUTO,
				    GT_AUTO, &mode) < 0) &&
			(mode.visible.x > MAXSCREENWIDTH ||
			 mode.visible.y > MAXSCREENHEIGHT)) {
					I_Error("Can't find a suitable mode\n",
						realwidth, realheight);
#endif
				}
	if (ggiSetMode(ggivis, &mode) < 0) {
		I_Error("LibGGI can't set any modes at all?!\n");
	}
    
	realwidth  = mode.visible.x;
	realheight = mode.visible.y;
	if (scale) {
		screenwidth  = realwidth / 2;
		screenheight = realheight / 2; 
	} else {
		screenwidth  = realwidth;
		screenheight = realheight;
	}
	
	pixelsize = (GT_SIZE(mode.graphtype)+7) / 8;
	if (mode.graphtype != GT_8BIT) {
		if ((palette = malloc(pixelsize*256)) == NULL) {
			I_Error("Unable to allocate memory?!\n");
		}
	}

	
	usedbuf = havedbuf = 0;
	stride = realwidth*pixelsize;
	if ((dbuf1 = ggiDBGetBuffer(ggivis, 0)) != NULL &&
	    (dbuf1->type & GGI_DB_SIMPLE_PLB) &&
	    (doublebuffer ? ((dbuf2 = ggiDBGetBuffer(ggivis, 1)) != NULL &&
			     (dbuf2->type & GGI_DB_SIMPLE_PLB)) : 1)) {
		havedbuf = 1;
		frameptr[0] = dbuf1->write;
		if (doublebuffer) {
			frameptr[1] = dbuf2->write;
		} else {
			frameptr[1] = frameptr[0];
		}
		stride = dbuf1->buffer.plb.stride;
		if (!scale && !palette && stride == pixelsize*realwidth){
			usedbuf = 1;
			pcscreen = frameptr[0];
			fprintf(stderr,
				"I_InitGraphics: Using DirectBuffer with");
			if (doublebuffer)	{
				fprintf(stderr, " doublebuffering\n");
			} else {
				fprintf(stderr, " singlebuffering\n");
			}
		}
	}
	if ((SCREEN0_PTR = malloc(screenwidth*screenheight)) == NULL) 
	{
			I_Error("Unable to allocate memory?!\n");
	}
	if (!usedbuf) {
		
		if (!havedbuf && (scale || palette)) {
			int linesize = pixelsize*realwidth;
			if (scale) linesize *= 4;
			if ((oneline = malloc(linesize)) == NULL) {
				I_Error("Unable to allocate memory?!\n");
			}
		}
		fprintf(stderr,
			"I_InitGraphics: Drawing into offscreen memory\n");
	}
	/* We will start drawing to frame 0, and start displaying frame 1 */
	if (doublebuffer)	{
		ggiSetWriteFrame(ggivis, 0);
		ggiSetDisplayFrame(ggivis, 1);
	}
	I_SetPalette(W_CacheLumpName("PLAYPAL", PU_CACHE));
	curframe = 0;
}
