/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <limits.h>

#include "pinentry.h"
#include "misc.h"

#ifdef WITH_QUALITY
#include <crack.h>
#endif

static pwm_t *gpwm;
static int gelapsed, gtimeout;
static gpg_error_t grc;

static gpg_error_t set_pinentry_strings(pwm_t *pwm, pwmd_pinentry_t which);

static void update_pinentry_settings(pwm_t *pwm)
{
    FILE *fp;
    char buf[LINE_MAX];
    char *p;
    struct passwd pw;
    char *pwbuf = _getpwuid(&pw);

    if (!pwbuf)
	return;

    snprintf(buf, sizeof(buf), "%s/.pwmd/pinentry.conf", pw.pw_dir);

    if ((fp = fopen(buf, "r")) == NULL) {
	pwmd_free(pwbuf);
	return;
    }

    while ((p = fgets(buf, sizeof(buf), fp)) != NULL) {
	char name[32], val[256];

	if (sscanf(p, " %31[a-zA-Z] = %255s", name, val) != 2)
	    continue;

	if (strcasecmp(name, "TTYNAME") == 0) {
	    if (pwm->pinentry_tty)
		pwmd_free(pwm->pinentry_tty);
	    pwm->pinentry_tty = pwmd_strdup(val);

	    if (!pwm->pinentry_tty)
		goto fail;
	}
	else if (strcasecmp(name, "TTYTYPE") == 0) {
	    if (pwm->pinentry_term)
		pwmd_free(pwm->pinentry_term);
	    pwm->pinentry_term = pwmd_strdup(val);
	    
	    if (!pwm->pinentry_term)
		goto fail;
	}
	else if (strcasecmp(name, "DISPLAY") == 0) {
	    if (pwm->pinentry_display)
		pwmd_free(pwm->pinentry_display);
	    pwm->pinentry_display = pwmd_strdup(val);

	    if (!pwm->pinentry_display)
		goto fail;
	}
	else if (strcasecmp(name, "PATH") == 0) {
	    if (pwm->pinentry_path)
		pwmd_free(pwm->pinentry_path);
	    pwm->pinentry_path = _expand_homedir(val, &pw);

	    if (pwm->pinentry_path)
		goto fail;
	}
    }

fail:
    pwmd_free(pwbuf);
    fclose(fp);
}

static gpg_error_t launch_pinentry(pwm_t *pwm)
{
    int rc;
    assuan_context_t ctx;
    int child_list[] = {-1};
    char *display = getenv("DISPLAY");
    const char *argv[10];
    const char **p = argv;
    int have_display = 0;
    char *tty = NULL;
    char *ttybuf = NULL;

    update_pinentry_settings(pwm);

    if (pwm->pinentry_display || display)
	have_display = 1;
    else {
	if (!pwm->pinentry_tty) {
	    ttybuf = pwmd_malloc(255);

	    if (!ttybuf)
		return gpg_error_from_errno(ENOMEM);

	    if (!ttyname_r(STDOUT_FILENO, ttybuf, 255))
		tty = ttybuf;
	    else
		pwmd_free(ttybuf);
	}
	else
	    tty = pwm->pinentry_tty;
    }

    if (!have_display && !tty)
	return GPG_ERR_ENOTTY;

    *p++ = "pinentry";
    *p++ = have_display ? "--display" : "--ttyname";
    *p++ = have_display ? pwm->pinentry_display ? pwm->pinentry_display : display : tty;

    if (pwm->lcctype) {
	*p++ = "--lc-ctype";
	*p++ = pwm->lcctype;
    }

    if (pwm->lcmessages) {
	*p++ = "--lc-messages";
	*p++ = pwm->lcmessages;
    }

    *p = NULL;

    if (!have_display) {
	*p++ = "--ttytype";
	*p++ = pwm->pinentry_term ? pwm->pinentry_term : getenv("TERM");
	*p = NULL;
    }

    rc = assuan_pipe_connect(&ctx,
	    pwm->pinentry_path ? pwm->pinentry_path : PINENTRY_PATH, argv,
	    child_list, 1);
    
    if (ttybuf)
	pwmd_free(ttybuf);

    if (rc)
	return rc;

    pwm->pid = assuan_get_pid(ctx);
    pwm->pctx = ctx;
    return set_pinentry_strings(pwm, 0);
}

static gpg_error_t pinentry_command(pwm_t *pwm, char **result, const char *cmd)
{
    if (!pwm->pctx) {
	gpg_error_t rc = launch_pinentry(pwm);

	if (rc)
	    return gpg_err_code(rc);
    }

    return _assuan_command(pwm, pwm->pctx, result, cmd);
}

#ifdef WITH_QUALITY
static int quality_cb(void *data, const char *line)
{
    pwm_t *pwm = data;
    int score = 0;
    char buf[5];
    const char *tmp;

    if (strncmp(line, "QUALITY ", 8))
	return GPG_ERR_INV_ARG;

    if (!(tmp = FascistCheck(line+8, CRACKLIB_DICT)))
	return assuan_send_data(pwm->pctx, "100", 3);

    if (!strcmp(tmp, N_("it is WAY too short")))
	score = 10;
    else if (!strcmp(tmp, N_("it is too short")))
	score = 20;
    else if (!strcmp(tmp, N_("it is all whitespace")))
	score = 25;
    else if (!strcmp(tmp, N_("it is based on your username")))
	score = 30;
    else if (!strcmp(tmp, N_("it does not contain enough DIFFERENT characters")))
	score = 35;
    else if (!strcmp(tmp, N_("it is based on a dictionary word")))
	score = 40;
    else if (!strcmp(tmp, N_("it is based upon your password entry")))
	score = 50;
    else if (!strcmp(tmp, N_("it's derived from your password entry")))
	score = 50;
    else if (!strcmp(tmp, N_("it is derived from your password entry")))
	score = 50;
    else if (!strcmp(tmp, N_("it is based on a (reversed) dictionary word")))
	score = 60;
    else if (!strcmp(tmp, N_("it is derivable from your password entry")))
	score = 70;
    else if (!strcmp(tmp, N_("it's derivable from your password entry")))
	score = 70;
    else if (!strcmp(tmp, N_("it looks like a National Insurance number.")))
	score = 80;
    else if (!strcmp(tmp, N_("it is too simplistic/systematic")))
	score = 90;
    else
	score = 0;

    snprintf(buf, sizeof(buf), "%i", score);
    return assuan_send_data(pwm->pctx, buf, strlen(buf));
}
#endif

static gpg_error_t set_pinentry_strings(pwm_t *pwm, pwmd_pinentry_t which)
{
    char *tmp, *desc = NULL;
    gpg_error_t rc;

    tmp = pwmd_malloc(ASSUAN_LINELENGTH+1);

    if (!tmp)
	return gpg_error_from_errno(ENOMEM);

    if (!pwm->title)
	pwm->title = pwmd_strdup_printf(N_("Password Manager Daemon: %s"),
		pwm->name ? pwm->name : "libpwmd");

    if (!pwm->title)
	goto fail_no_mem;

    if (!pwm->prompt)
	pwm->prompt = pwmd_strdup(N_("Passphrase:"));

    if (!pwm->prompt)
	goto fail_no_mem;

    if (!pwm->desc && (which == PWMD_PINENTRY_OPEN || which == PWMD_PINENTRY_SAVE)) {
	if (which == PWMD_PINENTRY_OPEN)
	    desc = pwmd_strdup_printf(N_("A passphrase is required to open the file \"%s\". Please enter the passphrase below."), pwm->filename);
	else
	    desc = pwmd_strdup_printf(N_("A passphrase is required to save to the file \"%s\". Please enter the passphrase below."), pwm->filename);

	if (!desc)
	    goto fail_no_mem;
    }

    if (pwm->desc)
	desc = pwm->desc;

    switch (which) {
	case PWMD_PINENTRY_OPEN:
	case PWMD_PINENTRY_SAVE:
	    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETERROR %s", desc);

	    if (pwm->desc != desc)
		pwmd_free(desc);
	    break;
	case PWMD_PINENTRY_OPEN_FAILED:
	    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETERROR %s",
		    N_("Invalid passphrase, please try again."));
	    break;
	case PWMD_PINENTRY_SAVE_CONFIRM:
	    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETERROR %s",
		    N_("Please type the passphrase again for confirmation."));
	    break;
	default:
	case PWMD_PINENTRY_DEFAULT:
	    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETERROR %s",
		    pwm->desc ? pwm->desc : "");
	    break;
    }

    rc = pinentry_command(pwm, NULL, tmp);

    if (rc) {
	pwmd_free(tmp);
	return rc;
    }

    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETPROMPT %s", pwm->prompt);
    rc = pinentry_command(pwm, NULL, tmp);

    if (rc) {
	pwmd_free(tmp);
	return rc;
    }

#ifdef WITH_QUALITY
    if (which == PWMD_PINENTRY_SAVE) {
	rc = pinentry_command(pwm, NULL, "SETQUALITYBAR");

	if (rc) {
	    pwmd_free(tmp);
	    return rc;
	}

	pwm->_inquire_func = quality_cb;
	pwm->_inquire_data = pwm;
    }
#endif

    snprintf(tmp, ASSUAN_LINELENGTH+1, "SETDESC %s", pwm->title);
    rc = pinentry_command(pwm, NULL, tmp);
    pwmd_free(tmp);
    return rc;

fail_no_mem:
    pwmd_free(tmp);
    return gpg_error_from_errno(ENOMEM);
}

static gpg_error_t terminate_pinentry(pwm_t *pwm)
{
    pid_t pid = pwm->pid;

    pwm->pid = -1;

    if (!pwm || pid == -1)
	return GPG_ERR_INV_ARG;

    if (kill(pid, 0) == 0) {
	if (kill(pid, SIGTERM) == -1) {
	    if (kill(pid, SIGKILL) == -1)
		return gpg_error_from_errno(errno);
	}
    }
    else
	return gpg_error_from_errno(errno);

    return 0;
}

void _pinentry_disconnect(pwm_t *pwm)
{
    if (pwm->pctx)
	assuan_disconnect(pwm->pctx);

    pwm->pctx = NULL;
    pwm->pid = -1;
}

/*
 * Only called from a child process.
 */
static void catchsig(int sig)
{
    switch (sig) {
	case SIGALRM:
	    if (++gelapsed >= gtimeout) {
		terminate_pinentry(gpwm);
		grc = GPG_ERR_TIMEOUT;
	    }
	    else
		alarm(1);
	    break;
	default:
	    break;
    }
}

static gpg_error_t do_getpin(pwm_t *pwm, char **result, pwmd_pinentry_t which)
{
    if (gtimeout) {
	signal(SIGALRM, catchsig);
	alarm(1);
    }

    if (result)
	*result = NULL;

    return pinentry_command(pwm, result,
	    which == PWMD_PINENTRY_CONFIRM ? "CONFIRM" : "GETPIN");
}

gpg_error_t _getpin(pwm_t *pwm, char **result, pwmd_pinentry_t which)
{
    grc = 0;
    gpg_error_t rc = set_pinentry_strings(pwm, which);

    if (rc) {
	_pinentry_disconnect(pwm);
	return rc;
    }

    if (result)
	*result = NULL;

    rc = do_getpin(pwm, result, which);

    /*
     * Since there was input cancel any timeout setting.
     */
    alarm(0);
    signal(SIGALRM, SIG_DFL);

    if (rc) {
	_pinentry_disconnect(pwm);

	/* This lets pwmd_open2() with PWMD_OPTION_PINENTRY_TIMEOUT work. Note
	 * that it is not thread safe do to the global variables. */
	if (rc == GPG_ERR_EOF)
	    rc = grc == GPG_ERR_TIMEOUT ? grc : GPG_ERR_CANCELED;
    }

    return rc;
}

gpg_error_t _do_save_getpin(pwm_t *pwm, char **password)
{
    int confirm = 0;
    gpg_error_t rc;
    char *result = NULL;

again:
    rc = _getpin(pwm, &result, confirm ? PWMD_PINENTRY_SAVE_CONFIRM : PWMD_PINENTRY_SAVE);

    if (rc) {
	_pinentry_disconnect(pwm);

	if (*password)
	    pwmd_free(*password);

	return rc;
    }

    if (!confirm++) {
	*password = result;
	goto again;
    }

    if (strcmp(*password, result)) {
	pwmd_free(*password);
	pwmd_free(result);
	confirm = 0;
	*password = NULL;
	goto again;
    }

    pwmd_free(result);
    _pinentry_disconnect(pwm);
    return 0;
}

static gpg_error_t do_local_getpin(pwm_t *pwm, char **password,
	pwmd_pinentry_t which)
{
    gpg_error_t rc;

    /* Another handle is using this pinentry method. Allowing this instance
     * would reset the timeout and global handle which wouldn't be good. */
    if (gpwm)
	return GPG_ERR_PIN_BLOCKED;

    if (pwm->pinentry_timeout != 0) {
	gpwm = pwm;
	gtimeout = abs(pwm->pinentry_timeout);
	gelapsed = 0;
    }

    rc = _getpin(pwm, password, which);

    /* Don't timeout when an invalid passphrase was entered. */
    gtimeout = 0;
    gpwm = NULL;
    return rc;
}

gpg_error_t _pinentry_open(pwm_t *pwm, const char *filename, char **password,
	int nb)
{
    gpg_error_t rc;

    if (pwm->filename)
	pwmd_free(pwm->filename);

    pwm->filename = pwmd_strdup(filename);

    if (!pwm->filename)
	return gpg_error_from_errno(ENOMEM);

    /* Get the passphrase using the LOCAL pinentry. */
    if (nb) {
	int p[2];
	pid_t pid;
	pwmd_nb_status_t pw;

	if (pipe(p) == -1)
	    return gpg_error_from_syserror();

#ifdef WITH_LIBPTH
	pid = pth_fork();
#else
	pid = fork();
#endif

	switch (pid) {
	    case 0:
		close(p[0]);
		pw.fd = p[0];

		if (pwm->pinentry_timeout != 0 && !pwm->pin_try) {
		    gpwm = pwm;
		    gtimeout = abs(pwm->pinentry_timeout);
		    gelapsed = 0;
		}

		pw.error = _getpin(pwm, password,
			!pwm->pin_try ? PWMD_PINENTRY_OPEN : PWMD_PINENTRY_OPEN_FAILED);
		_pinentry_disconnect(pwm);

		if (gtimeout && gelapsed >= gtimeout)
		    pw.error = GPG_ERR_TIMEOUT;

		if (!pw.error) {
		    snprintf(pw.password, sizeof(pw.password), "%s",
			    *password);
		    pwmd_free(*password);
		}

#ifdef WITH_LIBPTH
		pth_write(p[1], &pw, sizeof(pw));
#else
		write(p[1], &pw, sizeof(pw));
#endif
		memset(&pw, 0, sizeof(pw));
		close(p[1]);
		_exit(0);
		break;
	    case -1:
		rc = gpg_error_from_syserror();
		close(p[0]);
		close(p[1]);
		return rc;
	    default:
		break;
	}

	close(p[1]);
	pwm->nb_fd = p[0];
	pwm->nb_pid = pid;
	return 0;
    }

    return do_local_getpin(pwm, password, PWMD_PINENTRY_OPEN);
}

gpg_error_t _pwmd_getpin(pwm_t *pwm, const char *filename, char **result,
	pwmd_pinentry_t which)
{
    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (which == PWMD_PINENTRY_CLOSE) {
	_pinentry_disconnect(pwm);
	return 0;
    }

    if (!result && which != PWMD_PINENTRY_CONFIRM)
	return GPG_ERR_INV_ARG;

    char *p = pwm->filename;
    pwm->filename = (char *)filename;
    gpg_error_t rc = do_local_getpin(pwm, result, which);
    pwm->filename = p;
    return rc;
}
