/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2007-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <string.h>
#include <libpwmd.h>
#include <assuan.h>
#include <sys/select.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libgen.h>
#include <termios.h>
#include <limits.h>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#ifdef HAVE_GETOPT_LONG
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#else
#include "getopt_long.h"
#endif

#ifdef HAVE_LIBREADLINE
#  if defined(HAVE_READLINE_READLINE_H)
#    include <readline/readline.h>
#  elif defined(HAVE_READLINE_H)
#    include <readline.h>
#  endif /* !defined(HAVE_READLINE_H) */
static int interactive_error;
static int interactive;
#endif /* HAVE_LIBREADLINE */

#ifdef HAVE_READLINE_HISTORY
#  if defined(HAVE_READLINE_HISTORY_H)
#    include <readline/history.h>
#  elif defined(HAVE_HISTORY_H)
#    include <history.h>
#  endif
#endif /* HAVE_READLINE_HISTORY */

#include "gettext.h"
#define N_(msgid)	gettext(msgid)

#include "mem.h"

#define DEFAULT_PORT	22

static int in_keyfile;
static int no_pinentry;
static pwm_t *pwm;
static char *filename;
static int save;
static int local_pin;
static int force_save;
static char *cipher;
static long iter;
static char *password;
static char *keyfile;
#ifdef DEBUG
static int method;
#endif
#ifdef WITH_TCP
char *host = NULL;
#endif

struct inquire_s {
    int fd;
    char *line;
    size_t len;
};

static gpg_error_t finalize();

static void show_error(gpg_error_t error)
{
    fprintf(stderr, "ERR %i %s\n", gpg_err_code(error), pwmd_strerror(error));
}

static void usage(const char *pn, int status)
{
    fprintf(status == EXIT_FAILURE ? stderr : stdout, N_(
	    "Read a PWMD protocol command from standard input.\n\n"
	    "Usage: pwmc [options] [file]\n"
#ifdef DEBUG
	    "   --debug <N>\n"
	    "       pinentry method (0=pwmd, 1=libpwmd, 2=pwmd async, "
		    "3=libpwmd async)\n"
#endif
	    "\n"
#ifdef WITH_TCP
	    "   --host, -h <hostname>\n"
	    "       connect to the specified hostname\n"
	    "\n"
	    "   --port <N>\n"
	    "       alterate port (22)\n"
	    "\n"
	    "   --user <username>\n"
	    "       SSH username (default is the invoking user)\n"
	    "\n"
	    "   --no-ssh-agent\n"
	    "       disable SSH agent (enabled when SSH_AGENT_PID is set)\n"
	    "\n"
	    "   --identity, -i <filename>\n"
	    "       SSH identity file (if not using the SSH agent)\n"
	    "\n"
	    "   --known-hosts, -k <filename>\n"
	    "       known hosts file (~/.ssh/known_hosts)\n"
	    "\n"
	    "   --get-hostkey, -g\n"
	    "       retrieve the remote SSH host key and exit\n"
	    "\n"
	    "   --ipv4, -4\n"
	    "       try connecting via IPv4 only\n"
	    "\n"
	    "   --ipv6, -6\n"
	    "       try connecting via IPv6 only\n"
	    "\n"
#endif
            "   --url <string>\n"
	    "       a url string to parse (see below)\n"
	    "\n"
	    "   --no-status\n"
	    "       disable showing of status messages from the server\n"
	    "\n"
	    "   --name, -n <string>\n"
	    "       set the client name\n"
	    "\n"
	    "   --socket <filename>\n"
	    "       local socket to connect to (~/.pwmd/socket)\n"
	    "\n"
	    "   --passphrase, -P <string>\n"
	    "       passphrase to use (disables pinentry use)\n"
	    "\n"
	    "   --key-file <filename>\n"
	    "       obtain the passphrase from the specified filename\n"
	    "\n"
	    "   --base64\n"
	    "       the passphrase is base64 encoded\n"
	    "\n"
	    "   --timeout <seconds>\n"
	    "       pinentry timeout\n"
	    "\n"
	    "   --tries <N>\n"
	    "       number of pinentry tries before failing (3)\n"
	    "\n"
	    "   --no-pinentry\n"
	    "       disable pinentry both remotely and locally\n"
	    "\n"
	    "   --pinentry <path>\n"
	    "       the full path to the pinentry binary (server default)\n"
	    "\n"
	    "   --ttyname, -y <path>\n"
	    "       tty that pinentry will use\n"
	    "\n"
	    "   --ttytype, -t <string>\n"
	    "       pinentry terminal type (default is $TERM)\n"
	    "\n"
	    "   --display, -d\n"
	    "       pinentry display (default is $DISPLAY)\n"
	    "\n"
	    "   --lc-ctype <string>\n"
	    "       locale setting for pinentry\n"
	    "\n"
	    "   --lc-messages <string>\n"
	    "       locale setting for pinentry\n"
	    "\n"
	    "   --local-pinentry\n"
	    "       force using a local pinentry\n"
	    "\n"
	    "   --interactive\n"
	    "       use a shell like interface to pwmd (allows more than one command)\n"
	    "\n"
	    "   --output-fd <FD>\n"
	    "       redirect command output to the specified file descriptor\n"
	    "\n"
	    "   --inquire <COMMAND>\n"
	    "       the specified command (with any options) uses a server inquire while\n"
	    "       command data is read via the inquire file descriptor (stdin)\n"
	    "\n"
	    "   --inquire-fd <FD>\n"
	    "       read inquire data from the specified file descriptor (stdin)\n"
	    "\n"
	    "   --inquire-line, -L <STRING>\n"
	    "       the initial line to send (i.e., element path) before the inquire data\n"
	    "\n"
	    "   --cipher <string>\n"
	    "       the cipher to use when saving (see pwmd(1))\n"
	    "\n"
	    "   --save, -S\n"
	    "       send the SAVE command before exiting\n"
	    "\n"
	    "   --force-save\n"
	    "       like --save but always ask for a passphrase\n"
	    "\n"
	    "   --iterations, -I <N>\n"
	    "       encrypt with the specified number of iterations\n"
	    "\n"
	    "   --version\n"
	    "   --help\n"));
    fprintf(status == EXIT_FAILURE ? stderr : stdout, N_(
		"\n"
		"A url string (specified with --url) may be in the form of:\n"
		"    file://[path/to/socket]\n"
#ifdef WITH_TCP
		"    ssh[46]://[username@]hostname[:port],identity[,known_hosts] --no-ssh-agent\n"
		"    or\n"
		"    ssh[46]://[username@]hostname[:port][,,known_hosts]\n"
#endif
		));
    exit(status);
}

static gpg_error_t inquire_cb(void *user, const char *cmd, gpg_error_t rc,
	char **data, size_t *len)
{
    struct inquire_s *inq = user;

    *data = NULL;
    *len = 0;

    if (rc)
	return rc;

    if (!in_keyfile) {
	fprintf(stderr, N_(
"------------------------------------------------------------------------------\n"
"Press CTRL-D to end. Press twice on a non-empty line. Pressing ENTER after\n"
"each line during the inquire will send that line including the newline\n"
"character.\n"
"------------------------------------------------------------------------------\n"
));
    }

    /* The first part of the command data. */
    if (inq->len) {
	*data = inq->line;
	*len = inq->len;
	inq->len = 0;
	return 0;
    }

    *len = read(inq->fd, inq->line, ASSUAN_LINELENGTH);

    if (*len == -1)
	return gpg_error_from_syserror();

    if (*len)
	*data = inq->line;

    return *len ? 0 : GPG_ERR_EOF;
}

static int status_msg_cb(void *data, const char *line)
{
    char *p = strchr(line, ' ');

    if (p && strchr(p, ' ') && *++p) {
	char *p1 = strchr(p, ' ');
	int a = atoi(p);

	if (p1) {
	    int b = atoi(p1);
	    char l[64] = {0};
	    int t = a && b ? a*100/b : 0;

	    strncpy(l, line, strlen(line)-strlen(p)-1);
	    fprintf(stderr, "\r%s %i/%i %i%%%s", l, a, b, t, a == b ? "\n" : "");
	    fflush(stderr);
	    return 0;
	}
    }

    fprintf(stderr, "%s\n", line);
    fflush(stderr);
    return 0;
}

static gpg_error_t process_cmd(pwm_t *pwm, char **result, int input, int once)
{
    gpg_error_t rc;
    pwmd_async_t s;

    do {
	int i, n;
	fd_set rfds;
	int nfds = 5;
	pwmd_fd_t pfds[nfds];
	struct timeval tv = {0, 50000};

#ifdef DEBUG
	fprintf(stderr, ".");
#endif
	FD_ZERO(&rfds);
	rc = pwmd_get_fds(pwm, pfds, &nfds);

	if (rc)
	    break;

	if (!nfds) {
	    s = pwmd_process(pwm, &rc, result);
	    break;
	}

	for (i = 0, n = 0; i < nfds; i++) {
	    FD_SET(pfds[i].fd, &rfds);
	    n = pfds[i].fd > n ? pfds[i].fd : n;
	}

	if (input)
	    FD_SET(STDIN_FILENO, &rfds);

	nfds = select(n+1, &rfds, NULL, NULL, &tv);

	if (nfds == -1) {
	    rc = gpg_error_from_errno(errno);
	    break;
	}

	if (input && FD_ISSET(STDIN_FILENO, &rfds)) {
	    rc = 0;
	    break;
	}

	s = pwmd_process(pwm, &rc, result);

	if (once)
	    break;
    } while (s == ASYNC_PROCESS);

#ifdef DEBUG
    fprintf(stderr, "\n");
#endif
    return rc;
}

static gpg_error_t get_password(char **result, pwmd_pinentry_t w, int echo)
{
    char buf[LINE_MAX] = {0}, *p;
    struct termios told, tnew;
    char *key = NULL;

    *result = NULL;

    if (!isatty(STDIN_FILENO)) {
	fprintf(stderr, N_("Input is not from a terminal! Failing.\n"));
	return GPG_ERR_ENOTTY;
    }

    if (!echo) {
	if (tcgetattr(STDIN_FILENO, &told) == -1)
	    return gpg_error_from_syserror();

	memcpy(&tnew, &told, sizeof(struct termios));
	tnew.c_lflag &= ~(ECHO);
	tnew.c_lflag |= ICANON|ECHONL;

	if (tcsetattr(STDIN_FILENO, TCSANOW, &tnew) == -1) {
	    int n = errno;

	    tcsetattr(STDIN_FILENO, TCSANOW, &told);
	    return gpg_error_from_errno(n);
	}
    }

    switch (w) {
	case PWMD_PINENTRY_OPEN:
	    fprintf(stderr, N_("Password for %s: "), filename);
	    break;
	case PWMD_PINENTRY_OPEN_FAILED:
	    fprintf(stderr, N_("Invalid password. Password for %s: "), filename);
	    break;
	case PWMD_PINENTRY_SAVE:
	    fprintf(stderr, N_("New password for %s: "), filename);
	    break;
	case PWMD_PINENTRY_SAVE_CONFIRM:
	    fprintf(stderr, N_("Confirm password: "));
	    break;
	default:
	    break;
    }

    if ((p = fgets(buf, sizeof(buf), stdin)) == NULL) {
	tcsetattr(STDIN_FILENO, TCSANOW, &told);
	return 0;
    }

    if (!echo)
	tcsetattr(STDIN_FILENO, TCSANOW, &told);

    if (feof(stdin)) {
	clearerr(stdin);
	return GPG_ERR_CANCELED;
    }

    p[strlen(p) - 1] = 0;

    if (buf[0]) {
	key = pwmd_strdup_printf("%s", p);
	memset(&buf, 0, sizeof(buf));

	if (!key)
	    return GPG_ERR_ENOMEM;
    }

    *result = key;
    return 0;
}

#ifdef WITH_TCP
static gpg_error_t knownhost_cb(void *data, const char *host, const char *key,
	size_t len)
{
    gpg_error_t rc;
    char *buf = pwmd_strdup_printf(N_("Password Manager Daemon: %s\n\nWhile attempting an SSH connection to %s there was a problem verifying it's hostkey against the known and trusted hosts file because it's hostkey was not found.\n\nWould you like to treat this connection as trusted for this and future connections by adding %s's hostkey to the known hosts file?"), (char *)data, host, host);

    if (no_pinentry && !isatty(STDIN_FILENO)) {
	fprintf(stderr, N_("Input is not from a terminal! Failing.\n"));
	pwmd_free(buf);
	return GPG_ERR_ENOTTY;
    }
    else if (no_pinentry) {
	for (char *p = buf, len = 0; *p; p++, len++) {
	    if (*p == '\n')
		len = 0;

	    if (len == 78) {
		char *t = p;

		while (!isspace(*(--p)));
		*p = '\n';
		p = t;
		len = 0;
	    }
	}

	fprintf(stderr, "%s\n\n", buf);
	pwmd_free(buf);

	do {
	    char *result;

	    fprintf(stderr, N_("Trust this connection [y/N]: "));
	    fflush(stderr);
	    rc = get_password(&result, PWMD_PINENTRY_CONFIRM, 1);

	    if (rc)
		return rc;

	    if (!result || !*result || *result == 'n' || *result == 'N') {
		if (result && *result)
		    pwmd_free(result);

		return GPG_ERR_NOT_CONFIRMED;
	    }

	    if ((*result == 'y' || *result == 'Y') && *(result+1) == 0) {
		pwmd_free(result);
		return 0;
	    }
	} while (1);
    }

    rc = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TITLE, buf);
    pwmd_free(buf);

    if (rc)
	return rc;

    return pwmd_getpin(pwm, NULL, NULL, PWMD_PINENTRY_CONFIRM);
}

static int is_remote_url(const char *str)
{
    if (str && (strstr(str, "file://") || strstr(str, "local://")))
	return 0;

    return !str ? 0 : 1;
}
#endif

static char *escape(const char *str)
{
    const char *p;
    char *buf = pwmd_malloc(ASSUAN_LINELENGTH+1), *b = buf;
    size_t len = 0;

    for (p = str; *p; p++, len++) {
	if (len == ASSUAN_LINELENGTH)
	    break;

	if (*p == '\\') {
	    switch (*++p) {
		case 't':
		    *b++ = '\t';
		    break;
		case 'n':
		    *b++ = '\n';
		    break;
		case 'v':
		    *b++ = '\v';
		    break;
		case 'b':
		    *b++ = '\b';
		    break;
		case 'f':
		    *b++ = '\f';
		    break;
		case 'r':
		    *b++ = '\r';
		    break;
		default:
		    *b++ = *p;
		    break;
	    }

	    if (!*p)
		break;

	    continue;
	}

	*b++ = *p;
    }

    *b = 0;
    return buf;
}

static gpg_error_t send_inquire(int fd, const char *command, const char *line)
{
    struct inquire_s inq;
    struct stat st;
    gpg_error_t rc;

    if (fstat(fd, &st) == -1)
	return gpg_error_from_syserror();

    memset(&inq, 0, sizeof(inq));
    inq.fd = fd;
    inq.line = pwmd_calloc(1, ASSUAN_LINELENGTH);
    inq.len = 0;

    if (!inq.line)
	return GPG_ERR_ENOMEM;

    if (line) {
	char *s = escape(line);

	strncpy(inq.line, s, ASSUAN_LINELENGTH-1);
	inq.len = strlen(s);
	pwmd_free(s);
    }

    rc = pwmd_setopt(pwm, PWMD_OPTION_INQUIRE_TOTAL,
	    st.st_size ? st.st_size+strlen(inq.line) : 0);

    if (rc)
	return rc;

    rc = pwmd_inquire(pwm, command, inquire_cb, &inq);
    pwmd_free(inq.line);
    return rc;
}

#ifdef HAVE_LIBREADLINE
static int interactive_hook(void)
{
    interactive_error = process_cmd(pwm, NULL, 0, 1);

    if (interactive_error)
	rl_event_hook = NULL;

    return 0;
}

static void print_help()
{
    fprintf(stderr, N_(
"------------------------------------------------------------------------------\n"
"Type HELP for protocol command help (since pwmd 2.19). Commands which use a\n"
"server inquire need a non-protocol command to send the data. These commands\n"
"including any options are:\n"
"\n"
"    INQUIRE <command>                    - to read from the terminal\n"
"    INQUIRE_FILE <filename> <command>    - to read the specified filename\n"
"\n"
"Elements are TAB delimited. Press CTRL-V then TAB to insert from the prompt.\n"
"\n"
"Press CTRL-D to quit.\n"
"------------------------------------------------------------------------------\n"
    ));
}

static gpg_error_t do_interactive()
{
    gpg_error_t rc = process_cmd(pwm, NULL, 0, 1);

    if (rc)
	return rc;

    fprintf(stderr, N_("WARNING: interactive mode doesn't use secure memory!\n"));
    print_help();
    rl_initialize();
    rl_event_hook = &interactive_hook;
    rl_set_keyboard_input_timeout(100000);

    for (;;) {
	char *line;
	char *result = NULL;

	line = readline("pwm> ");

	if (interactive_error)
	    return interactive_error;

	if (!line) {
	    rc = finalize();

	    if (!rc)
		return 0;

	    if (rc != GPG_ERR_CANCELED && rc != GPG_ERR_EOF)
		fprintf(stderr, "ERR %i: %s\n", rc, pwmd_strerror(rc));

	    continue;
	}
	else if (!*line) {
	    free(line);
	    continue;
	}

#ifdef HAVE_READLINE_HISTORY
	add_history(line);
#endif

	/* INQUIRE <COMMAND [options ...]> */
	if (!strncasecmp(line, "inquire ", 8))
	    rc = send_inquire(STDIN_FILENO, line+8, NULL);
	/* INQUIRE_FILE <FILENAME> <COMMAND [options ...]> */
	else if (!strncasecmp(line, "inquire_file ", 13)) {
	    char *p = strchr(line+13, ' ');
	    char *f = p ? line+13 : NULL;
	    int fd;

	    if (!f || !*f || !*p) {
		fprintf(stderr, N_("Syntax error. Usage: INQUIRE_FILE <filename> <command>\n"));
		free(line);
		continue;
	    }

	    f[strlen(f) - strlen(p++)] = 0;
	    fd = open(f, O_RDONLY);

	    if (fd == -1) {
		fprintf(stderr, "%s\n", strerror(errno));
		free(line);
		continue;
	    }
	    else {
		rc = send_inquire(fd, p, NULL);
		close(fd);
	    }
	}
	else
	    rc = pwmd_command(pwm, &result, line);

	free(line);

	if (rc)
	    fprintf(stderr, "ERR %i: %s\n", rc, pwmd_strerror(rc));
	else if (result && *result)
	    printf("%s%s", result, result[strlen(result)-1] != '\n' ? "\n" : "");

	if (result)
	    pwmd_free(result);
    }

    return 0;
}
#endif

static gpg_error_t finalize()
{
    gpg_error_t rc = 0;
    int quit = 0;

#ifdef HAVE_LIBREADLINE
    if (!force_save && interactive) {
	char *p, buf[16];
	int finished = 0;

	fprintf(stderr, "\n");

	do {
	    fprintf(stderr, N_("(c)ancel/(f)orget password/(s)ave/(Q)uit/(S)ave and quit/(h)elp?: "));
	    p = fgets(buf, sizeof(buf), stdin);

	    if (feof(stdin)) {
		clearerr(stdin);
		return GPG_ERR_EOF;
	    }

	    switch (*p) {
		case 'h':
		    print_help();
		    break;
		case 'c':
		    return GPG_ERR_CANCELED;
		case 'Q':
		    return 0;
		case 'f':
		    rc = pwmd_command(pwm, NULL, "CLEARCACHE %s", filename);

		    if (rc)
			return rc;

		    interactive_hook();
		    break;
		case 'S':
		    quit = 1;
		case 's':
		    save = 1;
		    finished = 1;
		    break;
		default:
		    break;
	    }
	} while (!finished);
    }
#endif

    if (save && !filename) {
	fprintf(stderr, N_("No filename was specified on the command line. Aborting.\n"));
	return GPG_ERR_CANCELED;
    }

    if (save && filename) {
	fprintf(stderr, "\n");
	fprintf(stderr, N_("Saving changes ...\n"));

	if (iter != -2) {
	    rc = pwmd_setopt(pwm, PWMD_OPTION_ITERATIONS, iter);

	    if (rc)
		return rc;
	}

	if (cipher) {
	    rc = pwmd_setopt(pwm, PWMD_OPTION_CIPHER, cipher);

	    if (rc)
		return rc;
	}

	if ((local_pin || no_pinentry) && !keyfile) {
	    char *p1;
again:
	    if (!force_save) {
		rc = pwmd_command(pwm, NULL, "ISCACHED %s", filename);

		if (rc && rc != GPG_ERR_NOT_FOUND &&
			rc != GPG_ERR_ENOENT)
		    return rc;
		else if (!rc)
		    goto do_save;
	    }

	    if (!no_pinentry) {
		rc = pwmd_getpin(pwm, filename, &p1, PWMD_PINENTRY_SAVE);

		if (rc)
		    return rc;

		rc = pwmd_getpin(pwm, filename, &password,
			PWMD_PINENTRY_SAVE_CONFIRM);

		if (rc) {
		    pwmd_free(p1);
		    return rc;
		}
	    }
	    else {
		rc = get_password(&p1, PWMD_PINENTRY_SAVE, 0);

		if (rc)
		    return rc;

		rc = get_password(&password, PWMD_PINENTRY_SAVE_CONFIRM, 0);

		if (rc) {
		    if (p1)
			pwmd_free(p1);

		    return rc;
		}
	    }

	    if ((p1 || password) && ((!p1 && password) || (!password && p1) ||
		    strcmp(p1, password))) {
		pwmd_free(p1);
		pwmd_free(password);
		password = NULL;
		goto again;
	    }

	    if (p1)
		pwmd_free(p1);

	    rc = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

	    if (password)
		pwmd_free(password);

	    if (rc)
		return rc;
	}

	if (force_save) {
	    rc = pwmd_command(pwm, NULL, "CLEARCACHE %s", filename);

	    if (rc)
		return rc;

	    if (!local_pin) {
		rc = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, NULL);

		if (rc)
		    return rc;
	    }
	}

do_save:
	if (keyfile) {
	    struct inquire_s inq;

	    memset(&inq, 0, sizeof(inq));
	    inq.fd = open(keyfile, O_RDONLY);

	    if (inq.fd == -1) {
		rc = gpg_error_from_syserror();
		return rc;
	    }

	    inq.line = pwmd_calloc(1, ASSUAN_LINELENGTH);

	    if (!inq.line) {
		rc = GPG_ERR_ENOMEM;
		return rc;
	    }

	    in_keyfile = 1;
	    rc = pwmd_save_inquire(pwm, inquire_cb, &inq);
	    in_keyfile = 0;
	    pwmd_free(inq.line);
	    close(inq.fd);
	}
	else {
#ifdef DEBUG
	    switch (method) {
		case 0:
		    rc = pwmd_save(pwm);
		    break;
		case 1:
		    rc = pwmd_save2(pwm);
		    break;
		case 2:
		    rc = pwmd_save_async(pwm);
		    break;
		case 3:
		    rc = pwmd_save_async2(pwm);
		    break;
	    }

	    if (!rc && method >= 2)
		rc = process_cmd(pwm, NULL, 0, 0);

#else
#if defined(WITH_PINENTRY) && defined(WITH_TCP)
	    if (host)
		rc = pwmd_save2(pwm);
	    else
#endif
		rc = pwmd_save(pwm);
#endif
	}
    }

    if (interactive)
	return rc ? rc : quit ? 0 : GPG_ERR_CANCELED;

    return rc;
}

int main(int argc, char *argv[])
{
    int connected = 0;
    gpg_error_t error;
    int opt;
    int base64 = 0;
    char *socketpath = NULL;
    char command[ASSUAN_LINELENGTH], *p;
    int ret = EXIT_SUCCESS;
    char *result = NULL;
    char *pinentry_path = NULL;
    char *display = NULL, *tty = NULL, *ttytype = NULL, *lcctype = NULL,
	 *lcmessages = NULL;
    int outfd = STDOUT_FILENO;
    FILE *outfp = stdout;
    int inquirefd = STDIN_FILENO;
    FILE *inquirefp = stdin;
    int show_status = 1;
    char *clientname = "pwmc";
    char *inquire = NULL;
    char *inquire_line = NULL;
    int have_iter = 0;
    int timeout = 0;
#ifdef WITH_TCP
    char *host = NULL;
    int port = DEFAULT_PORT;
    char *username = NULL;
    char *ident = NULL;
    char *known_hosts = NULL;
    int get = 0;
    int prot = PWMD_IP_ANY;
    int use_ssh_agent = 1;
#endif
    int tries = 0;
    int local_tries = 3;
    int try = 0;
    pwmd_socket_t s;
    int lock_on_open = 1;
#ifdef DEBUG
    fd_set rfds;
#endif
    char *url_string = NULL;
    /* The order is important. */
    enum {
#ifdef DEBUG
	OPT_DEBUG,
#endif
#ifdef WITH_TCP
	OPT_HOST, OPT_PORT, OPT_IDENTITY, OPT_KNOWN_HOSTS, OPT_USER,
	OPT_GET_HOSTKEY, OPT_IPV4, OPT_IPV6, OPT_USE_SSH_AGENT,
#endif
	OPT_URL, OPT_LOCAL, OPT_FORCE_SAVE, OPT_TTYNAME, OPT_TTYTYPE,
	OPT_DISPLAY, OPT_LC_CTYPE, OPT_LC_MESSAGES, OPT_TIMEOUT, OPT_TRIES,
	OPT_PINENTRY, OPT_PASSPHRASE, OPT_KEYFILE, OPT_BASE64, OPT_SOCKET,
	OPT_NOLOCK, OPT_SAVE, OPT_ITERATIONS, OPT_OUTPUT_FD, OPT_INQUIRE,
	OPT_INQUIRE_FD, OPT_INQUIRE_LINE, OPT_NO_STATUS, OPT_NAME,
	OPT_VERSION, OPT_HELP, OPT_CIPHER, OPT_NO_PINENTRY,
#ifdef HAVE_LIBREADLINE
	OPT_INTERACTIVE,
#endif
    };
    const struct option long_opts[] = {
#ifdef DEBUG
        { "debug", 1, 0, 0 },
#endif
#ifdef WITH_TCP
	{ "host", 1, 0, 'h' },
	{ "port", 1, 0, 'p' },
	{ "identity", 1, 0, 'i' },
	{ "known-hosts", 1, 0, 'k' },
	{ "user", 1, 0, 'u' },
	{ "get-hostkey", 0, 0, 'g' },
	{ "ipv4", 0, 0, '4' },
	{ "ipv6", 0, 0, '6' },
	{ "no-ssh-agent", 0, 0, 0 },
#endif
	{ "url", 1, 0, 0 },
	{ "local-pinentry", 0, 0 },
	{ "force-save", 0, 0 },
	{ "ttyname", 1, 0, 'y' },
	{ "ttytype", 1, 0, 't' },
	{ "display", 1, 0, 'd' },
	{ "lc-ctype", 1, 0, 0 },
	{ "lc-messages", 1, 0, 0 },
	{ "timeout", 1, 0, 0 },
	{ "tries", 1, 0, 0 },
	{ "pinentry", 1, 0, 0 },
	{ "passphrase", 1, 0, 'P' },
	{ "key-file", 1, 0, 0 },
	{ "base64", 0, 0, 0 },
	{ "socket", 1, 0, 0 },
	{ "no-lock", 0, 0, 0 },
	{ "save", 0, 0, 'S' },
	{ "iterations", 1, 0, 'I' },
	{ "output-fd", 1, 0, 0 },
	{ "inquire", 1, 0, 0 },
	{ "inquire-fd", 1, 0, 0 },
	{ "inquire-line", 1, 0, 'L' },
	{ "no-status", 0, 0, 0 },
	{ "name", 1, 0, 'n' },
	{ "version", 0, 0, 0 },
	{ "help", 0, 0, 0 },
	{ "cipher", 1, 0, 0 },
	{ "no-pinentry", 0, 0, 0 },
#ifdef HAVE_LIBREADLINE
	{ "interactive", 0, 0 },
#endif
	{ 0, 0, 0, 0}
    };
#ifdef WITH_TCP
    const char *optstring = "L:46h:p:i:k:u:gy:t:d:P:I:Sn:";
#else
    const char *optstring = "L:y:t:d:P:I:Sn:";
#endif
    int opt_index = 0;

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain("libpwmd", LOCALEDIR);
#endif

#ifdef HAVE_LIBREADLINE
    if (!strcmp(basename(argv[0]), "pwmsh"))
	interactive = 1;
#endif

    iter = -2;

    while ((opt = getopt_long(argc, argv, optstring, long_opts, &opt_index)) != -1) {
	switch (opt) {
	    /* Handle long options without a short option part. */
	    case 0:
		switch (opt_index) {
#ifdef DEBUG
		    case OPT_DEBUG:
			method = atoi(optarg);

			if (method > 3)
			    method = 3;
			break;
#endif
#ifdef WITH_TCP
		    case OPT_USE_SSH_AGENT:
			use_ssh_agent = 0;
			break;
#endif
		    case OPT_KEYFILE:
			keyfile = optarg;
			break;
		    case OPT_BASE64:
			base64 = 1;
			break;
		    case OPT_NOLOCK:
			lock_on_open = 0;
			break;
		    case OPT_URL:
			url_string = optarg;
			break;
		    case OPT_LOCAL:
			local_pin = 1;
			break;
		    case OPT_FORCE_SAVE:
			save = force_save = 1;
			break;
		    case OPT_LC_CTYPE:
			lcctype = pwmd_strdup(optarg);
			break;
		    case OPT_LC_MESSAGES:
			lcmessages = pwmd_strdup(optarg);
			break;
		    case OPT_TIMEOUT:
			timeout = atoi(optarg);
			break;
		    case OPT_TRIES:
			tries = atoi(optarg);
			local_tries = tries;
			break;
		    case OPT_SOCKET:
			socketpath = pwmd_strdup(optarg);
			break;
		    case OPT_INQUIRE:
			inquire = optarg;
			break;
		    case OPT_INQUIRE_FD:
			inquirefd = atoi(optarg);
			inquirefp = fdopen(inquirefd, "r");

			if (!inquirefp) {
			    pwmd_free(password);
			    err(EXIT_FAILURE, "%i", inquirefd);
			}
			break;
		    case OPT_OUTPUT_FD:
			outfd = atoi(optarg);
			outfp = fdopen(outfd, "w");

			if (!outfp) {
			    pwmd_free(password);
			    err(EXIT_FAILURE, "%i", outfd);
			}
			break;
		    case OPT_NO_STATUS:
			show_status = 0;
			break;
		    case OPT_VERSION:
			pwmd_free(password);
			printf("%s (pwmc)\n%s\n\n"
				"Compile-time features:\n"
#ifdef WITH_TCP
				"+SSH "
#else
				"-SSH "
#endif
#ifdef WITH_PINENTRY
				"+PINENTRY "
#else
				"-PINENTRY "
#endif
#ifdef WITH_QUALITY
				"+CRACK "
#else
				"-CRACK "
#endif
#ifdef MEM_DEBUG
				"+MEM_DEBUG "
#else
				"-MEM_DEBUG "
#endif
#ifdef HAVE_LIBREADLINE
				"+INTERACTIVE "
#else
				"-INTERACTIVE "
#endif
				"\n"
				, PACKAGE_STRING, PACKAGE_BUGREPORT);
			exit(EXIT_SUCCESS);
		    case OPT_PINENTRY:
			pinentry_path = optarg;
			break;
		    case OPT_HELP:
			usage(argv[0], EXIT_SUCCESS);
		    case OPT_CIPHER:
			cipher = optarg;
			break;
		    case OPT_NO_PINENTRY:
			no_pinentry = 1;
			break;
#ifdef HAVE_LIBREADLINE
		    case OPT_INTERACTIVE:
			interactive = 1;
			break;
#endif
		    default:
			usage(argv[0], EXIT_FAILURE);
		}

		break;
#ifdef WITH_TCP
	    case '4':
		prot = PWMD_IPV4;
		break;
	    case '6':
		prot = PWMD_IPV6;
		break;
	    case 'h':
		host = pwmd_strdup(optarg);
		break;
	    case 'p':
		port = atoi(optarg);
		break;
	    case 'i':
		ident = pwmd_strdup(optarg);
		break;
	    case 'u':
		username = pwmd_strdup(optarg);
		break;
	    case 'k':
		known_hosts = pwmd_strdup(optarg);
		break;
	    case 'g':
		get = 1;
		break;
#endif
	    case 'L':
		inquire_line = optarg;
		break;
	    case 'y':
		tty = optarg;
		break;
	    case 't':
		ttytype = optarg;
		break;
	    case 'd':
		display = optarg;
		break;
	    case 'S':
		save = 1;
		break;
	    case 'I':
		iter = strtol(optarg, NULL, 10);
		have_iter = 1;
		break;
	    case 'P':
		password = pwmd_strdup(optarg);
		memset(optarg, 0, strlen(optarg));
		break;
	    case 'n':
		clientname = optarg;
		break;
	    default:
		pwmd_free(password);
		usage(argv[0], EXIT_FAILURE);
	}
    }

#ifdef HAVE_LIBREADLINE
    if (interactive && !isatty(STDIN_FILENO)) {
	pwmd_free(password);
	usage(argv[0], EXIT_FAILURE);
    }
#endif

#ifdef DEBUG
    if (!url_string) {
#endif
#ifdef WITH_TCP
    if (host && !get && !ident) {
	pwmd_free(password);
	usage(argv[0], EXIT_FAILURE);
    }

    if (get && !host) {
	pwmd_free(password);
	usage(argv[0], EXIT_FAILURE);
    }
#endif
#ifdef DEBUG
    }
#endif

    filename = argv[optind];
    pwmd_init();
    pwm = pwmd_new(clientname);
#ifdef DEBUG
    FD_ZERO(&rfds);
#endif
    
#ifdef WITH_TCP
    if (host || is_remote_url(url_string)) {
#ifndef WITH_PINENTRY
	no_pinentry = 1;
#endif
#ifndef DEBUG
	local_pin = 1;
#endif

	if (prot != PWMD_IP_ANY) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_IP_VERSION, prot);

	    if (error)
		goto done;
	}

	error = pwmd_setopt(pwm, PWMD_OPTION_KNOWNHOST_CB, knownhost_cb);

	if (error)
	    goto done;

	error = pwmd_setopt(pwm, PWMD_OPTION_KNOWNHOST_DATA, clientname);

	if (error)
	    goto done;

	if (!getenv("SSH_AGENT_PID") || ident)
	    use_ssh_agent = 0;

	error = pwmd_setopt(pwm, PWMD_OPTION_SSH_AGENT, use_ssh_agent);

	if (error)
	    goto done;

	fprintf(stderr, N_("Connecting ...\n"));

#ifdef DEBUG
	if (method >= 2) {
	    if (get) {
		char *hostkey;

		error = pwmd_get_hostkey_async(pwm, host, port);

		if (error)
		    errx(EXIT_FAILURE, "%s: %s", host, pwmd_strerror(error));

		error = process_cmd(pwm, &hostkey, 0, 0);

		if (error)
		    goto done;

		printf("%s", hostkey);
		pwmd_free(hostkey);
		pwmd_free(password);
		pwmd_close(pwm);
		exit(EXIT_SUCCESS);
	    }

	    if (url_string)
		error = pwmd_connect_url_async(pwm, url_string);
	    else
		error = pwmd_ssh_connect_async(pwm, host, port, ident, username,
			known_hosts);

	    if (error)
		goto done;

	    error = process_cmd(pwm, NULL, 0, 0);

	    if (error)
		goto done;
	}
	else {
#endif
	    if (get) {
		char *hostkey;

		error = pwmd_get_hostkey(pwm, host, port, &hostkey);

		if (error)
		    goto done;

		printf("%s", hostkey);
		pwmd_free(hostkey);
		pwmd_free(password);
		pwmd_close(pwm);
		exit(EXIT_SUCCESS);
	    }

	    if (url_string)
		error = pwmd_connect_url(pwm, url_string);
	    else
		error = pwmd_ssh_connect(pwm, host, port, ident, username, known_hosts);

	    if (error)
		goto done;
#ifdef DEBUG
	}
#endif
    }
    else {
#endif
	if (url_string)
	    error = pwmd_connect_url(pwm, url_string);
	else
	    error = pwmd_connect(pwm, socketpath);

	if (error)
	    goto done;
#ifdef WITH_TCP
    }
#endif

    fprintf(stderr, N_("Connected.\n"));
    connected = 1;
    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TITLE, NULL);
    error = pwmd_socket_type(pwm, &s);

    if (error)
	goto done;

#ifndef DEBUG
    if (s == PWMD_SOCKET_SSH && force_save && !local_pin) {
	error = GPG_ERR_WRONG_KEY_USAGE;
	goto done;
    }
#endif

    if (have_iter) {
	error = pwmd_command(pwm, &result, "VERSION");

	if (error)
	    goto done;

	pwmd_free(result);

	if (iter < 0) {
	    pwmd_free(password);
	    pwmd_close(pwm);
	    usage(argv[0], EXIT_FAILURE);
	}
    }

    if (filename && lock_on_open) {
	error = pwmd_setopt(pwm, PWMD_OPTION_LOCK_ON_OPEN, 1);

	if (error)
	    goto done;
    }

    if (base64) {
	error = pwmd_setopt(pwm, PWMD_OPTION_BASE64, 1);

	if (error)
	    goto done;
    }

    if (timeout > 0) {
	error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TIMEOUT, timeout);

	if (error)
	    goto done;
    }

    if (no_pinentry) {
	error = pwmd_setopt(pwm, PWMD_OPTION_NO_PINENTRY, 1);

	if (error)
	    goto done;
    }

    if ((no_pinentry || local_pin) && filename && !keyfile && !password) {
	error = pwmd_command(pwm, NULL, "ISCACHED %s", filename);

	if (error == GPG_ERR_NOT_FOUND) {
local_password:
	    if (try++ == local_tries)
		goto done;

	    if (password)
		pwmd_free(password);

	    password = NULL;

	    if (!no_pinentry) {

		if (try > 1)
		    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TIMEOUT, 0);

		error = pwmd_getpin(pwm, filename, &password,
			try > 1 ? PWMD_PINENTRY_OPEN_FAILED : PWMD_PINENTRY_OPEN);

		if (error)
		    goto done;

		if (!password)
		    password = pwmd_strdup("");
	    }
	    else {
		error = get_password(&password,try > 1 ? PWMD_PINENTRY_OPEN_FAILED : PWMD_PINENTRY_OPEN, 0);
	    }

	    if (error)
		goto done;

	    error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

	    if (error)
		goto done;

	    if (try > 1)
		goto do_open;
	}
	else if (error && error != GPG_ERR_ENOENT)
	    goto done;
    }

    error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

    if (error)
	goto done;

    if (!no_pinentry) {
	if (pinentry_path) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_PATH, pinentry_path);

	    if (error)
		goto done;
	}

	if (display) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_DISPLAY, display);

	    if (error)
		goto done;
	}

	if (tty) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TTY, tty);

	    if (error)
		goto done;
	}

	if (ttytype) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TERM, ttytype);

	    if (error)
		goto done;
	}

	if (lcctype) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_LC_CTYPE, lcctype);

	    if (error)
		goto done;
	}

	if (lcmessages) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_LC_MESSAGES,
		    lcmessages);

	    if (error)
		goto done;
	}

	if (tries > 0) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TRIES, tries);

	    if (error)
		goto done;
	}
    }

    if (show_status) {
	error = pwmd_setopt(pwm, PWMD_OPTION_STATUS_CB, status_msg_cb);

	if (error)
	    goto done;
    }

do_open:
    if (filename) {
	fprintf(stderr, N_("Opening data file \"%s\" ...\n"), filename);

#ifdef WITH_TCP
	if (keyfile && (!local_pin || host || is_remote_url(url_string))) {
	    struct inquire_s inq;

	    memset(&inq, 0, sizeof(inq));
	    inq.fd = open(keyfile, O_RDONLY);

	    if (inq.fd == -1) {
		error = gpg_error_from_syserror();
		goto done;
	    }

	    inq.line = pwmd_calloc(1, ASSUAN_LINELENGTH);

	    if (!inq.line) {
		error = GPG_ERR_ENOMEM;
		goto done;
	    }

	    in_keyfile = 1;
	    error = pwmd_open_inquire(pwm, filename, inquire_cb, &inq);
	    in_keyfile = 0;
	    pwmd_free(inq.line);
	    close(inq.fd);

	    if (error)
		goto done;

	    goto read_command;
	}
#endif
#ifdef DEBUG
	switch (method) {
	    case 0:
		error = pwmd_open(pwm, filename);
		break;
	    case 1:
		error = pwmd_open2(pwm, filename);
		break;
	    case 2:
		error = pwmd_open_async(pwm, filename);

		break;
	    case 3:
		error = pwmd_open_async2(pwm, filename);
		break;
	}

	if (error && (local_pin || no_pinentry) &&
		error == GPG_ERR_INV_PASSPHRASE)
	    goto local_password;

	if (error)
	    goto done;

	if (method >= 2)
	    error = process_cmd(pwm, &result, 0, 0);
#else
#ifdef WITH_PINENTRY
	if (local_pin)
	    error = pwmd_open2(pwm, filename);
	else
#endif
	    error = pwmd_open(pwm, filename);

	if (error && (local_pin || no_pinentry) &&
		error == GPG_ERR_INV_PASSPHRASE)
	    goto local_password;
#endif

	if (error)
	    goto done;
    }

#ifdef WITH_TCP
read_command:
#endif
#ifdef HAVE_LIBREADLINE
    if (interactive) {
	if (!force_save)
	    save = 0;

	error = do_interactive();
	goto do_exit;
    }
#endif

    if (inquire) {
	int fd = fileno(inquirefp);

	if (fd == -1) {
	    error = gpg_error_from_syserror();
	    goto done;
	}

	error = send_inquire(fd, inquire, inquire_line);
	close(fd);
	goto done;
    }

    fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK);
    ssize_t n;

    for (;;) {
	error = process_cmd(pwm, NULL, 1, 0);

	if (error)
	    goto done;

	n = read(STDIN_FILENO, command, sizeof(command));

	if (n == -1) {
	    if (errno == EAGAIN)
		continue;

	    error = gpg_error_from_errno(errno);
	    goto done;
	}
	else if (!n)
	    goto done;

	p = command;
	command[n] = 0;
	break;
    }

    if (!p || !*p)
	goto done;

    if (strcasecmp(p, "BYE") == 0)
	goto done;

    error = pwmd_command(pwm, &result, command);

    if (error)
	goto done;

    if (result) {
	fwrite(result, 1, strlen(result), outfp);
	fflush(outfp);
	pwmd_free(result);
    }

done:
    if (password)
	pwmd_free(password);

    password = NULL;

    if (!error)
	error = finalize();

do_exit:
    memset(command, 0, sizeof(command));

    if (error) {
	show_error(error);
	ret = EXIT_FAILURE;
    }

    pwmd_close(pwm);

    if (connected)
	fprintf(stderr, N_("Connection closed.\n"));

    if (socketpath)
	pwmd_free(socketpath);

    exit(ret);
}
