/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <err.h>
#include <pwd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <time.h>
#include <pthread.h>
#include <limits.h>

#ifdef WITH_LIBPTH
#include <pth.h>
#endif

#include "types.h"
#include "misc.h"
#include "ssh.h"

static gpg_error_t ssh_connect_finalize(pwm_t *pwm);
int read_hook_common(assuan_context_t ctx, assuan_fd_t fd, void *data,
	size_t len, ssize_t *ret);
int write_hook_common(assuan_context_t ctx, assuan_fd_t fd, const void *data,
	size_t len, ssize_t *ret);

static void close_agent(pwmd_tcp_conn_t *conn)
{
    if (conn->agent) {
	libssh2_agent_disconnect(conn->agent);
	libssh2_agent_free(conn->agent);
	conn->agent = NULL;
    }
}

static void ssh_deinit(pwmd_tcp_conn_t *conn)
{
    if (!conn)
	return;

    close_agent(conn);
    /* Fixes error messages in the pwmd log. */
    libssh2_channel_wait_closed(conn->channel);

    if (conn->channel) {
        libssh2_channel_close(conn->channel);
        libssh2_channel_free(conn->channel);
    }

    if (conn->kh) {
	libssh2_knownhost_free(conn->kh);
	conn->kh = NULL;
    }

    if (conn->session) {
	libssh2_session_disconnect(conn->session, N_("libpwmd saying bye!"));
	libssh2_session_free(conn->session);
    }

    conn->session = NULL;
    conn->channel = NULL;
    _free_ssh_conn(conn);
}

ssize_t read_hook_ssh(pwm_t *pwm, assuan_fd_t fd, void *data, size_t len)
{
    return libssh2_channel_read(pwm->tcp_conn->channel, data, len);
}

ssize_t write_hook_ssh(pwm_t *pwm, assuan_fd_t fd, const void *data, size_t len)
{
    ssize_t ret;

    /* libassuan cannot handle EAGAIN when doing writes. */
    do {
	ret = libssh2_channel_write(pwm->tcp_conn->channel, data, len);

	if (ret == LIBSSH2_ERROR_EAGAIN) {
#ifdef WITH_LIBPTH
	    pth_usleep(50000);
#else
	    usleep(50000);
#endif
	}
    } while (ret == LIBSSH2_ERROR_EAGAIN);

    return ret;
}

void _free_ssh_conn(pwmd_tcp_conn_t *conn)
{
    if (!conn)
	return;

    if (conn->username) {
	pwmd_free(conn->username);
	conn->username = NULL;
    }

    if (conn->known_hosts) {
	pwmd_free(conn->known_hosts);
	conn->known_hosts = NULL;
    }

    if (conn->identity) {
	pwmd_free(conn->identity);
	conn->identity = NULL;
    }

    if (conn->identity_pub) {
	pwmd_free(conn->identity_pub);
	conn->identity_pub = NULL;
    }

    if (conn->host) {
	pwmd_free(conn->host);
	conn->host = NULL;
    }

    if (conn->hostkey) {
	pwmd_free(conn->hostkey);
	conn->hostkey = NULL;
    }

    if (conn->chan) {
	ares_destroy(conn->chan);
	conn->chan = NULL;
    }

    if (!conn->session && conn->fd >= 0) {
	close(conn->fd);
	conn->fd = -1;
    }

    if (conn->session)
	ssh_deinit(conn);
    else
	pwmd_free(conn);
}

/*
 * Sets common options from both pwmd_ssh_connect() and
 * pwmd_ssh_connect_async().
 */
static gpg_error_t init_tcp_conn(pwmd_tcp_conn_t **dst, const char *host,
	int port, const char *identity, const char *user,
	const char *known_hosts, int get, int resume, int use_agent)
{
    pwmd_tcp_conn_t *conn = *dst;
    gpg_error_t rc = 0;
    char *pwbuf = NULL;

    if (get) {
	if (resume) {
	    if (host)
		return GPG_ERR_INV_STATE;

	    return 0;
	}

	if (!host || !*host)
	    return GPG_ERR_INV_ARG;
    }
    else if (!resume) {
	if (!host || !*host || (!use_agent && (!identity || !*identity)))
	    return GPG_ERR_INV_ARG;
    }
    else if (resume) {
	if (host)
	    return GPG_ERR_INV_STATE;

	if (!use_agent && (!identity || !*identity))
	    return GPG_ERR_INV_ARG;
    }

    if ((known_hosts && !*known_hosts) || (!use_agent && identity && !*identity))
	return GPG_ERR_INV_ARG;

    if (!resume) {
	conn = pwmd_calloc(1, sizeof(pwmd_tcp_conn_t));

	if (!conn)
	    return gpg_error_from_errno(ENOMEM);
    }

    if (!get) {
	struct passwd pw;

	pwbuf = _getpwuid(&pw);

	if (!pwbuf) {
	    rc = gpg_error_from_errno(errno);
	    goto fail;
	}

	if (conn->username)
	    pwmd_free(conn->username);

	conn->username = pwmd_strdup(user ? user : pw.pw_name);

	if (!conn->username) {
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	if (conn->identity)
	    pwmd_free(conn->identity);

	conn->identity = NULL;

	if (conn->identity_pub)
	    pwmd_free(conn->identity_pub);

	conn->identity_pub = NULL;

	if (identity) {
	    conn->identity = _expand_homedir((char *)identity, &pw);

	    if (!conn->identity) {
		rc = gpg_error_from_errno(ENOMEM);
		goto fail;
	    }

	    conn->identity_pub = pwmd_strdup_printf("%s.pub", conn->identity);

	    if (!conn->identity_pub) {
		rc = gpg_error_from_errno(ENOMEM);
		goto fail;
	    }
	}

	if (conn->known_hosts)
	    pwmd_free(conn->known_hosts);

	if (!known_hosts)
	    known_hosts = "~/.ssh/known_hosts";

	conn->known_hosts = _expand_homedir((char *)known_hosts, &pw);

	if (!conn->known_hosts) {
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	pwmd_free(pwbuf);
    }

    if (!resume) {
	conn->port = port;
	conn->host = pwmd_strdup(host);

	if (!conn->host) {
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	*dst = conn;
    }

    return 0;

fail:
    if (pwbuf)
	pwmd_free(pwbuf);

    _free_ssh_conn(conn);
    return rc;
}

static gpg_error_t do_connect(pwm_t *pwm, int prot, void *addr)
{
    struct sockaddr_in their_addr;

    pwm->tcp_conn->fd = socket(prot, SOCK_STREAM, 0);

    if (pwm->tcp_conn->fd == -1)
	return gpg_error_from_syserror();

    if (pwm->tcp_conn->async)
	fcntl(pwm->tcp_conn->fd, F_SETFL, O_NONBLOCK);

    pwm->cmd = ASYNC_CMD_CONNECT;
    their_addr.sin_family = prot;
    their_addr.sin_port = htons(pwm->tcp_conn->port == -1 ? 22 : pwm->tcp_conn->port);
    their_addr.sin_addr = *((struct in_addr *)addr);
    pwm->tcp_conn->addr = *((struct in_addr *)addr);
    pwm->tcp_conn->addrtype = prot;
    memset(their_addr.sin_zero, '\0', sizeof their_addr.sin_zero);

#ifdef WITH_LIBPTH
    if (pth_connect(pwm->tcp_conn->fd, (struct sockaddr *)&their_addr,
		sizeof(their_addr)) == -1)
#else
    if (connect(pwm->tcp_conn->fd, (struct sockaddr *)&their_addr,
		sizeof(their_addr)) == -1)
#endif
	return gpg_error_from_syserror();

    return 0;
}

static gpg_error_t ares_error_to_pwmd(int status)
{
    if (status != ARES_SUCCESS && status != ARES_EDESTRUCTION)
	warnx("%s", ares_strerror(status));

    switch (status) {
	case ARES_EDESTRUCTION:
	    return GPG_ERR_CANCELED;
	case ARES_ENODATA:
	case ARES_EFORMERR:
	case ARES_ENOTFOUND:
	    return GPG_ERR_UNKNOWN_HOST;
	case ARES_ESERVFAIL:
	     return GPG_ERR_EHOSTDOWN;
	case ARES_ETIMEOUT:
	    return GPG_ERR_TIMEOUT;
	case ARES_ENOMEM:
	    return gpg_error_from_errno(ENOMEM);
	case ARES_ECONNREFUSED:
	    return GPG_ERR_ECONNREFUSED;
	default:
	    /* FIXME ??? */
	    return GPG_ERR_EHOSTUNREACH;
    }

    return ARES_SUCCESS;
}

static void dns_resolve_cb(void *arg, int status, int timeouts,
	unsigned char *abuf, int alen)
{
    pwm_t *pwm = arg;
    int rc;
    struct hostent *he;

    if (status != ARES_SUCCESS) {
	pwm->tcp_conn->rc = ares_error_to_pwmd(status);
	return;
    }

    /* Check for an IPv6 address first. */
    if (pwm->prot == PWMD_IP_ANY || pwm->prot == PWMD_IPV6)
	rc = ares_parse_aaaa_reply(abuf, alen, &he, NULL, NULL);
    else
	rc = ares_parse_a_reply(abuf, alen, &he, NULL, NULL);

    if (rc != ARES_SUCCESS) {
	if (pwm->prot != PWMD_IP_ANY || rc != ARES_ENODATA) {
	    pwm->tcp_conn->rc = ares_error_to_pwmd(status);
	    return;
	}

	rc = ares_parse_a_reply(abuf, alen, &he, NULL, NULL);

	if (rc != ARES_SUCCESS) {
	    pwm->tcp_conn->rc = ares_error_to_pwmd(status);
	    return;
	}
    }

    pwm->tcp_conn->rc = do_connect(pwm, he->h_addrtype, he->h_addr);
    ares_free_hostent(he);
}

gpg_error_t _do_pwmd_ssh_connect_async(pwm_t *pwm, const char *host,
	int port, const char *identity, const char *user,
	const char *known_hosts, pwmd_async_cmd_t which)
{
    pwmd_tcp_conn_t *conn;
    gpg_error_t rc;
    int resume = 0;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_ASS_NESTED_COMMANDS;

    /* Resume an existing connection that may have been started from
     * pwmd_get_hostkey(). */
    if (pwm->tcp_conn) {
	resume = 1;
	conn = pwm->tcp_conn;
    }

    rc = init_tcp_conn(&conn, host, port, identity, user, known_hosts,
	    which == ASYNC_CMD_HOSTKEY ? 1 : 0, resume, pwm->use_agent);

    if (rc)
	return rc;

    conn->async = 1;
    pwm->tcp_conn = conn;
    pwm->tcp_conn->cmd = which;
    pwm->cmd = resume ? ASYNC_CMD_CONNECT : ASYNC_CMD_DNS;
    pwm->state = ASYNC_PROCESS;

    if (!resume) {
	struct in_addr addr;

	ares_init(&pwm->tcp_conn->chan);
	
	if (inet_pton(AF_INET, pwm->tcp_conn->host, &addr)) {
	    pwm->tcp_conn->rc = do_connect(pwm, AF_INET, &addr);
	    return 0;
	}
	else if (inet_pton(AF_INET6, pwm->tcp_conn->host, &addr)) {
	    pwm->tcp_conn->rc = do_connect(pwm, AF_INET6, &addr);
	    return 0;
	}

	ares_query(pwm->tcp_conn->chan, pwm->tcp_conn->host, ns_c_any,
		ns_t_any, dns_resolve_cb, pwm);
    }
    else {
	/* There may not be any pending data waiting to be read from the SSH
	 * FD so resume the connection here instead of from pwmd_process(). */
	rc = _setup_ssh_session(pwm);

	if (rc == GPG_ERR_EAGAIN)
	    rc = 0;
    }

    return rc;
}

static void *ssh_malloc(size_t size, void **data)
{
    return pwmd_malloc(size);
}

static void ssh_free(void *ptr, void **data)
{
    pwmd_free(ptr);
}

static void *ssh_realloc(void *ptr, size_t size, void **data)
{
    return pwmd_realloc(ptr, size);
}

gpg_error_t _setup_ssh_agent(pwm_t *pwm)
{
    int n;

    if (pwm->tcp_conn->state != SSH_AGENT) {
	n = libssh2_agent_connect(pwm->tcp_conn->agent);

	if (n) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_NO_AGENT;
	}

	n = libssh2_agent_list_identities(pwm->tcp_conn->agent);

	if (n) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_KEYRING_OPEN;
	}

	pwm->tcp_conn->state = SSH_AGENT;
	n = libssh2_agent_get_identity(pwm->tcp_conn->agent,
		&pwm->tcp_conn->agent_identity, pwm->tcp_conn->agent_identity_prev);

	if (n > 0) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_NO_SECKEY;
	}
	else if (n < 0) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_AGENT;
	}
    }

    for (;;) {
	n = libssh2_agent_userauth(pwm->tcp_conn->agent,
		pwm->tcp_conn->username, pwm->tcp_conn->agent_identity);

	if (!n)
	    break;
	else if (n == LIBSSH2_ERROR_EAGAIN) {
	    return GPG_ERR_EAGAIN;
	}

	pwm->tcp_conn->agent_identity_prev = pwm->tcp_conn->agent_identity;
	n = libssh2_agent_get_identity(pwm->tcp_conn->agent,
		&pwm->tcp_conn->agent_identity,
		pwm->tcp_conn->agent_identity_prev);

	if (n > 0) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_NO_SECKEY;
	}
	else if (n < 0) {
	    _free_ssh_conn(pwm->tcp_conn);
	    pwm->tcp_conn = NULL;
	    return GPG_ERR_AGENT;
	}
    }

    return _setup_ssh_channel(pwm);
}

gpg_error_t _setup_ssh_auth(pwm_t *pwm)
{
    int n;
    
    pwm->tcp_conn->state = SSH_AUTH;

    if (pwm->use_agent)
	return _setup_ssh_agent(pwm);

    n = libssh2_userauth_publickey_fromfile(pwm->tcp_conn->session,
	    pwm->tcp_conn->username, pwm->tcp_conn->identity_pub,
	    pwm->tcp_conn->identity, NULL);

    if (n == LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_EAGAIN;
    else if (n) {
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
	return GPG_ERR_BAD_SECKEY;
    }

    return _setup_ssh_channel(pwm);
}

gpg_error_t _setup_ssh_authlist(pwm_t *pwm)
{
    char *userauth;
    int n;

    pwm->tcp_conn->state = SSH_AUTHLIST;
    userauth = libssh2_userauth_list(pwm->tcp_conn->session,
	    pwm->tcp_conn->username, strlen(pwm->tcp_conn->username));
    n = libssh2_session_last_errno(pwm->tcp_conn->session);

    if (!userauth && n == LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_EAGAIN;
    else if (!userauth || !strstr(userauth, "publickey")) {
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
	return GPG_ERR_BAD_PIN_METHOD;
    }
    else if (n && n != LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_ASS_SERVER_START;

    return _setup_ssh_auth(pwm);
}

static void add_knownhost(pwm_t *pwm, const char *host, const char *key,
	size_t len, int type, struct libssh2_knownhost **dst)
{
    char *buf;

    if (pwm->tcp_conn->port != -1 && pwm->tcp_conn->port != 22) {
	buf = pwmd_malloc(256);
	snprintf(buf, 256, "[%s]:%i", host, pwm->tcp_conn->port);
    }
    else
	buf = pwmd_strdup(host);

    char *tbuf = pwmd_strdup_printf("libpwmd-%li", time(NULL));
    libssh2_knownhost_addc(pwm->tcp_conn->kh, buf, NULL, key, len, tbuf,
	    strlen(tbuf), type, dst);
    pwmd_free(tbuf);
    pwmd_free(buf);
}

static gpg_error_t check_known_hosts(pwm_t *pwm)
{
    size_t len;
    int type;
    const char *key;
    gpg_error_t rc = 0;
    int n;
    struct libssh2_knownhost *kh;

    key = libssh2_session_hostkey(pwm->tcp_conn->session, &len, &type);

    while (!libssh2_knownhost_get(pwm->tcp_conn->kh, &kh, NULL))
	libssh2_knownhost_del(pwm->tcp_conn->kh, kh);

    n = libssh2_knownhost_readfile(pwm->tcp_conn->kh,
	    pwm->tcp_conn->known_hosts, LIBSSH2_KNOWNHOST_FILE_OPENSSH);

    if (n < 0 && pwm->tcp_conn->cmd != ASYNC_CMD_HOSTKEY &&
	    n != LIBSSH2_ERROR_FILE)
	return GPG_ERR_BAD_CERT;

    n = libssh2_knownhost_checkp(pwm->tcp_conn->kh, pwm->tcp_conn->host,
	    pwm->tcp_conn->port, (char *)key, len,
	    LIBSSH2_KNOWNHOST_TYPE_PLAIN|LIBSSH2_KNOWNHOST_KEYENC_RAW,
	    &pwm->tcp_conn->hostent);

    type = type == LIBSSH2_HOSTKEY_TYPE_RSA ?
	LIBSSH2_KNOWNHOST_KEY_SSHRSA : LIBSSH2_KNOWNHOST_KEY_SSHDSS;

    switch (n) {
	case LIBSSH2_KNOWNHOST_CHECK_MATCH:
	    break;
	case LIBSSH2_KNOWNHOST_CHECK_NOTFOUND:
	    if (pwm->tcp_conn->cmd != ASYNC_CMD_HOSTKEY) {
		if (!pwm->kh_cb)
		    rc = GPG_ERR_NOT_CONFIRMED;
		else
		    rc = pwm->kh_cb(pwm->kh_data, pwm->tcp_conn->host, key,
			    len);

		if (rc)
		    return rc;
	    }

	    add_knownhost(pwm, pwm->tcp_conn->host, key, len,
		    LIBSSH2_KNOWNHOST_TYPE_PLAIN |
		    LIBSSH2_KNOWNHOST_KEYENC_RAW | type,
		    &pwm->tcp_conn->hostent);

	    /* Adds both the IP and hostname. */
	    char *buf = pwmd_malloc(255);

	    if (buf) {
		const char *p = inet_ntop(pwm->tcp_conn->addrtype,
			&pwm->tcp_conn->addr, buf, 255);

		if (p && strcmp(pwm->tcp_conn->host, p)) {
		    struct libssh2_knownhost *kh, *pkh = NULL;
		    int match = 0;

		    while (!libssh2_knownhost_get(pwm->tcp_conn->kh, &kh, pkh)) {
			pkh = kh;

			if (kh->name && !strcmp(kh->name, p)) {
			    match = 1;
			    break;
			}
		    }

		    if (!match)
			add_knownhost(pwm, p, key, len,
				LIBSSH2_KNOWNHOST_TYPE_PLAIN |
				LIBSSH2_KNOWNHOST_KEYENC_RAW | type,
				&pwm->tcp_conn->hostent_ip);
		}

		pwmd_free(buf);
	    }

	    /* It's not an error if writing the new host file fails since
	     * there isn't a way to notify the user. The hostkey is still
	     * valid though. */
	    if (pwm->tcp_conn->cmd != ASYNC_CMD_HOSTKEY) {
		char *tmp = tempnam(NULL, "khost");

		if (!tmp)
		    return 0;

		if (!libssh2_knownhost_writefile(pwm->tcp_conn->kh, tmp,
			LIBSSH2_KNOWNHOST_FILE_OPENSSH)) {
		    char *buf;
		    FILE *ifp, *ofp;

		    buf = pwmd_malloc(LINE_MAX);

		    if (!buf) {
			unlink(tmp);
			free(tmp);
			return 0;
		    }

		    ifp = fopen(tmp, "r");

		    if (!ifp)
			goto done;

		    ofp = fopen(pwm->tcp_conn->known_hosts, "w+");

		    if (!ofp)
			goto done;

		    while ((fgets(buf, LINE_MAX, ifp))) {
			if (fprintf(ofp, "%s", buf) < 0)
			    break;
		    }

done:
		    if (ifp)
			fclose(ifp);

		    if (ofp)
			fclose(ofp);

		    pwmd_free(buf);
		}

		unlink(tmp);
		free(tmp);
	    }

	    return 0;
	case LIBSSH2_KNOWNHOST_CHECK_MISMATCH:
	case LIBSSH2_KNOWNHOST_CHECK_FAILURE:
	    return GPG_ERR_BAD_CERT;
    }

    return 0;
}

static gpg_error_t verify_hostkey(pwm_t *pwm)
{
    gpg_error_t rc;
    size_t outlen;
    char *buf;

    if (!pwm->tcp_conn->kh) {
	pwm->tcp_conn->kh = libssh2_knownhost_init(pwm->tcp_conn->session);
    }

    if (!pwm->tcp_conn->kh)
	return GPG_ERR_ENOMEM;

    rc = check_known_hosts(pwm);

    if (rc)
	return rc;

    buf = pwmd_malloc(LINE_MAX);

    if (!buf)
	return gpg_error_from_errno(ENOMEM);

    if (libssh2_knownhost_writeline(pwm->tcp_conn->kh, pwm->tcp_conn->hostent,
		buf, LINE_MAX, &outlen, LIBSSH2_KNOWNHOST_FILE_OPENSSH)) {
	pwmd_free(buf);
	return gpg_error_from_errno(ENOMEM);
    }

    if (pwm->tcp_conn->hostkey)
	pwmd_free(pwm->tcp_conn->hostkey);

    pwm->tcp_conn->hostkey = NULL;

    if (pwm->tcp_conn->hostent_ip) {
	char *buf2 = pwmd_malloc(LINE_MAX);

	if (!buf2) {
	    pwmd_free(buf);
	    return gpg_error_from_errno(ENOMEM);
	}

	if (libssh2_knownhost_writeline(pwm->tcp_conn->kh,
		    pwm->tcp_conn->hostent_ip, buf2, LINE_MAX, &outlen,
		    LIBSSH2_KNOWNHOST_FILE_OPENSSH)) {
	    pwmd_free(buf);
	    pwmd_free(buf2);
	    return gpg_error_from_errno(ENOMEM);
	}

	pwm->tcp_conn->hostkey = pwmd_strdup_printf("%s%s", buf, buf2);
	pwmd_free(buf);
	pwmd_free(buf2);

	if (!pwm->tcp_conn->hostkey)
	    return gpg_error_from_errno(ENOMEM);
    }
    else
	pwm->tcp_conn->hostkey = buf;

    if (pwm->tcp_conn->cmd == ASYNC_CMD_HOSTKEY) {
	libssh2_knownhost_del(pwm->tcp_conn->kh, pwm->tcp_conn->hostent);

	if (pwm->tcp_conn->hostent_ip)
	    libssh2_knownhost_del(pwm->tcp_conn->kh, pwm->tcp_conn->hostent_ip);

	pwm->tcp_conn->hostent = NULL;
	pwm->tcp_conn->hostent_ip = NULL;
	pwm->tcp_conn->state = SSH_RESUME;
	return 0;
    }

    return _setup_ssh_authlist(pwm);
}

gpg_error_t _setup_ssh_channel(pwm_t *pwm)
{
    int n;
    gpg_error_t rc = 0;

    close_agent(pwm->tcp_conn);

    pwm->tcp_conn->state = SSH_CHANNEL;
    pwm->tcp_conn->channel =
	libssh2_channel_open_session(pwm->tcp_conn->session);
    n = libssh2_session_last_errno(pwm->tcp_conn->session);

    if (!pwm->tcp_conn->channel && n == LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_EAGAIN;

    if (!pwm->tcp_conn->channel) {
	rc = GPG_ERR_ASS_SERVER_START;
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
	return rc;
    }

    return _setup_ssh_shell(pwm);
}

gpg_error_t _setup_ssh_shell(pwm_t *pwm)
{
    int n;
    gpg_error_t rc;
    
    pwm->tcp_conn->state = SSH_SHELL;
    n = libssh2_channel_shell(pwm->tcp_conn->channel);

    if (n == LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_EAGAIN;
    else if (n) {
	rc = GPG_ERR_ASS_SERVER_START;
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
	return rc;
    }

    return ssh_connect_finalize(pwm);
}

static gpg_error_t ssh_connect_finalize(pwm_t *pwm)
{
    gpg_error_t rc;
    assuan_context_t ctx;
    struct assuan_io_hooks io_hooks = {read_hook_common, write_hook_common};

    assuan_set_io_hooks(&io_hooks);
    rc = assuan_socket_connect_fd(&ctx, pwm->tcp_conn->fd, 0, pwm);

    if (rc)
	goto fail;

    pwm->ctx = ctx;
    rc = _connect_finalize(pwm);

    if (rc)
	goto fail;

    return 0;

fail:
    _free_ssh_conn(pwm->tcp_conn);
    pwm->tcp_conn = NULL;
    return gpg_err_code(rc);
}

gpg_error_t _setup_ssh_init(pwm_t *pwm)
{
    int n;
    
    /* Resuming an SSH connection which may have been initially created with
     * pwmd_get_hostkey(). */
    if (pwm->tcp_conn->state == SSH_RESUME)
	goto done;

    pwm->tcp_conn->state = SSH_INIT;
    n = libssh2_session_startup(pwm->tcp_conn->session, pwm->tcp_conn->fd);

    if (n == LIBSSH2_ERROR_EAGAIN)
	return GPG_ERR_EAGAIN;
    else if (n) {
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
	return GPG_ERR_ASSUAN_SERVER_FAULT;
    }

done:
    return verify_hostkey(pwm);
}

gpg_error_t _setup_ssh_session(pwm_t *pwm)
{
    gpg_error_t rc;

    if (!pwm->tcp_conn->session) {
	pwm->tcp_conn->session = libssh2_session_init_ex(ssh_malloc, ssh_free,
		ssh_realloc, NULL);
#if LIBSSH2_VERSION_NUM >= 0x010208
	libssh2_session_flag(pwm->tcp_conn->session, LIBSSH2_FLAG_COMPRESS, 1);
#endif

	if (pwm->use_agent)
	    pwm->tcp_conn->agent = libssh2_agent_init(pwm->tcp_conn->session);
    }

    if (!pwm->tcp_conn->session) {
	rc = gpg_error_from_errno(ENOMEM);
	goto fail;
    }

    libssh2_session_set_blocking(pwm->tcp_conn->session, 0);
    return _setup_ssh_init(pwm);

fail:
    _free_ssh_conn(pwm->tcp_conn);
    pwm->tcp_conn = NULL;
    return gpg_err_code(rc);
}

static void gethostbyname_cb(void *arg, int status, int timeouts,
	struct hostent *he)
{
    pwm_t *pwm = arg;

    if (status != ARES_SUCCESS) {
	pwm->tcp_conn->rc = ares_error_to_pwmd(status);
	return;
    }

    pwm->tcp_conn->rc = do_connect(pwm, he->h_addrtype, he->h_addr);
}

gpg_error_t _do_pwmd_ssh_connect(pwm_t *pwm, const char *host, int port,
	const char *identity, const char *user, const char *known_hosts, int get)
{
    pwmd_tcp_conn_t *conn;
    gpg_error_t rc;
    int resume = 0;
    struct in_addr addr;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_INV_STATE;

    if (pwm->tcp_conn) {
	pwm->tcp_conn->async = 0;
	resume = 1;
	conn = pwm->tcp_conn;
    }

    rc = init_tcp_conn(&conn, host, port, identity, user, known_hosts, get,
	    resume, pwm->use_agent);

    if (rc)
	return rc;

    pwm->tcp_conn = conn;
    pwm->tcp_conn->cmd = get ? ASYNC_CMD_HOSTKEY : ASYNC_CMD_NONE;
    pwm->cmd = ASYNC_CMD_NONE;

    if (resume)
	goto done;

    pwm->cmd = ASYNC_CMD_DNS;

    if (inet_pton(AF_INET, pwm->tcp_conn->host, &addr)) {
	rc = do_connect(pwm, AF_INET, &addr);

	if (rc)
	    goto fail;

	goto done;
    }
    else if (inet_pton(AF_INET6, pwm->tcp_conn->host, &addr)) {
	rc = do_connect(pwm, AF_INET6, &addr);

	if (rc)
	    goto fail;

	goto done;
    }

    ares_init(&pwm->tcp_conn->chan);
    ares_gethostbyname(pwm->tcp_conn->chan, pwm->tcp_conn->host,
	    pwm->prot == PWMD_IP_ANY ||
	    pwm->prot == PWMD_IPV4 ? AF_INET : AF_INET6,
	    gethostbyname_cb, pwm);

    /* gethostbyname_cb() may have already been called. */
    if (pwm->tcp_conn->rc) {
	rc = pwm->tcp_conn->rc;
	goto fail;
    }

    /*
     * Fake a blocking DNS lookup. libcares does a better job than
     * getaddrinfo().
     */
    do {
	fd_set rfds, wfds;
	int n;
	struct timeval tv;

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	n = ares_fds(pwm->tcp_conn->chan, &rfds, &wfds);
	ares_timeout(pwm->tcp_conn->chan, NULL, &tv);

	if (!n)
	    break;

#ifdef WITH_LIBPTH
	n = pth_select(n, &rfds, &wfds, NULL, &tv);
#else
	n = select(n, &rfds, &wfds, NULL, &tv);
#endif

	if (n == -1) {
	    rc = gpg_error_from_syserror();
	    goto fail;
	}
	else if (n == 0) {
	    rc = GPG_ERR_TIMEOUT;
	    goto fail;
	}

	ares_process(pwm->tcp_conn->chan, &rfds, &wfds);

	if (pwm->tcp_conn->rc)
	    break;
    } while (pwm->cmd == ASYNC_CMD_DNS);

    if (pwm->tcp_conn->rc) {
	rc = pwm->tcp_conn->rc;
	goto fail;
    }

done:
    rc = _setup_ssh_session(pwm);
    pwm->cmd = ASYNC_CMD_NONE;

    if (pwm->tcp_conn)
	pwm->tcp_conn->cmd = ASYNC_CMD_NONE;

fail:
    return rc;
}

/*
 * ssh[46]://[username@]hostname[:port],identity[,known_hosts]
 *
 * Any missing parameters are checked for in init_tcp_conn().
 */
gpg_error_t _parse_ssh_url(char *str, char **host, int *port, char **user,
	char **identity, char **known_hosts)
{
    char *p;
    char *t;
    int len;

    *host = *user = *identity = *known_hosts = NULL;
    *port = -1;
    p = strrchr(str, '@');

    if (p) {
	len = strlen(str)-strlen(p)+1;
	*user = pwmd_malloc(len);

	if (!*user)
	    return gpg_error_from_errno(ENOMEM);

	snprintf(*user, len, "%s", str);
	p++;
    }
    else
	p = str;

    t = strchr(p, ':');

    if (t) {
	len = strlen(p)-strlen(t)+1;
	*host = pwmd_malloc(len);

	if (!*host)
	    return gpg_error_from_errno(ENOMEM);

	snprintf(*host, len, "%s", p);
	t++;
	*port = atoi(t);

	if (*t == '-')
	    t++;

	while (*t && isdigit(*t))
	    t++;

	p = t;
    }

    t = strchr(p, ',');

    if (t) {
	char *t2;

	if (!*host) {
	    len = strlen(p)-strlen(t)+1;
	    *host = pwmd_malloc(len);

	    if (!*host)
		return gpg_error_from_errno(ENOMEM);

	    snprintf(*host, len, "%s", p);
	}

	t++;
	t2 = strchr(t, ',');

	if (t2)
	    len = strlen(t)-strlen(t2)+1;
	else
	    len = strlen(t)+1;

	*identity = pwmd_malloc(len);

	if (!*identity)
	    return gpg_error_from_errno(ENOMEM);

	snprintf(*identity, len, "%s", t);

	if (t2) {
	    t2++;
	    t += len+1;
	    len = strlen(t2)+1;
	    *known_hosts = pwmd_malloc(len);

	    if (!*known_hosts)
		return gpg_error_from_errno(ENOMEM);

	    snprintf(*known_hosts, len, "%s", t2);
	}
    }
    else {
	if (!*host) {
	    len = strlen(p)+1;
	    *host = pwmd_malloc(len);

	    if (!*host)
		return gpg_error_from_errno(ENOMEM);

	    snprintf(*host, len, "%s", p);
	}
    }

    return 0;
}

void _ssh_disconnect(pwm_t *pwm)
{
    ssh_deinit(pwm->tcp_conn);
    pwm->tcp_conn = NULL;
}
