/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* daemon logger
*
****************************************************************************
****************************************************************************
***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/ll_bt.h"
#include "misc.h"
#include "sm.h"

/***************************************************************************
* log atoms
*
* E: = error
* W: = Warning
* I: = Information
***************************************************************************/
void print_main_log(char *entry,char *log_file)
{
  FILE *fp;
  char str[STR_LEN];

  sprintf(str,"%s/log/%s",VAR_PATH,log_file);
  if((fp = fopen(str,"a")) != NULL)
  {
    fprintf(fp,"%s\n",entry);
    fclose(fp);
  }
}

void compose_log_msg(char *str,char stat,char *msg,int srl)
{
  struct tm *tm_st;
  int t;

  t = time(NULL);
  tm_st = localtime((time_t *)&t);
  if(tm_st != NULL)
  {
    char *mon_tab[] = {"Jan","Feb","Mar","Apr","May","Jun",
                       "Jul","Aug","Sep","Oct","Nov","Dec"};

    sprintf(str,"%s%02d %02d:%02d:%02d %c%1d: %s",
      mon_tab[tm_st->tm_mon],tm_st->tm_mday,
      tm_st->tm_hour,tm_st->tm_min,tm_st->tm_sec,stat,srl,msg);
  }
}

/***************************************************************************
* main logs
***************************************************************************/
void log_msg_main_file(char stat,char *msg,int srl,char *log_file)
{
  if(us_hr.flags.flag.run_level <= srl)
  {
    char str[STR_LEN];

    compose_log_msg(str,stat,msg,srl);
    print_main_log(str,log_file);
  }
}

void log_msg_main(char stat,char *msg,int srl)
{
  log_msg_main_file(stat,msg,srl,"Main.log");
}

/***************************************************************************
* user logging
***************************************************************************/
void log_msg_user_file(char stat,char *msg,int srl,char *user,char *log_file)
{
  FILE *fp;
  char str[STR_LEN];

  umask(0);
  sprintf(str,"%s/log/%s",VAR_PATH,user);
  mkdir(str,0755);
  umask(022);

  sprintf(str,"%s/log/%s/%s",VAR_PATH,user,log_file);
  if((fp = fopen(str,"a")) != NULL)
  {
    compose_log_msg(str,stat,msg,srl);
    fprintf(fp,"%s\n",str);
    fclose(fp);
  }
}

void log_msg_user_only(char stat,char *msg,int srl,char *user)
{
  log_msg_user_file(stat,msg,srl,user,"session.log");
}

void log_msg_user(char stat,char *msg,int srl,char *user)
{
  char str[STR_LEN];

  log_msg_user_only(stat,msg,srl,user);
  sprintf(str,"'%s' %s",user,msg);
  log_msg_main(stat,str,srl);
}

void log_msg_all_users(char stat,char *msg,int srl)
{
  tll *ll;

  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    log_msg_user_only(stat,msg,srl,um->user);
  }
}

/***************************************************************************
* error logging (used after a libc routine returns -1)
***************************************************************************/
void log_error(char *msg)
{
  char str[STR_LEN];

  sprintf(str,"%s: %s",msg,strerror(errno));
  log_msg_main('E',str,1);
}

/*
 * this is namely the ll_bt call back error routine.
 */
void fatal_malloc_error(void)
{
  log_msg_main('E',"Fatal malloc() error. Unable to alloc memory. Aborting.",1);
  exit(-1);
}

/***************************************************************************
* produces a session volume summary on users log file
***************************************************************************/
void log_user_stats(char *user,ttinfo *tinfo)
{
  char str[STR_LEN],nif_str[STR_SMALL],time_str[STR_SMALL];

  switch(tinfo->flags.flag.type)
  {

    case TINFO_TYPE_UNKNOWN:
      strcpy(nif_str,unknown_label);
      break;

    case TINFO_TYPE_LOGIN:
      strcpy(nif_str,"shell");
      break;

    default:
      strcpy(nif_str,tinfo->nif);
      break;
  }

  sprintf(str,"--> Session Stats (%s:%d:%s)",
    tinfo->tty,tinfo->pid,nif_str);
  log_msg_user_only('I',str,1,user);

  str_hh_mm(tinfo->t_used,time_str);
  sprintf(str,"--> Time: %-10s   UL: %-8d bytes  DL: %-8d bytes",
    time_str,tinfo->ul,tinfo->dl);

  log_msg_user_only('I',str,1,user);
}

/***************************************************************************
* log gstat
***************************************************************************/
void log_gstat(char *log_file)
{
  char str[STR_LEN],str_st[STR_SMALL],str_dt[STR_SMALL],str_tt[STR_SMALL];

  str_hh_mm(gstat.avg_s_time,str_st);
  str_hh_mm(gstat.avg_d_time,str_dt);
  str_hh_mm(gstat.avg_t_time,str_tt);

  sprintf(str,
    "%5dk "   /*mem used*/
    "%4d "    /*total users*/
    "%3d "    /*no active*/
    "%2d "    /*no kons*/
    "%3d "    /*No free dialins*/

    "%4d "    /*Daily Connections*/
    "%6d "    /*Total Connections*/
    "%4d "    /*Avg Daily Connections*/
    "%6d "    /*Avg Total Connections*/

    "%5s "    /*Avg Session Time*/
    "%5s "    /*Avg Daily Time*/
    "%8s "    /*Avg Total Time*/

    "%13.2f " /*Daily Traffic*/
    "%13.2f " /*Total Traffic*/
    "%13.2f " /*Avg Session Traffic*/
    "%13.2f " /*Avg Daily Traffic*/
    "%13.2f " /*Avg Total Traffic*/,

  gstat.mem_used,gstat.total_users,gstat.no_active,gstat.no_kons,
  gstat.no_free_dialins,
  gstat.d_connections,gstat.t_connections,
  gstat.avg_d_connections,gstat.avg_t_connections,
  str_st,str_dt,str_tt,
  rnd2meg64(gstat.d_traffic),rnd2meg64(gstat.t_traffic),
  rnd2meg64(gstat.avg_s_traffic),rnd2meg64(gstat.avg_d_traffic),
  rnd2meg64(gstat.avg_t_traffic));

  log_msg_main_file('I',str,1,log_file);
}

/***************************************************************************
*
***************************************************************************/
void log_gstat_24hr(char *log_file)
{
  register int x;
  char str[STR_LEN] = "";

  for(x = 0;x < 24;x++)
  {
    char str2[STR_SMALL];

    sprintf(str2,"%3d ",gstat.hour_tally[x]);
    strcat(str,str2);
  }
  log_msg_main_file('I',str,1,log_file);
}

/***************************************************************************
*
***************************************************************************/
void log_daily(tum *um)
{
  char str[STR_LEN],tt_str[STR_SMALL],dt_str[STR_SMALL];

  str_hh_mm(um->tt_used,tt_str);
  str_hh_mm(um->dt_used,dt_str);
  sprintf(str,  
    "TT: %-10s TUL: %-13.2f TDL: %-13.2f  "
    "DT: %-10s DUL: %-13.2f DDL: %-13.2f",
    tt_str,
    rnd2meg64(um->t_ul),
    rnd2meg64(um->t_dl),
    dt_str,
    rnd2meg64(um->d_ul),
    rnd2meg64(um->d_dl));

  log_msg_user_file('I',str,1,um->user,"daily.log");
}

void log_total(tum *um)
{
  char str[STR_LEN],tt_str[STR_SMALL];

  str_hh_mm(um->tt_used,tt_str);
  sprintf(str,
    "TT: %-10s TUL: %-13.2f TDL: %-13.2f",
    tt_str,rnd2meg64(um->t_ul),rnd2meg64(um->t_dl));

  log_msg_user_file('I',str,1,um->user,"total.log");
}

/***************************************************************************
*
***************************************************************************/
void log_shutdown(void)
{
  tll *ll;
  char *msg = "Daemon Shutting Down.";

  log_msg_main('I',msg,1);
  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    log_msg_user_only('I',msg,1,um->user);
    if(um->no_active)
    {
      tll *ll;

      ll = um->tinfo_ll;
      for(;;)
      {
        ttinfo *tinfo;

        tinfo = ll_read_entry(&ll);
        if(tinfo == NULL)
          break;

        log_user_stats(um->user,tinfo);
      }
    }
  }
}
