/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* the monitor, the heart of clobberd
*
****************************************************************************
****************************************************************************
***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <utmp.h>
#include <pwd.h>
#include <dirent.h>
#include <time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/ll_bt.h"
#include "misc.h"
#include "log.h"
#include "sm.h"
#include "lock.h"
#include "notify.h"
#include "get_if_stats.h"
#include "clobber_user.h"
#include "scan.h"
#include "net_server.h"

/***************************************************************************
* now we aggragate all the tinfo volumes. NB/ this must be done while
* um record not locked.
***************************************************************************/
void update_um_data(tum *um)
{
  tll *ll_ptr;
  register int t,d;

  /*
   * calculate time increment
   */
  t = time(NULL);
  d = t - um->old_time;
  um->tt_used += d;
  um->dt_used += d;
  um->old_time = t;

  /*
   * tally tinfo data, aggregate it to um data
   */
  ll_ptr = um->tinfo_ll;
  um->last_active = 0;
  for(;;)
  {
    ttinfo *tinfo;

    tinfo = ll_read_entry(&ll_ptr);
    if(tinfo == NULL)
      break;

    /*
     * process volumes if terminal has a nif
     */
    if((tinfo->flags.flag.type != TINFO_TYPE_UNKNOWN) &&
       (tinfo->flags.flag.type != TINFO_TYPE_LOGIN))
    {
      (um->t_ul) += tinfo->delta_ul;
      (um->t_dl) += tinfo->delta_dl;
      (um->d_ul) += tinfo->delta_ul;
      (um->d_dl) += tinfo->delta_dl;
    }

    /*
     * take the least active terminal
     */
    if(tinfo->last_active > um->last_active)
      um->last_active = tinfo->last_active;
  }
}

/***************************************************************************
*
***************************************************************************/
void mon_entry(tus *us,tum *um)
{
  struct tm *tm_s;
  int t;

  void log_clobber_user(void)
  {
    log_msg_user('I',"End of Grace period. Dested.",1,um->user);
    clobber_user(um);
  }

  t = time(NULL);
  tm_s = localtime((time_t *)&t);
  if(tm_s != NULL)
  {
    register int x;

/*clobber if clobber flag set (user is banned)*/
    if(us->flags.flag.banned)
    {
      notify_user(us,um,CLOB_INDEX);
      log_msg_user('I',"Banned. Notified & Dested.",1,um->user);
      clobber_user(um);
      return;
    }

/*check for multiple logins*/
    if((um->no_active > 1) && us->flags.flag.clobber_mult_logins)
    {
      notify_user(us,um,CLOB_MULTIPLE_INDEX);
      log_msg_user('I',"Mult logins. Notified & Dested.",1,um->user);
      clobber_user(um);
      return;
    }

/*check expiry date*/
    if(!us->flags.flag.exempt_expiry_date)
    {
      register unsigned int a,b;

      a = us->expiry_date.day +
          (us->expiry_date.month * 31) +
          (us->expiry_date.year * 31 * 12);
      b = tm_s->tm_mday +
          ((tm_s->tm_mon + 1) * 31) +
          ((tm_s->tm_year + 1900) * 31 * 12);
      if(b >= a)
      {
        notify_user(us,um,ACCT_EXPIRED_INDEX);
        log_msg_user('I',"Account Expired. Notified & Dested.",1,um->user);
        clobber_user(um);
        return;
      }
    }

/*check if user is still active*/
    if(!us->flags.flag.exempt_act_timeout)
      if((t - um->last_active) >= us->act_timeout)
      {
        notify_user(us,um,UACT_TIMEOUT_INDEX);
        log_msg_user('I',"Activity Timeout. Notified & Dested.",1,um->user);
        clobber_user(um);
        return;
      }

/*set niceness*/
    if(!us->flags.flag.exempt_nice)
    {
      struct passwd *pw;

      pw = getpwnam(um->user);
      if(pw != NULL)
        setpriority(PRIO_USER,pw->pw_uid,us->nice_factor);
    }

/*set time zone*/
    if(!us->flags.flag.exempt_daily_time)
    {
      int dct[] = {6,0,1,2,3,4,5};

      um->dt_zone_limit = -1;
      um->dt_set_limit = 0;
      for(x = (DT_LIM_SIZE - 1);x >= 0;x--)
        if(us->dt_lim[x].d_fr != -1)
        {
          register int pass = 0;

          if((dct[tm_s->tm_wday] >= us->dt_lim[x].d_fr) &&
             (dct[tm_s->tm_wday] <= us->dt_lim[x].d_to))
            pass++;
          else
            if(us->dt_lim[x].d_fr > us->dt_lim[x].d_to)
              pass++;
          if((tm_s->tm_hour >= us->dt_lim[x].t_fr) &&
             (tm_s->tm_hour <= us->dt_lim[x].t_to))
            pass++;
          else
            if(us->dt_lim[x].t_fr > us->dt_lim[x].t_to)
              pass++;
          if(pass == 2)
          {
            um->dt_zone_limit = x;
            um->dt_set_limit = us->dt_lim[um->dt_zone_limit].limit;
            break;
          }
        }
    }

/*check total time*/
    if(!us->flags.flag.exempt_total_time)
    {
      if(um->tt_used >= us->tt_lim)
      {
        log_clobber_user();
        return;
      }
      if(!um->flags.flag.tt_graced)
        if(um->tt_used >= (us->tt_lim - us_hr.grace_time))
        {
          notify_user(us,um,EXCEEDED_TT_INDEX);
          log_msg_user('I',"Total Time grace. Notified.",1,um->user);
          um->flags.flag.tt_graced = 1;
        }
    }

/*check daily time*/
    if(!us->flags.flag.exempt_daily_time)
    {
      if(gstat.no_free_dialins < us_hr.min_free_dialins)
      {
        if(um->dt_used >= um->dt_set_limit)
        {
          log_clobber_user();
          return;
        }
        if(!um->flags.flag.dt_graced)
          if(um->dt_used >= (um->dt_set_limit - us_hr.grace_time))
          {
            notify_user(us,um,EXCEEDED_DT_INDEX);
            log_msg_user('I',"Daily Time grace. Notified.",1,um->user);
            um->flags.flag.dt_graced = 1;
          }
      }
    }

/*check total uploads limit*/
    if(!us->flags.flag.exempt_total_ul_lim)
    {
      register int total_uploads;

      total_uploads = rnd2meg64(um->t_ul);
      if(total_uploads >= us->t_ul_lim)
      {
        log_clobber_user();
        return;
      }
      if(!um->flags.flag.t_ul_graced)
        if(total_uploads >= (us->t_ul_lim - us_hr.grace_volume))
        { 
          notify_user(us,um,EXCEEDED_TNET_INDEX);
          log_msg_user('I',"Total Uploads grace. Notified.",1,um->user);
          um->flags.flag.t_ul_graced = 1;
        }
    }

/*check total dnloads limit*/
    if(!us->flags.flag.exempt_total_dl_lim)
    {
      register int total_dnloads;

      total_dnloads = rnd2meg64(um->t_dl);
      if(total_dnloads >= us->t_dl_lim)
      { 
        log_clobber_user();
        return;
      }
      if(!um->flags.flag.t_dl_graced)
        if(total_dnloads >= (us->t_dl_lim - us_hr.grace_volume))
        {
          notify_user(us,um,EXCEEDED_TNET_INDEX);
          log_msg_user('I',"Total Downloads grace. Notified.",1,um->user);
          um->flags.flag.t_dl_graced = 1;
        }
    }

/*check daily uploads limit*/
    if(!us->flags.flag.exempt_daily_ul_lim)
    {
      register int daily_uploads;

      daily_uploads = rnd2meg64(um->d_ul);
      if(daily_uploads >= us->d_ul_lim)
      {
        log_clobber_user();
        return;
      }
      if(!um->flags.flag.d_ul_graced)
        if(daily_uploads >= (us->d_ul_lim - us_hr.grace_volume))
        { 
          notify_user(us,um,EXCEEDED_DNET_INDEX);
          log_msg_user('I',"Daily Uploads grace. Notified.",1,um->user);
          um->flags.flag.d_ul_graced = 1;
        }
    }

/*check daily dnloads limit*/
    if(!us->flags.flag.exempt_daily_dl_lim)
    {
      register int daily_dnloads;

      daily_dnloads = rnd2meg64(um->d_dl);
      if(daily_dnloads >= us->d_dl_lim)
      {
        log_clobber_user();
        return;
      }
      if(!um->flags.flag.d_dl_graced)
        if(daily_dnloads >= (us->d_dl_lim - us_hr.grace_volume))
        {
          notify_user(us,um,EXCEEDED_DNET_INDEX);
          log_msg_user('I',"Daily Downloads grace. Notified.",1,um->user);
          um->flags.flag.d_dl_graced = 1;
        }
    }
  }
}

/***************************************************************************
* now we know who is online (scan_utmp() told us this), how we can monitor
* those online users.
***************************************************************************/
void monitor(void)
{
  tll *um_ll2;

  if(us_hr_locked())
  {
    log_msg_main('W',"us_hr locked. monitoring ignored",1);
    return;
  }

  um_ll2 = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&um_ll2);
    if(um == NULL)
      break;

    if(nbs_user_locked(um->user))
      log_msg_user('W',"locked (non-abs). monitoring for user ignored.",
        1,um->user);
    else
    {
      tus us;
      register int us_f = 0;

      /*
       * us records aren't applicable to nif's for the moment.
       */
      if(um->flags.flag.type == TUM_TYPE_USER)
        us_f = find_effective_us(um->user,&us);

      if(um->no_active > 0)
      {

        /*
         * this routine must be run while records aren't locked.
         * which is why it is here.
         */
        update_um_data(um);

        if(us_f)
        {

          /*
           * if the entry is a nif, then don't do any further
           * monitoring. at the moment, it is illogical to
           * monitor and clobber nifs, because they are not
           * users.
           */
          if(um->flags.flag.type == TUM_TYPE_USER)
            mon_entry(&us,um);
        }
        else
          um->dt_zone_limit = -1;
      }
      else
      {
        if(us_f)
          if(us.flags.flag.instant_purge_dt)
          {
            um->dt_used = 0;
            um->d_ul = 0;
            um->d_dl = 0;
          }
      }
    }
  }
}
