/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* network server (responds to clobberedit and clobbertool sessions over
* TCP/IP internetworking)
*
* notes about structure locking..
* -------------------------------
*
* locking is needed to arbitrate who can modify data during given time
* frame. the rule is that only one thing can modify a record at a time. if
* you get lots of different things reading/modifying a structure, data will
* become corrupt -- see database theory.
*
* in the daemon itself, when a structure is locked, it won't touch it, as
* the data being locked is still undergoing modification, and it's contents
* while locked is meaningless.
*
* before version 4.11, locking was implemented, but clients were able
* to "cheat" and update records without the record being locked.
*
* in version 4.11 and above, the daemon will deny access to a record if it
* hasn't first been locked. this applies to the following cmds:
*
*  - cmd_sushr
*  - cmd_sus
*  - cmd_sum
*  - cmd_dus
*  - cmd_dum
*  - cmd_nus
*
* if a record is locked by a connection, and the following commands are
* used to retrieve that record, they will fail.
*
*  - cmd_gushr
*  - cmd_gus
*  - cmd_gum
*
* Nb2/ for smooth server operation. don't lock records any longer than
* necessary.
*
****************************************************************************
****************************************************************************
***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/ll_bt.h"
#include "misc.h"
#include "log.h"
#include "sm.h"
#include "lock.h"
#include "monitor.h"
#include "net_bits.h"
#include "net_serv_adm.h"

tus_hr tmp_us_hr; 
tus    tmp_us;
tum    tmp_um;

/***************************************************************************
* tests that client and server versions match. if they don't then it's
* client connection is dropped.
***************************************************************************/
void version_handshake(tkon *k,char *param)
{
  char str[STR_LEN];

  if(!strcmp(param,VERSION))
  {
    k->flags.flag.version_hs = 1;
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Version handshake (passed).",k->kid);
    log_msg_main('I',str,0);
  }
  else
  {
    s_close_kon(k);
    sprintf(str,"@%d cmd- Version handshake (failed). Incorrect client version.",
      k->kid);
    log_msg_main('I',str,1);
  }
}

/***************************************************************************
*
***************************************************************************/
void quit(tkon *k)
{
  char str[STR_LEN];

  sprintf(str,"@%d cmd- Quit.",k->kid);
  log_msg_main('I',str,0);
  s_net_write_status(k,CO_SUCCESSFUL);
  s_close_kon(k);
}

/***************************************************************************
*
***************************************************************************/
void auth(tkon *k,char *pw)
{
  char str[STR_LEN];

  if(!strcmp(boot_st.passwd,pw))
  {
    k->flags.flag.authorised = 1;
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Password auth. (passed).",k->kid);
    log_msg_main('I',str,0);
    return;
  }
  else
    sprintf(str,"@%d cmd- Password auth. (failed) (Passwd:%s).",
      k->kid,pw == NULL ? none_label : pw);

  s_net_write_status(k,CO_OPERATION_FAILED);
  log_msg_main('I',str,1);
}

/***************************************************************************
* us hr functions
***************************************************************************/
void get_us_hr(tkon *k)
{
  char str[STR_BIG] = "";

  /*
   * if us_hr isn't locked by anyone accept connection k, then then it's
   * ok to read it.
   */
  if(!us_hr_locked() || kon_locked_us_hr(k))
  {
    comp_net_us_hr(&us_hr,str);
    s_net_write(k,str);

    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Get US_HR record.",k->kid);
    log_msg_main('I',str,0);
    return;
  }

  /*
   * if it is locked, then it's contents it being modified by another
   * connection, and therefore meaningless.
   */
  s_net_write_status(k,CO_RECORD_LOCKED);
  sprintf(str,"@%d cmd- Get US_HR record rejected. Record locked.",k->kid);
  log_msg_main('I',str,1);
}

void set_us_hr(tkon *k)
{
  char str[STR_LEN];

  /*
   * the us_hr records *must* be locked by the connection inorder for
   * the connection to modify the record.
   */
  if(kon_locked_us_hr(k))
  {
    memset(&tmp_us_hr,0,sizeof(tus_hr));
    k->flags.flag.mode = TKON_MODE_S_US_HR;

    sprintf(str,"@%d cmd- Set US_HR record (start).",k->kid);
    log_msg_main('I',str,0);
    return;
  }

  s_net_write_status(k,CO_OPERATION_FAILED);
  sprintf(str,"@%d cmd- Set US_HR rejected. Not locked.",k->kid);
  log_msg_main('I',str,1);
}

void do_lock_us_hr(tkon *k)
{
  char str[STR_LEN];

  if(lock_us_hr(k))
  {
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Lock US_HR record.",k->kid);
    log_msg_main('I',str,0);
    return;
  }

  s_net_write_status(k,CO_OPERATION_FAILED);
  sprintf(str,"@%d cmd- Lock US_HR record rejected. Already locked.",k->kid);
  log_msg_main('I',str,1);
}

void do_unlock_us_hr(tkon *k)
{
  char str[STR_LEN];

  unlock_us_hr(k);
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Unlock US_HR record.",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
char us_type_char(int c)
{
  switch(c)
  {
    case TUS_RECORD_TYPE_GROUP:
      return '@';
    case TUS_RECORD_TYPE_LINK:
      return '^';
    case TUS_RECORD_TYPE_UNION:
      return '~';
  }
  return ' ';
}

/*
 * for a normal userlist, send NULL as a groupname param.
 */
void get_us_users(tkon *k,char *groupname)
{
  char str[STR_LEN];
  tll *ll;

  ll = us_ll;
  for(;;)
  {
    tus *us;

    us = ll_read_entry(&ll);
    if(us == NULL)
      break;

    if(groupname != NULL)
    {
      if((us->flags.flag.record_type == TUS_RECORD_TYPE_LINK) ||
         (us->flags.flag.record_type == TUS_RECORD_TYPE_UNION))
      {
        if(strcmp(us->ident.link,groupname))
          continue;
      }
      else
        continue;
    }

    sprintf(str,":%c%s\n",
      us_type_char(us->flags.flag.record_type),us->user);
    s_net_write(k,str);
  }
  s_net_write_status(k,CO_SUCCESSFUL);

  if(groupname == NULL)
    sprintf(str,"@%d cmd- Get US users.",k->kid);
  else
    sprintf(str,"@%d cmd- Get US users (by group).",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
* actual us
***************************************************************************/
void get_us_actual(tkon *k,char *user)
{
  char str[STR_BIG] = "";

  if(!user_locked(user) || kon_locked_user(k,user))
  {
    tus *us;

    us = find_us(user);
    if(us != NULL)   
    {
      comp_net_us(us,str);
      s_net_write(k,str);

      s_net_write_status(k,CO_SUCCESSFUL);
      sprintf(str,"@%d cmd- Get US record(act) (user:%s).",k->kid,user);
      log_msg_main('I',str,0);
    }
    else
    {
      s_net_write_status(k,CO_RECORD_NOT_FOUND);
      sprintf(str,"@%d cmd- Get US record(act) rejected. Not found. (user:%s).",
        k->kid,user);
      log_msg_main('I',str,0);
    }
 
    return;
  }

  /*
   * if locked, then return error.
   */
  s_net_write_status(k,CO_RECORD_LOCKED);
  sprintf(str,"@%d cmd- Get US record(acc) rejected. User locked. (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

/***************************************************************************
* effective us.
***************************************************************************/
void get_us_effective(tkon *k,char *user)
{
  char str[STR_BIG] = "";

  if(!nbs_user_locked(user))
  {
    tus us_st;
    register int us_f;

    us_f = find_effective_us(user,&us_st);
    if(us_f)
    {
      comp_net_us(&us_st,str);
      s_net_write(k,str);

      s_net_write_status(k,CO_SUCCESSFUL);
      sprintf(str,"@%d cmd- Get US record(eff) (user:%s).",k->kid,user);
      log_msg_main('I',str,0);
    }
    else
    {
      s_net_write_status(k,CO_RECORD_NOT_FOUND);
      sprintf(str,"@%d cmd- Get US record(eff) rejected. Not found. (user:%s).",
        k->kid,user);
      log_msg_main('I',str,0);
    }
    return;
  }

  /*
   * if locked, then return error.
   */
  s_net_write_status(k,CO_RECORD_LOCKED);
  sprintf(str,"@%d cmd- Get US record(eff) rejected. Member or Link locked. (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

/***************************************************************************
*
***************************************************************************/
void set_us(tkon *k)
{
  char str[STR_LEN];

  memset(&tmp_us,0,sizeof(tus));
  k->flags.flag.mode = TKON_MODE_S_US;
  sprintf(str,"@%d cmd- Set US record (start).",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void do_lock_us(tkon *k,char *user)
{
  char str[STR_LEN];

  if(lock_user(k,user))
  {
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Lock user (user:%s).",k->kid,user);
    log_msg_main('I',str,0);
    return;
  }

  s_net_write_status(k,CO_OPERATION_FAILED);
  sprintf(str,"@%d cmd- Lock user rejected. Already locked. (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

void do_unlock_us(tkon *k)
{
  char str[STR_LEN];

  sprintf(str,"@%d cmd- Unlock user (user:%s).",k->kid,k->user_lock);
  unlock_user(k);
  s_net_write_status(k,CO_SUCCESSFUL);
  log_msg_main('I',str,0);
}

/***************************************************************************
* this function is used to create new users. it's function is to check if
* the the new user already exists. it will return ok if it doesn't, which
* means it's ok to create it or error, if it does.
***************************************************************************/
void new_us(tkon *k,char *user)
{
  char str[STR_LEN];

  /*
   * if user already exists, then return error.
   */
  if(find_us(user) != NULL)
  {
    s_net_write_status(k,CO_OPERATION_FAILED);
    sprintf(str,"@%d cmd- Check new user rejected. Already exists (user:%s).",
      k->kid,user);
    log_msg_main('I',str,1);
    return;
  }

  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Check new user passed (user:%s).",k->kid,user);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void delete_us(tkon *k,char *user)
{
  char str[STR_LEN];

  if(kon_locked_user(k,user))
  {
    remove_us(user);
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Delete US record (user:%s).",k->kid,user);
    log_msg_main('I',str,0);
    return;
  }

  s_net_write_status(k,CO_OPERATION_FAILED);
  sprintf(str,"@%d cmd- Delete US rejected. Not locked (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

/***************************************************************************
*
***************************************************************************/
void get_us_groups(tkon *k)
{
  char str[STR_LEN];
  tll *ll;
   
  ll = us_ll;
  for(;;)
  {
    tus *us;
    
    us = ll_read_entry(&ll);
    if(us == NULL)
      break;
 
    if(us->flags.flag.record_type != TUS_RECORD_TYPE_GROUP)
      continue;

    sprintf(str,":@%s\n",us->user);
    s_net_write(k,str);
  }
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Get US groups.",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
char um_active_char(tum *um)
{
  if(um->flags.flag.type == TUM_TYPE_NIF)
  {
    if(um->no_active)
      return '$';

    return '%';
  }

  switch(um->no_active)
  {
    case 0:
      return ' ';
    case 1:
      return '*';
  }
  return '+';
}

void get_um_users(tkon *k)
{
  char str[STR_LEN];
  tll *ll;

  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    sprintf(str,":%c%s\n",um_active_char(um),um->user);
    s_net_write(k,str);
  }
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Get UM users.",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void get_um(tkon *k,char *user)
{
  char str[STR_BIG] = "";

  /*
   * if user locked by any other connection, then don't read.
   */
  if(!user_locked(user) || kon_locked_user(k,user))
  {
    tum *um;

    um = find_um(user);
    if(um != NULL)
    {
      comp_net_um(um,str);
      s_net_write(k,str);

      s_net_write_status(k,CO_SUCCESSFUL);
      sprintf(str,"@%d cmd- Get UM record (user:%s).",k->kid,user);
    }
    else
    {
      s_net_write_status(k,CO_RECORD_NOT_FOUND);
      sprintf(str,"@%d cmd- Get UM record rejected. Not found. (user:%s).",
        k->kid,user);
    }
    log_msg_main('I',str,0);
    return;
  }

  /*
   * if locked, then return error.
   */
  s_net_write_status(k,CO_RECORD_LOCKED);
  sprintf(str,"@%d cmd- Get UM record rejected. User locked. (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

/***************************************************************************
*
***************************************************************************/
void set_um(tkon *k)
{
  char str[STR_LEN];

  memset(&tmp_um,0,sizeof(tum));
  k->flags.flag.mode = TKON_MODE_S_UM;
  sprintf(str,"@%d cmd- Set UM record (start).",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void delete_um(tkon *k,char *user)
{
  char str[STR_LEN];

  if(kon_locked_user(k,user))
  {
    remove_um(user);
    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Delete UM record (user:%s).",k->kid,user);
    log_msg_main('I',str,0);
    return;
  }

  s_net_write_status(k,CO_OPERATION_FAILED);
  sprintf(str,"@%d cmd- Delete UM rejected. Not locked (user:%s).",
    k->kid,user);
  log_msg_main('I',str,1);
}

/***************************************************************************
* Get (request) tinfo struct
***************************************************************************/
void get_tinfo(tkon *k,char *param)
{
  char str[STR_BIG];
  tum *um;

  um = find_um(param);
  if(um != NULL)
  {
    tll *ll_ptr;

    ll_ptr = um->tinfo_ll;
    for(;;)
    {
      ttinfo *tinfo;

      tinfo = ll_read_entry(&ll_ptr);
      if(tinfo == NULL)
        break;

      s_net_write(k,":@\n");
      str[0] = 0;
      comp_net_tinfo(tinfo,str);
      s_net_write(k,str);
      s_net_write(k,":#\n");
    }

    s_net_write_status(k,CO_SUCCESSFUL);
    sprintf(str,"@%d cmd- Get terminal info (user:%s).",k->kid,param);
    log_msg_main('I',str,0);  
  }
  else
  {
    s_net_write_status(k,CO_OPERATION_FAILED);
    sprintf(str,"@%d cmd- Get terminal info Failed. User not found. (user:%s).",
      k->kid,param);
    log_msg_main('I',str,0);
  }
}

/***************************************************************************
*
***************************************************************************/
void get_kon_stats(tkon *k)
{
  char str[STR_LEN];
  register int x;

  for(x = 0;x < boot_st.max_kons;x++)
    if(kon[x].s != -1)
    {
      char age_str[STR_SMALL],idle_str[STR_SMALL];
      char ac = ' ';
      register int t;

      if(kon[x].flags.flag.version_hs)
        ac = '%';
      if(kon[x].flags.flag.authorised)
        ac = '*';

      t = time(NULL);
      str_hh_mm(t - kon[x].created,age_str);
      str_hh_mm(t - kon[x].last_active,idle_str);
      sprintf(str,":%3d %-15s %6d %c %7s %4s %c %s\n",
        kon[x].kid,
        kon[x].hostname,
        kon[x].port,
        ac,
        age_str,
        idle_str,
        kon[x].flags.flag.us_hr_locked ? '*' : ' ',
        kon[x].user_lock);
      s_net_write(k,str);
    }

  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Get connection stats.",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void get_stime(tkon *k)
{
  char str[STR_LEN];

  sprintf(str,":%x %x\n",us_hr.sleep_time,(int)time(NULL));
  s_net_write(k,str);
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Get sleep time.",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
void force_save(tkon *k)
{
  char str[STR_LEN];

  save_data_file();
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Saved datafile (forced).",k->kid);
  log_msg_main('I',str,0);
}

void end_server(tkon *k)
{
  char str[STR_LEN];

  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Server Shutdown.",k->kid);
  log_msg_main('I',str,1);

  log_shutdown();
  save_database();
  exit(0);
}

/***************************************************************************
* get General Statistics
***************************************************************************/
void get_general_stats(tkon *k)
{
  char str[STR_BIG] = "";

  comp_net_gstat(&gstat,str);
  s_net_write(k,str);
  s_net_write_status(k,CO_SUCCESSFUL);
  sprintf(str,"@%d cmd- Get General Statistics.",k->kid);
  log_msg_main('I',str,0);  
}

/***************************************************************************
* get user list (in order of total time used)
***************************************************************************/

/**************************************
* creates a UM table from the UM linked
* list. UM table consists of a set of
* pointers to UM. we put them in a 
* table so that we can use the qsort()
* method of sorting.
**************************************/
void um_list_2_tab(tum ***tab,int *tab_size)
{
  tll *ll;

  *tab = NULL;
  *tab_size = 0;
  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    if(um->flags.flag.type == TUM_TYPE_NIF)
      continue;

    *tab = realloc(*tab,(*tab_size + 1) * sizeof(tum *));
    if(*tab == NULL)
      memory_error_cb();

    (*tab)[*tab_size] = um;
    (*tab_size)++;
  }
}

/**************************************
* get user list (Total Time Used)
**************************************/
void get_ul_ttu(tkon *k)
{
  tum **um_tab;
  int um_tab_size;
  char str[STR_LEN];

  int compar(const void *e1, const void *e2)
  {
    return (*((tum **)e2))->tt_used - (*((tum **)e1))->tt_used;
  }

  if(records_are_locked())
  {
    s_net_write_status(k,CO_RECORD_LOCKED);
    sprintf(str,"@%d cmd- List User (TTU) Rejected. Records are locked.",
      k->kid);
    log_msg_main('I',str,1);
    return;
  }

  um_list_2_tab(&um_tab,&um_tab_size);
  if(um_tab != NULL)
  {
    register int x;

    qsort(um_tab,um_tab_size,sizeof(tum *),compar);
    for(x = 0;x < um_tab_size;x++)
    {
      char t_str[STR_SMALL];

      str_hh_mm(um_tab[x]->tt_used,t_str);
      sprintf(str,":%c%-32s %s\n",
        um_active_char(um_tab[x]),um_tab[x]->user,t_str);
      s_net_write(k,str);
    }
    free(um_tab);
  }
  s_net_write_status(k,CO_SUCCESSFUL);

  sprintf(str,"@%d cmd- List User (TTU).",k->kid);
  log_msg_main('I',str,0);
}

/**************************************
* get user list (Total Network Volume List)
**************************************/
void get_ul_tnv(tkon *k)
{
  tum **um_tab;
  int um_tab_size;
  char str[STR_LEN];

  int compar(const void *e1, const void *e2)
  {
    register int traf1,traf2;

    traf1 = (*((tum **)e1))->t_ul + (*((tum **)e1))->t_dl;
    traf2 = (*((tum **)e2))->t_ul + (*((tum **)e2))->t_dl;
    return traf2 - traf1;
  }

  if(records_are_locked())
  {
    s_net_write_status(k,CO_RECORD_LOCKED);
    sprintf(str,"@%d cmd- List User (TNV) Rejected. Records are locked.",
      k->kid);
    log_msg_main('I',str,1);
    return;
  }

  um_list_2_tab(&um_tab,&um_tab_size);
  if(um_tab != NULL)
  {
    register int x;

    qsort(um_tab,um_tab_size,sizeof(tum *),compar);
    for(x = 0;x < um_tab_size;x++)
    {
      sprintf(str,":%c%-32s %-13.2f\n",
        um_active_char(um_tab[x]),um_tab[x]->user,
        rnd2meg64(um_tab[x]->t_ul + um_tab[x]->t_dl));
      s_net_write(k,str);
    }
    free(um_tab);
  }
  s_net_write_status(k,CO_SUCCESSFUL);

  sprintf(str,"@%d cmd- List User (TNV).",k->kid);
  log_msg_main('I',str,0);
}

/**************************************
* get user list (Connection Efficiency)
**************************************/
void get_ul_cef(tkon *k)
{
  tum **um_tab;
  int um_tab_size;
  char str[STR_LEN];

  float calc_coeff(int tt,int conns)
  {
    if(conns == 0)
      return 0;
    return (float)tt / (float)conns;
  }

  int compar(const void *e1, const void *e2)
  {
    float c1,c2;

    c1 = calc_coeff(
      (*((tum **)e1))->tt_used,(*((tum **)e1))->t_connections);

    c2 = calc_coeff(
      (*((tum **)e2))->tt_used,(*((tum **)e2))->t_connections);

    return (int)((c2 - c1) * 100);
  }

  if(records_are_locked())
  {
    s_net_write_status(k,CO_RECORD_LOCKED);
    sprintf(str,"@%d cmd- List User (CE) Rejected. Records are locked.",
      k->kid);
    log_msg_main('I',str,1);
    return;
  }

  um_list_2_tab(&um_tab,&um_tab_size);
  if(um_tab != NULL)
  {
    register int x;

    qsort(um_tab,um_tab_size,sizeof(tum *),compar);
    for(x = 0;x < um_tab_size;x++)
    {
      char t_str[STR_SMALL]; 

      str_hh_mm(
        calc_coeff(um_tab[x]->tt_used,um_tab[x]->t_connections),t_str);
      sprintf(str,":%c%-32s %s\n",
        um_active_char(um_tab[x]),um_tab[x]->user,t_str);
      s_net_write(k,str);
    }
    free(um_tab);
  }
  s_net_write_status(k,CO_SUCCESSFUL);

  sprintf(str,"@%d cmd- List User (CE).",k->kid);
  log_msg_main('I',str,0);
}

/***************************************************************************
*
***************************************************************************/
char *net_get_param(char *cmd,char *str)
{
  register int si;

  si = strlen(cmd) + 1;
  if(strlen(str) > si)
    return str + si;
  return NULL;
}

void cmd_mode(tkon *k,char *ins)
{
  char str[STR_LEN],*param,cmd_str[STR_LEN_LN + 1];

  void missing_param(void)
  {
    s_net_write_status(k,CO_CMD_MISSING_PARAM);
    sprintf(str,"@%d cmd missing parameter",k->kid);
    log_msg_main('I',str,1);
  }

  sscanf(ins,"%s",cmd_str);

  /*******************************************************************
  * initially, this is the only cmd clobberd will accept.
  * as of 4.11, clobberd will require a version handshake.
  *******************************************************************/
  if(!strcmp(cmd_str,cmd_ver))
  {
    param = net_get_param(cmd_ver,ins);
    if(param != NULL)
    {
      version_handshake(k,param);
      return;
    }
  }

  /******************************************************************
  * once it's been checked that the client and server are the
  * same version, normal communication can begin.
  ******************************************************************/
  if(k->flags.flag.version_hs)
  { 

    /******************************************************************
    * the following cmds can be accessed without authorisation.
    ******************************************************************/

    /**************************************
    * us_hr "General Settings" cmds
    **************************************/
    if(!strcmp(cmd_str,cmd_gushr))
    {
      get_us_hr(k);
      return;
    }

    /**************************************
    * us "User Settings" cmds
    **************************************/
    if(!strcmp(cmd_str,cmd_gusu))
    {
      get_us_users(k,NULL);
      return;
    }

    if(!strcmp(cmd_str,cmd_gusg))
    {
      get_us_groups(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_gusug))
    {
      param = net_get_param(cmd_gusug,ins);
      if(param != NULL)
      {
        get_us_users(k,param);
        return;
      }
      missing_param();
      return;
    }

    if(!strcmp(cmd_str,cmd_gus))   /*actual us*/
    {
      param = net_get_param(cmd_gus,ins);
      if(param != NULL)
      {
        get_us_actual(k,param);
        return;
      }
      missing_param();
      return;
    }

    if(!strcmp(cmd_str,cmd_gus2)) /*effective*/
    {
      param = net_get_param(cmd_gus2,ins);
      if(param != NULL)
      {
        get_us_effective(k,param);
        return;
      }
      missing_param();
      return;
    }

    /**************************************
    * um "User Monitor" cmds
    **************************************/
    if(!strcmp(cmd_str,cmd_gumu))
    {
      get_um_users(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_gum))
    {
      param = net_get_param(cmd_gum,ins);
      if(param != NULL)
      {
        get_um(k,net_get_param(cmd_gum,ins));
        return;
      }
      missing_param();
      return;
    }

    /**************************************
    * misc cmds
    **************************************/
    if(!strcmp(cmd_str,cmd_quit))
    {
      quit(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_pass))
    {
      param = net_get_param(cmd_pass,ins);
      if(param != NULL)
      {
        auth(k,param);
        return;
      }
      missing_param();
      return;
    }

    if(!strcmp(cmd_str,cmd_stime))
    {
      get_stime(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_gtinf))
    {
      param = net_get_param(cmd_gtinf,ins);
      if(param != NULL)
      {
        get_tinfo(k,param);
        return;
      }
      missing_param();
      return;
    }

    if(!strcmp(cmd_str,cmd_stat))
    {
      get_kon_stats(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_gstat))
    {
      get_general_stats(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_ul_ttu))
    {
      get_ul_ttu(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_ul_tnv))
    {
      get_ul_tnv(k);
      return;
    }

    if(!strcmp(cmd_str,cmd_ul_cef))
    {
      get_ul_cef(k);
      return;
    }


    /*****************************************************************
    * these are authorised only commands.
    *****************************************************************/
    if(k->flags.flag.authorised)
    {

      /**************************************
      * us_hr "General Settings" cmds
      **************************************/
      if(!strcmp(cmd_str,cmd_sushr))
      {
        set_us_hr(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_lushr))
      {
        do_lock_us_hr(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_uushr))
      {
        do_unlock_us_hr(k);
        return;
      }

      /**************************************
      * us "User Settings" cmds
      **************************************/
      if(!strcmp(cmd_str,cmd_sus))
      {
        set_us(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_luser))
      {
        param = net_get_param(cmd_luser,ins);
        if(param != NULL)
        {
          do_lock_us(k,param);
          return;
        }
        missing_param();
        return;
      }

      if(!strcmp(cmd_str,cmd_uuser))
      {
        do_unlock_us(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_nus))
      {
        param = net_get_param(cmd_nus,ins);
        if(param != NULL)
        {
          new_us(k,param);
          return;
        }
        missing_param();
        return;
      }

      if(!strcmp(cmd_str,cmd_dus))
      {
        param = net_get_param(cmd_dus,ins);
        if(param != NULL)
        {
          delete_us(k,param);
          return;
        }
        missing_param();
        return;
      }

      /**************************************
      * um "User Monitor" cmds
      **************************************/
      if(!strcmp(cmd_str,cmd_sum))
      {
        set_um(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_dum))
      {
        param = net_get_param(cmd_dum,ins);
        if(param != NULL)
        {
          delete_um(k,param);
          return;
        }
        missing_param();
        return;
      }

      /**************************************
      * misc cmds
      **************************************/
      if(!strcmp(cmd_str,cmd_tpurg))
      {
        total_purge(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_dpurg))
      {
        daily_purge(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_save))
      {
        force_save(k);
        return;
      }

      if(!strcmp(cmd_str,cmd_die))
        end_server(k);

      if(!strcmp(cmd_str,cmd_arch))
      {
        archive(k);
        return;
      }

      sprintf(str,"@%d cmd unknown.",k->kid);
    }
    else
      sprintf(str,"@%d cmd unauthorised or unknown.",k->kid);

    s_net_write_status(k,CO_UNKNOWN_CMD);
    log_msg_main('I',str,1);
  }
  else
  {
    s_close_kon(k);
    sprintf(str,"@%d cmd invalid from unknown client. Killing connection.",
      k->kid);
    log_msg_main('I',str,1);
  }
}

/***************************************************************************
* the following three commands will gradually collect texterised records
* from a net socket, and decode them into a structure (depending on what's
* being received.
***************************************************************************/

/***************************************
*
***************************************/
void set_us_hr_mode(tkon *k,char *ins)
{
  if(!strcmp(ins,cmd_ret))
  {
    char str[STR_LEN];

    memcpy(&us_hr,&tmp_us_hr,sizeof(tus_hr));
    k->flags.flag.mode = TKON_MODE_CMD;
    s_net_write_status(k,CO_SUCCESSFUL);

    sprintf(str,"@%d cmd- Set US_HR record (finish).",k->kid);
    log_msg_main('I',str,0);
  }
  else
    decomp_net_us_hr(&tmp_us_hr,ins);
}

/***************************************
*
***************************************/
void set_us_mode(tkon *k,char *ins)
{
  char str[STR_LEN];

  if(!strcmp(ins,cmd_ret))
  {
    if(tmp_us.user[0])
    {

      /*
       * packet is now received and decoded into tmp_us
       *
       * check that the user is locked, if not then we
       * reject it. the client hasn't followed the rules.
       */
      if(kon_locked_user(k,tmp_us.user))
      {
        tus *u;

        u = find_us(tmp_us.user);
        if(u == NULL)
          append_us(&tmp_us);
        else
          memcpy(u,&tmp_us,sizeof(tus));

        s_net_write_status(k,CO_SUCCESSFUL);
        sprintf(str,"@%d cmd- Set US record (finish) (user:%s).",
          k->kid,tmp_us.user);
        log_msg_main('I',str,0);
      }
      else
      {
        s_net_write_status(k,CO_OPERATION_FAILED);
        sprintf(str,"@%d cmd- Set US rejected. Not locked (user:%s).",
          k->kid,tmp_us.user);
        log_msg_main('I',str,1);
      }
    }
    else
    {
      s_net_write_status(k,CO_OPERATION_FAILED);
      sprintf(str,"@%d cmd- Set US rejected. Username blank.",k->kid);  
      log_msg_main('I',str,1);
    }

    k->flags.flag.mode = TKON_MODE_CMD;
  }  
  else
    decomp_net_us(&tmp_us,ins);
}

/***************************************
*
***************************************/
void set_um_mode(tkon *k,char *ins)
{
  char str[STR_LEN];

  if(!strcmp(ins,cmd_ret))
  {
    if(tmp_um.user[0])
    {
      if(kon_locked_user(k,tmp_um.user))
      { 
        tum *u;

        u = find_um(tmp_um.user);
        if(u != NULL)
        {
          u->d_connections = tmp_um.d_connections;
          u->t_connections = tmp_um.t_connections;
          u->tt_used = tmp_um.tt_used;
          u->dt_used = tmp_um.dt_used;
          u->t_ul = tmp_um.t_ul;
          u->t_dl = tmp_um.t_dl;
          u->d_ul = tmp_um.d_ul;
          u->d_dl = tmp_um.d_dl;
          u->flags = tmp_um.flags;

          s_net_write_status(k,CO_SUCCESSFUL);
          sprintf(str,"@%d cmd- Set UM record (finish) (user:%s).",
            k->kid,tmp_um.user);
          log_msg_main('I',str,0);
        }
        else
        {
          s_net_write_status(k,CO_RECORD_NOT_FOUND);
          sprintf(str,
            "@%d cmd- Set UM Record rejected. User not found (user:%s).",
            k->kid,tmp_um.user);
          log_msg_main('I',str,1);
        }
      }
      else
      {
        s_net_write_status(k,CO_OPERATION_FAILED);
        sprintf(str,"@%d cmd- Set UM rejected. Not locked (user:%s).",
          k->kid,tmp_um.user);
        log_msg_main('I',str,1);
      }
    }
    else
    {
      s_net_write_status(k,CO_OPERATION_FAILED);
      sprintf(str,"@%d cmd- Set UM rejected. Username blank.",k->kid);
      log_msg_main('I',str,1);
    }

    k->flags.flag.mode = TKON_MODE_CMD;
  }  
  else
    decomp_net_um(&tmp_um,ins);
}

/***************************************************************************
* this is the main routine to service any active network connections.
* this is a very important one as well.
***************************************************************************/
void service_kons(void)
{
  register int n;

  for(n = 0;n < boot_st.max_kons;n++)
    if(kon[n].s != -1)
    {
      register int sustain;

      /*
       * as network activity usually happens in short spurts, having
       * a sustain period where a socket is constantly being serviced 
       * will improved connection response.
       */
      for(sustain = 0;sustain < NETWORK_SUSTAIN;sustain++)
      {
        char ins[STR_LEN];

        /*
         * read a line from the a connection stream
         */
        if(s_net_readln(&(kon[n]),ins,STR_LEN_LN) > 0)
        {
          kon[n].last_active = time(NULL);

          switch(kon[n].flags.flag.mode)
          {
            case TKON_MODE_CMD:
              cmd_mode(&(kon[n]),ins);
              break;
            case TKON_MODE_S_US_HR:
              set_us_hr_mode(&(kon[n]),ins);
              break;
            case TKON_MODE_S_US:
              set_us_mode(&(kon[n]),ins);
              break;
            case TKON_MODE_S_UM:
              set_um_mode(&(kon[n]),ins);
              break;
          }
        }
        else
        {

          /*
           * if nothing is happening, then check how active the
           * connection is. if it's been a really long time since
           * it was used, then nuke it.
           *
           * we also break out of sustain loop and service next
           * connection.
           */
          if((time(NULL) - kon[n].last_active) > boot_st.kon_timeout)
          {
            sprintf(ins,"[%d] Connection timeout.",kon[n].kid);
            log_msg_main('I',ins,1);
            s_close_kon(&(kon[n]));
          }
          break;
        }

        /*
         * if connection has ended, exit sustain loop
         */
        if(kon[n].s == -1)
          break;
      }
    }
}
