/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* these modules scan the utmp file for active users. it scans the
* /proc/net/dev sf for active nifs'.
*
****************************************************************************
****************************************************************************
***************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <utmp.h>
#include <pwd.h>
#include <time.h>
#include <sys/stat.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/ll_bt.h"
#include "misc.h"
#include "log.h"
#include "sm.h"
#include "get_if_stats.h"
#include "net_bits.h"

int modem_lenient_f = 0;

/***************************************************************************
* get shell stats (if a shell user)
***************************************************************************/
void get_shell_stats(ttinfo *tinfo)
{
  struct stat stbuf;
  char str[STR_LEN];

  strcpy(str,"/dev/");
  strncat(str,tinfo->tty,STR_TTY_LEN);
  if(!stat(str,&stbuf))
  {
    /* If it's a login terminal, then all we need to do is
     * find out how active it is.
     */
    tinfo->flags.flag.type = TINFO_TYPE_LOGIN;
    if(stbuf.st_mtime > tinfo->last_active)
      tinfo->last_active = stbuf.st_mtime;
  }
}

/***************************************************************************
* remove any tinfo entries that are logged out of
***************************************************************************/
void check_defunc_tinfo(tum *um)
{
  tll *ll_ptr;

  void remove_term(ttinfo *tinfo)
  {
    tll *p_next;

    p_next = ll_ptr->next;
    remove_tinfo(&(um->tinfo_ll),tinfo->pid);
    ll_ptr = p_next;
  }

  ll_ptr = um->tinfo_ll;
  for(;;)
  {
    ttinfo *tinfo;

    if(ll_ptr == NULL)
      return;
    tinfo = ll_ptr->entry;

    /*
     * if the tinfo linked list belongs to a nif, then we try to find
     * it in our cached gnif's, if it's not found, then we blow that
     * entry out of the water.
     *
     * if it's a normal user um entry, then handle it like a normal
     * terminal- if their terminal isn't in the utmp file, then
     * they are logged off. blow entry out of water.
     */
    if(um->flags.flag.type == TUM_TYPE_NIF)
    {
      if(find_nif(tinfo->nif) == NULL)
      {
        /*
         * nif's don't contribute to general stats, due to not being
         * users. also, ppp users generate a nif entry, therefore,
         * there's a condition where there is a double up of 
         * general statistic recording, and gstats cane become bummed.
         */
        log_msg_user('I',"nif down.",1,um->user);
        log_user_stats(um->user,tinfo);
        remove_term(tinfo);
        continue;
      }
    }
    else
    {
      struct utmp *up;
      register int match = 0;

      setutent();
      for(;;)
      {
        up = getutent();
        if(up == NULL)
          break;
        if(up->ut_type == USER_PROCESS)
          if(tinfo->pid == up->ut_pid)
          {
            match = 1;
            break;
          }
      }
      endutent();

      if(!match)
      {
        char str[STR_LEN];
        register int s_traffic;

        sprintf(str,"logoff from %s:%d.",tinfo->tty,tinfo->pid);
        log_msg_user('I',str,1,um->user);
        log_user_stats(um->user,tinfo);

        /*
         * update gstats
         */
        s_traffic = tinfo->ul + tinfo->dl;
        gstat.d_traffic += s_traffic;
        gstat.t_traffic += s_traffic;
        gstat.avg_s_time = (gstat.avg_s_time + tinfo->t_used) >> 1;
        gstat.avg_s_traffic = (gstat.avg_s_traffic + s_traffic) >> 1;

        remove_term(tinfo);
        continue;
      }
    }

    ll_ptr = ll_ptr->next;
  }
}

/***************************************************************************
*
***************************************************************************/
void inc_time_counter(int *time_counter,int *old_time)
{
  register int t;

  t = time(NULL);
  (*time_counter) += t - *old_time;
  *old_time = t;
}

/***************************************************************************
*
***************************************************************************/
void create_new_tinfo(tum *um,int pid,char *tty,char *nif_name,int nif_type)
{
  ttinfo entry;
  register int t;

  entry.pid = pid;
  strcpy(entry.tty,tty);
  strcpy(entry.nif,nif_name);
  entry.t_used = 0;
  t = time(NULL);
  entry.last_active = t;
  entry.old_time = t;
  entry.flags.flag.type = nif_type;
  entry.flags.flag.nif_primed = 0;
  entry.ul = 0;  
  entry.dl = 0;
  entry.delta_ul = 0;
  entry.delta_dl = 0;
  append_tinfo(&(um->tinfo_ll),&entry);
}

/***************************************************************************
* gathers terminal data of a given active user
***************************************************************************/
void stat_terminal(tum *um,struct utmp *utmp_ptr)
{
  ttinfo *tinfo;

  tinfo = find_tinfo_pid(&(um->tinfo_ll),utmp_ptr->ut_pid);
  if(tinfo == NULL)
  {
    char str[STR_LEN],tty[STR_TTY_LEN + 1];
    struct tm *tm_st;
    int t;

    strncpy(tty,utmp_ptr->ut_line,STR_TTY_LEN);
    tty[STR_TTY_LEN] = 0;
    create_new_tinfo(um,utmp_ptr->ut_pid,tty,none_label,TINFO_TYPE_UNKNOWN);
    tinfo = find_tinfo_pid(&(um->tinfo_ll),utmp_ptr->ut_pid);

    sprintf(str,"logon to %s:%d.",tty,utmp_ptr->ut_pid);
    log_msg_user('I',str,1,um->user);

    (um->d_connections)++;
    (um->t_connections)++;

    /*
     * update gstats
     */
    gstat.d_connections++;
    gstat.t_connections++;

    t = time(NULL);
    tm_st = localtime((time_t *)&t);
    if(tm_st != NULL)
      gstat.hour_tally[tm_st->tm_hour]++;
  }

  if(tinfo != NULL)
  {
    inc_time_counter(&(tinfo->t_used),&(tinfo->old_time));

    /*
     * process terminal, find out what type it is, if it's ppp, get
     * update volumes
     */
    tinfo->flags.flag.type = TINFO_TYPE_UNKNOWN;
    getcalc_ppp_nif_stats(tinfo,utmp_ptr); /*if any*/
    if(tinfo->flags.flag.type == TINFO_TYPE_UNKNOWN)
      get_shell_stats(tinfo);      /*if any*/
  }
}

/***************************************************************************
* yes, i know what you're thinking- network interfaces don't have terminals.
* however, the easiest way to intergrate nif monitoring into the existing
* frame work of clobberd is to pretend they do have terminals. this allows
* the existing code to process nif's like normal users.
*
* Also nb/ there is a condition that this code can't properly deal with.
* in linux there is no way to distingish between one NIF session,
* and another (there is no such thing as a network interface ID, or
* session counter within linux). therefore, if a NIF (ie eth0), goes up
* and down fast enough so that this code can't detect the "down" state,
* then clobberd won't detect that the NIF is a new session. therefore
* the new session won't be primed, and deltas won't be inited.
* the result is that volumes will go backward, and minus values will
* result.
*
***************************************************************************/
void stat_nif_terminal(tum *um,tnif *nif)
{
  ttinfo *tinfo;

  tinfo = find_tinfo_nif(&(um->tinfo_ll),um->user);
  if(tinfo == NULL)
  {
    create_new_tinfo(um,0,"",um->user,TINFO_TYPE_GNIF);
    tinfo = find_tinfo_nif(&(um->tinfo_ll),um->user);

    log_msg_user('I',"nif up.",1,um->user);
  }

  if(tinfo != NULL)
  {
    inc_time_counter(&(tinfo->t_used),&(tinfo->old_time));
    getcalc_nif_stats(tinfo,nif);
  }
}

/***************************************************************************
*
***************************************************************************/
int user_exempt(char *user)
{
  register int x;

  for(x = 0;x < boot_st.exempt_user_size;x++)
    if(!strcmp(boot_st.exempt_user[x],user))
      return 1;
  return 0;
}

void create_new_um(char *username,int um_type)
{
  tum u;

  u.flags.flag.type = um_type;
  strcpy(u.user,username);
  u.d_connections = 0;
  u.t_connections = 0;
  u.tt_used = 0;
  u.dt_used = u.tt_used;
  u.t_ul = 0;
  u.t_dl = 0;
  u.d_ul = 0;
  u.d_dl = 0;
  u.no_active = 0;
  u.tinfo_ll = NULL;
  init_um(&u);
  append_um(&u);
}

/***************************************************************************
* gathers information about active users and stores it in um linked list
*
* if you want to understand and modify clobberd, to say recognise more
* connection types, you must understand this routine, and it's
* subroutines. this is a very important part of clobberd.
***************************************************************************/
void scan_utmp(void)
{
  struct utmp *utmp_ptr;
  int garbage_f = 0;

  /*
   * we now scan the utmp file. if it's a new user, then we create
   * a new UM entry for them. for all users, we increment the
   * no_actives variable and log their terminal by running stat_terminal()
   */
  setutent();
  for(;;)
  {
    utmp_ptr = getutent();
    if(utmp_ptr == NULL)
      break;
    if(utmp_ptr->ut_type == USER_PROCESS)
    {
      char tn[STR_USER_LEN + 1];

      strncpy(tn,utmp_ptr->ut_user,STR_USER_LEN);
      tn[STR_USER_LEN] = 0;

      /*
       * find out if entry is a user in /etc/passwd. if not, then it's
       * garbage, and ignored.
       */
      if(getpwnam(tn) == NULL)
        garbage_f = 1;
      else
        if(!user_exempt(tn))
        {
          tum *um;

          um = find_um(tn);
          if(um == NULL)
          {
            create_new_um(tn,TUM_TYPE_USER);
            um = find_um(tn);
            log_msg_user('I',"New user.",1,tn);
          }

          if(um != NULL)
          {
            (um->no_active)++;
            stat_terminal(um,utmp_ptr);
          }
        }
    }
  }
  endutent();

  if(garbage_f)
    log_msg_main('W',"Garbage found in utmp file",0);
}

/***************************************************************************
* scans /proc/net/dev
***************************************************************************/
void scan_nifs(void)
{
  tll *ll;
  tnif *nif = NULL;

  ll = nif_ll;
  for(;;)
  {
    tum *um;

    nif = ll_read_entry(&ll);
    if(nif == NULL)
      break;

    um = find_um(nif->name);
    if(um == NULL)
    {
      create_new_um(nif->name,TUM_TYPE_NIF);
      um = find_um(nif->name);
      log_msg_user('I',"New nif.",1,nif->name);
    }

    if(um != NULL)
    {
      (um->no_active)++;
      stat_nif_terminal(um,nif);
    }
  }
}

/***************************************************************************
* find number of free dialin terminals
***************************************************************************/
void dialins_free(void)
{
  gstat.no_free_dialins = -1;
  if(!us_hr.flags.flag.min_free_dialin_exempt)
  {
    register int x;
    char str[STR_LEN];

    /*
     * to find the no of free modems, we compare how many "add_tty:"'s
     * are in the utmp file
     */
    gstat.no_free_dialins = boot_st.mfd_tty_t_size;
    for(x = 0;x < boot_st.mfd_tty_t_size;x++)
    {
      struct utmp *up;

      setutent();
      for(;;)
      {
        up = getutent();
        if(up == NULL)
          break;
        if(up->ut_type == USER_PROCESS)
          if(!strncmp(up->ut_line,boot_st.mfd_tty_t[x],STR_TTY_LEN))
          {
            gstat.no_free_dialins--;
            break;
          }
      }
      endutent();
    }
    if(gstat.no_free_dialins >= us_hr.min_free_dialins)
    {
      if(!modem_lenient_f)
      {
        modem_lenient_f = 1;
        sprintf(str,"Modem Lenancy on (Dialin Stat: %d/%d).",
          gstat.no_free_dialins,us_hr.min_free_dialins);
        log_msg_main('I',str,1);
      }
    }
    else
    {
      if(modem_lenient_f)
      {
        modem_lenient_f = 0;
        sprintf(str,"Modem Lenancy off (Dialin Stat: %d/%d).",
          gstat.no_free_dialins,us_hr.min_free_dialins);
        log_msg_main('I',str,1);
      }
    }
  }
  else
    if(modem_lenient_f)
    {
      modem_lenient_f = 0;
      log_msg_main('I',"Modem Lenancy exempt",1);
    }
}

/***************************************************************************
*
***************************************************************************/
void scan(void)
{
  tll *ll;

  cache_user_nifs();
  cache_nifs();

  /*
   * at the begining of every scan, we prepare our existing um data, by
   * zeroing the no_actives variable and also removing any non-existing
   * entries in each um's tinfo linked list.
   */
  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    um->no_active = 0;
    check_defunc_tinfo(um);
  }

  scan_utmp();
  scan_nifs();

  /*
   * finally, we go through um data. any UM's that are not active, we
   * init their UM structures so as to prepare them for when the user
   * logs in again.
   */
  ll = um_ll;
  for(;;)
  {
    tum *um;
    
    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    if(um->no_active == 0)
      init_um(um);
  }

  ll_free(&user_nif_ll);
  ll_free(&nif_ll);

  dialins_free();
}

/***************************************************************************
*
***************************************************************************/
void system_scan(void)
{
  FILE *fp;
  char str[STR_LEN + 1] = "";
  tll *ll;
  register int x;

  gstat.mem_used = 0;
  gstat.total_users = 0;
  gstat.no_active = 0;
  gstat.no_kons = 0;

  sprintf(str,"/proc/%d/stat",getpid());
  fp = fopen(str,"r");
  {
    fgets(str,STR_LEN,fp);
    if(str[0])
      sscanf(str,"%*d %*s %*c %*d %*d %*d %*d %*d %*d %*d %*d %*d %*d "
                 "%*d %*d %*d %*d %*d %*d %*d %*d %*d %d",(&gstat.mem_used));
    fclose(fp);
  }
  gstat.mem_used >>= 10;

  ll = um_ll;
  for(;;)
  {
    tum *um;

    um = ll_read_entry(&ll);
    if(um == NULL)
      break;

    gstat.total_users++;
    if(um->flags.flag.type == TUM_TYPE_USER)
      if(um->no_active)
        gstat.no_active++;
  }

  for(x = 0;x < boot_st.max_kons;x++)
    if(kon[x].s != -1)
      gstat.no_kons++;
}
