/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* file io, structure, linked list, btree code etc
*
****************************************************************************
****************************************************************************
***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/ll_bt.h"
#include "misc.h"
#include "log.h"

tll    *user_nif_ll = NULL;
tll    *nif_ll = NULL;
tus_hr us_hr;
tll    *us_ll = NULL;
tbt    *us_bt = NULL;
tll    *um_ll = NULL;
tgstat gstat;

/***************************************************************************
* fast cache for ppp lock file info in /var/run
* as from 4.9, access this info many many times during a user scan..
*
* (unsorted- linked list usually small 2-40 entries)
***************************************************************************/
void append_user_nif(char *nif_name,int pid)
{
  tuser_nif user_nif;

  strcpy(user_nif.nif_name,nif_name);
  user_nif.pid = pid;

  ll_append_unsorted(&user_nif_ll,&user_nif,sizeof(tuser_nif));
}

tuser_nif *find_user_nif(int pid)
{
  tll *ret;

  int cmp_func(tll *n)
  {
    if(((tuser_nif *)(n->entry))->pid == pid)
      return 1;
    return 0;
  }

  ret = ll_find(&user_nif_ll,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

/***************************************************************************
* caches /proc/net/dev entries for semi fast lookups.
*
* (unsorted- linked list usually small 2-40 entries)
***************************************************************************/
void append_nif(char *name,int rx,int tx)
{
  tnif nif;

  strcpy(nif.name,name);
  nif.rx = rx;
  nif.tx = tx;

  ll_append_unsorted(&nif_ll,&nif,sizeof(tnif));
}

tnif *find_nif(char *name)
{
  tll *ret;

  int cmp_func(tll *n)
  {
    if(!strcmp(((tnif *)(n->entry))->name,name))
      return 1;
    return 0;
  }

  ret = ll_find(&nif_ll,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

/***************************************************************************
*  manage terminal info data (linked list within um linked list)
*
* (unsorted- linked list usually very small. 1-3 entries)
***************************************************************************/
void append_tinfo(tll **um_entry,ttinfo *entry)
{
  ll_append_unsorted(um_entry,entry,sizeof(ttinfo));
}

ttinfo *find_tinfo_pid(tll **um_entry,int pid)
{
  tll *ret;

  int cmp_func(tll *n)
  {
    if(((ttinfo *)(n->entry))->pid == pid)
      return 1;
    return 0;
  }

  ret = ll_find(um_entry,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

ttinfo *find_tinfo_nif(tll **um_entry,char *nif)
{
  tll *ret;

  int cmp_func(tll *n)
  {
    if(!strcmp(((ttinfo *)(n->entry))->nif,nif))
      return 1;
    return 0;
  }

  ret = ll_find(um_entry,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

void remove_tinfo(tll **um_entry,int pid)
{
  int cmp_func(tll *n)
  {
    if(((ttinfo *)(n->entry))->pid == pid)
      return 1;
    return 0;
  }

  ll_remove(um_entry,cmp_func);
}

/***************************************************************************
* manages user settings data as a linked list and as a binary tree for 
* fast searches.
***************************************************************************/
void append_us(tus *entry)
{
  tll *node;

  int cmp_func_ll(tll *n)
  {
    if(strcmp(entry->user,((tus *)(n->entry))->user) < 0)
      return 1;
    return 0;
  }

  int cmp_func_bt(tbt *n)
  {
    return strcmp(entry->user,((tus *)(n->entry))->user);
  }

  node = ll_append_sorted(&us_ll,entry,sizeof(tus),cmp_func_ll);
  if(node != NULL)
    bt_append(&us_bt,node->entry,cmp_func_bt);
}

/*
 * this routine finds the "actual us entry". if the entry it's trying to
 * find doesn't exist, then it returns NULL.
 *
 * this routine is mostly used in net_server.c so that clobberedit can
 * perform maintenance. you wouldn't normally use this routine.
 *
 */
tus *find_us(char *user)
{
  tbt *ret;

  int cmp_func(tbt *n)
  {
    return strcmp(user,((tus *)(n->entry))->user);
  }

  ret = bt_find(&us_bt,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

/*
 * this routine finds the "effective us entry". it will first try to find
 * the entry. if it doesn't exist, then it will return the default us
 * entry (that's if it's enabled). if an entry exists, then it will
 * determine what it is. if it's a normal entry or a group entry, then it
 * will return it. if it's a link, it will locate the linked us entry
 * (which has to be a group entry). if it finds it, and the us entry is
 * flagged as a group, it will return that entry. if it's not, then it
 * will return the default us setting (if it's exists). if it's a union
 * link, then it will find the linked entry, and do a union of the two,
 * and return the results.
 *
 * this is used by reporting and monitor functions.
 */
int find_effective_us(char *user,tus *ret)
{
  tus *e;

  int refer_to_default(void)
  {
    if(us_hr.flags.flag.default_us_enabled)
    {
      memcpy(ret,&(us_hr.default_us),sizeof(tus));
      ret->flags.flag.record_type = TUS_RECORD_TYPE_EFFECTIVE;
      return 1;
    }
    return 0;
  }

  e = find_us(user);
  if(e == NULL)
    return refer_to_default();
  else
  {
    tus *l;

    switch(e->flags.flag.record_type)
    {
      case TUS_RECORD_TYPE_NORMAL:
      case TUS_RECORD_TYPE_GROUP:
      default:
        memcpy(ret,e,sizeof(tus));
        break;

      case TUS_RECORD_TYPE_LINK:
        l = find_us(e->ident.link);
        if(l == NULL)
          return refer_to_default();
        if(l->flags.flag.record_type != TUS_RECORD_TYPE_GROUP)
          return refer_to_default();
        memcpy(ret,l,sizeof(tus));
        strcpy(ret->user,e->user);
        break;

      case TUS_RECORD_TYPE_UNION:
        l = find_us(e->ident.link);
        if(l == NULL)
          return refer_to_default();
        if(l->flags.flag.record_type != TUS_RECORD_TYPE_GROUP)
          return refer_to_default();

        /*
         * the union operation
         */
        ret->flags.all = 0;  /*clear all flags. namely the exemption flags*/
        strcpy(ret->user,e->user);
        strcpy(ret->ident.email,l->ident.email);
        ret->flags.flag.email_type = e->flags.flag.email_type;
        ret->flags.flag.flash_method = e->flags.flag.flash_method;

        if(e->flags.flag.exempt_total_time)
        {
          if(l->flags.flag.exempt_total_time)
            ret->flags.flag.exempt_total_time = 1;
          else
            ret->tt_lim = l->tt_lim;
        }
        else
          ret->tt_lim = e->tt_lim;

        if(e->flags.flag.exempt_daily_time)
        {
          if(l->flags.flag.exempt_daily_time)
            ret->flags.flag.exempt_daily_time = 1;
          else
            memcpy(&(ret->dt_lim),&(l->dt_lim),sizeof(tdt_lim) * DT_LIM_SIZE);
        }
        else
          memcpy(&(ret->dt_lim),&(e->dt_lim),sizeof(tdt_lim) * DT_LIM_SIZE);

        if(e->flags.flag.exempt_expiry_date)
        {
          if(l->flags.flag.exempt_expiry_date)
            ret->flags.flag.exempt_expiry_date = 1;
          else
            memcpy(&(ret->expiry_date),&(l->expiry_date),sizeof(tdate_st));
        }
        else
          memcpy(&(ret->expiry_date),&(e->expiry_date),sizeof(tdate_st));

        if(e->flags.flag.exempt_nice)
        {
          if(l->flags.flag.exempt_nice)
            ret->flags.flag.exempt_nice = 1;
          else
            ret->nice_factor = l->nice_factor;
        }
        else
          ret->nice_factor = e->nice_factor;

        if(e->flags.flag.exempt_total_ul_lim)
        {
          if(l->flags.flag.exempt_total_ul_lim)
            ret->flags.flag.exempt_total_ul_lim = 1;
          else
            ret->t_ul_lim = l->t_ul_lim;
        }
        else
          ret->t_ul_lim = e->t_ul_lim;

        if(e->flags.flag.exempt_total_dl_lim)
        {
          if(l->flags.flag.exempt_total_dl_lim)
            ret->flags.flag.exempt_total_dl_lim = 1;
          else
            ret->t_dl_lim = l->t_dl_lim;
        }
        else
          ret->t_dl_lim = e->t_dl_lim;

        if(e->flags.flag.exempt_daily_ul_lim)
        {
          if(l->flags.flag.exempt_daily_ul_lim)
            ret->flags.flag.exempt_daily_ul_lim = 1;
          else
            ret->d_ul_lim = l->d_ul_lim;
        }
        else
          ret->d_ul_lim = e->d_ul_lim;

        if(e->flags.flag.exempt_daily_dl_lim)
        {
          if(l->flags.flag.exempt_daily_dl_lim)
            ret->flags.flag.exempt_daily_dl_lim = 1;
          else
            ret->d_dl_lim = l->d_dl_lim;
        }
        else
          ret->d_dl_lim = e->d_dl_lim;

        if(e->flags.flag.exempt_act_timeout)
        {
          if(l->flags.flag.exempt_act_timeout)
            ret->flags.flag.exempt_act_timeout = 1;
          else
            ret->act_timeout = l->act_timeout;
        }
        else
          ret->act_timeout = e->act_timeout;

        ret->flags.flag.instant_purge_dt = e->flags.flag.instant_purge_dt;
        ret->flags.flag.clobber_mult_logins = e->flags.flag.clobber_mult_logins;
        ret->flags.flag.banned = e->flags.flag.banned;

        break;
    }

    ret->flags.flag.record_type = TUS_RECORD_TYPE_EFFECTIVE;
    return 1;
  }
  return 0;
}

void remove_us(char *user)
{
  int cmp_func_ll(tll *n)
  {
    if(!strcmp(user,((tus *)(n->entry))->user))
      return 1;
    return 0;
  }

  int cmp_func_bt(tbt *n)
  {
    return strcmp(user,((tus *)(n->entry))->user);
  }

  int cmp_func_bt2(tbt *t,tbt *b)
  {
    return strcmp(((tus *)(b->entry))->user,((tus *)(t->entry))->user);
  }

  bt_remove(&us_bt,cmp_func_bt,cmp_func_bt2);
  ll_remove(&us_ll,cmp_func_ll);
}

/***************************************************************************
* manages user monitor data  (sorted)
***************************************************************************/
void append_um(tum *entry)
{
  int cmp_func(tll *n)
  {
    if(strcmp(entry->user,((tum *)(n->entry))->user) < 0)
      return 1;
    return 0;
  }

  ll_append_sorted(&um_ll,entry,sizeof(tum),cmp_func);
}

tum *find_um(char *user)
{
  tll *ret;

  int cmp_func(tll *n)
  {
    if(!strcmp(((tum *)(n->entry))->user,user))
      return 1;
    return 0;
  }

  ret = ll_find(&um_ll,cmp_func);
  if(ret == NULL)
    return NULL;
  return ret->entry;
}

void remove_um(char *user)
{
  int cmp_func(tll *n)
  {  
    if(!strcmp(((tum *)(n->entry))->user,user))
    {

      /*
       * not only delete the um entry, but the entire tinfo ll hanging
       * off the um entry
       */
      ll_free(&(((tum *)(n->entry))->tinfo_ll));
      return 1;
    }
    return 0;
  }

  ll_remove(&um_ll,cmp_func);
}

/***************************************************************************
* these are all temperary fields that aren't saved to disk they are within
* clobberd daemon only and must be inited
***************************************************************************/
void init_um(tum *um)
{
  register int t;

  um->flags.flag.tt_graced = 0;
  um->flags.flag.dt_graced = 0;
  um->flags.flag.t_ul_graced = 0;
  um->flags.flag.t_dl_graced = 0;
  um->flags.flag.d_ul_graced = 0;
  um->flags.flag.d_dl_graced = 0;

  t = time(NULL);
  um->old_time = t;
  um->dt_zone_limit = -1;
  um->dt_set_limit = 0;
  um->last_active = t;
}

/***************************************************************************
*
***************************************************************************/
int load_data_file(void)
{
  FILE *fp;
  char ins[STR_LEN + 1];
  int mode = 0;
  tus tmp_us;
  tum tmp_um;

  if((fp = fopen(VAR_PATH "/clobberd.db","r")) == NULL)
  {
    log_error("load_data_file()");
    return 0;
  }
  us_hr_clr(&us_hr);
  us_clr(&tmp_us);
  memset(&tmp_um,0,sizeof(tum));
  memset(&gstat,0,sizeof(tgstat));
  while(!feof(fp))
  {
    if(freadln(fp,ins))
    {
      if(ins[0] == ';')
      {
        char cmd[STR_LEN + 1];

        sscanf(ins,"%s",cmd);

        if(!strcmp(cmd,cmd__ok) || !strcmp(cmd,cmd_ret))
        {
          switch(mode)
          {
            case 2:
              append_us(&tmp_us);
              us_clr(&tmp_us);
              break;   
            case 3:
              init_um(&tmp_um);
              append_um(&tmp_um);
              memset(&tmp_um,0,sizeof(tum));
              break;
          }
          mode = 0;
        }
        else
        {
          if(!strcmp(cmd,cmd_gushr))
            mode = 1;
          else if(!strcmp(cmd,cmd_gus))
            mode = 2;
          else if(!strcmp(cmd,cmd_gum))
            mode = 3;
          else if(!strcmp(cmd,cmd_gstat))
            mode = 4;
        }
      }
      switch(mode)
      {
        case 1:
          decomp_net_us_hr(&us_hr,ins);
          break;
        case 2:
          decomp_net_us(&tmp_us,ins);
          break;
        case 3:
          decomp_net_um(&tmp_um,ins);
          break;
        case 4:
          decomp_net_gstat(&gstat,ins);
          break;
      }
    }
  }
  fclose(fp);
  return 1;
}

/***************************************************************************
*
***************************************************************************/
int save_data_file(void)
{
  FILE *fp;
  char str[STR_BIG] = "";

  void save_us_ll(tll *ll)
  {
    for(;;)
    {
      tus *us;

      us = ll_read_entry(&ll);
      if(us == NULL)
        break;

      fprintf(fp,"\n%s\n",cmd_gus);
      str[0] = 0;
      comp_net_us(us,str);
      fputs(str,fp);
      fprintf(fp,"%s\n",cmd_ret);
    }
  }

  void save_um_ll(tll *ll)
  {
    for(;;)
    {
      tum *um;

      um = ll_read_entry(&ll);
      if(um == NULL)
        break;

      fprintf(fp,"\n%s\n",cmd_gum);
      str[0] = 0;
      comp_net_um(um,str);
      fputs(str,fp);
      fprintf(fp,"%s\n",cmd_ret);
    }
  }

  if((fp = fopen(VAR_PATH "/clobberd.db","w")) == NULL)
  {
    log_error("save_data_file()");
    return 0;
  }
  fprintf(fp,
     "# Clobberd " VERSION "\n"
     "# \n"
     "# Do NOT edit this file directly while clobberd is running.\n"
     "# Please maintain database via clobberd TCP DBM interface.\n\n");

  fprintf(fp,"%s\n",cmd_gushr);
  str[0] = 0;
  comp_net_us_hr(&us_hr,str);
  fputs(str,fp);
  fprintf(fp,"%s\n",cmd_ret);

  save_us_ll(us_ll);
  save_um_ll(um_ll);

  fprintf(fp,"\n%s\n",cmd_gstat);
  str[0] = 0;
  comp_net_gstat(&gstat,str);
  fputs(str,fp);
  fprintf(fp,"%s\n",cmd_ret);

  fclose(fp);
  return 1;
}
